/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.security.util.crypto;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.security.Provider;
import java.security.Security;
import java.util.Map;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.nifi.processor.exception.ProcessException;
import org.apache.nifi.processor.io.StreamCallback;
import org.apache.nifi.security.util.EncryptionMethod;
import org.apache.nifi.security.util.KeyDerivationFunction;
import org.apache.nifi.security.util.crypto.PasswordBasedEncryptor;
import org.apache.nifi.security.util.crypto.RandomIVPBECipherProvider;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;

public class PasswordBasedEncryptorTest {
    private static final String TEST_RESOURCES_PATH = "src/test/resources/TestEncryptContent";
    private static final char[] FILE_PASSWORD = "thisIsABadPassword".toCharArray();
    private static final Path TEST_SALTED_PATH = Paths.get(String.format("%s/salted_128_raw.enc", "src/test/resources/TestEncryptContent"), new String[0]);
    private static final Path TEST_UNSALTED_PATH = Paths.get(String.format("%s/unsalted_128_raw.enc", "src/test/resources/TestEncryptContent"), new String[0]);
    private static final Path TEST_PLAIN_PATH = Paths.get(String.format("%s/plain.txt", "src/test/resources/TestEncryptContent"), new String[0]);
    private static final byte[] PLAINTEXT = new byte[]{9, 8, 7, 6, 5, 4, 3, 2, 1, 0};
    private static final char[] PASSWORD = new char[]{'a', 'b', 'c', 'd', 'e', 'f', 'g'};
    private static final int SALT_LENGTH = RandomIVPBECipherProvider.SALT_DELIMITER.length;
    private static final String INITIALIZATION_VECTOR_LENGTH = Integer.toString(16);
    private static final String IV_ATTRIBUTE = "iv";
    private static final EncryptionMethod PBE_ENCRYPTION_METHOD = EncryptionMethod.SHA256_256AES;
    private static final EncryptionMethod KDF_ENCRYPTION_METHOD = EncryptionMethod.AES_GCM;

    @Test
    public void testEncryptDecryptLegacy() throws IOException {
        PasswordBasedEncryptor encryptor = new PasswordBasedEncryptor(PBE_ENCRYPTION_METHOD, PASSWORD, KeyDerivationFunction.NIFI_LEGACY);
        this.assertEncryptDecryptMatched(encryptor);
    }

    @Test
    public void testEncryptDecryptOpenSsl() throws IOException {
        PasswordBasedEncryptor encryptor = new PasswordBasedEncryptor(PBE_ENCRYPTION_METHOD, PASSWORD, KeyDerivationFunction.OPENSSL_EVP_BYTES_TO_KEY);
        this.assertEncryptDecryptMatched(encryptor);
    }

    @Test
    public void testEncryptDecryptBcrypt() throws IOException {
        PasswordBasedEncryptor encryptor = new PasswordBasedEncryptor(KDF_ENCRYPTION_METHOD, PASSWORD, KeyDerivationFunction.BCRYPT);
        this.assertEncryptDecryptMatched(encryptor);
    }

    @Test
    public void testEncryptDecryptScrypt() throws IOException {
        PasswordBasedEncryptor encryptor = new PasswordBasedEncryptor(KDF_ENCRYPTION_METHOD, PASSWORD, KeyDerivationFunction.SCRYPT);
        this.assertEncryptDecryptMatched(encryptor);
    }

    @Test
    public void testEncryptDecryptPbkdf2() throws IOException {
        PasswordBasedEncryptor encryptor = new PasswordBasedEncryptor(KDF_ENCRYPTION_METHOD, PASSWORD, KeyDerivationFunction.PBKDF2);
        this.assertEncryptDecryptMatched(encryptor);
    }

    @Test
    public void testDecryptOpenSslSalted() throws IOException {
        PasswordBasedEncryptor encryptor = new PasswordBasedEncryptor(EncryptionMethod.MD5_128AES, FILE_PASSWORD, KeyDerivationFunction.OPENSSL_EVP_BYTES_TO_KEY);
        byte[] plainBytes = Files.readAllBytes(TEST_PLAIN_PATH);
        byte[] encryptedBytes = Files.readAllBytes(TEST_SALTED_PATH);
        this.assertDecryptMatched(encryptor, encryptedBytes, plainBytes);
    }

    @Test
    public void testDecryptOpenSslUnsalted() throws IOException {
        PasswordBasedEncryptor encryptor = new PasswordBasedEncryptor(EncryptionMethod.MD5_128AES, FILE_PASSWORD, KeyDerivationFunction.OPENSSL_EVP_BYTES_TO_KEY);
        byte[] plainBytes = Files.readAllBytes(TEST_PLAIN_PATH);
        byte[] encryptedBytes = Files.readAllBytes(TEST_UNSALTED_PATH);
        this.assertDecryptMatched(encryptor, encryptedBytes, plainBytes);
    }

    @Test
    public void testEncryptDecryptArgon2SkippedSaltMissing() throws IOException {
        PasswordBasedEncryptor encryptor = new PasswordBasedEncryptor(KDF_ENCRYPTION_METHOD, PASSWORD, KeyDerivationFunction.ARGON2);
        StreamCallback decryption = encryptor.getDecryptionCallback();
        byte[] encryptedBytes = this.encryptBytes(encryptor);
        ByteArrayInputStream encryptedInputStream = new ByteArrayInputStream(encryptedBytes);
        long skipped = encryptedInputStream.skip(SALT_LENGTH);
        Assertions.assertEquals((long)SALT_LENGTH, (long)skipped);
        ByteArrayOutputStream decryptedOutputStream = new ByteArrayOutputStream();
        Assertions.assertThrows(ProcessException.class, () -> decryption.process((InputStream)encryptedInputStream, (OutputStream)decryptedOutputStream));
    }

    @Test
    public void testEncryptDecryptArgon2SaltDelimiterMissing() throws IOException {
        PasswordBasedEncryptor encryptor = new PasswordBasedEncryptor(KDF_ENCRYPTION_METHOD, PASSWORD, KeyDerivationFunction.ARGON2);
        StreamCallback decryption = encryptor.getDecryptionCallback();
        byte[] encryptedBytes = this.encryptBytes(encryptor);
        byte[] delimiterRemoved = ArrayUtils.removeElements((byte[])encryptedBytes, (byte[])RandomIVPBECipherProvider.SALT_DELIMITER);
        ByteArrayInputStream encryptedInputStream = new ByteArrayInputStream(delimiterRemoved);
        ByteArrayOutputStream decryptedOutputStream = new ByteArrayOutputStream();
        Assertions.assertThrows(ProcessException.class, () -> decryption.process((InputStream)encryptedInputStream, (OutputStream)decryptedOutputStream));
    }

    @Test
    public void testEncryptDecryptArgon2InitializationVectorMissing() throws IOException {
        PasswordBasedEncryptor encryptor = new PasswordBasedEncryptor(KDF_ENCRYPTION_METHOD, PASSWORD, KeyDerivationFunction.ARGON2);
        StreamCallback decryption = encryptor.getDecryptionCallback();
        StreamCallback encryption = encryptor.getEncryptionCallback();
        ByteArrayOutputStream encryptedOutputStream = new ByteArrayOutputStream();
        encryption.process((InputStream)new ByteArrayInputStream(PLAINTEXT), (OutputStream)encryptedOutputStream);
        byte[] encryptedBytes = encryptedOutputStream.toByteArray();
        String initializationVectorAttribute = (String)encryptor.flowfileAttributes.get(this.getAttributeName(IV_ATTRIBUTE));
        byte[] initializationVector = initializationVectorAttribute.getBytes(StandardCharsets.UTF_8);
        byte[] encryptedBytesUpdated = ArrayUtils.removeElements((byte[])encryptedBytes, (byte[])initializationVector);
        ByteArrayInputStream encryptedInputStream = new ByteArrayInputStream(encryptedBytesUpdated);
        ByteArrayOutputStream decryptedOutputStream = new ByteArrayOutputStream();
        Assertions.assertThrows(ProcessException.class, () -> decryption.process((InputStream)encryptedInputStream, (OutputStream)decryptedOutputStream));
    }

    @Test
    public void testEncryptDecryptArgon2InitializationVectorDelimiterMissing() throws IOException {
        PasswordBasedEncryptor encryptor = new PasswordBasedEncryptor(KDF_ENCRYPTION_METHOD, PASSWORD, KeyDerivationFunction.ARGON2);
        StreamCallback decryption = encryptor.getDecryptionCallback();
        byte[] encryptedBytes = this.encryptBytes(encryptor);
        byte[] encryptedBytesUpdated = ArrayUtils.removeElements((byte[])encryptedBytes, (byte[])RandomIVPBECipherProvider.IV_DELIMITER);
        ByteArrayInputStream encryptedInputStream = new ByteArrayInputStream(encryptedBytesUpdated);
        ByteArrayOutputStream decryptedOutputStream = new ByteArrayOutputStream();
        Assertions.assertThrows(ProcessException.class, () -> decryption.process((InputStream)encryptedInputStream, (OutputStream)decryptedOutputStream));
    }

    private byte[] encryptBytes(PasswordBasedEncryptor encryptor) throws IOException {
        StreamCallback encryption = encryptor.getEncryptionCallback();
        ByteArrayOutputStream encryptedOutputStream = new ByteArrayOutputStream();
        encryption.process((InputStream)new ByteArrayInputStream(PLAINTEXT), (OutputStream)encryptedOutputStream);
        return encryptedOutputStream.toByteArray();
    }

    private void assertEncryptDecryptMatched(PasswordBasedEncryptor encryptor) throws IOException {
        StreamCallback encryption = encryptor.getEncryptionCallback();
        StreamCallback decryption = encryptor.getDecryptionCallback();
        ByteArrayOutputStream encryptedOutputStream = new ByteArrayOutputStream();
        encryption.process((InputStream)new ByteArrayInputStream(PLAINTEXT), (OutputStream)encryptedOutputStream);
        ByteArrayInputStream encryptedInputStream = new ByteArrayInputStream(encryptedOutputStream.toByteArray());
        ByteArrayOutputStream decryptedOutputStream = new ByteArrayOutputStream();
        decryption.process((InputStream)encryptedInputStream, (OutputStream)decryptedOutputStream);
        byte[] decrypted = decryptedOutputStream.toByteArray();
        Assertions.assertArrayEquals((byte[])PLAINTEXT, (byte[])decrypted);
        this.assertFlowFileAttributesFound(encryptor.flowfileAttributes);
    }

    private void assertFlowFileAttributesFound(Map<String, String> attributes) {
        Assertions.assertTrue((boolean)attributes.containsKey(this.getAttributeName("algorithm")));
        Assertions.assertTrue((boolean)attributes.containsKey(this.getAttributeName("timestamp")));
        Assertions.assertTrue((boolean)attributes.containsKey(this.getAttributeName("cipher_text_length")));
        Assertions.assertEquals((Object)"decrypted", (Object)attributes.get(this.getAttributeName("action")));
        Assertions.assertEquals((Object)Integer.toString(PLAINTEXT.length), (Object)attributes.get(this.getAttributeName("plaintext_length")));
        Assertions.assertTrue((boolean)attributes.containsKey(this.getAttributeName("salt")));
        Assertions.assertTrue((boolean)attributes.containsKey(this.getAttributeName("salt_length")));
        Assertions.assertTrue((boolean)attributes.containsKey(this.getAttributeName(IV_ATTRIBUTE)));
        Assertions.assertEquals((Object)INITIALIZATION_VECTOR_LENGTH, (Object)attributes.get(this.getAttributeName("iv_length")));
        Assertions.assertTrue((boolean)attributes.containsKey(this.getAttributeName("kdf")));
    }

    private void assertDecryptMatched(PasswordBasedEncryptor encryptor, byte[] encrypted, byte[] expected) throws IOException {
        StreamCallback decryption = encryptor.getDecryptionCallback();
        ByteArrayOutputStream decryptedOutputStream = new ByteArrayOutputStream();
        decryption.process((InputStream)new ByteArrayInputStream(encrypted), (OutputStream)decryptedOutputStream);
        byte[] decrypted = decryptedOutputStream.toByteArray();
        Assertions.assertArrayEquals((byte[])expected, (byte[])decrypted);
    }

    private String getAttributeName(String name) {
        return String.format("encryptcontent.%s", name);
    }

    static {
        Security.addProvider((Provider)new BouncyCastleProvider());
    }
}

