/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.processors.standard.util;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.UUID;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.nifi.components.PropertyDescriptor;
import org.apache.nifi.context.PropertyContext;
import org.apache.nifi.logging.ComponentLog;
import org.apache.nifi.processors.standard.util.FileInfo;
import org.apache.nifi.processors.standard.util.SFTPTransfer;
import org.apache.nifi.util.MockPropertyContext;
import org.apache.sshd.common.file.FileSystemFactory;
import org.apache.sshd.common.file.virtualfs.VirtualFileSystemFactory;
import org.apache.sshd.common.keyprovider.KeyPairProvider;
import org.apache.sshd.server.SshServer;
import org.apache.sshd.server.keyprovider.SimpleGeneratorHostKeyProvider;
import org.apache.sshd.sftp.server.SftpSubsystemFactory;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.io.TempDir;
import org.mockito.Mockito;

public class TestServerSFTPTransfer {
    private static final String LOCALHOST = "127.0.0.1";
    private static final String USERNAME = "user";
    private static final String PASSWORD = UUID.randomUUID().toString();
    private static final String DIR_1 = "dir1";
    private static final String DIR_2 = "dir2";
    private static final String LINKED_DIRECTORY = "linked-directory";
    private static final String LINKED_FILE = "linked-file";
    private static final String EMPTY_DIRECTORY = "dir4";
    private static final String DIR_1_CHILD_1 = "child1";
    private static final String DIR_1_CHILD_2 = "child2";
    private static final String FILE_1 = "file1.txt";
    private static final String FILE_2 = "file2.txt";
    private static final String DOT_FILE = ".foo.txt";
    private static final boolean FILTERING_ENABLED = true;
    @TempDir
    File serverDirectory;
    private SshServer sshServer;

    @BeforeEach
    public void setupFiles() throws IOException {
        this.writeFile(DIR_1, DIR_1_CHILD_1, FILE_1);
        this.writeFile(DIR_1, DIR_1_CHILD_1, FILE_2);
        this.writeFile(DIR_1, DIR_1_CHILD_1, DOT_FILE);
        this.writeFile(DIR_1, DIR_1_CHILD_2, FILE_1);
        this.writeFile(DIR_1, DIR_1_CHILD_2, FILE_2);
        this.writeFile(DIR_1, DIR_1_CHILD_2, DOT_FILE);
        this.writeFile(DIR_2, FILE_1);
        this.writeFile(DIR_2, FILE_2);
        this.writeFile(DIR_2, DOT_FILE);
        File linkedDirectory = new File(this.serverDirectory, LINKED_DIRECTORY);
        File linkedDirectoryTarget = new File(this.serverDirectory.getAbsolutePath(), DIR_1);
        Files.createSymbolicLink(linkedDirectory.toPath(), linkedDirectoryTarget.toPath(), new FileAttribute[0]);
        File secondDirectory = new File(this.serverDirectory, DIR_2);
        File linkedFile = new File(this.serverDirectory, LINKED_FILE);
        File linkedFileTarget = new File(secondDirectory, FILE_1);
        Files.createSymbolicLink(linkedFile.toPath(), linkedFileTarget.toPath(), new FileAttribute[0]);
        File emptyDirectory = new File(this.serverDirectory, EMPTY_DIRECTORY);
        Assertions.assertTrue((boolean)emptyDirectory.mkdirs());
        this.startServer();
    }

    @AfterEach
    public void stopServer() throws IOException {
        this.sshServer.stop(true);
    }

    @Test
    public void testGetListingSimple() throws IOException {
        Map<PropertyDescriptor, String> properties = this.createBaseProperties();
        properties.put(SFTPTransfer.REMOTE_PATH, DIR_2);
        try (SFTPTransfer transfer = this.createSFTPTransfer(properties);){
            List listing = transfer.getListing(true);
            Assertions.assertNotNull((Object)listing);
            Assertions.assertEquals((int)2, (int)listing.size());
            FileInfo file1Info = listing.stream().filter(f -> f.getFileName().equals(FILE_1)).findFirst().orElse(null);
            Assertions.assertNotNull((Object)file1Info);
            Assertions.assertFalse((boolean)file1Info.isDirectory());
            FileInfo file2Info = listing.stream().filter(f -> f.getFileName().equals(FILE_2)).findFirst().orElse(null);
            Assertions.assertNotNull((Object)file2Info);
            Assertions.assertFalse((boolean)file2Info.isDirectory());
        }
    }

    @Test
    public void testGetListingSimpleWithDotFiles() throws IOException {
        Map<PropertyDescriptor, String> properties = this.createBaseProperties();
        properties.put(SFTPTransfer.REMOTE_PATH, DIR_2);
        properties.put(SFTPTransfer.IGNORE_DOTTED_FILES, "false");
        try (SFTPTransfer transfer = this.createSFTPTransfer(properties);){
            List listing = transfer.getListing(true);
            Assertions.assertNotNull((Object)listing);
            Assertions.assertEquals((int)3, (int)listing.size());
            FileInfo dotFileInfo = listing.stream().filter(f -> f.getFileName().equals(DOT_FILE)).findFirst().orElse(null);
            Assertions.assertNotNull((Object)dotFileInfo);
        }
    }

    @Test
    public void testGetListingWithoutRecursiveSearch() throws IOException {
        Map<PropertyDescriptor, String> properties = this.createBaseProperties();
        properties.put(SFTPTransfer.REMOTE_PATH, DIR_1);
        properties.put(SFTPTransfer.RECURSIVE_SEARCH, "false");
        try (SFTPTransfer transfer = this.createSFTPTransfer(properties);){
            List listing = transfer.getListing(true);
            Assertions.assertNotNull((Object)listing);
            Assertions.assertEquals((int)0, (int)listing.size());
        }
    }

    @Test
    public void testGetListingWithRecursiveSearch() throws IOException {
        Map<PropertyDescriptor, String> properties = this.createBaseProperties();
        properties.put(SFTPTransfer.REMOTE_PATH, DIR_1);
        properties.put(SFTPTransfer.RECURSIVE_SEARCH, "true");
        try (SFTPTransfer transfer = this.createSFTPTransfer(properties);){
            List listing = transfer.getListing(true);
            Assertions.assertNotNull((Object)listing);
            Assertions.assertEquals((int)4, (int)listing.size());
        }
    }

    @Test
    public void testGetListingWithoutSymlinks() throws IOException {
        Map<PropertyDescriptor, String> properties = this.createBaseProperties();
        properties.put(SFTPTransfer.RECURSIVE_SEARCH, "true");
        properties.put(SFTPTransfer.FOLLOW_SYMLINK, "false");
        try (SFTPTransfer transfer = this.createSFTPTransfer(properties);){
            List listing = transfer.getListing(true);
            Assertions.assertNotNull((Object)listing);
            Assertions.assertEquals((int)6, (int)listing.size());
        }
    }

    @Test
    public void testGetListingWithSymlinks() throws IOException {
        Map<PropertyDescriptor, String> properties = this.createBaseProperties();
        properties.put(SFTPTransfer.RECURSIVE_SEARCH, "true");
        properties.put(SFTPTransfer.FOLLOW_SYMLINK, "true");
        try (SFTPTransfer transfer = this.createSFTPTransfer(properties);){
            List listing = transfer.getListing(true);
            Assertions.assertNotNull((Object)listing);
            Assertions.assertEquals((int)11, (int)listing.size());
        }
    }

    @Test
    public void testGetListingWithBatchSize() throws IOException {
        List listing;
        Map<PropertyDescriptor, String> properties = this.createBaseProperties();
        properties.put(SFTPTransfer.REMOTE_PATH, DIR_1);
        properties.put(SFTPTransfer.RECURSIVE_SEARCH, "true");
        try (SFTPTransfer transfer = this.createSFTPTransfer(properties);){
            listing = transfer.getListing(true);
            Assertions.assertNotNull((Object)listing);
            Assertions.assertEquals((int)4, (int)listing.size());
        }
        properties.put(SFTPTransfer.REMOTE_POLL_BATCH_SIZE, "2");
        transfer = this.createSFTPTransfer(properties);
        try {
            listing = transfer.getListing(true);
            Assertions.assertNotNull((Object)listing);
            Assertions.assertEquals((int)2, (int)listing.size());
        }
        finally {
            if (transfer != null) {
                transfer.close();
            }
        }
    }

    @Test
    public void testGetListingWithFileFilter() throws IOException {
        String fileFilterRegex = "file1.*";
        Map<PropertyDescriptor, String> properties = this.createBaseProperties();
        properties.put(SFTPTransfer.REMOTE_PATH, DIR_1);
        properties.put(SFTPTransfer.RECURSIVE_SEARCH, "true");
        properties.put(SFTPTransfer.FILE_FILTER_REGEX, "file1.*");
        try (SFTPTransfer transfer = this.createSFTPTransfer(properties);){
            List listing = transfer.getListing(true);
            Assertions.assertNotNull((Object)listing);
            Assertions.assertEquals((int)2, (int)listing.size());
            listing.forEach(f -> Assertions.assertTrue((boolean)f.getFileName().matches("file1.*")));
        }
    }

    @Test
    public void testGetListingWithPathFilter() throws IOException {
        String remotePath = ".";
        String pathFilterRegex = "dir1/child1";
        Map<PropertyDescriptor, String> properties = this.createBaseProperties();
        properties.put(SFTPTransfer.REMOTE_PATH, ".");
        properties.put(SFTPTransfer.RECURSIVE_SEARCH, "true");
        properties.put(SFTPTransfer.PATH_FILTER_REGEX, "dir1/child1");
        try (SFTPTransfer transfer = this.createSFTPTransfer(properties);){
            List listing = transfer.getListing(true);
            Assertions.assertNotNull((Object)listing);
            Assertions.assertEquals((int)2, (int)listing.size());
        }
    }

    @Test
    public void testGetListingWhenRemotePathDoesNotExist() throws IOException {
        Map<PropertyDescriptor, String> properties = this.createBaseProperties();
        properties.put(SFTPTransfer.REMOTE_PATH, "DOES-NOT-EXIST");
        properties.put(SFTPTransfer.RECURSIVE_SEARCH, "true");
        try (SFTPTransfer transfer = this.createSFTPTransfer(properties);){
            Assertions.assertThrows(FileNotFoundException.class, () -> transfer.getListing(true));
        }
    }

    @Test
    public void testDeleteFileWithoutPath() throws IOException {
        Map<PropertyDescriptor, String> properties = this.createBaseProperties();
        properties.put(SFTPTransfer.REMOTE_PATH, DIR_2);
        try (SFTPTransfer transfer = this.createSFTPTransfer(properties);){
            List listing = transfer.getListing(true);
            Assertions.assertNotNull((Object)listing);
            Assertions.assertEquals((int)2, (int)listing.size());
            for (FileInfo fileInfo : listing) {
                transfer.deleteFile(null, null, fileInfo.getFullPathFileName());
            }
            List listingAfterDelete = transfer.getListing(true);
            Assertions.assertNotNull((Object)listingAfterDelete);
            Assertions.assertEquals((int)0, (int)listingAfterDelete.size());
        }
    }

    @Test
    public void testDeleteFileWithPath() throws IOException {
        Map<PropertyDescriptor, String> properties = this.createBaseProperties();
        properties.put(SFTPTransfer.REMOTE_PATH, DIR_2);
        try (SFTPTransfer transfer = this.createSFTPTransfer(properties);){
            List listing = transfer.getListing(true);
            Assertions.assertNotNull((Object)listing);
            Assertions.assertEquals((int)2, (int)listing.size());
            for (FileInfo fileInfo : listing) {
                String filename = fileInfo.getFileName();
                String path = fileInfo.getFullPathFileName().replace(filename, "");
                transfer.deleteFile(null, path, filename);
            }
            List listingAfterDelete = transfer.getListing(true);
            Assertions.assertNotNull((Object)listingAfterDelete);
            Assertions.assertEquals((int)0, (int)listingAfterDelete.size());
        }
    }

    @Test
    public void testDeleteFileWhenDoesNotExist() throws IOException {
        Map<PropertyDescriptor, String> properties = this.createBaseProperties();
        try (SFTPTransfer transfer = this.createSFTPTransfer(properties);){
            Assertions.assertThrows(FileNotFoundException.class, () -> transfer.deleteFile(null, null, "foo/bar/does-not-exist.txt"));
        }
    }

    @Test
    public void testDeleteDirectory() throws IOException {
        Map<PropertyDescriptor, String> properties = this.createBaseProperties();
        properties.put(SFTPTransfer.REMOTE_PATH, EMPTY_DIRECTORY);
        try (SFTPTransfer transfer = this.createSFTPTransfer(properties);){
            List listing = transfer.getListing(true);
            Assertions.assertNotNull((Object)listing);
            Assertions.assertEquals((int)0, (int)listing.size());
            transfer.deleteDirectory(null, EMPTY_DIRECTORY);
            Assertions.assertThrows(FileNotFoundException.class, () -> transfer.getListing(true));
        }
    }

    @Test
    public void testDeleteDirectoryWhenDoesNotExist() throws IOException {
        Map<PropertyDescriptor, String> properties = this.createBaseProperties();
        try (SFTPTransfer transfer = this.createSFTPTransfer(properties);){
            Assertions.assertThrows(IOException.class, () -> transfer.deleteDirectory(null, "DOES-NOT-EXIST"));
        }
    }

    @Test
    public void testEnsureDirectoryExistsSimple() throws IOException {
        String remotePath = "DOES-NOT-EXIST";
        Map<PropertyDescriptor, String> properties = this.createBaseProperties();
        properties.put(SFTPTransfer.REMOTE_PATH, "DOES-NOT-EXIST");
        try (SFTPTransfer transfer = this.createSFTPTransfer(properties);){
            Assertions.assertThrows(FileNotFoundException.class, () -> transfer.getListing(true));
            String absolutePath = transfer.getAbsolutePath(null, "DOES-NOT-EXIST");
            transfer.ensureDirectoryExists(null, new File(absolutePath));
            List listing = transfer.getListing(true);
            Assertions.assertNotNull((Object)listing);
            Assertions.assertEquals((int)0, (int)listing.size());
        }
    }

    @Test
    public void testEnsureDirectoryExistsMultipleLevels() throws IOException {
        String remotePath = "A/B/C";
        Map<PropertyDescriptor, String> properties = this.createBaseProperties();
        properties.put(SFTPTransfer.REMOTE_PATH, "A/B/C");
        try (SFTPTransfer transfer = this.createSFTPTransfer(properties);){
            Assertions.assertThrows(FileNotFoundException.class, () -> transfer.getListing(true));
            String absolutePath = transfer.getAbsolutePath(null, "A/B/C");
            transfer.ensureDirectoryExists(null, new File(absolutePath));
            List listing = transfer.getListing(true);
            Assertions.assertNotNull((Object)listing);
            Assertions.assertEquals((int)0, (int)listing.size());
        }
    }

    @Test
    public void testEnsureDirectoryExistsWhenAlreadyExists() throws IOException {
        Map<PropertyDescriptor, String> properties = this.createBaseProperties();
        properties.put(SFTPTransfer.REMOTE_PATH, DIR_2);
        try (SFTPTransfer transfer = this.createSFTPTransfer(properties);){
            List listing = transfer.getListing(true);
            Assertions.assertNotNull((Object)listing);
            Assertions.assertEquals((int)2, (int)listing.size());
            String absolutePath = transfer.getAbsolutePath(null, DIR_2);
            transfer.ensureDirectoryExists(null, new File(absolutePath));
        }
    }

    @Test
    public void testEnsureDirectoryExistsWithDirectoryListingDisabled() throws IOException {
        String remotePath = "DOES-NOT-EXIST";
        Map<PropertyDescriptor, String> properties = this.createBaseProperties();
        properties.put(SFTPTransfer.REMOTE_PATH, "DOES-NOT-EXIST");
        properties.put(SFTPTransfer.DISABLE_DIRECTORY_LISTING, "true");
        try (SFTPTransfer transfer = this.createSFTPTransfer(properties);){
            Assertions.assertThrows(FileNotFoundException.class, () -> transfer.getListing(true));
            String absolutePath = transfer.getAbsolutePath(null, "DOES-NOT-EXIST");
            transfer.ensureDirectoryExists(null, new File(absolutePath));
            List listing = transfer.getListing(true);
            Assertions.assertNotNull((Object)listing);
            Assertions.assertEquals((int)0, (int)listing.size());
        }
    }

    @Test
    public void testEnsureDirectoryExistsWithDirectoryListingDisabledAndAlreadyExists() throws IOException {
        Map<PropertyDescriptor, String> properties = this.createBaseProperties();
        properties.put(SFTPTransfer.REMOTE_PATH, DIR_2);
        properties.put(SFTPTransfer.DISABLE_DIRECTORY_LISTING, "true");
        try (SFTPTransfer transfer = this.createSFTPTransfer(properties);){
            List listing = transfer.getListing(true);
            Assertions.assertNotNull((Object)listing);
            Assertions.assertEquals((int)2, (int)listing.size());
            String absolutePath = transfer.getAbsolutePath(null, DIR_2);
            Assertions.assertThrows(IOException.class, () -> transfer.ensureDirectoryExists(null, new File(absolutePath)));
        }
    }

    @Test
    public void testEnsureDirectoryExistsWithDirectoryListingDisabledAndParentDoesNotExist() throws IOException {
        String remotePath = "A/B/C";
        Map<PropertyDescriptor, String> properties = this.createBaseProperties();
        properties.put(SFTPTransfer.REMOTE_PATH, "A/B/C");
        properties.put(SFTPTransfer.DISABLE_DIRECTORY_LISTING, "true");
        try (SFTPTransfer transfer = this.createSFTPTransfer(properties);){
            Assertions.assertThrows(FileNotFoundException.class, () -> transfer.getListing(true));
            String absolutePath = transfer.getAbsolutePath(null, "A/B/C");
            transfer.ensureDirectoryExists(null, new File(absolutePath));
        }
    }

    @Test
    public void testGetRemoteFileInfo() throws IOException {
        Map<PropertyDescriptor, String> properties = this.createBaseProperties();
        try (SFTPTransfer transfer = this.createSFTPTransfer(properties);){
            FileInfo fileInfo = transfer.getRemoteFileInfo(null, DIR_2, FILE_1);
            Assertions.assertNotNull((Object)fileInfo);
            Assertions.assertEquals((Object)FILE_1, (Object)fileInfo.getFileName());
        }
    }

    @Test
    public void testGetRemoteFileInfoWhenPathDoesNotExist() throws IOException {
        Map<PropertyDescriptor, String> properties = this.createBaseProperties();
        try (SFTPTransfer transfer = this.createSFTPTransfer(properties);){
            FileInfo fileInfo = transfer.getRemoteFileInfo(null, "DOES-NOT-EXIST", FILE_1);
            Assertions.assertNull((Object)fileInfo);
        }
    }

    @Test
    public void testGetRemoteFileInfoWhenFileDoesNotExist() throws IOException {
        Map<PropertyDescriptor, String> properties = this.createBaseProperties();
        try (SFTPTransfer transfer = this.createSFTPTransfer(properties);){
            FileInfo fileInfo = transfer.getRemoteFileInfo(null, DIR_2, "DOES-NOT-EXIST");
            Assertions.assertNull((Object)fileInfo);
        }
    }

    @Test
    public void testGetRemoteFileInfoWhenFileIsADirectory() throws IOException {
        Map<PropertyDescriptor, String> properties = this.createBaseProperties();
        try (SFTPTransfer transfer = this.createSFTPTransfer(properties);){
            FileInfo fileInfo = transfer.getRemoteFileInfo(null, DIR_1, DIR_1_CHILD_1);
            Assertions.assertNull((Object)fileInfo);
        }
    }

    @Test
    public void testRename() throws IOException {
        Map<PropertyDescriptor, String> properties = this.createBaseProperties();
        try (SFTPTransfer transfer = this.createSFTPTransfer(properties);){
            String source = "dir2/file1.txt";
            String target = "dir2/file1.txt-RENAMED";
            FileInfo targetInfoBefore = transfer.getRemoteFileInfo(null, DIR_2, "file1.txt-RENAMED");
            Assertions.assertNull((Object)targetInfoBefore);
            transfer.rename(null, "dir2/file1.txt", "dir2/file1.txt-RENAMED");
            FileInfo targetInfoAfter = transfer.getRemoteFileInfo(null, DIR_2, "file1.txt-RENAMED");
            Assertions.assertNotNull((Object)targetInfoAfter);
        }
    }

    @Test
    public void testRenameWhenSourceDoesNotExist() throws IOException {
        Map<PropertyDescriptor, String> properties = this.createBaseProperties();
        try (SFTPTransfer transfer = this.createSFTPTransfer(properties);){
            String source = "dir2/DOES-NOT-EXIST";
            String target = "dir2/file1.txt-RENAMED";
            Assertions.assertThrows(FileNotFoundException.class, () -> transfer.rename(null, "dir2/DOES-NOT-EXIST", "dir2/file1.txt-RENAMED"));
        }
    }

    @Test
    public void testRenameWhenTargetAlreadyExists() throws IOException {
        Map<PropertyDescriptor, String> properties = this.createBaseProperties();
        try (SFTPTransfer transfer = this.createSFTPTransfer(properties);){
            String source = "dir2/file1.txt";
            String target = "dir2/file2.txt";
            FileInfo targetInfoBefore = transfer.getRemoteFileInfo(null, DIR_2, FILE_2);
            Assertions.assertNotNull((Object)targetInfoBefore);
            Assertions.assertThrows(IOException.class, () -> transfer.rename(null, "dir2/file1.txt", "dir2/file2.txt"));
        }
    }

    @Test
    public void testPutWithPermissions() throws IOException {
        String permissions = "rw-rw-rw-";
        Map<PropertyDescriptor, String> properties = this.createBaseProperties();
        properties.put(SFTPTransfer.PERMISSIONS, "rw-rw-rw-");
        String filename = "test-put-simple.txt";
        String fileContent = "this is a test";
        try (SFTPTransfer transfer = this.createSFTPTransfer(properties);
             ByteArrayInputStream in = new ByteArrayInputStream("this is a test".getBytes(StandardCharsets.UTF_8));){
            FileInfo fileInfoBefore = transfer.getRemoteFileInfo(null, EMPTY_DIRECTORY, "test-put-simple.txt");
            Assertions.assertNull((Object)fileInfoBefore);
            String fullPath = transfer.put(null, EMPTY_DIRECTORY, "test-put-simple.txt", (InputStream)in);
            Assertions.assertNotNull((Object)fullPath);
            FileInfo fileInfoAfter = transfer.getRemoteFileInfo(null, EMPTY_DIRECTORY, "test-put-simple.txt");
            Assertions.assertNotNull((Object)fileInfoAfter);
            Assertions.assertEquals((Object)"rw-rw-rw-", (Object)fileInfoAfter.getPermissions());
            File writtenFile = new File(this.serverDirectory, "dir4/test-put-simple.txt");
            String retrievedContent = IOUtils.toString((URI)writtenFile.toURI(), (Charset)StandardCharsets.UTF_8);
            Assertions.assertEquals((Object)"this is a test", (Object)retrievedContent);
        }
    }

    @Test
    public void testPutWithTempFilename() throws IOException {
        String permissions = "rw-rw-rw-";
        Map<PropertyDescriptor, String> properties = this.createBaseProperties();
        properties.put(SFTPTransfer.PERMISSIONS, "rw-rw-rw-");
        properties.put(SFTPTransfer.TEMP_FILENAME, "temp-file.txt");
        String filename = "test-put-simple.txt";
        String fileContent = "this is a test";
        try (SFTPTransfer transfer = this.createSFTPTransfer(properties);
             ByteArrayInputStream in = new ByteArrayInputStream("this is a test".getBytes(StandardCharsets.UTF_8));){
            FileInfo fileInfoBefore = transfer.getRemoteFileInfo(null, EMPTY_DIRECTORY, "test-put-simple.txt");
            Assertions.assertNull((Object)fileInfoBefore);
            String fullPath = transfer.put(null, EMPTY_DIRECTORY, "test-put-simple.txt", (InputStream)in);
            Assertions.assertNotNull((Object)fullPath);
            FileInfo fileInfoAfter = transfer.getRemoteFileInfo(null, EMPTY_DIRECTORY, "test-put-simple.txt");
            Assertions.assertNotNull((Object)fileInfoAfter);
            Assertions.assertEquals((Object)"rw-rw-rw-", (Object)fileInfoAfter.getPermissions());
        }
    }

    @Test
    public void testPutWithLastModifiedTime() throws IOException, ParseException {
        String permissions = "rw-rw-rw-";
        String lastModifiedTime = "2019-09-01T11:11:11-0500";
        SimpleDateFormat formatter = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ssZ", Locale.US);
        long expectedLastModifiedTime = formatter.parse("2019-09-01T11:11:11-0500").getTime();
        Map<PropertyDescriptor, String> properties = this.createBaseProperties();
        properties.put(SFTPTransfer.PERMISSIONS, "rw-rw-rw-");
        properties.put(SFTPTransfer.LAST_MODIFIED_TIME, "2019-09-01T11:11:11-0500");
        String filename = "test-put-simple.txt";
        String fileContent = "this is a test";
        try (SFTPTransfer transfer = this.createSFTPTransfer(properties);
             ByteArrayInputStream in = new ByteArrayInputStream("this is a test".getBytes(StandardCharsets.UTF_8));){
            FileInfo fileInfoBefore = transfer.getRemoteFileInfo(null, EMPTY_DIRECTORY, "test-put-simple.txt");
            Assertions.assertNull((Object)fileInfoBefore);
            String fullPath = transfer.put(null, EMPTY_DIRECTORY, "test-put-simple.txt", (InputStream)in);
            Assertions.assertNotNull((Object)fullPath);
            FileInfo fileInfoAfter = transfer.getRemoteFileInfo(null, EMPTY_DIRECTORY, "test-put-simple.txt");
            Assertions.assertNotNull((Object)fileInfoAfter);
            Assertions.assertEquals((Object)"rw-rw-rw-", (Object)fileInfoAfter.getPermissions());
            Assertions.assertEquals((long)expectedLastModifiedTime, (long)fileInfoAfter.getLastModifiedTime());
        }
    }

    @Test
    public void testPutWhenFileAlreadyExists() throws IOException {
        String permissions = "rw-rw-rw-";
        Map<PropertyDescriptor, String> properties = this.createBaseProperties();
        properties.put(SFTPTransfer.PERMISSIONS, "rw-rw-rw-");
        String fileContent = "this is a test";
        try (SFTPTransfer transfer = this.createSFTPTransfer(properties);
             ByteArrayInputStream in = new ByteArrayInputStream("this is a test".getBytes(StandardCharsets.UTF_8));){
            FileInfo fileInfoBefore = transfer.getRemoteFileInfo(null, DIR_2, FILE_1);
            Assertions.assertNotNull((Object)fileInfoBefore);
            Assertions.assertThrows(IOException.class, () -> transfer.put(null, DIR_2, FILE_1, in));
        }
    }

    @Test
    public void testPutWhenDirectoryDoesNotExist() throws IOException {
        String permissions = "rw-rw-rw-";
        Map<PropertyDescriptor, String> properties = this.createBaseProperties();
        properties.put(SFTPTransfer.PERMISSIONS, "rw-rw-rw-");
        String fileContent = "this is a test";
        try (SFTPTransfer transfer = this.createSFTPTransfer(properties);
             ByteArrayInputStream in = new ByteArrayInputStream("this is a test".getBytes(StandardCharsets.UTF_8));){
            Assertions.assertThrows(IOException.class, () -> transfer.put(null, "DOES-NOT-EXIST", FILE_1, in));
        }
    }

    private Map<PropertyDescriptor, String> createBaseProperties() {
        HashMap<PropertyDescriptor, String> properties = new HashMap<PropertyDescriptor, String>();
        properties.put(SFTPTransfer.HOSTNAME, LOCALHOST);
        properties.put(SFTPTransfer.PORT, Integer.toString(this.sshServer.getPort()));
        properties.put(SFTPTransfer.USERNAME, USERNAME);
        properties.put(SFTPTransfer.PASSWORD, PASSWORD);
        properties.put(SFTPTransfer.STRICT_HOST_KEY_CHECKING, Boolean.FALSE.toString());
        return properties;
    }

    private SFTPTransfer createSFTPTransfer(Map<PropertyDescriptor, String> properties) {
        MockPropertyContext propertyContext = new MockPropertyContext(properties);
        ComponentLog logger = (ComponentLog)Mockito.mock(ComponentLog.class);
        return new SFTPTransfer((PropertyContext)propertyContext, logger);
    }

    private void startServer() throws IOException {
        this.sshServer = SshServer.setUpDefaultServer();
        this.sshServer.setHost(LOCALHOST);
        this.sshServer.setPasswordAuthenticator((username, password, serverSession) -> USERNAME.equals(username) && PASSWORD.equals(password));
        this.sshServer.setKeyPairProvider((KeyPairProvider)new SimpleGeneratorHostKeyProvider());
        this.sshServer.setFileSystemFactory((FileSystemFactory)new VirtualFileSystemFactory(this.serverDirectory.toPath()));
        this.sshServer.setSubsystemFactories(Collections.singletonList(new SftpSubsystemFactory()));
        this.sshServer.start();
    }

    private void writeFile(String ... pathElements) throws IOException {
        Path path = Paths.get(this.serverDirectory.getAbsolutePath(), pathElements);
        File parentFile = path.toFile().getParentFile();
        FileUtils.forceMkdir((File)parentFile);
        byte[] contents = path.toFile().getAbsolutePath().getBytes(StandardCharsets.UTF_8);
        Files.write(path, contents, new OpenOption[0]);
    }
}

