/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.processors.jslt;

import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.core.JsonParseException;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonToken;
import com.fasterxml.jackson.core.ObjectCodec;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectWriter;
import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.schibsted.spt.data.jslt.Expression;
import com.schibsted.spt.data.jslt.Parser;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.StringReader;
import java.io.UncheckedIOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import org.apache.nifi.annotation.behavior.InputRequirement;
import org.apache.nifi.annotation.behavior.SideEffectFree;
import org.apache.nifi.annotation.behavior.SupportsBatching;
import org.apache.nifi.annotation.behavior.SystemResource;
import org.apache.nifi.annotation.behavior.SystemResourceConsideration;
import org.apache.nifi.annotation.behavior.WritesAttribute;
import org.apache.nifi.annotation.documentation.CapabilityDescription;
import org.apache.nifi.annotation.documentation.Tags;
import org.apache.nifi.annotation.lifecycle.OnScheduled;
import org.apache.nifi.annotation.lifecycle.OnShutdown;
import org.apache.nifi.annotation.lifecycle.OnStopped;
import org.apache.nifi.components.DescribedValue;
import org.apache.nifi.components.PropertyDescriptor;
import org.apache.nifi.components.PropertyValue;
import org.apache.nifi.components.ValidationContext;
import org.apache.nifi.components.ValidationResult;
import org.apache.nifi.components.resource.ResourceCardinality;
import org.apache.nifi.components.resource.ResourceReference;
import org.apache.nifi.components.resource.ResourceType;
import org.apache.nifi.expression.ExpressionLanguageScope;
import org.apache.nifi.flowfile.FlowFile;
import org.apache.nifi.flowfile.attributes.CoreAttributes;
import org.apache.nifi.processor.AbstractProcessor;
import org.apache.nifi.processor.ProcessContext;
import org.apache.nifi.processor.ProcessSession;
import org.apache.nifi.processor.Relationship;
import org.apache.nifi.processor.exception.ProcessException;
import org.apache.nifi.processor.util.StandardValidators;
import org.apache.nifi.util.StopWatch;

@SideEffectFree
@SupportsBatching
@Tags(value={"json", "jslt", "transform"})
@InputRequirement(value=InputRequirement.Requirement.INPUT_REQUIRED)
@SystemResourceConsideration(resource=SystemResource.MEMORY)
@WritesAttribute(attribute="mime.type", description="Always set to application/json")
@CapabilityDescription(value="Applies a JSLT transformation to the FlowFile JSON payload. A new FlowFile is created with transformed content and is routed to the 'success' relationship. If the JSLT transform fails, the original FlowFile is routed to the 'failure' relationship.")
public class JSLTTransformJSON
extends AbstractProcessor {
    public static String JSLT_FILTER_DEFAULT = ". != null and . != {} and . != []";
    public static final PropertyDescriptor JSLT_TRANSFORM = new PropertyDescriptor.Builder().name("jslt-transform-transformation").displayName("JSLT Transformation").description("JSLT Transformation for transform of JSON data. Any NiFi Expression Language present will be evaluated first to get the final transform to be applied. The JSLT Tutorial provides an overview of supported expressions: https://github.com/schibsted/jslt/blob/master/tutorial.md").expressionLanguageSupported(ExpressionLanguageScope.FLOWFILE_ATTRIBUTES).addValidator(StandardValidators.NON_EMPTY_VALIDATOR).identifiesExternalResource(ResourceCardinality.SINGLE, ResourceType.TEXT, new ResourceType[]{ResourceType.FILE}).required(true).build();
    public static final PropertyDescriptor TRANSFORMATION_STRATEGY = new PropertyDescriptor.Builder().name("jslt-transform-transformation-strategy").displayName("Transformation Strategy").description("Whether to apply the JSLT transformation to the entire FlowFile contents or each JSON object in the root-level array").required(true).allowableValues(TransformationStrategy.class).defaultValue(TransformationStrategy.ENTIRE_FLOWFILE.getValue()).build();
    public static final PropertyDescriptor PRETTY_PRINT = new PropertyDescriptor.Builder().name("jslt-transform-pretty_print").displayName("Pretty Print").description("Apply pretty-print formatting to the output of the JSLT transform").required(true).allowableValues(new String[]{"true", "false"}).defaultValue("false").build();
    public static final PropertyDescriptor TRANSFORM_CACHE_SIZE = new PropertyDescriptor.Builder().name("jslt-transform-cache-size").displayName("Transform Cache Size").description("Compiling a JSLT Transform can be fairly expensive. Ideally, this will be done only once. However, if the Expression Language is used in the transform, we may need a new Transform for each FlowFile. This value controls how many of those Transforms we cache in memory in order to avoid having to compile the Transform each time.").expressionLanguageSupported(ExpressionLanguageScope.NONE).addValidator(StandardValidators.POSITIVE_INTEGER_VALIDATOR).defaultValue("1").required(true).build();
    public static final PropertyDescriptor RESULT_FILTER = new PropertyDescriptor.Builder().name("jslt-transform-result-filter").displayName("Transform Result Filter").description("A filter for output JSON results using a JSLT expression. This property supports changing the default filter, which removes JSON objects with null values, empty objects and empty arrays from the output JSON. This JSLT must return true for each JSON object to be included and false for each object to be removed. Using a filter value of \"true\" to disables filtering.").addValidator(StandardValidators.NON_EMPTY_VALIDATOR).required(true).defaultValue(JSLT_FILTER_DEFAULT).build();
    public static final Relationship REL_SUCCESS = new Relationship.Builder().name("success").description("The FlowFile with transformed content will be routed to this relationship").build();
    public static final Relationship REL_FAILURE = new Relationship.Builder().name("failure").description("If a FlowFile fails processing for any reason (for example, the FlowFile is not valid JSON), it will be routed to this relationship").build();
    private static final List<PropertyDescriptor> descriptors;
    private static final Set<Relationship> relationships;
    private static final ObjectMapper jsonObjectMapper;
    private Cache<String, Expression> transformCache;

    public Set<Relationship> getRelationships() {
        return relationships;
    }

    public final List<PropertyDescriptor> getSupportedPropertyDescriptors() {
        return descriptors;
    }

    protected Collection<ValidationResult> customValidate(ValidationContext validationContext) {
        ArrayList<ValidationResult> results = new ArrayList<ValidationResult>(super.customValidate(validationContext));
        PropertyValue transformProperty = validationContext.getProperty(JSLT_TRANSFORM);
        if (transformProperty.isExpressionLanguagePresent()) {
            ValidationResult.Builder transformBuilder = new ValidationResult.Builder().subject(JSLT_TRANSFORM.getDisplayName());
            results.add(transformBuilder.valid(true).build());
        } else {
            results.add(this.validateJSLT(JSLT_TRANSFORM, transformProperty));
        }
        PropertyValue filterProperty = validationContext.getProperty(RESULT_FILTER);
        results.add(this.validateJSLT(RESULT_FILTER, filterProperty));
        return results;
    }

    private ValidationResult validateJSLT(PropertyDescriptor property, PropertyValue value) {
        ValidationResult.Builder builder = new ValidationResult.Builder().subject(property.getDisplayName());
        try {
            String transform = this.readTransform(value);
            this.getJstlExpression(transform, null);
            builder.valid(true);
        }
        catch (RuntimeException e) {
            String explanation = String.format("%s not valid: %s", property.getDisplayName(), e.getMessage());
            builder.valid(false).explanation(explanation);
        }
        return builder.build();
    }

    @OnScheduled
    public void onScheduled(ProcessContext context) {
        int maxTransformsToCache = context.getProperty(TRANSFORM_CACHE_SIZE).asInteger();
        this.transformCache = Caffeine.newBuilder().maximumSize((long)maxTransformsToCache).build();
        PropertyValue transformProperty = context.getProperty(JSLT_TRANSFORM);
        PropertyValue filterProperty = context.getProperty(RESULT_FILTER);
        if (!transformProperty.isExpressionLanguagePresent()) {
            try {
                String transform = this.readTransform(transformProperty);
                this.transformCache.put((Object)transform, (Object)this.getJstlExpression(transform, filterProperty.getValue()));
            }
            catch (RuntimeException e) {
                throw new ProcessException("JSLT Transform compilation failed", (Throwable)e);
            }
        }
    }

    public void onTrigger(ProcessContext context, ProcessSession session) throws ProcessException {
        FlowFile original = session.get();
        if (original == null) {
            return;
        }
        TransformationStrategy transformationStrategy = TransformationStrategy.valueOf(context.getProperty(TRANSFORMATION_STRATEGY).getValue());
        StopWatch stopWatch = new StopWatch(true);
        PropertyValue transformProperty = context.getProperty(JSLT_TRANSFORM);
        PropertyValue filterProperty = context.getProperty(RESULT_FILTER);
        JsonFactory jsonFactory = new JsonFactory();
        try {
            String transform = this.readTransform(transformProperty, original);
            Expression jsltExpression = (Expression)this.transformCache.get((Object)transform, currString -> this.getJstlExpression(transform, filterProperty.getValue()));
            boolean prettyPrint = context.getProperty(PRETTY_PRINT).asBoolean();
            FlowFile transformed = session.write(original, (inputStream, outputStream) -> {
                Object firstJsonNode;
                JsonParser jsonParser;
                boolean topLevelArray = false;
                if (TransformationStrategy.EACH_OBJECT.equals((Object)transformationStrategy)) {
                    jsonParser = jsonFactory.createParser(inputStream);
                    jsonParser.setCodec((ObjectCodec)jsonObjectMapper);
                    JsonToken token = jsonParser.nextToken();
                    if (token == JsonToken.START_ARRAY) {
                        token = jsonParser.nextToken();
                        topLevelArray = true;
                    }
                    firstJsonNode = token == JsonToken.START_OBJECT ? (JsonNode)jsonParser.readValueAsTree() : null;
                } else {
                    firstJsonNode = this.readJson(inputStream);
                    jsonParser = null;
                }
                ObjectWriter writer = prettyPrint ? jsonObjectMapper.writerWithDefaultPrettyPrinter() : jsonObjectMapper.writer();
                JsonGenerator jsonGenerator = writer.createGenerator(outputStream);
                if (topLevelArray) {
                    jsonGenerator.writeStartArray();
                }
                JsonNode nextNode = firstJsonNode;
                do {
                    JsonNode outputObject;
                    JsonNode transformedJson;
                    if ((transformedJson = jsltExpression.apply(nextNode)) == null || transformedJson.isNull()) {
                        this.getLogger().warn("JSLT Transform resulted in no data {}", new Object[]{original});
                        outputObject = null;
                    } else {
                        outputObject = transformedJson;
                    }
                    if (outputObject == null) continue;
                    jsonGenerator.writeObject((Object)outputObject);
                } while ((nextNode = this.getNextJsonNode(transformationStrategy, jsonParser)) != null);
                if (topLevelArray) {
                    jsonGenerator.writeEndArray();
                }
                jsonGenerator.flush();
            });
            transformed = session.putAttribute(transformed, CoreAttributes.MIME_TYPE.key(), "application/json");
            session.transfer(transformed, REL_SUCCESS);
            session.getProvenanceReporter().modifyContent(transformed, "Modified With " + transform, stopWatch.getElapsed(TimeUnit.MILLISECONDS));
            stopWatch.stop();
            this.getLogger().debug("JSLT Transform completed {}", new Object[]{original});
        }
        catch (Exception e) {
            this.getLogger().error("JSLT Transform failed {}", new Object[]{original, e});
            session.transfer(original, REL_FAILURE);
        }
    }

    @OnStopped
    @OnShutdown
    public void onStopped() {
        if (this.transformCache != null) {
            this.transformCache.cleanUp();
        }
    }

    private Expression getJstlExpression(String transform, String jsltFilter) {
        Parser parser = new Parser((Reader)new StringReader(transform)).withSource("<inline>");
        if (jsltFilter != null && !jsltFilter.isEmpty() && !jsltFilter.equals(JSLT_FILTER_DEFAULT)) {
            parser = parser.withObjectFilter(jsltFilter);
        }
        return parser.compile();
    }

    private JsonNode readJson(InputStream in) throws IOException {
        try {
            return jsonObjectMapper.readTree(in);
        }
        catch (JsonParseException e) {
            throw new IOException("Could not parse data as JSON", e);
        }
    }

    private String readTransform(PropertyValue propertyValue, FlowFile flowFile) {
        String transform = propertyValue.isExpressionLanguagePresent() ? propertyValue.evaluateAttributeExpressions(flowFile).getValue() : this.readTransform(propertyValue);
        return transform;
    }

    private String readTransform(PropertyValue propertyValue) {
        String string;
        ResourceReference resourceReference = propertyValue.asResource();
        if (resourceReference == null) {
            return propertyValue.getValue();
        }
        BufferedReader reader = new BufferedReader(new InputStreamReader(resourceReference.read()));
        try {
            string = reader.lines().collect(Collectors.joining());
        }
        catch (Throwable throwable) {
            try {
                try {
                    reader.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (IOException e) {
                throw new UncheckedIOException("Read JSLT Transform failed", e);
            }
        }
        reader.close();
        return string;
    }

    protected JsonNode getNextJsonNode(TransformationStrategy transformationStrategy, JsonParser jsonParser) throws IOException {
        if (TransformationStrategy.ENTIRE_FLOWFILE.equals((Object)transformationStrategy)) {
            return null;
        }
        return this.getJsonNode(jsonParser);
    }

    private JsonNode getJsonNode(JsonParser jsonParser) throws IOException {
        JsonToken token;
        block4: while (true) {
            if ((token = jsonParser.nextToken()) == null) {
                return null;
            }
            switch (token) {
                case START_ARRAY: 
                case END_ARRAY: 
                case END_OBJECT: {
                    continue block4;
                }
                case START_OBJECT: {
                    return (JsonNode)jsonParser.readValueAsTree();
                }
            }
            break;
        }
        throw new IOException("Expected to get a JSON Object but got a token of type " + token.name());
    }

    static {
        jsonObjectMapper = new ObjectMapper();
        descriptors = Collections.unmodifiableList(Arrays.asList(JSLT_TRANSFORM, TRANSFORMATION_STRATEGY, PRETTY_PRINT, TRANSFORM_CACHE_SIZE, RESULT_FILTER));
        relationships = Collections.unmodifiableSet(new LinkedHashSet<Relationship>(Arrays.asList(REL_SUCCESS, REL_FAILURE)));
    }

    static enum TransformationStrategy implements DescribedValue
    {
        ENTIRE_FLOWFILE("Entire FlowFile", "Apply transformation to entire FlowFile content JSON"),
        EACH_OBJECT("Each JSON Object", "Apply transformation each JSON Object in an array");

        private final String displayName;
        private final String description;

        private TransformationStrategy(String displayName, String description) {
            this.displayName = displayName;
            this.description = description;
        }

        public String getValue() {
            return this.name();
        }

        public String getDisplayName() {
            return this.displayName;
        }

        public String getDescription() {
            return this.description;
        }
    }
}

