/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.processors.cipher;

import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.security.InvalidKeyException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Base64;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.function.Function;
import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;
import org.apache.nifi.annotation.behavior.InputRequirement;
import org.apache.nifi.annotation.behavior.SupportsBatching;
import org.apache.nifi.annotation.behavior.WritesAttribute;
import org.apache.nifi.annotation.behavior.WritesAttributes;
import org.apache.nifi.annotation.documentation.CapabilityDescription;
import org.apache.nifi.annotation.documentation.Tags;
import org.apache.nifi.annotation.lifecycle.OnScheduled;
import org.apache.nifi.components.PropertyDescriptor;
import org.apache.nifi.components.ValidationContext;
import org.apache.nifi.components.ValidationResult;
import org.apache.nifi.expression.ExpressionLanguageScope;
import org.apache.nifi.flowfile.FlowFile;
import org.apache.nifi.processor.AbstractProcessor;
import org.apache.nifi.processor.ProcessContext;
import org.apache.nifi.processor.ProcessSession;
import org.apache.nifi.processor.Relationship;
import org.apache.nifi.processor.exception.ProcessException;
import org.apache.nifi.processor.util.StandardValidators;
import org.bouncycastle.util.encoders.Hex;

@SupportsBatching
@InputRequirement(value=InputRequirement.Requirement.INPUT_REQUIRED)
@Tags(value={"Authentication", "Signing", "MAC", "HMAC"})
@CapabilityDescription(value="Calculates a Message Authentication Code using the provided Secret Key and compares it with the provided MAC property")
@WritesAttributes(value={@WritesAttribute(attribute="mac.calculated", description="Calculated Message Authentication Code encoded by the selected encoding"), @WritesAttribute(attribute="mac.encoding", description="The Encoding of the Hashed Message Authentication Code"), @WritesAttribute(attribute="mac.algorithm", description="Hashed Message Authentication Code Algorithm")})
public class VerifyContentMAC
extends AbstractProcessor {
    protected static final String HMAC_SHA256 = "HmacSHA256";
    protected static final String HMAC_SHA512 = "HmacSHA512";
    protected static final PropertyDescriptor MAC_ALGORITHM = new PropertyDescriptor.Builder().name("mac-algorithm").displayName("Message Authentication Code Algorithm").description("Hashed Message Authentication Code Function").allowableValues(new String[]{"HmacSHA256", "HmacSHA512"}).required(true).addValidator(StandardValidators.NON_EMPTY_VALIDATOR).build();
    protected static final PropertyDescriptor MAC_ENCODING = new PropertyDescriptor.Builder().name("message-authentication-code-encoding").displayName("Message Authentication Code Encoding").description("Encoding of the Message Authentication Code").required(true).addValidator(StandardValidators.NON_EMPTY_VALIDATOR).allowableValues(new String[]{Encoding.HEXADECIMAL.name(), Encoding.BASE64.name()}).defaultValue(Encoding.HEXADECIMAL.name()).build();
    protected static final PropertyDescriptor MAC = new PropertyDescriptor.Builder().name("message-authentication-code").displayName("Message Authentication Code").description("The MAC to compare with the calculated value").required(true).addValidator(StandardValidators.NON_EMPTY_VALIDATOR).expressionLanguageSupported(ExpressionLanguageScope.FLOWFILE_ATTRIBUTES).build();
    protected static final PropertyDescriptor SECRET_KEY_ENCODING = new PropertyDescriptor.Builder().name("secret-key-encoding").displayName("Secret Key Encoding").description("Encoding of the Secret Key").required(true).addValidator(StandardValidators.NON_EMPTY_VALIDATOR).allowableValues(new String[]{Encoding.UTF8.name(), Encoding.HEXADECIMAL.name(), Encoding.BASE64.name()}).defaultValue(Encoding.HEXADECIMAL.name()).build();
    protected static final PropertyDescriptor SECRET_KEY = new PropertyDescriptor.Builder().name("secret-key").displayName("Secret Key").description("Secret key to calculate the hash").required(true).addValidator(StandardValidators.NON_EMPTY_VALIDATOR).sensitive(true).build();
    protected static final Relationship FAILURE = new Relationship.Builder().name("failure").description("Signature Verification Failed").build();
    protected static final Relationship SUCCESS = new Relationship.Builder().name("success").description("Signature Verification Succeeded").build();
    protected static final String MAC_CALCULATED_ATTRIBUTE = "mac.calculated";
    protected static final String MAC_ALGORITHM_ATTRIBUTE = "mac.algorithm";
    protected static final String MAC_ENCODING_ATTRIBUTE = "mac.encoding";
    private static final List<PropertyDescriptor> PROPERTIES = Collections.unmodifiableList(Arrays.asList(MAC_ALGORITHM, MAC_ENCODING, MAC, SECRET_KEY_ENCODING, SECRET_KEY));
    private static final Set<Relationship> RELATIONSHIPS = Collections.unmodifiableSet(new HashSet<Relationship>(Arrays.asList(SUCCESS, FAILURE)));
    private static final int BUFFER_SIZE = 512000;
    private SecretKeySpec secretKeySpec;
    private String macAlgorithm;
    private String macEncoding;

    @OnScheduled
    public void setUp(ProcessContext context) {
        this.macAlgorithm = context.getProperty(MAC_ALGORITHM).getValue();
        this.macEncoding = context.getProperty(MAC_ENCODING).getValue();
        String secretKeyEncoding = context.getProperty(SECRET_KEY_ENCODING).getValue();
        String inputSecretKey = context.getProperty(SECRET_KEY).getValue();
        byte[] secretKey = Encoding.valueOf(secretKeyEncoding).decode(inputSecretKey);
        this.secretKeySpec = new SecretKeySpec(secretKey, this.macAlgorithm);
    }

    public Set<Relationship> getRelationships() {
        return RELATIONSHIPS;
    }

    public void onTrigger(ProcessContext context, ProcessSession session) {
        FlowFile flowFile = session.get();
        if (flowFile == null) {
            return;
        }
        String macEncoded = context.getProperty(MAC).evaluateAttributeExpressions(flowFile).getValue();
        try {
            byte[] macDecoded = Encoding.valueOf(this.macEncoding).decode(macEncoded);
            byte[] macCalculated = this.getCalculatedMac(session, flowFile);
            flowFile = this.setFlowFileAttributes(session, flowFile, macCalculated);
            if (MessageDigest.isEqual(macDecoded, macCalculated)) {
                session.transfer(flowFile, SUCCESS);
            } else {
                this.getLogger().info("Verification Failed with Message Authentication Code Algorithm [{}]", new Object[]{this.macAlgorithm});
                session.transfer(flowFile, FAILURE);
            }
        }
        catch (Exception e) {
            this.getLogger().error("Processing Failed with Message Authentication Code Algorithm [{}]", new Object[]{this.macAlgorithm, e});
            session.transfer(flowFile, FAILURE);
        }
    }

    protected List<PropertyDescriptor> getSupportedPropertyDescriptors() {
        return PROPERTIES;
    }

    protected Collection<ValidationResult> customValidate(ValidationContext validationContext) {
        ArrayList<ValidationResult> results = new ArrayList<ValidationResult>(super.customValidate(validationContext));
        String secretKeyEncoding = validationContext.getProperty(SECRET_KEY_ENCODING).getValue();
        String encodedSecretKey = validationContext.getProperty(SECRET_KEY).getValue();
        try {
            Encoding.valueOf(secretKeyEncoding).decode(encodedSecretKey);
        }
        catch (Exception e) {
            results.add(new ValidationResult.Builder().valid(false).subject(SECRET_KEY.getDisplayName()).explanation("The provided Secret Key is not a valid " + secretKeyEncoding + " value").build());
        }
        return results;
    }

    private FlowFile setFlowFileAttributes(ProcessSession session, FlowFile flowFile, byte[] calculatedMac) {
        HashMap<String, String> attributes = new HashMap<String, String>();
        attributes.put(MAC_ALGORITHM_ATTRIBUTE, this.macAlgorithm);
        attributes.put(MAC_ENCODING_ATTRIBUTE, this.macEncoding);
        attributes.put(MAC_CALCULATED_ATTRIBUTE, Encoding.valueOf(this.macEncoding).encode(calculatedMac));
        return session.putAllAttributes(flowFile, attributes);
    }

    private Mac getInitializedMac() {
        try {
            Mac mac = Mac.getInstance(this.macAlgorithm);
            mac.init(this.secretKeySpec);
            return mac;
        }
        catch (InvalidKeyException | NoSuchAlgorithmException e) {
            throw new ProcessException("HMAC initialization failed", (Throwable)e);
        }
    }

    private byte[] getCalculatedMac(ProcessSession session, FlowFile flowFile) {
        Mac mac = this.getInitializedMac();
        byte[] contents = new byte[512000];
        try (InputStream is = session.read(flowFile);){
            int readSize;
            while ((readSize = is.read(contents)) != -1) {
                mac.update(contents, 0, readSize);
            }
        }
        catch (IOException e) {
            throw new ProcessException("File processing failed", (Throwable)e);
        }
        return mac.doFinal();
    }

    static enum Encoding {
        HEXADECIMAL(Hex::toHexString, Hex::decode),
        BASE64(value -> Base64.getEncoder().encodeToString((byte[])value), value -> Base64.getDecoder().decode((String)value)),
        UTF8(value -> new String((byte[])value, StandardCharsets.UTF_8), value -> value.getBytes(StandardCharsets.UTF_8));

        private final Function<byte[], String> encodeFunction;
        private final Function<String, byte[]> decodeFunction;

        private Encoding(Function<byte[], String> encodeFunction, Function<String, byte[]> decodeFunction) {
            this.decodeFunction = decodeFunction;
            this.encodeFunction = encodeFunction;
        }

        public byte[] decode(String value) {
            return this.decodeFunction.apply(value);
        }

        public String encode(byte[] value) {
            return this.encodeFunction.apply(value);
        }
    }
}

