/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.bootstrap.util;

import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.security.GeneralSecurityException;
import java.security.KeyStore;
import java.security.cert.Certificate;
import java.time.LocalDate;
import java.time.temporal.ChronoUnit;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Properties;
import java.util.Set;
import java.util.stream.Collectors;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.nifi.security.util.KeyStoreUtils;
import org.apache.nifi.security.util.StandardTlsConfiguration;
import org.apache.nifi.security.util.TlsConfiguration;
import org.bouncycastle.util.IPAddress;
import org.slf4j.Logger;

public class SecureNiFiConfigUtil {
    private static final int CERT_DURATION_DAYS = 60;
    private static final String LOCALHOST_NAME = "localhost";
    private static final String PROPERTY_VALUE_PATTERN = "%s=%s";

    private SecureNiFiConfigUtil() {
    }

    private static boolean fileExists(String filename) {
        return StringUtils.isNotEmpty((CharSequence)filename) && Paths.get(filename, new String[0]).toFile().exists();
    }

    private static boolean isHttpsSecurityConfiguredWithEmptyPasswords(Properties nifiProperties) {
        if (StringUtils.isEmpty((CharSequence)nifiProperties.getProperty("nifi.web.https.port", ""))) {
            return false;
        }
        String keystorePath = nifiProperties.getProperty("nifi.security.keystore", "");
        String truststorePath = nifiProperties.getProperty("nifi.security.truststore", "");
        if (StringUtils.isEmpty((CharSequence)keystorePath) || StringUtils.isEmpty((CharSequence)truststorePath)) {
            return false;
        }
        String keystorePassword = nifiProperties.getProperty("nifi.security.keystorePasswd", "");
        String truststorePassword = nifiProperties.getProperty("nifi.security.truststorePasswd", "");
        return !StringUtils.isNotEmpty((CharSequence)keystorePassword) && !StringUtils.isNotEmpty((CharSequence)truststorePassword);
    }

    public static void configureSecureNiFiProperties(String nifiPropertiesFilename, Logger cmdLogger) throws IOException, RuntimeException {
        File propertiesFile = new File(nifiPropertiesFilename);
        Properties nifiProperties = SecureNiFiConfigUtil.loadProperties(propertiesFile);
        if (!SecureNiFiConfigUtil.isHttpsSecurityConfiguredWithEmptyPasswords(nifiProperties)) {
            cmdLogger.debug("Skipping Apache Nifi certificate generation.");
            return;
        }
        String keystorePath = nifiProperties.getProperty("nifi.security.keystore", "");
        String truststorePath = nifiProperties.getProperty("nifi.security.truststore", "");
        boolean keystoreExists = SecureNiFiConfigUtil.fileExists(keystorePath);
        boolean truststoreExists = SecureNiFiConfigUtil.fileExists(truststorePath);
        if (!keystoreExists && !truststoreExists) {
            TlsConfiguration tlsConfiguration;
            cmdLogger.info("Generating Self-Signed Certificate: Expires on {}", (Object)LocalDate.now().plus(60L, ChronoUnit.DAYS));
            try {
                String[] subjectAlternativeNames = SecureNiFiConfigUtil.getSubjectAlternativeNames(nifiProperties, cmdLogger);
                tlsConfiguration = KeyStoreUtils.createTlsConfigAndNewKeystoreTruststore((TlsConfiguration)StandardTlsConfiguration.fromNiFiProperties((Properties)nifiProperties), (int)60, (String[])subjectAlternativeNames);
                KeyStore keyStore = KeyStoreUtils.loadKeyStore((String)tlsConfiguration.getKeystorePath(), (char[])tlsConfiguration.getKeystorePassword().toCharArray(), (String)tlsConfiguration.getKeystoreType().getType());
                Enumeration<String> aliases = keyStore.aliases();
                while (aliases.hasMoreElements()) {
                    String alias = aliases.nextElement();
                    Certificate certificate = keyStore.getCertificate(alias);
                    if (certificate == null) continue;
                    String sha256 = DigestUtils.sha256Hex((byte[])certificate.getEncoded());
                    cmdLogger.info("Generated Self-Signed Certificate SHA-256: {}", (Object)sha256.toUpperCase(Locale.ROOT));
                }
            }
            catch (GeneralSecurityException e) {
                throw new RuntimeException(e);
            }
            Files.move(Paths.get(tlsConfiguration.getKeystorePath(), new String[0]), Paths.get(keystorePath, new String[0]), StandardCopyOption.REPLACE_EXISTING);
            Files.move(Paths.get(tlsConfiguration.getTruststorePath(), new String[0]), Paths.get(truststorePath, new String[0]), StandardCopyOption.REPLACE_EXISTING);
            SecureNiFiConfigUtil.updateProperties(propertiesFile, tlsConfiguration);
            cmdLogger.debug("Generated Keystore [{}] Truststore [{}]", (Object)keystorePath, (Object)truststorePath);
        } else if (!keystoreExists && truststoreExists) {
            cmdLogger.warn("Truststore file {} already exists.  Apache NiFi will not generate keystore and truststore separately.", (Object)truststorePath);
        } else if (keystoreExists && !truststoreExists) {
            cmdLogger.warn("Keystore file {} already exists.  Apache NiFi will not generate keystore and truststore separately.", (Object)keystorePath);
        }
    }

    private static String[] getSubjectAlternativeNames(Properties nifiProperties, Logger cmdLogger) {
        HashSet<String> dnsSubjectAlternativeNames = new HashSet<String>();
        try {
            dnsSubjectAlternativeNames.add(InetAddress.getLocalHost().getHostName());
        }
        catch (UnknownHostException e) {
            cmdLogger.debug("Could not add localhost hostname as certificate SAN", (Throwable)e);
        }
        SecureNiFiConfigUtil.addSubjectAlternativeName(nifiProperties, "nifi.remote.input.host", dnsSubjectAlternativeNames);
        SecureNiFiConfigUtil.addSubjectAlternativeName(nifiProperties, "nifi.web.https.host", dnsSubjectAlternativeNames);
        SecureNiFiConfigUtil.addSubjectAlternativeName(nifiProperties, "nifi.web.proxy.host", dnsSubjectAlternativeNames);
        SecureNiFiConfigUtil.addSubjectAlternativeName(nifiProperties, "nifi.cluster.load.balance.host", dnsSubjectAlternativeNames);
        dnsSubjectAlternativeNames.remove(LOCALHOST_NAME);
        return dnsSubjectAlternativeNames.toArray(new String[dnsSubjectAlternativeNames.size()]);
    }

    private static void addSubjectAlternativeName(Properties nifiProperties, String propertyName, Set<String> dnsSubjectAlternativeNames) {
        String hostValue = nifiProperties.getProperty(propertyName, "");
        if (!hostValue.isEmpty() && !IPAddress.isValid((String)hostValue)) {
            dnsSubjectAlternativeNames.add(hostValue);
        }
    }

    private static String getPropertyLine(String name, String value) {
        return String.format(PROPERTY_VALUE_PATTERN, name, value);
    }

    private static void updateProperties(File propertiesFile, TlsConfiguration tlsConfiguration) throws IOException {
        Path propertiesFilePath = propertiesFile.toPath();
        List<String> lines = Files.readAllLines(propertiesFilePath);
        List updatedLines = lines.stream().map(line -> {
            if (line.startsWith("nifi.security.keystorePasswd")) {
                return SecureNiFiConfigUtil.getPropertyLine("nifi.security.keystorePasswd", tlsConfiguration.getKeystorePassword());
            }
            if (line.startsWith("nifi.security.truststorePasswd")) {
                return SecureNiFiConfigUtil.getPropertyLine("nifi.security.truststorePasswd", tlsConfiguration.getTruststorePassword());
            }
            if (line.startsWith("nifi.security.keyPasswd")) {
                return SecureNiFiConfigUtil.getPropertyLine("nifi.security.keyPasswd", tlsConfiguration.getKeystorePassword());
            }
            if (line.startsWith("nifi.security.keystoreType")) {
                return SecureNiFiConfigUtil.getPropertyLine("nifi.security.keystoreType", tlsConfiguration.getKeystoreType().getType());
            }
            if (line.startsWith("nifi.security.truststoreType")) {
                return SecureNiFiConfigUtil.getPropertyLine("nifi.security.truststoreType", tlsConfiguration.getTruststoreType().getType());
            }
            return line;
        }).collect(Collectors.toList());
        Files.write(propertiesFilePath, updatedLines, new OpenOption[0]);
    }

    private static Properties loadProperties(File propertiesFile) {
        Properties properties = new Properties();
        try (FileReader reader = new FileReader(propertiesFile);){
            properties.load(reader);
        }
        catch (IOException e) {
            String message = String.format("Failed to read NiFi Properties [%s]", propertiesFile);
            throw new UncheckedIOException(message, e);
        }
        return properties;
    }
}

