/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.bootstrap;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.nifi.attribute.expression.language.StandardPropertyValue;
import org.apache.nifi.bootstrap.notification.NotificationContext;
import org.apache.nifi.bootstrap.notification.NotificationInitializationContext;
import org.apache.nifi.bootstrap.notification.NotificationService;
import org.apache.nifi.bootstrap.notification.NotificationType;
import org.apache.nifi.bootstrap.notification.NotificationValidationContext;
import org.apache.nifi.components.PropertyDescriptor;
import org.apache.nifi.components.PropertyValue;
import org.apache.nifi.components.ValidationResult;
import org.apache.nifi.components.resource.ResourceContext;
import org.apache.nifi.components.resource.ResourceReferenceFactory;
import org.apache.nifi.components.resource.StandardResourceContext;
import org.apache.nifi.components.resource.StandardResourceReferenceFactory;
import org.apache.nifi.parameter.ParameterLookup;
import org.apache.nifi.registry.VariableRegistry;
import org.apache.nifi.xml.processing.parsers.StandardDocumentProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

public class NotificationServiceManager {
    private static final Logger logger = LoggerFactory.getLogger(NotificationServiceManager.class);
    private final Map<String, ConfiguredNotificationService> servicesById = new HashMap<String, ConfiguredNotificationService>();
    private final Map<NotificationType, List<ConfiguredNotificationService>> servicesByNotificationType = new HashMap<NotificationType, List<ConfiguredNotificationService>>();
    private final ScheduledExecutorService notificationExecutor;
    private int maxAttempts = 5;
    private final VariableRegistry variableRegistry;

    public NotificationServiceManager() {
        this(VariableRegistry.ENVIRONMENT_SYSTEM_REGISTRY);
    }

    NotificationServiceManager(VariableRegistry variableRegistry) {
        this.variableRegistry = variableRegistry;
        this.notificationExecutor = Executors.newScheduledThreadPool(1, new ThreadFactory(){

            @Override
            public Thread newThread(Runnable r) {
                Thread t = Executors.defaultThreadFactory().newThread(r);
                t.setName("Notification Service Dispatcher");
                t.setDaemon(true);
                return t;
            }
        });
    }

    public void setMaxNotificationAttempts(int maxAttempts) {
        this.maxAttempts = maxAttempts;
    }

    public void loadNotificationServices(File servicesFile) throws IOException {
        HashMap<String, ConfiguredNotificationService> serviceMap = new HashMap<String, ConfiguredNotificationService>();
        try (FileInputStream fis = new FileInputStream(servicesFile);
             BufferedInputStream in = new BufferedInputStream(fis);){
            StandardDocumentProvider documentProvider = new StandardDocumentProvider();
            Document doc = documentProvider.parse((InputStream)in);
            List<Element> serviceElements = NotificationServiceManager.getChildElementsByTagName(doc.getDocumentElement(), "service");
            logger.debug("Found {} service elements", (Object)serviceElements.size());
            for (Element serviceElement : serviceElements) {
                ConfiguredNotificationService config = this.createService(serviceElement);
                NotificationService service = config.getService();
                if (service == null) continue;
                String id = service.getIdentifier();
                if (serviceMap.containsKey(id)) {
                    logger.error("Found two different Notification Services configured with the same ID: '{}'. Loaded the first service.", (Object)id);
                    continue;
                }
                NotificationValidationContext validationContext = new NotificationValidationContext(this.buildNotificationContext(config), this.variableRegistry);
                Collection validationResults = service.validate(validationContext);
                ArrayList<String> invalidReasons = new ArrayList<String>();
                for (ValidationResult result : validationResults) {
                    if (result.isValid()) continue;
                    invalidReasons.add(result.toString());
                }
                if (!invalidReasons.isEmpty()) {
                    logger.warn("Configured Notification Service {} is not valid for the following reasons: {}", (Object)service, invalidReasons);
                }
                serviceMap.put(id, config);
            }
        }
        logger.info("Successfully loaded the following {} services: {}", (Object)serviceMap.size(), serviceMap.keySet());
        this.servicesById.clear();
        this.servicesById.putAll(serviceMap);
    }

    public void notify(final NotificationType type, final String subject, final String message) {
        List<ConfiguredNotificationService> configs = this.servicesByNotificationType.get((Object)type);
        if (configs == null || configs.isEmpty()) {
            return;
        }
        for (final ConfiguredNotificationService config : configs) {
            final NotificationService service = config.getService();
            final AtomicInteger attemptCount = new AtomicInteger(0);
            this.notificationExecutor.submit(new Runnable(){

                @Override
                public void run() {
                    NotificationValidationContext validationContext = new NotificationValidationContext(NotificationServiceManager.this.buildNotificationContext(config), NotificationServiceManager.this.variableRegistry);
                    Collection validationResults = service.validate(validationContext);
                    ArrayList<String> invalidReasons = new ArrayList<String>();
                    for (ValidationResult result : validationResults) {
                        if (result.isValid()) continue;
                        invalidReasons.add(result.toString());
                    }
                    boolean failure = false;
                    if (invalidReasons.isEmpty()) {
                        NotificationContext context = NotificationServiceManager.this.buildNotificationContext(config);
                        try {
                            service.notify(context, type, subject, message);
                            logger.info("Successfully sent notification of type {} to {}", (Object)type, (Object)service);
                        }
                        catch (Throwable t) {
                            logger.error("Failed to send notification of type {} to {} with Subject {} due to {}. Will ", new Object[]{type, service == null ? "Unknown Notification Service" : service.toString(), subject, t.toString()});
                            logger.error("", t);
                            failure = true;
                        }
                    } else {
                        logger.warn("Notification Service {} is not valid for the following reasons: {}", (Object)service, invalidReasons);
                        failure = true;
                    }
                    int attempts = attemptCount.incrementAndGet();
                    if (failure) {
                        if (attempts < NotificationServiceManager.this.maxAttempts) {
                            logger.info("After failing to send notification to {} {} times, will attempt again in 1 minute", (Object)service, (Object)attempts);
                            NotificationServiceManager.this.notificationExecutor.schedule(this, 1L, TimeUnit.MINUTES);
                        } else {
                            logger.info("After failing to send notification of type {} to {} {} times, will no longer attempt to send notification", new Object[]{type, service, attempts});
                        }
                    }
                }
            });
            if (!NotificationType.NIFI_STOPPED.equals((Object)type)) continue;
            while (attemptCount.get() == 0) {
                try {
                    Thread.sleep(1000L);
                }
                catch (InterruptedException interruptedException) {}
            }
        }
    }

    private NotificationContext buildNotificationContext(final ConfiguredNotificationService config) {
        return new NotificationContext(){

            @Override
            public PropertyValue getProperty(PropertyDescriptor descriptor) {
                PropertyDescriptor fullPropDescriptor = config.getService().getPropertyDescriptor(descriptor.getName());
                if (fullPropDescriptor == null) {
                    return null;
                }
                String configuredValue = config.getProperties().get(fullPropDescriptor.getName());
                if (configuredValue == null) {
                    configuredValue = fullPropDescriptor.getDefaultValue();
                }
                StandardResourceContext resourceContext = new StandardResourceContext((ResourceReferenceFactory)new StandardResourceReferenceFactory(), descriptor);
                return new StandardPropertyValue((ResourceContext)resourceContext, configuredValue, null, ParameterLookup.EMPTY, NotificationServiceManager.this.variableRegistry);
            }

            @Override
            public Map<PropertyDescriptor, String> getProperties() {
                HashMap<PropertyDescriptor, String> props = new HashMap<PropertyDescriptor, String>();
                Map<String, String> configuredProps = config.getProperties();
                NotificationService service = config.getService();
                ArrayList configuredPropertyDescriptors = new ArrayList(service.getPropertyDescriptors());
                configuredProps.forEach((key, value) -> {
                    PropertyDescriptor propertyDescriptor = config2.service.getPropertyDescriptor((String)key);
                    props.put(config2.service.getPropertyDescriptor((String)key), (String)value);
                    configuredPropertyDescriptors.remove(propertyDescriptor);
                });
                for (PropertyDescriptor descriptor : configuredPropertyDescriptors) {
                    props.put(descriptor, descriptor.getDefaultValue());
                }
                return props;
            }
        };
    }

    public void registerNotificationService(NotificationType type, String serviceId) {
        ConfiguredNotificationService service = this.servicesById.get(serviceId);
        if (service == null) {
            throw new IllegalArgumentException("No Notification Service exists with ID " + serviceId);
        }
        List<ConfiguredNotificationService> services = this.servicesByNotificationType.get((Object)type);
        if (services == null) {
            services = new ArrayList<ConfiguredNotificationService>();
            this.servicesByNotificationType.put(type, services);
        }
        services.add(service);
    }

    private ConfiguredNotificationService createService(Element serviceElement) {
        Object serviceObject;
        Class<?> clazz;
        Element idElement = NotificationServiceManager.getChild(serviceElement, "id");
        if (idElement == null) {
            logger.error("Found configuration for Notification Service with no 'id' element; this service cannot be referenced so it will not be loaded");
            return null;
        }
        final String serviceId = idElement.getTextContent().trim();
        logger.debug("Loading Notification Service with ID {}", (Object)serviceId);
        Element classElement = NotificationServiceManager.getChild(serviceElement, "class");
        if (classElement == null) {
            logger.error("Found configuration for Notification Service with no 'class' element; Service ID is '{}'. This service annot be loaded", (Object)serviceId);
            return null;
        }
        String className = classElement.getTextContent().trim();
        try {
            clazz = Class.forName(className);
        }
        catch (Exception e) {
            logger.error("Found configuration for Notification Service with ID '{}' and Class '{}' but could not load class.", (Object)serviceId, (Object)className);
            logger.error("", (Throwable)e);
            return null;
        }
        if (!NotificationService.class.isAssignableFrom(clazz)) {
            logger.error("Found configuration for Notification Service with ID '{}' and Class '{}' but class is not a Notification Service.", (Object)serviceId, (Object)className);
            return null;
        }
        try {
            serviceObject = clazz.newInstance();
        }
        catch (Exception e) {
            logger.error("Found configuration for Notification Service with ID '{}' and Class '{}' but could not instantiate Notification Service.", (Object)serviceId, (Object)className);
            logger.error("", (Throwable)e);
            return null;
        }
        final HashMap<String, String> propertyValues = new HashMap<String, String>();
        List<Element> propertyElements = NotificationServiceManager.getChildElementsByTagName(serviceElement, "property");
        for (Element propertyElement : propertyElements) {
            String propName = propertyElement.getAttribute("name");
            if (propName == null || propName.trim().isEmpty()) {
                logger.warn("Found configuration for Notification Service with ID '{}' that has property value configured but no name for the property.", (Object)serviceId);
                continue;
            }
            String propValue = propertyElement.getTextContent().trim();
            propertyValues.put(propName, propValue);
        }
        NotificationService service = (NotificationService)serviceObject;
        try {
            service.initialize(new NotificationInitializationContext(){

                public PropertyValue getProperty(PropertyDescriptor descriptor) {
                    String propName = descriptor.getName();
                    String value = (String)propertyValues.get(propName);
                    if (value == null) {
                        value = descriptor.getDefaultValue();
                    }
                    StandardResourceContext resourceContext = new StandardResourceContext((ResourceReferenceFactory)new StandardResourceReferenceFactory(), descriptor);
                    return new StandardPropertyValue((ResourceContext)resourceContext, value, null, ParameterLookup.EMPTY, NotificationServiceManager.this.variableRegistry);
                }

                public Map<String, String> getAllProperties() {
                    return Collections.unmodifiableMap(propertyValues);
                }

                @Override
                public String getIdentifier() {
                    return serviceId;
                }
            });
        }
        catch (Exception e) {
            logger.error("Failed to load Notification Service with ID '{}'", (Object)serviceId);
            logger.error("", (Throwable)e);
        }
        return new ConfiguredNotificationService(service, propertyValues);
    }

    public static Element getChild(Element element, String tagName) {
        List<Element> children = NotificationServiceManager.getChildElementsByTagName(element, tagName);
        if (children.isEmpty()) {
            return null;
        }
        if (children.size() > 1) {
            return null;
        }
        return children.get(0);
    }

    public static List<Element> getChildElementsByTagName(Element element, String tagName) {
        ArrayList<Element> matches = new ArrayList<Element>();
        NodeList nodeList = element.getChildNodes();
        for (int i = 0; i < nodeList.getLength(); ++i) {
            Element child;
            Node node = nodeList.item(i);
            if (!(node instanceof Element) || !(child = (Element)nodeList.item(i)).getNodeName().equals(tagName)) continue;
            matches.add(child);
        }
        return matches;
    }

    private static class ConfiguredNotificationService {
        private final NotificationService service;
        private final Map<String, String> properties;

        public ConfiguredNotificationService(NotificationService service, Map<String, String> properties) {
            this.service = service;
            this.properties = properties;
        }

        public NotificationService getService() {
            return this.service;
        }

        public Map<String, String> getProperties() {
            return this.properties;
        }
    }
}

