/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.trogdor.common;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import org.apache.kafka.clients.admin.Admin;
import org.apache.kafka.clients.admin.MockAdminClient;
import org.apache.kafka.clients.admin.NewTopic;
import org.apache.kafka.clients.admin.TopicDescription;
import org.apache.kafka.common.KafkaFuture;
import org.apache.kafka.common.Node;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.TopicPartitionInfo;
import org.apache.kafka.common.errors.TopicExistsException;
import org.apache.kafka.common.errors.UnknownTopicOrPartitionException;
import org.apache.kafka.common.utils.Utils;
import org.apache.kafka.trogdor.common.WorkerUtils;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class WorkerUtilsTest {
    private static final Logger log = LoggerFactory.getLogger(WorkerUtilsTest.class);
    private final Node broker1 = new Node(0, "testHost-1", 1234);
    private final Node broker2 = new Node(1, "testHost-2", 1234);
    private final Node broker3 = new Node(1, "testHost-3", 1234);
    private final List<Node> cluster = Arrays.asList(this.broker1, this.broker2, this.broker3);
    private final List<Node> singleReplica = Collections.singletonList(this.broker1);
    private static final String TEST_TOPIC = "test-topic-1";
    private static final short TEST_REPLICATION_FACTOR = 1;
    private static final int TEST_PARTITIONS = 1;
    private static final NewTopic NEW_TEST_TOPIC = new NewTopic("test-topic-1", 1, 1);
    private MockAdminClient adminClient;

    @BeforeEach
    public void setUp() {
        this.adminClient = new MockAdminClient(this.cluster, this.broker1);
    }

    @Test
    public void testCreateOneTopic() throws Throwable {
        Map<String, NewTopic> newTopics = Collections.singletonMap(TEST_TOPIC, NEW_TEST_TOPIC);
        WorkerUtils.createTopics((Logger)log, (Admin)this.adminClient, newTopics, (boolean)true);
        Assertions.assertEquals(Collections.singleton(TEST_TOPIC), (Object)this.adminClient.listTopics().names().get());
        Assertions.assertEquals((Object)new TopicDescription(TEST_TOPIC, false, Collections.singletonList(new TopicPartitionInfo(0, this.broker1, this.singleReplica, Collections.emptyList()))), (Object)((KafkaFuture)this.adminClient.describeTopics(Collections.singleton(TEST_TOPIC)).topicNameValues().get(TEST_TOPIC)).get());
    }

    @Test
    public void testCreateRetriesOnTimeout() throws Throwable {
        this.adminClient.timeoutNextRequest(1);
        WorkerUtils.createTopics((Logger)log, (Admin)this.adminClient, Collections.singletonMap(TEST_TOPIC, NEW_TEST_TOPIC), (boolean)true);
        Assertions.assertEquals((Object)new TopicDescription(TEST_TOPIC, false, Collections.singletonList(new TopicPartitionInfo(0, this.broker1, this.singleReplica, Collections.emptyList()))), (Object)((KafkaFuture)this.adminClient.describeTopics(Collections.singleton(TEST_TOPIC)).topicNameValues().get(TEST_TOPIC)).get());
    }

    @Test
    public void testCreateZeroTopicsDoesNothing() throws Throwable {
        WorkerUtils.createTopics((Logger)log, (Admin)this.adminClient, Collections.emptyMap(), (boolean)true);
        Assertions.assertEquals((int)0, (int)((Set)this.adminClient.listTopics().names().get()).size());
    }

    @Test
    public void testCreateTopicsFailsIfAtLeastOneTopicExists() throws Throwable {
        this.adminClient.addTopic(false, TEST_TOPIC, Collections.singletonList(new TopicPartitionInfo(0, this.broker1, this.singleReplica, Collections.emptyList())), null);
        HashMap<String, NewTopic> newTopics = new HashMap<String, NewTopic>();
        newTopics.put(TEST_TOPIC, NEW_TEST_TOPIC);
        newTopics.put("another-topic", new NewTopic("another-topic", 1, 1));
        newTopics.put("one-more-topic", new NewTopic("one-more-topic", 1, 1));
        Assertions.assertThrows(TopicExistsException.class, () -> WorkerUtils.createTopics((Logger)log, (Admin)this.adminClient, (Map)newTopics, (boolean)true));
    }

    @Test
    public void testExistingTopicsMustHaveRequestedNumberOfPartitions() throws Throwable {
        ArrayList<TopicPartitionInfo> tpInfo = new ArrayList<TopicPartitionInfo>();
        tpInfo.add(new TopicPartitionInfo(0, this.broker1, this.singleReplica, Collections.emptyList()));
        tpInfo.add(new TopicPartitionInfo(1, this.broker2, this.singleReplica, Collections.emptyList()));
        this.adminClient.addTopic(false, TEST_TOPIC, tpInfo, null);
        Assertions.assertThrows(RuntimeException.class, () -> WorkerUtils.createTopics((Logger)log, (Admin)this.adminClient, Collections.singletonMap(TEST_TOPIC, NEW_TEST_TOPIC), (boolean)false));
    }

    @Test
    public void testExistingTopicsNotCreated() throws Throwable {
        String existingTopic = "existing-topic";
        ArrayList<TopicPartitionInfo> tpInfo = new ArrayList<TopicPartitionInfo>();
        tpInfo.add(new TopicPartitionInfo(0, this.broker1, this.singleReplica, Collections.emptyList()));
        tpInfo.add(new TopicPartitionInfo(1, this.broker2, this.singleReplica, Collections.emptyList()));
        tpInfo.add(new TopicPartitionInfo(2, this.broker3, this.singleReplica, Collections.emptyList()));
        this.adminClient.addTopic(false, "existing-topic", tpInfo, null);
        WorkerUtils.createTopics((Logger)log, (Admin)this.adminClient, Collections.singletonMap("existing-topic", new NewTopic("existing-topic", tpInfo.size(), 1)), (boolean)false);
        Assertions.assertEquals(Collections.singleton("existing-topic"), (Object)this.adminClient.listTopics().names().get());
    }

    @Test
    public void testCreatesNotExistingTopics() throws Throwable {
        Assertions.assertEquals((int)0, (int)((Set)this.adminClient.listTopics().names().get()).size());
        WorkerUtils.createTopics((Logger)log, (Admin)this.adminClient, Collections.singletonMap(TEST_TOPIC, NEW_TEST_TOPIC), (boolean)false);
        Assertions.assertEquals(Collections.singleton(TEST_TOPIC), (Object)this.adminClient.listTopics().names().get());
        Assertions.assertEquals((Object)new TopicDescription(TEST_TOPIC, false, Collections.singletonList(new TopicPartitionInfo(0, this.broker1, this.singleReplica, Collections.emptyList()))), (Object)((KafkaFuture)this.adminClient.describeTopics(Collections.singleton(TEST_TOPIC)).topicNameValues().get(TEST_TOPIC)).get());
    }

    @Test
    public void testCreatesOneTopicVerifiesOneTopic() throws Throwable {
        String existingTopic = "existing-topic";
        ArrayList<TopicPartitionInfo> tpInfo = new ArrayList<TopicPartitionInfo>();
        tpInfo.add(new TopicPartitionInfo(0, this.broker1, this.singleReplica, Collections.emptyList()));
        tpInfo.add(new TopicPartitionInfo(1, this.broker2, this.singleReplica, Collections.emptyList()));
        this.adminClient.addTopic(false, "existing-topic", tpInfo, null);
        HashMap<String, NewTopic> topics = new HashMap<String, NewTopic>();
        topics.put("existing-topic", new NewTopic("existing-topic", tpInfo.size(), 1));
        topics.put(TEST_TOPIC, NEW_TEST_TOPIC);
        WorkerUtils.createTopics((Logger)log, (Admin)this.adminClient, topics, (boolean)false);
        Assertions.assertEquals((Object)Utils.mkSet((Object[])new String[]{"existing-topic", TEST_TOPIC}), (Object)this.adminClient.listTopics().names().get());
    }

    @Test
    public void testCreateNonExistingTopicsWithZeroTopicsDoesNothing() throws Throwable {
        WorkerUtils.createTopics((Logger)log, (Admin)this.adminClient, Collections.emptyMap(), (boolean)false);
        Assertions.assertEquals((int)0, (int)((Set)this.adminClient.listTopics().names().get()).size());
    }

    @Test
    public void testAddConfigsToPropertiesAddsAllConfigs() {
        Properties props = new Properties();
        props.put("bootstrap.servers", "localhost:9092");
        props.put("acks", "all");
        Properties resultProps = new Properties();
        resultProps.putAll((Map<?, ?>)props);
        resultProps.put("client.id", "test-client");
        resultProps.put("linger.ms", "1000");
        WorkerUtils.addConfigsToProperties((Properties)props, Collections.singletonMap("client.id", "test-client"), Collections.singletonMap("linger.ms", "1000"));
        Assertions.assertEquals((Object)resultProps, (Object)props);
    }

    @Test
    public void testCommonConfigOverwritesDefaultProps() {
        Properties props = new Properties();
        props.put("bootstrap.servers", "localhost:9092");
        props.put("acks", "all");
        Properties resultProps = new Properties();
        resultProps.put("bootstrap.servers", "localhost:9092");
        resultProps.put("acks", "1");
        resultProps.put("linger.ms", "1000");
        WorkerUtils.addConfigsToProperties((Properties)props, Collections.singletonMap("acks", "1"), Collections.singletonMap("linger.ms", "1000"));
        Assertions.assertEquals((Object)resultProps, (Object)props);
    }

    @Test
    public void testClientConfigOverwritesBothDefaultAndCommonConfigs() {
        Properties props = new Properties();
        props.put("bootstrap.servers", "localhost:9092");
        props.put("acks", "all");
        Properties resultProps = new Properties();
        resultProps.put("bootstrap.servers", "localhost:9092");
        resultProps.put("acks", "0");
        WorkerUtils.addConfigsToProperties((Properties)props, Collections.singletonMap("acks", "1"), Collections.singletonMap("acks", "0"));
        Assertions.assertEquals((Object)resultProps, (Object)props);
    }

    @Test
    public void testGetMatchingTopicPartitionsCorrectlyMatchesExactTopicName() throws Throwable {
        String topic1 = "existing-topic";
        String topic2 = "another-topic";
        this.makeExistingTopicWithOneReplica("existing-topic", 10);
        this.makeExistingTopicWithOneReplica("another-topic", 20);
        Collection topicPartitions = WorkerUtils.getMatchingTopicPartitions((Admin)this.adminClient, (String)"another-topic", (int)0, (int)2);
        Assertions.assertEquals((Object)Utils.mkSet((Object[])new TopicPartition[]{new TopicPartition("another-topic", 0), new TopicPartition("another-topic", 1), new TopicPartition("another-topic", 2)}), new HashSet(topicPartitions));
    }

    @Test
    public void testGetMatchingTopicPartitionsCorrectlyMatchesTopics() throws Throwable {
        String topic1 = "test-topic";
        String topic2 = "another-test-topic";
        String topic3 = "one-more";
        this.makeExistingTopicWithOneReplica("test-topic", 10);
        this.makeExistingTopicWithOneReplica("another-test-topic", 20);
        this.makeExistingTopicWithOneReplica("one-more", 30);
        Collection topicPartitions = WorkerUtils.getMatchingTopicPartitions((Admin)this.adminClient, (String)".*-topic$", (int)0, (int)1);
        Assertions.assertEquals((Object)Utils.mkSet((Object[])new TopicPartition[]{new TopicPartition("test-topic", 0), new TopicPartition("test-topic", 1), new TopicPartition("another-test-topic", 0), new TopicPartition("another-test-topic", 1)}), new HashSet(topicPartitions));
    }

    private void makeExistingTopicWithOneReplica(String topicName, int numPartitions) {
        ArrayList<TopicPartitionInfo> tpInfo = new ArrayList<TopicPartitionInfo>();
        int brokerIndex = 0;
        for (int i = 0; i < numPartitions; ++i) {
            Node broker = this.cluster.get(brokerIndex);
            tpInfo.add(new TopicPartitionInfo(i, broker, this.singleReplica, Collections.emptyList()));
            brokerIndex = (brokerIndex + 1) % this.cluster.size();
        }
        this.adminClient.addTopic(false, topicName, tpInfo, null);
    }

    @Test
    public void testVerifyTopics() throws Throwable {
        Map<String, NewTopic> newTopics = Collections.singletonMap(TEST_TOPIC, NEW_TEST_TOPIC);
        WorkerUtils.createTopics((Logger)log, (Admin)this.adminClient, newTopics, (boolean)true);
        this.adminClient.setFetchesRemainingUntilVisible(TEST_TOPIC, 2);
        WorkerUtils.verifyTopics((Logger)log, (Admin)this.adminClient, Collections.singleton(TEST_TOPIC), Collections.singletonMap(TEST_TOPIC, NEW_TEST_TOPIC), (int)3, (long)1L);
        this.adminClient.setFetchesRemainingUntilVisible(TEST_TOPIC, 100);
        Assertions.assertThrows(UnknownTopicOrPartitionException.class, () -> WorkerUtils.verifyTopics((Logger)log, (Admin)this.adminClient, Collections.singleton(TEST_TOPIC), Collections.singletonMap(TEST_TOPIC, NEW_TEST_TOPIC), (int)2, (long)1L));
    }
}

