/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.snapshot;

import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.Iterator;
import java.util.Optional;
import java.util.stream.IntStream;
import org.apache.kafka.common.record.CompressionType;
import org.apache.kafka.common.record.MemoryRecords;
import org.apache.kafka.common.record.Record;
import org.apache.kafka.common.record.RecordBatch;
import org.apache.kafka.common.record.SimpleRecord;
import org.apache.kafka.common.record.UnalignedFileRecords;
import org.apache.kafka.common.record.UnalignedMemoryRecords;
import org.apache.kafka.common.record.UnalignedRecords;
import org.apache.kafka.common.utils.BufferSupplier;
import org.apache.kafka.common.utils.CloseableIterator;
import org.apache.kafka.common.utils.Utils;
import org.apache.kafka.raft.OffsetAndEpoch;
import org.apache.kafka.snapshot.FileRawSnapshotReader;
import org.apache.kafka.snapshot.FileRawSnapshotWriter;
import org.apache.kafka.snapshot.Snapshots;
import org.apache.kafka.test.TestUtils;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

public final class FileRawSnapshotTest {
    private Path tempDir = null;

    @BeforeEach
    public void setUp() {
        this.tempDir = TestUtils.tempDirectory().toPath();
    }

    @AfterEach
    public void tearDown() throws IOException {
        Utils.delete((File)this.tempDir.toFile());
    }

    @Test
    public void testWritingSnapshot() throws IOException {
        OffsetAndEpoch offsetAndEpoch = new OffsetAndEpoch(10L, 3);
        int bufferSize = 256;
        int numberOfBatches = 10;
        int expectedSize = 0;
        try (FileRawSnapshotWriter snapshot = FileRawSnapshotTest.createSnapshotWriter(this.tempDir, offsetAndEpoch);){
            Assertions.assertEquals((long)0L, (long)snapshot.sizeInBytes());
            UnalignedMemoryRecords records = FileRawSnapshotTest.buildRecords(ByteBuffer.wrap(FileRawSnapshotTest.randomBytes(bufferSize)));
            for (int i = 0; i < numberOfBatches; ++i) {
                snapshot.append(records);
                expectedSize += records.sizeInBytes();
            }
            Assertions.assertEquals((long)expectedSize, (long)snapshot.sizeInBytes());
            snapshot.freeze();
        }
        Assertions.assertTrue((boolean)Files.exists(Snapshots.snapshotPath((Path)this.tempDir, (OffsetAndEpoch)offsetAndEpoch), new LinkOption[0]));
        Assertions.assertEquals((long)expectedSize, (long)Files.size(Snapshots.snapshotPath((Path)this.tempDir, (OffsetAndEpoch)offsetAndEpoch)));
    }

    @Test
    public void testWriteReadSnapshot() throws IOException {
        OffsetAndEpoch offsetAndEpoch = new OffsetAndEpoch(10L, 3);
        int bufferSize = 256;
        int numberOfBatches = 10;
        ByteBuffer expectedBuffer = ByteBuffer.wrap(FileRawSnapshotTest.randomBytes(bufferSize));
        try (FileRawSnapshotWriter snapshot = FileRawSnapshotTest.createSnapshotWriter(this.tempDir, offsetAndEpoch);){
            UnalignedMemoryRecords records = FileRawSnapshotTest.buildRecords(expectedBuffer);
            for (int i = 0; i < numberOfBatches; ++i) {
                snapshot.append(records);
            }
            snapshot.freeze();
        }
        snapshot = FileRawSnapshotReader.open((Path)this.tempDir, (OffsetAndEpoch)offsetAndEpoch);
        try {
            int countBatches = 0;
            int countRecords = 0;
            Iterator batches = Utils.covariantCast((Iterator)snapshot.records().batchIterator());
            while (batches.hasNext()) {
                RecordBatch batch = (RecordBatch)batches.next();
                ++countBatches;
                CloseableIterator records = batch.streamingIterator((BufferSupplier)new BufferSupplier.GrowableBufferSupplier());
                while (records.hasNext()) {
                    Record record = (Record)records.next();
                    ++countRecords;
                    Assertions.assertFalse((boolean)record.hasKey());
                    Assertions.assertTrue((boolean)record.hasValue());
                    Assertions.assertEquals((int)bufferSize, (int)record.value().remaining());
                    Assertions.assertEquals((Object)expectedBuffer, (Object)record.value());
                }
            }
            Assertions.assertEquals((int)numberOfBatches, (int)countBatches);
            Assertions.assertEquals((int)numberOfBatches, (int)countRecords);
        }
        finally {
            if (snapshot != null) {
                snapshot.close();
            }
        }
    }

    @Test
    public void testPartialWriteReadSnapshot() throws IOException {
        Path tempDir = TestUtils.tempDirectory().toPath();
        OffsetAndEpoch offsetAndEpoch = new OffsetAndEpoch(10L, 3);
        ByteBuffer records = FileRawSnapshotTest.buildRecords(ByteBuffer.wrap(Utils.utf8((String)"foo"))).buffer();
        ByteBuffer expectedBuffer = ByteBuffer.wrap(records.array());
        ByteBuffer buffer1 = expectedBuffer.duplicate();
        buffer1.position(0);
        buffer1.limit(expectedBuffer.limit() / 2);
        ByteBuffer buffer2 = expectedBuffer.duplicate();
        buffer2.position(expectedBuffer.limit() / 2);
        buffer2.limit(expectedBuffer.limit());
        try (FileRawSnapshotWriter snapshot = FileRawSnapshotTest.createSnapshotWriter(tempDir, offsetAndEpoch);){
            snapshot.append(new UnalignedMemoryRecords(buffer1));
            snapshot.append(new UnalignedMemoryRecords(buffer2));
            snapshot.freeze();
        }
        snapshot = FileRawSnapshotReader.open((Path)tempDir, (OffsetAndEpoch)offsetAndEpoch);
        try {
            int totalSize = Math.toIntExact(snapshot.sizeInBytes());
            Assertions.assertEquals((int)expectedBuffer.remaining(), (int)totalSize);
            UnalignedFileRecords record1 = (UnalignedFileRecords)snapshot.slice(0L, totalSize / 2);
            UnalignedFileRecords record2 = (UnalignedFileRecords)snapshot.slice((long)(totalSize / 2), totalSize - totalSize / 2);
            Assertions.assertEquals((Object)buffer1, (Object)TestUtils.toBuffer((UnalignedRecords)record1));
            Assertions.assertEquals((Object)buffer2, (Object)TestUtils.toBuffer((UnalignedRecords)record2));
            ByteBuffer readBuffer = ByteBuffer.allocate(record1.sizeInBytes() + record2.sizeInBytes());
            readBuffer.put(TestUtils.toBuffer((UnalignedRecords)record1));
            readBuffer.put(TestUtils.toBuffer((UnalignedRecords)record2));
            readBuffer.flip();
            Assertions.assertEquals((Object)expectedBuffer, (Object)readBuffer);
        }
        finally {
            if (snapshot != null) {
                snapshot.close();
            }
        }
    }

    @Test
    public void testBatchWriteReadSnapshot() throws IOException {
        OffsetAndEpoch offsetAndEpoch = new OffsetAndEpoch(10L, 3);
        int bufferSize = 256;
        int batchSize = 3;
        int numberOfBatches = 10;
        try (FileRawSnapshotWriter snapshot = FileRawSnapshotTest.createSnapshotWriter(this.tempDir, offsetAndEpoch);){
            for (int i = 0; i < numberOfBatches; ++i) {
                ByteBuffer[] buffers = (ByteBuffer[])IntStream.range(0, batchSize).mapToObj(ignore -> ByteBuffer.wrap(FileRawSnapshotTest.randomBytes(bufferSize))).toArray(ByteBuffer[]::new);
                snapshot.append(FileRawSnapshotTest.buildRecords(buffers));
            }
            snapshot.freeze();
        }
        snapshot = FileRawSnapshotReader.open((Path)this.tempDir, (OffsetAndEpoch)offsetAndEpoch);
        try {
            int countBatches = 0;
            int countRecords = 0;
            Iterator batches = Utils.covariantCast((Iterator)snapshot.records().batchIterator());
            while (batches.hasNext()) {
                RecordBatch batch = (RecordBatch)batches.next();
                ++countBatches;
                CloseableIterator records = batch.streamingIterator((BufferSupplier)new BufferSupplier.GrowableBufferSupplier());
                while (records.hasNext()) {
                    Record record = (Record)records.next();
                    ++countRecords;
                    Assertions.assertFalse((boolean)record.hasKey());
                    Assertions.assertTrue((boolean)record.hasValue());
                    Assertions.assertEquals((int)bufferSize, (int)record.value().remaining());
                }
            }
            Assertions.assertEquals((int)numberOfBatches, (int)countBatches);
            Assertions.assertEquals((int)(numberOfBatches * batchSize), (int)countRecords);
        }
        finally {
            if (snapshot != null) {
                snapshot.close();
            }
        }
    }

    @Test
    public void testBufferWriteReadSnapshot() throws IOException {
        OffsetAndEpoch offsetAndEpoch = new OffsetAndEpoch(10L, 3);
        int bufferSize = 256;
        int batchSize = 3;
        int numberOfBatches = 10;
        int expectedSize = 0;
        try (FileRawSnapshotWriter snapshot = FileRawSnapshotTest.createSnapshotWriter(this.tempDir, offsetAndEpoch);){
            for (int i = 0; i < numberOfBatches; ++i) {
                ByteBuffer[] buffers = (ByteBuffer[])IntStream.range(0, batchSize).mapToObj(ignore -> ByteBuffer.wrap(FileRawSnapshotTest.randomBytes(bufferSize))).toArray(ByteBuffer[]::new);
                UnalignedMemoryRecords records = FileRawSnapshotTest.buildRecords(buffers);
                snapshot.append(records);
                expectedSize += records.sizeInBytes();
            }
            Assertions.assertEquals((long)expectedSize, (long)snapshot.sizeInBytes());
            snapshot.freeze();
        }
        Assertions.assertTrue((boolean)Files.exists(Snapshots.snapshotPath((Path)this.tempDir, (OffsetAndEpoch)offsetAndEpoch), new LinkOption[0]));
        Assertions.assertEquals((long)expectedSize, (long)Files.size(Snapshots.snapshotPath((Path)this.tempDir, (OffsetAndEpoch)offsetAndEpoch)));
        snapshot = FileRawSnapshotReader.open((Path)this.tempDir, (OffsetAndEpoch)offsetAndEpoch);
        try {
            int countBatches = 0;
            int countRecords = 0;
            Iterator batches = Utils.covariantCast((Iterator)snapshot.records().batchIterator());
            while (batches.hasNext()) {
                RecordBatch batch = (RecordBatch)batches.next();
                ++countBatches;
                CloseableIterator records = batch.streamingIterator((BufferSupplier)new BufferSupplier.GrowableBufferSupplier());
                while (records.hasNext()) {
                    Record record = (Record)records.next();
                    ++countRecords;
                    Assertions.assertFalse((boolean)record.hasKey());
                    Assertions.assertTrue((boolean)record.hasValue());
                    Assertions.assertEquals((int)bufferSize, (int)record.value().remaining());
                }
            }
            Assertions.assertEquals((int)numberOfBatches, (int)countBatches);
            Assertions.assertEquals((int)(numberOfBatches * batchSize), (int)countRecords);
        }
        finally {
            if (snapshot != null) {
                snapshot.close();
            }
        }
    }

    @Test
    public void testAbortedSnapshot() throws IOException {
        OffsetAndEpoch offsetAndEpoch = new OffsetAndEpoch(20L, 2);
        int bufferSize = 256;
        int numberOfBatches = 10;
        try (FileRawSnapshotWriter snapshot = FileRawSnapshotTest.createSnapshotWriter(this.tempDir, offsetAndEpoch);){
            UnalignedMemoryRecords records = FileRawSnapshotTest.buildRecords(ByteBuffer.wrap(FileRawSnapshotTest.randomBytes(bufferSize)));
            for (int i = 0; i < numberOfBatches; ++i) {
                snapshot.append(records);
            }
        }
        Assertions.assertFalse((boolean)Files.exists(Snapshots.snapshotPath((Path)this.tempDir, (OffsetAndEpoch)offsetAndEpoch), new LinkOption[0]));
        Assertions.assertEquals((long)0L, (long)Files.list(Snapshots.snapshotDir((Path)this.tempDir)).count());
    }

    @Test
    public void testAppendToFrozenSnapshot() throws IOException {
        OffsetAndEpoch offsetAndEpoch = new OffsetAndEpoch(10L, 3);
        int bufferSize = 256;
        int numberOfBatches = 10;
        try (FileRawSnapshotWriter snapshot = FileRawSnapshotTest.createSnapshotWriter(this.tempDir, offsetAndEpoch);){
            UnalignedMemoryRecords records = FileRawSnapshotTest.buildRecords(ByteBuffer.wrap(FileRawSnapshotTest.randomBytes(bufferSize)));
            for (int i = 0; i < numberOfBatches; ++i) {
                snapshot.append(records);
            }
            snapshot.freeze();
            Assertions.assertThrows(RuntimeException.class, () -> snapshot.append(records));
        }
        Assertions.assertTrue((boolean)Files.exists(Snapshots.snapshotPath((Path)this.tempDir, (OffsetAndEpoch)offsetAndEpoch), new LinkOption[0]));
        Assertions.assertTrue((Files.size(Snapshots.snapshotPath((Path)this.tempDir, (OffsetAndEpoch)offsetAndEpoch)) > (long)(bufferSize * numberOfBatches) ? 1 : 0) != 0);
    }

    @Test
    public void testCreateSnapshotWithSameId() {
        int i;
        UnalignedMemoryRecords records;
        OffsetAndEpoch offsetAndEpoch = new OffsetAndEpoch(20L, 2);
        int bufferSize = 256;
        int numberOfBatches = 1;
        try (FileRawSnapshotWriter snapshot = FileRawSnapshotTest.createSnapshotWriter(this.tempDir, offsetAndEpoch);){
            records = FileRawSnapshotTest.buildRecords(ByteBuffer.wrap(FileRawSnapshotTest.randomBytes(bufferSize)));
            for (i = 0; i < numberOfBatches; ++i) {
                snapshot.append(records);
            }
            snapshot.freeze();
        }
        snapshot = FileRawSnapshotTest.createSnapshotWriter(this.tempDir, offsetAndEpoch);
        try {
            records = FileRawSnapshotTest.buildRecords(ByteBuffer.wrap(FileRawSnapshotTest.randomBytes(bufferSize)));
            for (i = 0; i < numberOfBatches; ++i) {
                snapshot.append(records);
            }
            snapshot.freeze();
        }
        finally {
            if (snapshot != null) {
                snapshot.close();
            }
        }
    }

    private static byte[] randomBytes(int size) {
        byte[] array = new byte[size];
        TestUtils.SEEDED_RANDOM.nextBytes(array);
        return array;
    }

    private static UnalignedMemoryRecords buildRecords(ByteBuffer ... buffers) {
        MemoryRecords records = MemoryRecords.withRecords((CompressionType)CompressionType.NONE, (SimpleRecord[])((SimpleRecord[])Arrays.stream(buffers).map(SimpleRecord::new).toArray(SimpleRecord[]::new)));
        return new UnalignedMemoryRecords(records.buffer());
    }

    private static FileRawSnapshotWriter createSnapshotWriter(Path dir, OffsetAndEpoch snapshotId) {
        return FileRawSnapshotWriter.create((Path)dir, (OffsetAndEpoch)snapshotId, Optional.empty());
    }
}

