/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.raft;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.util.Collections;
import java.util.Optional;
import java.util.OptionalInt;
import java.util.OptionalLong;
import java.util.Random;
import java.util.Set;
import org.apache.kafka.common.utils.LogContext;
import org.apache.kafka.common.utils.MockTime;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.common.utils.Utils;
import org.apache.kafka.raft.CandidateState;
import org.apache.kafka.raft.ElectionState;
import org.apache.kafka.raft.FollowerState;
import org.apache.kafka.raft.LeaderState;
import org.apache.kafka.raft.LogOffsetMetadata;
import org.apache.kafka.raft.MockQuorumStateStore;
import org.apache.kafka.raft.MockableRandom;
import org.apache.kafka.raft.OffsetAndEpoch;
import org.apache.kafka.raft.QuorumState;
import org.apache.kafka.raft.QuorumStateStore;
import org.apache.kafka.raft.ResignedState;
import org.apache.kafka.raft.UnattachedState;
import org.apache.kafka.raft.VotedState;
import org.apache.kafka.raft.internals.BatchAccumulator;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;
import org.mockito.Mockito;

public class QuorumStateTest {
    private final int localId = 0;
    private final int logEndEpoch = 0;
    private final MockQuorumStateStore store = new MockQuorumStateStore();
    private final MockTime time = new MockTime();
    private final int electionTimeoutMs = 5000;
    private final int fetchTimeoutMs = 10000;
    private final MockableRandom random = new MockableRandom(1L);
    private BatchAccumulator<?> accumulator = (BatchAccumulator)Mockito.mock(BatchAccumulator.class);

    private QuorumState buildQuorumState(Set<Integer> voters) {
        return this.buildQuorumState(OptionalInt.of(0), voters);
    }

    private QuorumState buildQuorumState(OptionalInt localId, Set<Integer> voters) {
        return new QuorumState(localId, voters, 5000, 10000, (QuorumStateStore)this.store, (Time)this.time, new LogContext(), (Random)this.random);
    }

    @Test
    public void testInitializePrimordialEpoch() throws IOException {
        Set voters = Utils.mkSet((Object[])new Integer[]{0});
        Assertions.assertNull((Object)this.store.readElectionState());
        QuorumState state = this.initializeEmptyState(voters);
        Assertions.assertTrue((boolean)state.isUnattached());
        Assertions.assertEquals((int)0, (int)state.epoch());
        state.transitionToCandidate();
        CandidateState candidateState = state.candidateStateOrThrow();
        Assertions.assertTrue((boolean)candidateState.isVoteGranted());
        Assertions.assertEquals((int)1, (int)candidateState.epoch());
    }

    @Test
    public void testInitializeAsUnattached() throws IOException {
        int node1 = 1;
        int node2 = 2;
        int epoch = 5;
        Set voters = Utils.mkSet((Object[])new Integer[]{0, node1, node2});
        this.store.writeElectionState(ElectionState.withUnknownLeader((int)epoch, (Set)voters));
        int jitterMs = 2500;
        this.random.mockNextInt(jitterMs);
        QuorumState state = this.buildQuorumState(voters);
        state.initialize(new OffsetAndEpoch(0L, 0));
        Assertions.assertTrue((boolean)state.isUnattached());
        UnattachedState unattachedState = state.unattachedStateOrThrow();
        Assertions.assertEquals((int)epoch, (int)unattachedState.epoch());
        Assertions.assertEquals((long)(5000 + jitterMs), (long)unattachedState.remainingElectionTimeMs(this.time.milliseconds()));
    }

    @Test
    public void testInitializeAsFollower() throws IOException {
        int node1 = 1;
        int node2 = 2;
        int epoch = 5;
        Set voters = Utils.mkSet((Object[])new Integer[]{0, node1, node2});
        this.store.writeElectionState(ElectionState.withElectedLeader((int)epoch, (int)node1, (Set)voters));
        QuorumState state = this.buildQuorumState(voters);
        state.initialize(new OffsetAndEpoch(0L, 0));
        Assertions.assertTrue((boolean)state.isFollower());
        Assertions.assertEquals((int)epoch, (int)state.epoch());
        FollowerState followerState = state.followerStateOrThrow();
        Assertions.assertEquals((int)epoch, (int)followerState.epoch());
        Assertions.assertEquals((int)node1, (int)followerState.leaderId());
        Assertions.assertEquals((long)10000L, (long)followerState.remainingFetchTimeMs(this.time.milliseconds()));
    }

    @Test
    public void testInitializeAsVoted() throws IOException {
        int node1 = 1;
        int node2 = 2;
        int epoch = 5;
        Set voters = Utils.mkSet((Object[])new Integer[]{0, node1, node2});
        this.store.writeElectionState(ElectionState.withVotedCandidate((int)epoch, (int)node1, (Set)voters));
        int jitterMs = 2500;
        this.random.mockNextInt(jitterMs);
        QuorumState state = this.buildQuorumState(voters);
        state.initialize(new OffsetAndEpoch(0L, 0));
        Assertions.assertTrue((boolean)state.isVoted());
        Assertions.assertEquals((int)epoch, (int)state.epoch());
        VotedState votedState = state.votedStateOrThrow();
        Assertions.assertEquals((int)epoch, (int)votedState.epoch());
        Assertions.assertEquals((int)node1, (int)votedState.votedId());
        Assertions.assertEquals((long)(5000 + jitterMs), (long)votedState.remainingElectionTimeMs(this.time.milliseconds()));
    }

    @Test
    public void testInitializeAsResignedCandidate() throws IOException {
        int node1 = 1;
        int node2 = 2;
        int epoch = 5;
        Set voters = Utils.mkSet((Object[])new Integer[]{0, node1, node2});
        ElectionState election = ElectionState.withVotedCandidate((int)epoch, (int)0, (Set)voters);
        this.store.writeElectionState(election);
        int jitterMs = 2500;
        this.random.mockNextInt(jitterMs);
        QuorumState state = this.buildQuorumState(voters);
        state.initialize(new OffsetAndEpoch(0L, 0));
        Assertions.assertTrue((boolean)state.isCandidate());
        Assertions.assertEquals((int)epoch, (int)state.epoch());
        CandidateState candidateState = state.candidateStateOrThrow();
        Assertions.assertEquals((int)epoch, (int)candidateState.epoch());
        Assertions.assertEquals((Object)election, (Object)candidateState.election());
        Assertions.assertEquals((Object)Utils.mkSet((Object[])new Integer[]{node1, node2}), (Object)candidateState.unrecordedVoters());
        Assertions.assertEquals((Object)Utils.mkSet((Object[])new Integer[]{0}), (Object)candidateState.grantingVoters());
        Assertions.assertEquals(Collections.emptySet(), (Object)candidateState.rejectingVoters());
        Assertions.assertEquals((long)(5000 + jitterMs), (long)candidateState.remainingElectionTimeMs(this.time.milliseconds()));
    }

    @Test
    public void testInitializeAsResignedLeader() throws IOException {
        int node1 = 1;
        int node2 = 2;
        int epoch = 5;
        Set voters = Utils.mkSet((Object[])new Integer[]{0, node1, node2});
        ElectionState election = ElectionState.withElectedLeader((int)epoch, (int)0, (Set)voters);
        this.store.writeElectionState(election);
        int jitterMs = 2500;
        this.random.mockNextInt(jitterMs);
        QuorumState state = this.buildQuorumState(voters);
        state.initialize(new OffsetAndEpoch(0L, 0));
        Assertions.assertFalse((boolean)state.isLeader());
        Assertions.assertEquals((int)epoch, (int)state.epoch());
        ResignedState resignedState = state.resignedStateOrThrow();
        Assertions.assertEquals((int)epoch, (int)resignedState.epoch());
        Assertions.assertEquals((Object)election, (Object)resignedState.election());
        Assertions.assertEquals((Object)Utils.mkSet((Object[])new Integer[]{node1, node2}), (Object)resignedState.unackedVoters());
        Assertions.assertEquals((long)(5000 + jitterMs), (long)resignedState.remainingElectionTimeMs(this.time.milliseconds()));
    }

    @Test
    public void testCandidateToCandidate() throws IOException {
        int node1 = 1;
        int node2 = 2;
        Set voters = Utils.mkSet((Object[])new Integer[]{0, node1, node2});
        Assertions.assertNull((Object)this.store.readElectionState());
        QuorumState state = this.initializeEmptyState(voters);
        state.transitionToCandidate();
        Assertions.assertTrue((boolean)state.isCandidate());
        Assertions.assertEquals((int)1, (int)state.epoch());
        CandidateState candidate1 = state.candidateStateOrThrow();
        candidate1.recordRejectedVote(node2);
        int backoffMs = 500;
        candidate1.startBackingOff(this.time.milliseconds(), (long)backoffMs);
        Assertions.assertTrue((boolean)candidate1.isBackingOff());
        Assertions.assertFalse((boolean)candidate1.isBackoffComplete(this.time.milliseconds()));
        this.time.sleep((long)(backoffMs - 1));
        Assertions.assertTrue((boolean)candidate1.isBackingOff());
        Assertions.assertFalse((boolean)candidate1.isBackoffComplete(this.time.milliseconds()));
        this.time.sleep(1L);
        Assertions.assertTrue((boolean)candidate1.isBackingOff());
        Assertions.assertTrue((boolean)candidate1.isBackoffComplete(this.time.milliseconds()));
        int jitterMs = 2500;
        this.random.mockNextInt(jitterMs);
        state.transitionToCandidate();
        Assertions.assertTrue((boolean)state.isCandidate());
        CandidateState candidate2 = state.candidateStateOrThrow();
        Assertions.assertEquals((int)2, (int)state.epoch());
        Assertions.assertEquals(Collections.singleton(0), (Object)candidate2.grantingVoters());
        Assertions.assertEquals(Collections.emptySet(), (Object)candidate2.rejectingVoters());
        Assertions.assertEquals((long)(5000 + jitterMs), (long)candidate2.remainingElectionTimeMs(this.time.milliseconds()));
    }

    @Test
    public void testCandidateToResigned() throws IOException {
        int node1 = 1;
        int node2 = 2;
        Set voters = Utils.mkSet((Object[])new Integer[]{0, node1, node2});
        Assertions.assertNull((Object)this.store.readElectionState());
        QuorumState state = this.initializeEmptyState(voters);
        state.transitionToCandidate();
        Assertions.assertTrue((boolean)state.isCandidate());
        Assertions.assertEquals((int)1, (int)state.epoch());
        Assertions.assertThrows(IllegalStateException.class, () -> state.transitionToResigned(Collections.singletonList(0)));
        Assertions.assertTrue((boolean)state.isCandidate());
    }

    @Test
    public void testCandidateToLeader() throws IOException {
        Set voters = Utils.mkSet((Object[])new Integer[]{0});
        Assertions.assertNull((Object)this.store.readElectionState());
        QuorumState state = this.initializeEmptyState(voters);
        state.transitionToCandidate();
        Assertions.assertTrue((boolean)state.isCandidate());
        Assertions.assertEquals((int)1, (int)state.epoch());
        state.transitionToLeader(0L, this.accumulator);
        LeaderState leaderState = state.leaderStateOrThrow();
        Assertions.assertTrue((boolean)state.isLeader());
        Assertions.assertEquals((int)1, (int)leaderState.epoch());
        Assertions.assertEquals(Optional.empty(), (Object)leaderState.highWatermark());
    }

    @Test
    public void testCandidateToLeaderWithoutGrantedVote() throws IOException {
        int otherNodeId = 1;
        Set voters = Utils.mkSet((Object[])new Integer[]{0, otherNodeId});
        QuorumState state = this.initializeEmptyState(voters);
        state.initialize(new OffsetAndEpoch(0L, 0));
        state.transitionToCandidate();
        Assertions.assertFalse((boolean)state.candidateStateOrThrow().isVoteGranted());
        Assertions.assertThrows(IllegalStateException.class, () -> state.transitionToLeader(0L, this.accumulator));
        state.candidateStateOrThrow().recordGrantedVote(otherNodeId);
        Assertions.assertTrue((boolean)state.candidateStateOrThrow().isVoteGranted());
        state.transitionToLeader(0L, this.accumulator);
        Assertions.assertTrue((boolean)state.isLeader());
    }

    @Test
    public void testCandidateToFollower() throws IOException {
        int otherNodeId = 1;
        Set voters = Utils.mkSet((Object[])new Integer[]{0, otherNodeId});
        QuorumState state = this.initializeEmptyState(voters);
        state.initialize(new OffsetAndEpoch(0L, 0));
        state.transitionToCandidate();
        state.transitionToFollower(5, otherNodeId);
        Assertions.assertEquals((int)5, (int)state.epoch());
        Assertions.assertEquals((Object)OptionalInt.of(otherNodeId), (Object)state.leaderId());
        Assertions.assertEquals((Object)ElectionState.withElectedLeader((int)5, (int)otherNodeId, (Set)voters), (Object)this.store.readElectionState());
    }

    @Test
    public void testCandidateToUnattached() throws IOException {
        int otherNodeId = 1;
        Set voters = Utils.mkSet((Object[])new Integer[]{0, otherNodeId});
        QuorumState state = this.initializeEmptyState(voters);
        state.initialize(new OffsetAndEpoch(0L, 0));
        state.transitionToCandidate();
        state.transitionToUnattached(5);
        Assertions.assertEquals((int)5, (int)state.epoch());
        Assertions.assertEquals((Object)OptionalInt.empty(), (Object)state.leaderId());
        Assertions.assertEquals((Object)ElectionState.withUnknownLeader((int)5, (Set)voters), (Object)this.store.readElectionState());
    }

    @Test
    public void testCandidateToVoted() throws IOException {
        int otherNodeId = 1;
        Set voters = Utils.mkSet((Object[])new Integer[]{0, otherNodeId});
        QuorumState state = this.initializeEmptyState(voters);
        state.initialize(new OffsetAndEpoch(0L, 0));
        state.transitionToCandidate();
        state.transitionToVoted(5, otherNodeId);
        Assertions.assertEquals((int)5, (int)state.epoch());
        Assertions.assertEquals((Object)OptionalInt.empty(), (Object)state.leaderId());
        VotedState followerState = state.votedStateOrThrow();
        Assertions.assertEquals((int)otherNodeId, (int)followerState.votedId());
        Assertions.assertEquals((Object)ElectionState.withVotedCandidate((int)5, (int)otherNodeId, (Set)voters), (Object)this.store.readElectionState());
    }

    @Test
    public void testCandidateToAnyStateLowerEpoch() throws IOException {
        int otherNodeId = 1;
        Set voters = Utils.mkSet((Object[])new Integer[]{0, otherNodeId});
        QuorumState state = this.initializeEmptyState(voters);
        state.initialize(new OffsetAndEpoch(0L, 0));
        state.transitionToUnattached(5);
        state.transitionToCandidate();
        Assertions.assertThrows(IllegalStateException.class, () -> state.transitionToUnattached(4));
        Assertions.assertThrows(IllegalStateException.class, () -> state.transitionToVoted(4, otherNodeId));
        Assertions.assertThrows(IllegalStateException.class, () -> state.transitionToFollower(4, otherNodeId));
        Assertions.assertEquals((int)6, (int)state.epoch());
        Assertions.assertEquals((Object)ElectionState.withVotedCandidate((int)6, (int)0, (Set)voters), (Object)this.store.readElectionState());
    }

    @Test
    public void testLeaderToLeader() throws IOException {
        Set voters = Utils.mkSet((Object[])new Integer[]{0});
        Assertions.assertNull((Object)this.store.readElectionState());
        QuorumState state = this.initializeEmptyState(voters);
        state.initialize(new OffsetAndEpoch(0L, 0));
        state.transitionToCandidate();
        state.transitionToLeader(0L, this.accumulator);
        Assertions.assertTrue((boolean)state.isLeader());
        Assertions.assertEquals((int)1, (int)state.epoch());
        Assertions.assertThrows(IllegalStateException.class, () -> state.transitionToLeader(0L, this.accumulator));
        Assertions.assertTrue((boolean)state.isLeader());
        Assertions.assertEquals((int)1, (int)state.epoch());
    }

    @Test
    public void testLeaderToResigned() throws IOException {
        Set voters = Utils.mkSet((Object[])new Integer[]{0});
        Assertions.assertNull((Object)this.store.readElectionState());
        QuorumState state = this.initializeEmptyState(voters);
        state.initialize(new OffsetAndEpoch(0L, 0));
        state.transitionToCandidate();
        state.transitionToLeader(0L, this.accumulator);
        Assertions.assertTrue((boolean)state.isLeader());
        Assertions.assertEquals((int)1, (int)state.epoch());
        state.transitionToResigned(Collections.singletonList(0));
        Assertions.assertTrue((boolean)state.isResigned());
        ResignedState resignedState = state.resignedStateOrThrow();
        Assertions.assertEquals((Object)ElectionState.withElectedLeader((int)1, (int)0, (Set)voters), (Object)resignedState.election());
        Assertions.assertEquals((int)1, (int)resignedState.epoch());
        Assertions.assertEquals(Collections.emptySet(), (Object)resignedState.unackedVoters());
    }

    @Test
    public void testLeaderToCandidate() throws IOException {
        Set voters = Utils.mkSet((Object[])new Integer[]{0});
        Assertions.assertNull((Object)this.store.readElectionState());
        QuorumState state = this.initializeEmptyState(voters);
        state.initialize(new OffsetAndEpoch(0L, 0));
        state.transitionToCandidate();
        state.transitionToLeader(0L, this.accumulator);
        Assertions.assertTrue((boolean)state.isLeader());
        Assertions.assertEquals((int)1, (int)state.epoch());
        Assertions.assertThrows(IllegalStateException.class, () -> ((QuorumState)state).transitionToCandidate());
        Assertions.assertTrue((boolean)state.isLeader());
        Assertions.assertEquals((int)1, (int)state.epoch());
    }

    @Test
    public void testLeaderToFollower() throws IOException {
        int otherNodeId = 1;
        Set voters = Utils.mkSet((Object[])new Integer[]{0, otherNodeId});
        QuorumState state = this.initializeEmptyState(voters);
        state.transitionToCandidate();
        state.candidateStateOrThrow().recordGrantedVote(otherNodeId);
        state.transitionToLeader(0L, this.accumulator);
        state.transitionToFollower(5, otherNodeId);
        Assertions.assertEquals((int)5, (int)state.epoch());
        Assertions.assertEquals((Object)OptionalInt.of(otherNodeId), (Object)state.leaderId());
        Assertions.assertEquals((Object)ElectionState.withElectedLeader((int)5, (int)otherNodeId, (Set)voters), (Object)this.store.readElectionState());
    }

    @Test
    public void testLeaderToUnattached() throws IOException {
        int otherNodeId = 1;
        Set voters = Utils.mkSet((Object[])new Integer[]{0, otherNodeId});
        QuorumState state = this.initializeEmptyState(voters);
        state.initialize(new OffsetAndEpoch(0L, 0));
        state.transitionToCandidate();
        state.candidateStateOrThrow().recordGrantedVote(otherNodeId);
        state.transitionToLeader(0L, this.accumulator);
        state.transitionToUnattached(5);
        Assertions.assertEquals((int)5, (int)state.epoch());
        Assertions.assertEquals((Object)OptionalInt.empty(), (Object)state.leaderId());
        Assertions.assertEquals((Object)ElectionState.withUnknownLeader((int)5, (Set)voters), (Object)this.store.readElectionState());
    }

    @Test
    public void testLeaderToVoted() throws IOException {
        int otherNodeId = 1;
        Set voters = Utils.mkSet((Object[])new Integer[]{0, otherNodeId});
        QuorumState state = this.initializeEmptyState(voters);
        state.initialize(new OffsetAndEpoch(0L, 0));
        state.transitionToCandidate();
        state.candidateStateOrThrow().recordGrantedVote(otherNodeId);
        state.transitionToLeader(0L, this.accumulator);
        state.transitionToVoted(5, otherNodeId);
        Assertions.assertEquals((int)5, (int)state.epoch());
        Assertions.assertEquals((Object)OptionalInt.empty(), (Object)state.leaderId());
        VotedState votedState = state.votedStateOrThrow();
        Assertions.assertEquals((int)otherNodeId, (int)votedState.votedId());
        Assertions.assertEquals((Object)ElectionState.withVotedCandidate((int)5, (int)otherNodeId, (Set)voters), (Object)this.store.readElectionState());
    }

    @Test
    public void testLeaderToAnyStateLowerEpoch() throws IOException {
        int otherNodeId = 1;
        Set voters = Utils.mkSet((Object[])new Integer[]{0, otherNodeId});
        QuorumState state = this.initializeEmptyState(voters);
        state.initialize(new OffsetAndEpoch(0L, 0));
        state.transitionToUnattached(5);
        state.transitionToCandidate();
        state.candidateStateOrThrow().recordGrantedVote(otherNodeId);
        state.transitionToLeader(0L, this.accumulator);
        Assertions.assertThrows(IllegalStateException.class, () -> state.transitionToUnattached(4));
        Assertions.assertThrows(IllegalStateException.class, () -> state.transitionToVoted(4, otherNodeId));
        Assertions.assertThrows(IllegalStateException.class, () -> state.transitionToFollower(4, otherNodeId));
        Assertions.assertEquals((int)6, (int)state.epoch());
        Assertions.assertEquals((Object)ElectionState.withElectedLeader((int)6, (int)0, (Set)voters), (Object)this.store.readElectionState());
    }

    @Test
    public void testCannotFollowOrVoteForSelf() throws IOException {
        Set voters = Utils.mkSet((Object[])new Integer[]{0});
        Assertions.assertNull((Object)this.store.readElectionState());
        QuorumState state = this.initializeEmptyState(voters);
        Assertions.assertThrows(IllegalStateException.class, () -> state.transitionToFollower(0, 0));
        Assertions.assertThrows(IllegalStateException.class, () -> state.transitionToVoted(0, 0));
    }

    @Test
    public void testUnattachedToLeaderOrResigned() throws IOException {
        int leaderId = 1;
        int epoch = 5;
        Set voters = Utils.mkSet((Object[])new Integer[]{0, leaderId});
        this.store.writeElectionState(ElectionState.withVotedCandidate((int)epoch, (int)leaderId, (Set)voters));
        QuorumState state = this.initializeEmptyState(voters);
        state.initialize(new OffsetAndEpoch(0L, 0));
        Assertions.assertTrue((boolean)state.isUnattached());
        Assertions.assertThrows(IllegalStateException.class, () -> state.transitionToLeader(0L, this.accumulator));
        Assertions.assertThrows(IllegalStateException.class, () -> state.transitionToResigned(Collections.emptyList()));
    }

    @Test
    public void testUnattachedToVotedSameEpoch() throws IOException {
        int otherNodeId = 1;
        Set voters = Utils.mkSet((Object[])new Integer[]{0, otherNodeId});
        QuorumState state = this.initializeEmptyState(voters);
        state.initialize(new OffsetAndEpoch(0L, 0));
        state.transitionToUnattached(5);
        int jitterMs = 2500;
        this.random.mockNextInt(5000, jitterMs);
        state.transitionToVoted(5, otherNodeId);
        VotedState votedState = state.votedStateOrThrow();
        Assertions.assertEquals((int)5, (int)votedState.epoch());
        Assertions.assertEquals((int)otherNodeId, (int)votedState.votedId());
        Assertions.assertEquals((Object)ElectionState.withVotedCandidate((int)5, (int)otherNodeId, (Set)voters), (Object)this.store.readElectionState());
        Assertions.assertEquals((long)(5000 + jitterMs), (long)votedState.remainingElectionTimeMs(this.time.milliseconds()));
    }

    @Test
    public void testUnattachedToVotedHigherEpoch() throws IOException {
        int otherNodeId = 1;
        Set voters = Utils.mkSet((Object[])new Integer[]{0, otherNodeId});
        QuorumState state = this.initializeEmptyState(voters);
        state.initialize(new OffsetAndEpoch(0L, 0));
        state.transitionToUnattached(5);
        state.transitionToVoted(8, otherNodeId);
        VotedState votedState = state.votedStateOrThrow();
        Assertions.assertEquals((int)8, (int)votedState.epoch());
        Assertions.assertEquals((int)otherNodeId, (int)votedState.votedId());
        Assertions.assertEquals((Object)ElectionState.withVotedCandidate((int)8, (int)otherNodeId, (Set)voters), (Object)this.store.readElectionState());
    }

    @Test
    public void testUnattachedToCandidate() throws IOException {
        int otherNodeId = 1;
        Set voters = Utils.mkSet((Object[])new Integer[]{0, otherNodeId});
        QuorumState state = this.initializeEmptyState(voters);
        state.initialize(new OffsetAndEpoch(0L, 0));
        state.transitionToUnattached(5);
        int jitterMs = 2500;
        this.random.mockNextInt(5000, jitterMs);
        state.transitionToCandidate();
        Assertions.assertTrue((boolean)state.isCandidate());
        CandidateState candidateState = state.candidateStateOrThrow();
        Assertions.assertEquals((int)6, (int)candidateState.epoch());
        Assertions.assertEquals((long)(5000 + jitterMs), (long)candidateState.remainingElectionTimeMs(this.time.milliseconds()));
    }

    @Test
    public void testUnattachedToUnattached() throws IOException {
        int otherNodeId = 1;
        Set voters = Utils.mkSet((Object[])new Integer[]{0, otherNodeId});
        QuorumState state = this.initializeEmptyState(voters);
        state.initialize(new OffsetAndEpoch(0L, 0));
        state.transitionToUnattached(5);
        long remainingElectionTimeMs = state.unattachedStateOrThrow().remainingElectionTimeMs(this.time.milliseconds());
        this.time.sleep(1000L);
        state.transitionToUnattached(6);
        UnattachedState unattachedState = state.unattachedStateOrThrow();
        Assertions.assertEquals((int)6, (int)unattachedState.epoch());
        Assertions.assertEquals((long)(remainingElectionTimeMs - 1000L), (long)unattachedState.remainingElectionTimeMs(this.time.milliseconds()));
    }

    @Test
    public void testUnattachedToFollowerSameEpoch() throws IOException {
        int otherNodeId = 1;
        Set voters = Utils.mkSet((Object[])new Integer[]{0, otherNodeId});
        QuorumState state = this.initializeEmptyState(voters);
        state.initialize(new OffsetAndEpoch(0L, 0));
        state.transitionToUnattached(5);
        state.transitionToFollower(5, otherNodeId);
        Assertions.assertTrue((boolean)state.isFollower());
        FollowerState followerState = state.followerStateOrThrow();
        Assertions.assertEquals((int)5, (int)followerState.epoch());
        Assertions.assertEquals((int)otherNodeId, (int)followerState.leaderId());
        Assertions.assertEquals((long)10000L, (long)followerState.remainingFetchTimeMs(this.time.milliseconds()));
    }

    @Test
    public void testUnattachedToFollowerHigherEpoch() throws IOException {
        int otherNodeId = 1;
        Set voters = Utils.mkSet((Object[])new Integer[]{0, otherNodeId});
        QuorumState state = this.initializeEmptyState(voters);
        state.initialize(new OffsetAndEpoch(0L, 0));
        state.transitionToUnattached(5);
        state.transitionToFollower(8, otherNodeId);
        Assertions.assertTrue((boolean)state.isFollower());
        FollowerState followerState = state.followerStateOrThrow();
        Assertions.assertEquals((int)8, (int)followerState.epoch());
        Assertions.assertEquals((int)otherNodeId, (int)followerState.leaderId());
        Assertions.assertEquals((long)10000L, (long)followerState.remainingFetchTimeMs(this.time.milliseconds()));
    }

    @Test
    public void testUnattachedToAnyStateLowerEpoch() throws IOException {
        int otherNodeId = 1;
        Set voters = Utils.mkSet((Object[])new Integer[]{0, otherNodeId});
        QuorumState state = this.initializeEmptyState(voters);
        state.initialize(new OffsetAndEpoch(0L, 0));
        state.transitionToUnattached(5);
        Assertions.assertThrows(IllegalStateException.class, () -> state.transitionToUnattached(4));
        Assertions.assertThrows(IllegalStateException.class, () -> state.transitionToVoted(4, otherNodeId));
        Assertions.assertThrows(IllegalStateException.class, () -> state.transitionToFollower(4, otherNodeId));
        Assertions.assertEquals((int)5, (int)state.epoch());
        Assertions.assertEquals((Object)ElectionState.withUnknownLeader((int)5, (Set)voters), (Object)this.store.readElectionState());
    }

    @Test
    public void testVotedToInvalidLeaderOrResigned() throws IOException {
        int node1 = 1;
        int node2 = 2;
        Set voters = Utils.mkSet((Object[])new Integer[]{0, node1, node2});
        QuorumState state = this.initializeEmptyState(voters);
        state.initialize(new OffsetAndEpoch(0L, 0));
        state.transitionToVoted(5, node1);
        Assertions.assertThrows(IllegalStateException.class, () -> state.transitionToLeader(0L, this.accumulator));
        Assertions.assertThrows(IllegalStateException.class, () -> state.transitionToResigned(Collections.emptyList()));
    }

    @Test
    public void testVotedToCandidate() throws IOException {
        int node1 = 1;
        int node2 = 2;
        Set voters = Utils.mkSet((Object[])new Integer[]{0, node1, node2});
        QuorumState state = this.initializeEmptyState(voters);
        state.initialize(new OffsetAndEpoch(0L, 0));
        state.transitionToVoted(5, node1);
        int jitterMs = 2500;
        this.random.mockNextInt(5000, jitterMs);
        state.transitionToCandidate();
        Assertions.assertTrue((boolean)state.isCandidate());
        CandidateState candidateState = state.candidateStateOrThrow();
        Assertions.assertEquals((int)6, (int)candidateState.epoch());
        Assertions.assertEquals((long)(5000 + jitterMs), (long)candidateState.remainingElectionTimeMs(this.time.milliseconds()));
    }

    @Test
    public void testVotedToVotedSameEpoch() throws IOException {
        int node1 = 1;
        int node2 = 2;
        Set voters = Utils.mkSet((Object[])new Integer[]{0, node1, node2});
        QuorumState state = this.initializeEmptyState(voters);
        state.initialize(new OffsetAndEpoch(0L, 0));
        state.transitionToUnattached(5);
        state.transitionToVoted(8, node1);
        Assertions.assertThrows(IllegalStateException.class, () -> state.transitionToVoted(8, node1));
        Assertions.assertThrows(IllegalStateException.class, () -> state.transitionToVoted(8, node2));
    }

    @Test
    public void testVotedToFollowerSameEpoch() throws IOException {
        int node1 = 1;
        int node2 = 2;
        Set voters = Utils.mkSet((Object[])new Integer[]{0, node1, node2});
        QuorumState state = this.initializeEmptyState(voters);
        state.initialize(new OffsetAndEpoch(0L, 0));
        state.transitionToVoted(5, node1);
        state.transitionToFollower(5, node2);
        FollowerState followerState = state.followerStateOrThrow();
        Assertions.assertEquals((int)5, (int)followerState.epoch());
        Assertions.assertEquals((int)node2, (int)followerState.leaderId());
        Assertions.assertEquals((Object)ElectionState.withElectedLeader((int)5, (int)node2, (Set)voters), (Object)this.store.readElectionState());
    }

    @Test
    public void testVotedToFollowerHigherEpoch() throws IOException {
        int node1 = 1;
        int node2 = 2;
        Set voters = Utils.mkSet((Object[])new Integer[]{0, node1, node2});
        QuorumState state = this.initializeEmptyState(voters);
        state.initialize(new OffsetAndEpoch(0L, 0));
        state.transitionToVoted(5, node1);
        state.transitionToFollower(8, node2);
        FollowerState followerState = state.followerStateOrThrow();
        Assertions.assertEquals((int)8, (int)followerState.epoch());
        Assertions.assertEquals((int)node2, (int)followerState.leaderId());
        Assertions.assertEquals((Object)ElectionState.withElectedLeader((int)8, (int)node2, (Set)voters), (Object)this.store.readElectionState());
    }

    @Test
    public void testVotedToUnattachedSameEpoch() throws IOException {
        int node1 = 1;
        int node2 = 2;
        Set voters = Utils.mkSet((Object[])new Integer[]{0, node1, node2});
        QuorumState state = this.initializeEmptyState(voters);
        state.initialize(new OffsetAndEpoch(0L, 0));
        state.transitionToVoted(5, node1);
        Assertions.assertThrows(IllegalStateException.class, () -> state.transitionToUnattached(5));
    }

    @Test
    public void testVotedToUnattachedHigherEpoch() throws IOException {
        int otherNodeId = 1;
        Set voters = Utils.mkSet((Object[])new Integer[]{0, otherNodeId});
        QuorumState state = this.initializeEmptyState(voters);
        state.initialize(new OffsetAndEpoch(0L, 0));
        state.transitionToVoted(5, otherNodeId);
        long remainingElectionTimeMs = state.votedStateOrThrow().remainingElectionTimeMs(this.time.milliseconds());
        this.time.sleep(1000L);
        state.transitionToUnattached(6);
        UnattachedState unattachedState = state.unattachedStateOrThrow();
        Assertions.assertEquals((int)6, (int)unattachedState.epoch());
        Assertions.assertEquals((long)(remainingElectionTimeMs - 1000L), (long)unattachedState.remainingElectionTimeMs(this.time.milliseconds()));
    }

    @Test
    public void testVotedToAnyStateLowerEpoch() throws IOException {
        int otherNodeId = 1;
        Set voters = Utils.mkSet((Object[])new Integer[]{0, otherNodeId});
        QuorumState state = this.initializeEmptyState(voters);
        state.initialize(new OffsetAndEpoch(0L, 0));
        state.transitionToVoted(5, otherNodeId);
        Assertions.assertThrows(IllegalStateException.class, () -> state.transitionToUnattached(4));
        Assertions.assertThrows(IllegalStateException.class, () -> state.transitionToVoted(4, otherNodeId));
        Assertions.assertThrows(IllegalStateException.class, () -> state.transitionToFollower(4, otherNodeId));
        Assertions.assertEquals((int)5, (int)state.epoch());
        Assertions.assertEquals((Object)ElectionState.withVotedCandidate((int)5, (int)otherNodeId, (Set)voters), (Object)this.store.readElectionState());
    }

    @Test
    public void testFollowerToFollowerSameEpoch() throws IOException {
        int node1 = 1;
        int node2 = 2;
        Set voters = Utils.mkSet((Object[])new Integer[]{0, node1, node2});
        QuorumState state = this.initializeEmptyState(voters);
        state.initialize(new OffsetAndEpoch(0L, 0));
        state.transitionToFollower(8, node2);
        Assertions.assertThrows(IllegalStateException.class, () -> state.transitionToFollower(8, node1));
        Assertions.assertThrows(IllegalStateException.class, () -> state.transitionToFollower(8, node2));
        FollowerState followerState = state.followerStateOrThrow();
        Assertions.assertEquals((int)8, (int)followerState.epoch());
        Assertions.assertEquals((int)node2, (int)followerState.leaderId());
        Assertions.assertEquals((Object)ElectionState.withElectedLeader((int)8, (int)node2, (Set)voters), (Object)this.store.readElectionState());
    }

    @Test
    public void testFollowerToFollowerHigherEpoch() throws IOException {
        int node1 = 1;
        int node2 = 2;
        Set voters = Utils.mkSet((Object[])new Integer[]{0, node1, node2});
        QuorumState state = this.initializeEmptyState(voters);
        state.initialize(new OffsetAndEpoch(0L, 0));
        state.transitionToFollower(8, node2);
        state.transitionToFollower(9, node1);
        FollowerState followerState = state.followerStateOrThrow();
        Assertions.assertEquals((int)9, (int)followerState.epoch());
        Assertions.assertEquals((int)node1, (int)followerState.leaderId());
        Assertions.assertEquals((Object)ElectionState.withElectedLeader((int)9, (int)node1, (Set)voters), (Object)this.store.readElectionState());
    }

    @Test
    public void testFollowerToLeaderOrResigned() throws IOException {
        int node1 = 1;
        int node2 = 2;
        Set voters = Utils.mkSet((Object[])new Integer[]{0, node1, node2});
        QuorumState state = this.initializeEmptyState(voters);
        state.initialize(new OffsetAndEpoch(0L, 0));
        state.transitionToFollower(8, node2);
        Assertions.assertThrows(IllegalStateException.class, () -> state.transitionToLeader(0L, this.accumulator));
        Assertions.assertThrows(IllegalStateException.class, () -> state.transitionToResigned(Collections.emptyList()));
    }

    @Test
    public void testFollowerToCandidate() throws IOException {
        int node1 = 1;
        int node2 = 2;
        Set voters = Utils.mkSet((Object[])new Integer[]{0, node1, node2});
        QuorumState state = this.initializeEmptyState(voters);
        state.initialize(new OffsetAndEpoch(0L, 0));
        state.transitionToFollower(8, node2);
        int jitterMs = 2500;
        this.random.mockNextInt(5000, jitterMs);
        state.transitionToCandidate();
        Assertions.assertTrue((boolean)state.isCandidate());
        CandidateState candidateState = state.candidateStateOrThrow();
        Assertions.assertEquals((int)9, (int)candidateState.epoch());
        Assertions.assertEquals((long)(5000 + jitterMs), (long)candidateState.remainingElectionTimeMs(this.time.milliseconds()));
    }

    @Test
    public void testFollowerToUnattachedSameEpoch() throws IOException {
        int node1 = 1;
        int node2 = 2;
        Set voters = Utils.mkSet((Object[])new Integer[]{0, node1, node2});
        QuorumState state = this.initializeEmptyState(voters);
        state.initialize(new OffsetAndEpoch(0L, 0));
        state.transitionToFollower(8, node2);
        Assertions.assertThrows(IllegalStateException.class, () -> state.transitionToUnattached(8));
    }

    @Test
    public void testFollowerToUnattachedHigherEpoch() throws IOException {
        int node1 = 1;
        int node2 = 2;
        Set voters = Utils.mkSet((Object[])new Integer[]{0, node1, node2});
        QuorumState state = this.initializeEmptyState(voters);
        state.initialize(new OffsetAndEpoch(0L, 0));
        state.transitionToFollower(8, node2);
        int jitterMs = 2500;
        this.random.mockNextInt(5000, jitterMs);
        state.transitionToUnattached(9);
        Assertions.assertTrue((boolean)state.isUnattached());
        UnattachedState unattachedState = state.unattachedStateOrThrow();
        Assertions.assertEquals((int)9, (int)unattachedState.epoch());
        Assertions.assertEquals((long)(5000 + jitterMs), (long)unattachedState.remainingElectionTimeMs(this.time.milliseconds()));
    }

    @Test
    public void testFollowerToVotedSameEpoch() throws IOException {
        int node1 = 1;
        int node2 = 2;
        Set voters = Utils.mkSet((Object[])new Integer[]{0, node1, node2});
        QuorumState state = this.initializeEmptyState(voters);
        state.initialize(new OffsetAndEpoch(0L, 0));
        state.transitionToFollower(8, node2);
        Assertions.assertThrows(IllegalStateException.class, () -> state.transitionToVoted(8, node1));
        Assertions.assertThrows(IllegalStateException.class, () -> state.transitionToVoted(8, 0));
        Assertions.assertThrows(IllegalStateException.class, () -> state.transitionToVoted(8, node2));
    }

    @Test
    public void testFollowerToVotedHigherEpoch() throws IOException {
        int node1 = 1;
        int node2 = 2;
        Set voters = Utils.mkSet((Object[])new Integer[]{0, node1, node2});
        QuorumState state = this.initializeEmptyState(voters);
        state.initialize(new OffsetAndEpoch(0L, 0));
        state.transitionToFollower(8, node2);
        int jitterMs = 2500;
        this.random.mockNextInt(5000, jitterMs);
        state.transitionToVoted(9, node1);
        Assertions.assertTrue((boolean)state.isVoted());
        VotedState votedState = state.votedStateOrThrow();
        Assertions.assertEquals((int)9, (int)votedState.epoch());
        Assertions.assertEquals((int)node1, (int)votedState.votedId());
        Assertions.assertEquals((long)(5000 + jitterMs), (long)votedState.remainingElectionTimeMs(this.time.milliseconds()));
    }

    @Test
    public void testFollowerToAnyStateLowerEpoch() throws IOException {
        int otherNodeId = 1;
        Set voters = Utils.mkSet((Object[])new Integer[]{0, otherNodeId});
        QuorumState state = this.initializeEmptyState(voters);
        state.initialize(new OffsetAndEpoch(0L, 0));
        state.transitionToFollower(5, otherNodeId);
        Assertions.assertThrows(IllegalStateException.class, () -> state.transitionToUnattached(4));
        Assertions.assertThrows(IllegalStateException.class, () -> state.transitionToVoted(4, otherNodeId));
        Assertions.assertThrows(IllegalStateException.class, () -> state.transitionToFollower(4, otherNodeId));
        Assertions.assertEquals((int)5, (int)state.epoch());
        Assertions.assertEquals((Object)ElectionState.withElectedLeader((int)5, (int)otherNodeId, (Set)voters), (Object)this.store.readElectionState());
    }

    @Test
    public void testCannotBecomeFollowerOfNonVoter() throws IOException {
        int otherNodeId = 1;
        int nonVoterId = 2;
        Set voters = Utils.mkSet((Object[])new Integer[]{0, otherNodeId});
        QuorumState state = this.initializeEmptyState(voters);
        state.initialize(new OffsetAndEpoch(0L, 0));
        Assertions.assertThrows(IllegalStateException.class, () -> state.transitionToVoted(4, nonVoterId));
        Assertions.assertThrows(IllegalStateException.class, () -> state.transitionToFollower(4, nonVoterId));
    }

    @Test
    public void testObserverCannotBecomeCandidateOrLeaderOrVoted() throws IOException {
        int otherNodeId = 1;
        Set voters = Utils.mkSet((Object[])new Integer[]{otherNodeId});
        QuorumState state = this.initializeEmptyState(voters);
        state.initialize(new OffsetAndEpoch(0L, 0));
        Assertions.assertTrue((boolean)state.isObserver());
        Assertions.assertThrows(IllegalStateException.class, () -> ((QuorumState)state).transitionToCandidate());
        Assertions.assertThrows(IllegalStateException.class, () -> state.transitionToLeader(0L, this.accumulator));
        Assertions.assertThrows(IllegalStateException.class, () -> state.transitionToVoted(5, otherNodeId));
    }

    @Test
    public void testObserverFollowerToUnattached() throws IOException {
        int node1 = 1;
        int node2 = 2;
        Set voters = Utils.mkSet((Object[])new Integer[]{node1, node2});
        QuorumState state = this.initializeEmptyState(voters);
        state.initialize(new OffsetAndEpoch(0L, 0));
        Assertions.assertTrue((boolean)state.isObserver());
        state.transitionToFollower(2, node1);
        state.transitionToUnattached(3);
        Assertions.assertTrue((boolean)state.isUnattached());
        UnattachedState unattachedState = state.unattachedStateOrThrow();
        Assertions.assertEquals((int)3, (int)unattachedState.epoch());
        Assertions.assertEquals((long)Long.MAX_VALUE, (long)unattachedState.electionTimeoutMs());
    }

    @Test
    public void testObserverUnattachedToFollower() throws IOException {
        int node1 = 1;
        int node2 = 2;
        Set voters = Utils.mkSet((Object[])new Integer[]{node1, node2});
        QuorumState state = this.initializeEmptyState(voters);
        state.initialize(new OffsetAndEpoch(0L, 0));
        Assertions.assertTrue((boolean)state.isObserver());
        state.transitionToUnattached(2);
        state.transitionToFollower(3, node1);
        Assertions.assertTrue((boolean)state.isFollower());
        FollowerState followerState = state.followerStateOrThrow();
        Assertions.assertEquals((int)3, (int)followerState.epoch());
        Assertions.assertEquals((int)node1, (int)followerState.leaderId());
        Assertions.assertEquals((long)10000L, (long)followerState.remainingFetchTimeMs(this.time.milliseconds()));
    }

    @Test
    public void testInitializeWithCorruptedStore() {
        QuorumStateStore stateStore = (QuorumStateStore)Mockito.mock(QuorumStateStore.class);
        ((QuorumStateStore)Mockito.doThrow(UncheckedIOException.class).when((Object)stateStore)).readElectionState();
        QuorumState state = this.buildQuorumState(Utils.mkSet((Object[])new Integer[]{0}));
        int epoch = 2;
        state.initialize(new OffsetAndEpoch(0L, epoch));
        Assertions.assertEquals((int)epoch, (int)state.epoch());
        Assertions.assertTrue((boolean)state.isUnattached());
        Assertions.assertFalse((boolean)state.hasLeader());
    }

    @Test
    public void testInconsistentVotersBetweenConfigAndState() throws IOException {
        int otherNodeId = 1;
        Set voters = Utils.mkSet((Object[])new Integer[]{0, otherNodeId});
        QuorumState state = this.initializeEmptyState(voters);
        int unknownVoterId = 2;
        Set stateVoters = Utils.mkSet((Object[])new Integer[]{0, otherNodeId, unknownVoterId});
        int epoch = 5;
        this.store.writeElectionState(ElectionState.withElectedLeader((int)epoch, (int)0, (Set)stateVoters));
        Assertions.assertThrows(IllegalStateException.class, () -> state.initialize(new OffsetAndEpoch(0L, 0)));
    }

    @Test
    public void testHasRemoteLeader() throws IOException {
        int otherNodeId = 1;
        Set voters = Utils.mkSet((Object[])new Integer[]{0, otherNodeId});
        QuorumState state = this.initializeEmptyState(voters);
        Assertions.assertFalse((boolean)state.hasRemoteLeader());
        state.transitionToCandidate();
        Assertions.assertFalse((boolean)state.hasRemoteLeader());
        state.candidateStateOrThrow().recordGrantedVote(otherNodeId);
        state.transitionToLeader(0L, this.accumulator);
        Assertions.assertFalse((boolean)state.hasRemoteLeader());
        state.transitionToUnattached(state.epoch() + 1);
        Assertions.assertFalse((boolean)state.hasRemoteLeader());
        state.transitionToVoted(state.epoch() + 1, otherNodeId);
        Assertions.assertFalse((boolean)state.hasRemoteLeader());
        state.transitionToFollower(state.epoch() + 1, otherNodeId);
        Assertions.assertTrue((boolean)state.hasRemoteLeader());
    }

    @Test
    public void testHighWatermarkRetained() throws IOException {
        int otherNodeId = 1;
        Set voters = Utils.mkSet((Object[])new Integer[]{0, otherNodeId});
        QuorumState state = this.initializeEmptyState(voters);
        state.transitionToFollower(5, otherNodeId);
        FollowerState followerState = state.followerStateOrThrow();
        followerState.updateHighWatermark(OptionalLong.of(10L));
        Optional<LogOffsetMetadata> highWatermark = Optional.of(new LogOffsetMetadata(10L));
        Assertions.assertEquals(highWatermark, (Object)state.highWatermark());
        state.transitionToUnattached(6);
        Assertions.assertEquals(highWatermark, (Object)state.highWatermark());
        state.transitionToVoted(7, otherNodeId);
        Assertions.assertEquals(highWatermark, (Object)state.highWatermark());
        state.transitionToCandidate();
        Assertions.assertEquals(highWatermark, (Object)state.highWatermark());
        CandidateState candidateState = state.candidateStateOrThrow();
        candidateState.recordGrantedVote(otherNodeId);
        Assertions.assertTrue((boolean)candidateState.isVoteGranted());
        state.transitionToLeader(10L, this.accumulator);
        Assertions.assertEquals(Optional.empty(), (Object)state.highWatermark());
    }

    @Test
    public void testInitializeWithEmptyLocalId() throws IOException {
        QuorumState state = this.buildQuorumState(OptionalInt.empty(), Utils.mkSet((Object[])new Integer[]{0, 1}));
        state.initialize(new OffsetAndEpoch(0L, 0));
        Assertions.assertTrue((boolean)state.isObserver());
        Assertions.assertFalse((boolean)state.isVoter());
        Assertions.assertThrows(IllegalStateException.class, () -> ((QuorumState)state).transitionToCandidate());
        Assertions.assertThrows(IllegalStateException.class, () -> state.transitionToVoted(1, 1));
        Assertions.assertThrows(IllegalStateException.class, () -> state.transitionToLeader(0L, this.accumulator));
        state.transitionToFollower(1, 1);
        Assertions.assertTrue((boolean)state.isFollower());
        state.transitionToUnattached(2);
        Assertions.assertTrue((boolean)state.isUnattached());
    }

    @Test
    public void testObserverInitializationFailsIfElectionStateHasVotedCandidate() {
        Set voters = Utils.mkSet((Object[])new Integer[]{0, 1});
        int epoch = 5;
        int votedId = 1;
        this.store.writeElectionState(ElectionState.withVotedCandidate((int)epoch, (int)votedId, (Set)voters));
        QuorumState state1 = this.buildQuorumState(OptionalInt.of(2), voters);
        Assertions.assertThrows(IllegalStateException.class, () -> state1.initialize(new OffsetAndEpoch(0L, 0)));
        QuorumState state2 = this.buildQuorumState(OptionalInt.empty(), voters);
        Assertions.assertThrows(IllegalStateException.class, () -> state2.initialize(new OffsetAndEpoch(0L, 0)));
    }

    private QuorumState initializeEmptyState(Set<Integer> voters) throws IOException {
        QuorumState state = this.buildQuorumState(voters);
        this.store.writeElectionState(ElectionState.withUnknownLeader((int)0, voters));
        state.initialize(new OffsetAndEpoch(0L, 0));
        return state;
    }
}

