package org.apache.kafka.clients.mapr;

import java.io.IOException;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.lang.reflect.Method;

public class GenericHFactory<T> {
  protected static final Map<String, Constructor<?>> CONSTRUCTOR_CACHE =
          new ConcurrentHashMap<>();

  @SuppressWarnings("unchecked")
  public static <T> T getImplementorInstance(String className,
                                  Object[] params, 
                                  Class<?>... classes) {
    StringBuilder suffix = new StringBuilder();
    if (classes != null && classes.length > 0) {
      for (Class<?> c : classes) {
        suffix.append("_").append(c.getName());
      }
    }

    try {
      String key = className + suffix;
      Constructor<?> method = CONSTRUCTOR_CACHE.get(key);
      if (method == null) {
        synchronized (CONSTRUCTOR_CACHE) {
          method = CONSTRUCTOR_CACHE.get(key);
          if (method == null) {
            Class<? extends T> clazz = (Class<? extends T>) Class.forName(className);
            method = clazz.getDeclaredConstructor(classes);
            method.setAccessible(true);
            CONSTRUCTOR_CACHE.put(key, method);
          }
        }
      }
      return (T) method.newInstance(params);
    }
    catch (Throwable t) {
      throw new RuntimeException(String.format("Error occurred while instantiating %s.\n==> %s.",
        className, getMessage(t)), t);
    }
  }

  /*
   * This API can be used to invoke a 'static' method - methodName from a given
   * class - className.
   */
  @SuppressWarnings("unchecked")
  public static <T> T runMethod(String className,
                     String methodName,
                     Object[] params,
                     Class[] paramTypes) {
    try {
      Class<? extends T> clazz = (Class<? extends T>) Class.forName(className);
      Method method = clazz.getDeclaredMethod(methodName, paramTypes);

      // Make the method accessible in case it is private and restore the accessibility at the end of invocation.
      boolean isMethodAccessible = method.isAccessible();

      method.setAccessible(true);
      Object obj = method.invoke(null, params);
      method.setAccessible(isMethodAccessible);

      return (T) obj;

    } catch (Throwable t) {
      throw new RuntimeException(String.format("Error occurred while invoking %s:%s.\n==> %s.",
        className, methodName, getMessage(t)), t);
    }
  }

  private static Object getMessage(Throwable t) {
    String msg = t.getMessage();
    while ((t instanceof InvocationTargetException || t instanceof RuntimeException || msg == null)
         && t.getCause() != null && t.getCause() != t) {
      t = t.getCause();
      msg = t.toString();
    }
    return msg;
  }

}
