/*
 * Decompiled with CFR 0.152.
 */
package io.druid.segment.realtime.firehose;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.maxmind.geoip2.DatabaseReader;
import com.maxmind.geoip2.exception.GeoIp2Exception;
import com.maxmind.geoip2.model.Omni;
import com.metamx.common.logger.Logger;
import io.druid.data.input.InputRow;
import io.druid.data.input.Row;
import io.druid.segment.realtime.firehose.IrcDecoder;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.InetAddress;
import java.net.URL;
import java.net.UnknownHostException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.zip.GZIPInputStream;
import org.apache.commons.io.FileUtils;
import org.joda.time.DateTime;
import org.joda.time.ReadableInstant;

class WikipediaIrcDecoder
implements IrcDecoder {
    static final Logger log = new Logger(WikipediaIrcDecoder.class);
    final DatabaseReader geoLookup;
    static final Pattern pattern = Pattern.compile(".*\\x0314\\[\\[\\x0307(.+?)\\x0314\\]\\]\\x034 (.*?)\\x0310.*\\x0302(http.+?)\\x03.+\\x0303(.+?)\\x03.+\\x03 (\\(([+-]\\d+)\\).*|.+) \\x0310(.+)\\x03.*");
    static final Pattern ipPattern = Pattern.compile("\\d+.\\d+.\\d+.\\d+");
    static final Pattern shortnamePattern = Pattern.compile("#(\\w\\w)\\..*");
    static final List<String> dimensionList = Lists.newArrayList((Object[])new String[]{"page", "language", "user", "unpatrolled", "newPage", "robot", "anonymous", "namespace", "continent", "country", "region", "city"});
    final Map<String, Map<String, String>> namespaces;
    final String geoIpDatabase;

    public WikipediaIrcDecoder(Map<String, Map<String, String>> namespaces) {
        this(namespaces, null);
    }

    @JsonCreator
    public WikipediaIrcDecoder(@JsonProperty(value="namespaces") Map<String, Map<String, String>> namespaces, @JsonProperty(value="geoIpDatabase") String geoIpDatabase) {
        if (namespaces == null) {
            namespaces = Maps.newHashMap();
        }
        this.namespaces = namespaces;
        this.geoIpDatabase = geoIpDatabase;
        this.geoLookup = geoIpDatabase != null ? this.openGeoIpDb(new File(geoIpDatabase)) : this.openDefaultGeoIpDb();
    }

    private DatabaseReader openDefaultGeoIpDb() {
        File geoDb = new File(System.getProperty("java.io.tmpdir"), this.getClass().getCanonicalName() + ".GeoLite2-City.mmdb");
        try {
            return this.openDefaultGeoIpDb(geoDb);
        }
        catch (RuntimeException e) {
            log.warn(e.getMessage() + " Attempting to re-download.", new Object[]{e});
            if (geoDb.exists() && !geoDb.delete()) {
                throw new RuntimeException("Could not delete geo db file [" + geoDb.getAbsolutePath() + "].");
            }
            return this.openDefaultGeoIpDb(geoDb);
        }
    }

    private DatabaseReader openDefaultGeoIpDb(File geoDb) {
        this.downloadGeoLiteDbToFile(geoDb);
        return this.openGeoIpDb(geoDb);
    }

    private DatabaseReader openGeoIpDb(File geoDb) {
        try {
            DatabaseReader reader = new DatabaseReader(geoDb);
            log.info("Using geo ip database at [%s].", new Object[]{geoDb});
            return reader;
        }
        catch (IOException e) {
            throw new RuntimeException("Could not open geo db at [" + geoDb.getAbsolutePath() + "].", e);
        }
    }

    private void downloadGeoLiteDbToFile(File geoDb) {
        if (geoDb.exists()) {
            return;
        }
        try {
            log.info("Downloading geo ip database to [%s]. This may take a few minutes.", new Object[]{geoDb.getAbsolutePath()});
            File tmpFile = File.createTempFile("druid", "geo");
            FileUtils.copyInputStreamToFile((InputStream)new GZIPInputStream(new URL("http://geolite.maxmind.com/download/geoip/database/GeoLite2-City.mmdb.gz").openStream()), (File)tmpFile);
            if (!tmpFile.renameTo(geoDb)) {
                throw new RuntimeException("Unable to move geo file to [" + geoDb.getAbsolutePath() + "]!");
            }
        }
        catch (IOException e) {
            throw new RuntimeException("Unable to download geo ip database.", e);
        }
    }

    @JsonProperty
    public Map<String, Map<String, String>> getNamespaces() {
        return this.namespaces;
    }

    @JsonProperty
    public String getGeoIpDatabase() {
        return this.geoIpDatabase;
    }

    @Override
    public InputRow decodeMessage(final DateTime timestamp, String channel, String msg) {
        final HashMap dimensions = Maps.newHashMap();
        final HashMap metrics = Maps.newHashMap();
        Matcher m = pattern.matcher(msg);
        if (!m.matches()) {
            throw new IllegalArgumentException("Invalid input format");
        }
        Matcher shortname = shortnamePattern.matcher(channel);
        if (shortname.matches()) {
            dimensions.put("language", shortname.group(1));
        }
        String page = m.group(1);
        String pageUrl = page.replaceAll("\\s", "_");
        dimensions.put("page", pageUrl);
        String user = m.group(4);
        Matcher ipMatch = ipPattern.matcher(user);
        boolean anonymous = ipMatch.matches();
        if (anonymous) {
            try {
                InetAddress ip = InetAddress.getByName(ipMatch.group());
                Omni lookup = this.geoLookup.omni(ip);
                dimensions.put("continent", lookup.getContinent().getName());
                dimensions.put("country", lookup.getCountry().getName());
                dimensions.put("region", lookup.getMostSpecificSubdivision().getName());
                dimensions.put("city", lookup.getCity().getName());
            }
            catch (UnknownHostException e) {
                log.error((Throwable)e, "invalid ip [%s]", new Object[]{ipMatch.group()});
            }
            catch (IOException e) {
                log.error((Throwable)e, "error looking up geo ip", new Object[0]);
            }
            catch (GeoIp2Exception e) {
                log.error((Throwable)e, "error looking up geo ip", new Object[0]);
            }
        }
        dimensions.put("user", user);
        String flags = m.group(2);
        dimensions.put("unpatrolled", Boolean.toString(flags.contains("!")));
        dimensions.put("newPage", Boolean.toString(flags.contains("N")));
        dimensions.put("robot", Boolean.toString(flags.contains("B")));
        dimensions.put("anonymous", Boolean.toString(anonymous));
        String[] parts = page.split(":");
        if (parts.length > 1 && !parts[1].startsWith(" ")) {
            Map<String, String> channelNamespaces = this.namespaces.get(channel);
            if (channelNamespaces != null && channelNamespaces.containsKey(parts[0])) {
                dimensions.put("namespace", channelNamespaces.get(parts[0]));
            } else {
                dimensions.put("namespace", "wikipedia");
            }
        } else {
            dimensions.put("namespace", "article");
        }
        float delta = m.group(6) != null ? Float.parseFloat(m.group(6)) : 0.0f;
        metrics.put("delta", Float.valueOf(delta));
        metrics.put("added", Float.valueOf(Math.max(delta, 0.0f)));
        metrics.put("deleted", Float.valueOf(Math.min(delta, 0.0f)));
        return new InputRow(){

            public List<String> getDimensions() {
                return dimensionList;
            }

            public long getTimestampFromEpoch() {
                return timestamp.getMillis();
            }

            public DateTime getTimestamp() {
                return timestamp;
            }

            public List<String> getDimension(String dimension) {
                String value = (String)dimensions.get(dimension);
                if (value != null) {
                    return ImmutableList.of((Object)value);
                }
                return ImmutableList.of();
            }

            public Object getRaw(String dimension) {
                return dimensions.get(dimension);
            }

            public float getFloatMetric(String metric) {
                return ((Float)metrics.get(metric)).floatValue();
            }

            public long getLongMetric(String metric) {
                return new Float(((Float)metrics.get(metric)).floatValue()).longValue();
            }

            public int compareTo(Row o) {
                return timestamp.compareTo((ReadableInstant)o.getTimestamp());
            }

            public String toString() {
                return "WikipediaRow{timestamp=" + timestamp + ", dimensions=" + dimensions + ", metrics=" + metrics + '}';
            }
        };
    }
}

