/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.security;

import com.google.common.annotations.VisibleForTesting;
import java.io.Closeable;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.PrintWriter;
import java.lang.reflect.InvocationTargetException;
import java.net.InetAddress;
import java.security.NoSuchAlgorithmException;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.regex.Pattern;
import javax.crypto.Cipher;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.io.Text;
import org.apache.hadoop.security.Credentials;
import org.apache.hadoop.security.SaslPropertiesResolver;
import org.apache.hadoop.security.SecurityUtil;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.security.authentication.util.KerberosUtil;
import org.apache.hadoop.security.token.Token;
import org.apache.hadoop.util.ExitUtil;
import org.apache.hadoop.util.Shell;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class KerberosDiags
implements Closeable {
    private static final Logger LOG = LoggerFactory.getLogger(KerberosDiags.class);
    public static final String KRB5_CCNAME = "KRB5CCNAME";
    public static final String JAVA_SECURITY_KRB5_CONF = "java.security.krb5.conf";
    public static final String JAVA_SECURITY_KRB5_REALM = "java.security.krb5.realm";
    public static final String SUN_SECURITY_KRB5_DEBUG = "sun.security.krb5.debug";
    public static final String SUN_SECURITY_SPNEGO_DEBUG = "sun.security.spnego.debug";
    public static final String SUN_SECURITY_JAAS_FILE = "java.security.auth.login.config";
    public static final String KERBEROS_KINIT_COMMAND = "hadoop.kerberos.kinit.command";
    public static final String HADOOP_AUTHENTICATION_IS_DISABLED = "Hadoop authentication is disabled";
    public static final String UNSET = "(unset)";
    public static final String NO_DEFAULT_REALM = "Cannot locate default realm";
    private final Configuration conf;
    private final List<String> services;
    private final PrintWriter out;
    private final File keytab;
    private final String principal;
    private final long minKeyLength;
    private final boolean securityRequired;
    public static final String CAT_JVM = "JVM";
    public static final String CAT_JAAS = "JAAS";
    public static final String CAT_CONFIG = "CONFIG";
    public static final String CAT_LOGIN = "LOGIN";
    public static final String CAT_KERBEROS = "KERBEROS";
    public static final String CAT_SASL = "SASL";

    public KerberosDiags(Configuration conf, PrintWriter out, List<String> services, File keytab, String principal, long minKeyLength, boolean securityRequired) {
        this.conf = conf;
        this.services = services;
        this.keytab = keytab;
        this.principal = principal;
        this.out = out;
        this.minKeyLength = minKeyLength;
        this.securityRequired = securityRequired;
    }

    @Override
    public void close() throws IOException {
        this.flush();
    }

    public boolean execute() throws Exception {
        String prop;
        this.title("Kerberos Diagnostics scan at %s", new Date(System.currentTimeMillis()));
        this.println("Hostname: %s", InetAddress.getLocalHost().getCanonicalHostName());
        this.validateKeyLength();
        this.println("JVM Kerberos Login Module = %s", KerberosUtil.getKrb5LoginModuleName());
        this.printDefaultRealm();
        this.title("System Properties", new Object[0]);
        String[] stringArray = new String[]{JAVA_SECURITY_KRB5_CONF, JAVA_SECURITY_KRB5_REALM, SUN_SECURITY_KRB5_DEBUG, SUN_SECURITY_SPNEGO_DEBUG, SUN_SECURITY_JAAS_FILE};
        int n = stringArray.length;
        int n2 = 0;
        while (n2 < n) {
            prop = stringArray[n2];
            this.printSysprop(prop);
            ++n2;
        }
        this.title("Environment Variables", new Object[0]);
        stringArray = new String[]{"HADOOP_JAAS_DEBUG", KRB5_CCNAME, "HADOOP_USER_NAME", "HADOOP_PROXY_USER", "HADOOP_TOKEN_FILE_LOCATION"};
        n = stringArray.length;
        n2 = 0;
        while (n2 < n) {
            String env = stringArray[n2];
            this.printEnv(env);
            ++n2;
        }
        stringArray = new String[]{KERBEROS_KINIT_COMMAND, "hadoop.security.authentication", "hadoop.security.authorization", "hadoop.kerberos.min.seconds.before.relogin", "hadoop.security.dns.interface", "hadoop.security.dns.nameserver", "hadoop.rpc.protection", "hadoop.security.saslproperties.resolver.class", "hadoop.security.crypto.codec.classes", "hadoop.security.group.mapping", "hadoop.security.impersonation.provider.class", "dfs.data.transfer.protection"};
        n = stringArray.length;
        n2 = 0;
        while (n2 < n) {
            prop = stringArray[n2];
            this.printConfOpt(prop);
            ++n2;
        }
        if (SecurityUtil.getAuthenticationMethod((Configuration)this.conf).equals((Object)UserGroupInformation.AuthenticationMethod.SIMPLE)) {
            this.println(HADOOP_AUTHENTICATION_IS_DISABLED);
            this.failif(this.securityRequired, CAT_CONFIG, HADOOP_AUTHENTICATION_IS_DISABLED, new Object[0]);
            return false;
        }
        this.validateKrb5File();
        this.validateSasl("hadoop.security.saslproperties.resolver.class");
        this.validateSasl("dfs.data.transfer.saslproperties.resolver.class");
        this.validateKinitExecutable();
        this.validateJAAS();
        boolean krb5Debug = this.getAndSet(SUN_SECURITY_KRB5_DEBUG);
        boolean spnegoDebug = this.getAndSet(SUN_SECURITY_SPNEGO_DEBUG);
        try {
            this.title("Logging in", new Object[0]);
            if (this.keytab != null) {
                this.dumpKeytab(this.keytab);
                this.loginFromKeytab();
            } else {
                UserGroupInformation loginUser = UserGroupInformation.getLoginUser();
                this.dumpUGI("Log in user", loginUser);
                this.validateUGI("Login user", loginUser);
                this.println("Ticket based login: %b", UserGroupInformation.isLoginTicketBased());
                this.println("Keytab based login: %b", UserGroupInformation.isLoginKeytabBased());
            }
            return true;
        }
        finally {
            System.setProperty(SUN_SECURITY_KRB5_DEBUG, Boolean.toString(krb5Debug));
            System.setProperty(SUN_SECURITY_SPNEGO_DEBUG, Boolean.toString(spnegoDebug));
        }
    }

    protected void validateKeyLength() throws NoSuchAlgorithmException {
        int aesLen = Cipher.getMaxAllowedKeyLength("AES");
        this.println("Maximum AES encryption key length %d bits", aesLen);
        this.failif((long)aesLen < this.minKeyLength, CAT_JVM, "Java Cryptography Extensions are not installed on this JVM. Maximum supported key length %s - minimum required %d", aesLen, this.minKeyLength);
    }

    protected void printDefaultRealm() {
        try {
            this.println("Default Realm = %s", KerberosUtil.getDefaultRealm());
        }
        catch (ClassNotFoundException | IllegalAccessException | NoSuchMethodException e) {
            throw new KerberosDiagsFailure(CAT_JVM, e, "Failed to invoke krb5.Config.getDefaultRealm: %s", e);
        }
        catch (InvocationTargetException e) {
            Throwable cause;
            Throwable throwable = cause = e.getCause() != null ? e.getCause() : e;
            if (cause.toString().contains(NO_DEFAULT_REALM)) {
                this.println("Host has no default realm");
                LOG.debug(cause.toString(), cause);
            }
            this.println("Kerberos.getDefaultRealm() failed: %s\n%s", cause, org.apache.hadoop.util.StringUtils.stringifyException((Throwable)cause));
        }
    }

    private void validateKrb5File() throws IOException {
        if (!Shell.WINDOWS) {
            String krb5name;
            this.title("Locating Kerberos configuration file", new Object[0]);
            String krbPath = "/etc/krb5.conf";
            String jvmKrbPath = System.getProperty(JAVA_SECURITY_KRB5_CONF);
            if (jvmKrbPath != null) {
                this.println("Setting kerberos path from sysprop %s: %s", JAVA_SECURITY_KRB5_CONF, jvmKrbPath);
                krbPath = jvmKrbPath;
            }
            if ((krb5name = System.getenv(KRB5_CCNAME)) != null) {
                this.println("Setting kerberos path from environment variable %s: %s", KRB5_CCNAME, krb5name);
                krbPath = krb5name;
                if (jvmKrbPath != null) {
                    this.println("Warning - both %s and %s were set - %s takes priority", JAVA_SECURITY_KRB5_CONF, KRB5_CCNAME, KRB5_CCNAME);
                }
            }
            File krbFile = new File(krbPath);
            this.println("Kerberos configuration file = %s", krbFile);
            this.failif(!krbFile.exists(), CAT_KERBEROS, "Kerberos configuration file %s not found", krbFile);
            this.dump(krbFile);
        }
    }

    public void dumpKeytab(File keytabFile) throws IOException {
        this.title("Examining keytab %s", keytabFile);
        File kt = keytabFile.getCanonicalFile();
        this.failif(!kt.exists(), CAT_CONFIG, "Keytab not found: %s", kt);
        this.failif(!kt.isFile(), CAT_CONFIG, "Keytab is not a valid file: %s", kt);
        String[] names = KerberosUtil.getPrincipalNames((String)keytabFile.getCanonicalPath(), (Pattern)Pattern.compile(".*"));
        this.println("keytab entry count: %d", names.length);
        String[] stringArray = names;
        int n = names.length;
        int n2 = 0;
        while (n2 < n) {
            String name = stringArray[n2];
            this.println("    %s", name);
            ++n2;
        }
        this.println("-----");
    }

    private void loginFromKeytab() throws IOException {
        if (this.keytab != null) {
            File kt = this.keytab.getCanonicalFile();
            this.println("Using keytab %s principal %s", kt, this.principal);
            String identity = this.principal;
            this.failif(StringUtils.isEmpty((String)this.principal), CAT_KERBEROS, "No principal defined", new Object[0]);
            UserGroupInformation ugi = UserGroupInformation.loginUserFromKeytabAndReturnUGI((String)this.principal, (String)kt.getPath());
            this.dumpUGI(identity, ugi);
            this.validateUGI(this.principal, ugi);
            this.title("Attempting to log in from keytab again", new Object[0]);
            UserGroupInformation.setShouldRenewImmediatelyForTests((boolean)true);
            ugi.reloginFromKeytab();
        } else {
            this.println("No keytab: logging is as current user");
        }
    }

    private void dumpUGI(String title, UserGroupInformation ugi) throws IOException {
        this.title(title, new Object[0]);
        this.println("UGI instance = %s", ugi);
        this.println("Has kerberos credentials: %b", ugi.hasKerberosCredentials());
        this.println("Authentication method: %s", ugi.getAuthenticationMethod());
        this.println("Real Authentication method: %s", ugi.getRealAuthenticationMethod());
        this.title("Group names", new Object[0]);
        String[] stringArray = ugi.getGroupNames();
        int n = stringArray.length;
        int n2 = 0;
        while (n2 < n) {
            String name = stringArray[n2];
            this.println(name);
            ++n2;
        }
        this.title("Credentials", new Object[0]);
        Credentials credentials = ugi.getCredentials();
        List secretKeys = credentials.getAllSecretKeys();
        this.title("Secret keys", new Object[0]);
        if (!secretKeys.isEmpty()) {
            for (Text secret : secretKeys) {
                this.println("%s", secret);
            }
        } else {
            this.println("(none)");
        }
        this.dumpTokens(ugi);
    }

    private void validateUGI(String messagePrefix, UserGroupInformation user) {
        this.failif(!user.hasKerberosCredentials(), CAT_LOGIN, "%s: No kerberos credentials for %s", messagePrefix, user);
        this.failif(user.getAuthenticationMethod() == null, CAT_LOGIN, "%s: Null AuthenticationMethod for %s", messagePrefix, user);
    }

    private void validateKinitExecutable() {
        String kinit = this.conf.getTrimmed(KERBEROS_KINIT_COMMAND, "");
        if (!kinit.isEmpty()) {
            File kinitPath = new File(kinit);
            this.println("%s = %s", KERBEROS_KINIT_COMMAND, kinitPath);
            if (kinitPath.isAbsolute()) {
                this.failif(!kinitPath.exists(), CAT_KERBEROS, "%s executable does not exist: %s", KERBEROS_KINIT_COMMAND, kinitPath);
                this.failif(!kinitPath.isFile(), CAT_KERBEROS, "%s path does not refer to a file: %s", KERBEROS_KINIT_COMMAND, kinitPath);
                this.failif(kinitPath.length() == 0L, CAT_KERBEROS, "%s file is empty: %s", KERBEROS_KINIT_COMMAND, kinitPath);
            } else {
                this.println("Executable %s is relative -must be on the PATH", kinit);
                this.printEnv("PATH");
            }
        }
    }

    private void validateSasl(String saslPropsResolverKey) {
        this.title("Resolving SASL property %s", saslPropsResolverKey);
        String saslPropsResolver = this.conf.getTrimmed(saslPropsResolverKey);
        try {
            Class resolverClass = this.conf.getClass(saslPropsResolverKey, SaslPropertiesResolver.class, SaslPropertiesResolver.class);
            this.println("Resolver is %s", resolverClass);
        }
        catch (RuntimeException e) {
            throw new KerberosDiagsFailure(CAT_SASL, e, "Failed to load %s class %s", saslPropsResolverKey, saslPropsResolver);
        }
    }

    private void validateJAAS() {
        String jaasFilename = System.getProperty(SUN_SECURITY_JAAS_FILE);
        if (jaasFilename != null) {
            this.title(CAT_JAAS, new Object[0]);
            File jaasFile = new File(jaasFilename);
            this.println("JAAS file is defined in %s: %s", SUN_SECURITY_JAAS_FILE, jaasFile);
            this.failif(!jaasFile.exists(), CAT_JAAS, "JAAS file does not exist: %s", jaasFile);
            this.failif(!jaasFile.isFile(), CAT_JAAS, "Specified JAAS file is not a file: %s", jaasFile);
        }
    }

    public void dumpTokens(UserGroupInformation user) {
        Collection tokens = user.getCredentials().getAllTokens();
        this.title("Token Count: %d", tokens.size());
        for (Token token : tokens) {
            this.println("Token %s", token.getKind());
        }
    }

    private boolean getAndSet(String sysprop) {
        boolean old = Boolean.getBoolean(sysprop);
        System.setProperty(sysprop, "true");
        return old;
    }

    private void flush() {
        if (this.out != null) {
            this.out.flush();
        } else {
            System.out.flush();
        }
        System.err.flush();
    }

    @VisibleForTesting
    public void println(String format, Object ... args) {
        this.println(KerberosDiags.format(format, args));
    }

    @VisibleForTesting
    private void println(String msg) {
        this.flush();
        if (this.out != null) {
            this.out.println(msg);
        } else {
            LOG.info(msg);
        }
        this.flush();
    }

    private void title(String format, Object ... args) {
        this.println("");
        this.println("");
        String msg = "== " + KerberosDiags.format(format, args) + " ==";
        this.println(msg);
        this.println("");
    }

    private void printSysprop(String property) {
        this.println("%s = \"%s\"", property, System.getProperty(property, UNSET));
    }

    private void printConfOpt(String option) {
        this.println("%s = \"%s\"", option, this.conf.get(option, UNSET));
    }

    private void printEnv(String variable) {
        String env = System.getenv(variable);
        this.println("%s = \"%s\"", variable, env != null ? env : UNSET);
    }

    public void dump(File file) throws IOException {
        Throwable throwable = null;
        Object var3_4 = null;
        try (FileInputStream in = new FileInputStream(file);){
            for (String line : IOUtils.readLines((InputStream)in)) {
                this.println("%s", line);
            }
        }
        catch (Throwable throwable2) {
            if (throwable == null) {
                throwable = throwable2;
            } else if (throwable != throwable2) {
                throwable.addSuppressed(throwable2);
            }
            throw throwable;
        }
        this.println("");
    }

    private void fail(String category, String message, Object ... args) throws KerberosDiagsFailure {
        throw new KerberosDiagsFailure(category, message, args);
    }

    private void failif(boolean condition, String category, String message, Object ... args) throws KerberosDiagsFailure {
        if (condition) {
            this.fail(category, message, args);
        }
    }

    public static String format(String message, Object ... args) {
        if (args.length == 0) {
            return message;
        }
        return String.format(message, args);
    }

    public static class KerberosDiagsFailure
    extends ExitUtil.ExitException {
        private final String category;

        public KerberosDiagsFailure(String category, String message) {
            super(41, String.valueOf(category) + ": " + message);
            this.category = category;
        }

        public KerberosDiagsFailure(String category, String message, Object ... args) {
            this(category, KerberosDiags.format(message, args));
        }

        public KerberosDiagsFailure(String category, Throwable throwable, String message, Object ... args) {
            this(category, message, args);
            this.initCause(throwable);
        }

        public String getCategory() {
            return this.category;
        }
    }
}

