/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hive.hcatalog.streaming.mutate.client.lock;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;
import java.util.Timer;
import java.util.concurrent.TimeUnit;
import org.apache.commons.lang3.StringUtils;
import org.apache.hadoop.hive.conf.HiveConf;
import org.apache.hadoop.hive.metastore.IMetaStoreClient;
import org.apache.hadoop.hive.metastore.LockComponentBuilder;
import org.apache.hadoop.hive.metastore.LockRequestBuilder;
import org.apache.hadoop.hive.metastore.api.DataOperationType;
import org.apache.hadoop.hive.metastore.api.LockComponent;
import org.apache.hadoop.hive.metastore.api.LockRequest;
import org.apache.hadoop.hive.metastore.api.LockResponse;
import org.apache.hadoop.hive.metastore.api.LockState;
import org.apache.hadoop.hive.metastore.api.Table;
import org.apache.hive.hcatalog.streaming.mutate.client.lock.HeartbeatFactory;
import org.apache.hive.hcatalog.streaming.mutate.client.lock.LockException;
import org.apache.hive.hcatalog.streaming.mutate.client.lock.LockFailureListener;
import org.apache.thrift.TException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Lock {
    private static final Logger LOG = LoggerFactory.getLogger(Lock.class);
    private static final double HEARTBEAT_FACTOR = 0.75;
    private static final int DEFAULT_HEARTBEAT_PERIOD = 275;
    private final IMetaStoreClient metaStoreClient;
    private final HeartbeatFactory heartbeatFactory;
    private final LockFailureListener listener;
    private final Collection<Table> sinks;
    private final Collection<Table> tables = new HashSet<Table>();
    private final int lockRetries;
    private final int retryWaitSeconds;
    private final String user;
    private final HiveConf hiveConf;
    private Timer heartbeat;
    private Long lockId;
    private Long transactionId;

    public Lock(IMetaStoreClient metaStoreClient, Options options) {
        this(metaStoreClient, new HeartbeatFactory(), options.hiveConf, options.listener, options.user, options.sources, options.sinks, options.lockRetries, options.retryWaitSeconds);
    }

    Lock(IMetaStoreClient metaStoreClient, HeartbeatFactory heartbeatFactory, HiveConf hiveConf, LockFailureListener listener, String user, Collection<Table> sources, Collection<Table> sinks, int lockRetries, int retryWaitSeconds) {
        this.metaStoreClient = metaStoreClient;
        this.heartbeatFactory = heartbeatFactory;
        this.hiveConf = hiveConf;
        this.user = user;
        this.listener = listener;
        this.lockRetries = lockRetries;
        this.retryWaitSeconds = retryWaitSeconds;
        this.sinks = sinks;
        this.tables.addAll(sources);
        this.tables.addAll(sinks);
        if (LockFailureListener.NULL_LISTENER.equals(listener)) {
            LOG.warn("No {} supplied. Data quality and availability cannot be assured.", (Object)LockFailureListener.class.getSimpleName());
        }
    }

    public void acquire() throws LockException {
        this.lockId = this.internalAcquire(null);
        this.initiateHeartbeat();
    }

    public void acquire(long transactionId) throws LockException {
        if (transactionId <= 0L) {
            throw new IllegalArgumentException("Invalid transaction id: " + transactionId);
        }
        this.lockId = this.internalAcquire(transactionId);
        this.transactionId = transactionId;
        this.initiateHeartbeat();
    }

    public void release() throws LockException {
        if (this.heartbeat != null) {
            this.heartbeat.cancel();
        }
        this.internalRelease();
    }

    public String getUser() {
        return this.user;
    }

    public String toString() {
        return "Lock [metaStoreClient=" + this.metaStoreClient + ", lockId=" + this.lockId + ", transactionId=" + this.transactionId + "]";
    }

    private long internalAcquire(Long transactionId) throws LockException {
        int attempts = 0;
        LockRequest request = this.buildLockRequest(transactionId);
        do {
            LockResponse response = null;
            try {
                response = this.metaStoreClient.lock(request);
            }
            catch (TException e) {
                throw new LockException("Unable to acquire lock for tables: [" + Lock.join(this.tables) + "]", e);
            }
            if (response == null) continue;
            LockState state = response.getState();
            if (state == LockState.NOT_ACQUIRED || state == LockState.ABORT) break;
            if (state == LockState.ACQUIRED) {
                LOG.debug("Acquired lock {}", (Object)response.getLockid());
                return response.getLockid();
            }
            if (state != LockState.WAITING) continue;
            try {
                Thread.sleep(TimeUnit.SECONDS.toMillis(this.retryWaitSeconds));
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
        } while (++attempts < this.lockRetries);
        throw new LockException("Could not acquire lock on tables: [" + Lock.join(this.tables) + "]");
    }

    private void internalRelease() {
        try {
            if (this.lockId != null && this.transactionId == null) {
                this.metaStoreClient.unlock(this.lockId.longValue());
                LOG.debug("Released lock {}", (Object)this.lockId);
                this.lockId = null;
            }
        }
        catch (TException e) {
            LOG.error("Lock " + this.lockId + " failed.", (Throwable)e);
            this.listener.lockFailed(this.lockId, this.transactionId, Lock.asStrings(this.tables), e);
        }
    }

    private LockRequest buildLockRequest(Long transactionId) {
        if (transactionId == null && !this.sinks.isEmpty()) {
            throw new IllegalArgumentException("Cannot sink to tables outside of a transaction: sinks=" + Lock.asStrings(this.sinks));
        }
        LockRequestBuilder requestBuilder = new LockRequestBuilder();
        for (Table table : this.tables) {
            LockComponentBuilder componentBuilder = new LockComponentBuilder().setDbName(table.getDbName()).setTableName(table.getTableName());
            if (this.sinks.contains(table)) {
                componentBuilder.setSemiShared().setOperationType(DataOperationType.UPDATE).setIsAcid(true);
            } else {
                componentBuilder.setShared().setOperationType(DataOperationType.INSERT).setIsAcid(true);
            }
            LockComponent component = componentBuilder.build();
            requestBuilder.addLockComponent(component);
        }
        if (transactionId != null) {
            requestBuilder.setTransactionId(transactionId.longValue());
        }
        LockRequest request = requestBuilder.setUser(this.user).build();
        return request;
    }

    private void initiateHeartbeat() {
        int heartbeatPeriod = this.getHeartbeatPeriod();
        LOG.debug("Heartbeat period {}s", (Object)heartbeatPeriod);
        this.heartbeat = this.heartbeatFactory.newInstance(this.metaStoreClient, this.listener, this.transactionId, this.tables, this.lockId, heartbeatPeriod);
    }

    private int getHeartbeatPeriod() {
        String txTimeoutSeconds;
        int heartbeatPeriod = 275;
        if (this.hiveConf != null && (txTimeoutSeconds = this.hiveConf.getVar(HiveConf.ConfVars.HIVE_TXN_TIMEOUT)) != null) {
            heartbeatPeriod = Math.max(1, (int)((double)Integer.parseInt(txTimeoutSeconds.substring(0, txTimeoutSeconds.length() - 1)) * 0.75));
        }
        return heartbeatPeriod;
    }

    Long getLockId() {
        return this.lockId;
    }

    Long getTransactionId() {
        return this.transactionId;
    }

    static String join(Iterable<? extends Object> values) {
        return StringUtils.join(values, (String)",");
    }

    static List<String> asStrings(Collection<Table> tables) {
        ArrayList<String> strings = new ArrayList<String>(tables.size());
        for (Table descriptor : tables) {
            strings.add(descriptor.getDbName() + "." + descriptor.getTableName());
        }
        return strings;
    }

    public static final class Options {
        Set<Table> sources = new LinkedHashSet<Table>();
        Set<Table> sinks = new LinkedHashSet<Table>();
        LockFailureListener listener = LockFailureListener.NULL_LISTENER;
        int lockRetries = 5;
        int retryWaitSeconds = 30;
        String user;
        HiveConf hiveConf;

        public Options addSourceTable(String databaseName, String tableName) {
            this.addTable(databaseName, tableName, this.sources);
            return this;
        }

        public Options addSinkTable(String databaseName, String tableName) {
            this.addTable(databaseName, tableName, this.sinks);
            return this;
        }

        private void addTable(String databaseName, String tableName, Set<Table> tables) {
            Options.checkNotNullOrEmpty(databaseName);
            Options.checkNotNullOrEmpty(tableName);
            Table table = new Table();
            table.setDbName(databaseName);
            table.setTableName(tableName);
            tables.add(table);
        }

        public Options user(String user) {
            Options.checkNotNullOrEmpty(user);
            this.user = user;
            return this;
        }

        public Options configuration(HiveConf hiveConf) {
            Options.checkNotNull(hiveConf);
            this.hiveConf = hiveConf;
            return this;
        }

        public Options lockFailureListener(LockFailureListener listener) {
            Options.checkNotNull(listener);
            this.listener = listener;
            return this;
        }

        public Options lockRetries(int lockRetries) {
            Options.checkArgument(lockRetries > 0);
            this.lockRetries = lockRetries;
            return this;
        }

        public Options retryWaitSeconds(int retryWaitSeconds) {
            Options.checkArgument(retryWaitSeconds > 0);
            this.retryWaitSeconds = retryWaitSeconds;
            return this;
        }

        private static void checkArgument(boolean value) {
            if (!value) {
                throw new IllegalArgumentException();
            }
        }

        private static void checkNotNull(Object value) {
            if (value == null) {
                throw new IllegalArgumentException();
            }
        }

        private static void checkNotNullOrEmpty(String value) {
            if (StringUtils.isBlank((CharSequence)value)) {
                throw new IllegalArgumentException();
            }
        }
    }
}

