/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.util;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.List;
import java.util.SortedSet;
import java.util.UUID;
import org.apache.hadoop.hbase.testclassification.SmallTests;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.KeyRange;
import org.apache.hadoop.hbase.util.RegionSplitCalculator;
import org.apache.hive.com.google.common.collect.ComparisonChain;
import org.apache.hive.com.google.common.collect.Multimap;
import org.apache.hive.org.apache.commons.logging.Log;
import org.apache.hive.org.apache.commons.logging.LogFactory;
import org.junit.Assert;
import org.junit.Test;
import org.junit.experimental.categories.Category;

@Category(value={SmallTests.class})
public class TestRegionSplitCalculator {
    private static final Log LOG = LogFactory.getLog(TestRegionSplitCalculator.class);
    Comparator<SimpleRange> cmp = new Comparator<SimpleRange>(){

        @Override
        public int compare(SimpleRange sr1, SimpleRange sr2) {
            ComparisonChain cc = ComparisonChain.start();
            cc = cc.compare(sr1.getStartKey(), sr2.getStartKey(), Bytes.BYTES_COMPARATOR);
            cc = cc.compare(sr1.getEndKey(), sr2.getEndKey(), RegionSplitCalculator.BYTES_COMPARATOR);
            cc = cc.compare(sr1.tiebreaker, sr2.tiebreaker);
            return cc.result();
        }
    };

    void checkDepths(SortedSet<byte[]> splits, Multimap<byte[], SimpleRange> regions, Integer ... depths) {
        Assert.assertEquals((long)splits.size(), (long)depths.length);
        int i = 0;
        for (byte[] k : splits) {
            Collection<SimpleRange> rs = regions.get(k);
            int sz = rs == null ? 0 : rs.size();
            Assert.assertEquals((long)depths[i].intValue(), (long)sz);
            ++i;
        }
    }

    String dump(SortedSet<byte[]> splits, Multimap<byte[], SimpleRange> regions) {
        StringBuilder sb = new StringBuilder();
        for (byte[] k : splits) {
            sb.append(Bytes.toString(k)).append(":\t");
            for (SimpleRange r : regions.get(k)) {
                sb.append(r.toString()).append("\t");
            }
            sb.append("\n");
        }
        String s = sb.toString();
        LOG.info("\n" + s);
        return s;
    }

    @Test
    public void testSplitCalculator() {
        SimpleRange a = new SimpleRange(Bytes.toBytes("A"), Bytes.toBytes("B"));
        SimpleRange b = new SimpleRange(Bytes.toBytes("B"), Bytes.toBytes("C"));
        SimpleRange c = new SimpleRange(Bytes.toBytes("C"), Bytes.toBytes("D"));
        RegionSplitCalculator<SimpleRange> sc = new RegionSplitCalculator<SimpleRange>(this.cmp);
        sc.add(a);
        sc.add(b);
        sc.add(c);
        Multimap<byte[], SimpleRange> regions = sc.calcCoverage();
        LOG.info("Standard");
        String res = this.dump(sc.getSplits(), regions);
        this.checkDepths(sc.getSplits(), regions, 1, 1, 1, 0);
        Assert.assertEquals((Object)res, (Object)"A:\t[A, B]\t\nB:\t[B, C]\t\nC:\t[C, D]\t\nD:\t\n");
    }

    @Test
    public void testSplitCalculatorNoEdge() {
        RegionSplitCalculator<SimpleRange> sc = new RegionSplitCalculator<SimpleRange>(this.cmp);
        Multimap<byte[], SimpleRange> regions = sc.calcCoverage();
        LOG.info("Empty");
        String res = this.dump(sc.getSplits(), regions);
        this.checkDepths(sc.getSplits(), regions, new Integer[0]);
        Assert.assertEquals((Object)"", (Object)res);
    }

    @Test
    public void testSplitCalculatorSingleEdge() {
        SimpleRange a = new SimpleRange(Bytes.toBytes("A"), Bytes.toBytes("B"));
        RegionSplitCalculator<SimpleRange> sc = new RegionSplitCalculator<SimpleRange>(this.cmp);
        sc.add(a);
        Multimap<byte[], SimpleRange> regions = sc.calcCoverage();
        LOG.info("Single edge");
        String res = this.dump(sc.getSplits(), regions);
        this.checkDepths(sc.getSplits(), regions, 1, 0);
        Assert.assertEquals((Object)"A:\t[A, B]\t\nB:\t\n", (Object)res);
    }

    @Test
    public void testSplitCalculatorDegenerateEdge() {
        SimpleRange a = new SimpleRange(Bytes.toBytes("A"), Bytes.toBytes("A"));
        RegionSplitCalculator<SimpleRange> sc = new RegionSplitCalculator<SimpleRange>(this.cmp);
        sc.add(a);
        Multimap<byte[], SimpleRange> regions = sc.calcCoverage();
        LOG.info("Single empty edge");
        String res = this.dump(sc.getSplits(), regions);
        this.checkDepths(sc.getSplits(), regions, 1);
        Assert.assertEquals((Object)"A:\t[A, A]\t\n", (Object)res);
    }

    @Test
    public void testSplitCalculatorCoverSplit() {
        SimpleRange a = new SimpleRange(Bytes.toBytes("A"), Bytes.toBytes("B"));
        SimpleRange b = new SimpleRange(Bytes.toBytes("B"), Bytes.toBytes("C"));
        SimpleRange c = new SimpleRange(Bytes.toBytes("A"), Bytes.toBytes("C"));
        RegionSplitCalculator<SimpleRange> sc = new RegionSplitCalculator<SimpleRange>(this.cmp);
        sc.add(a);
        sc.add(b);
        sc.add(c);
        Multimap<byte[], SimpleRange> regions = sc.calcCoverage();
        LOG.info("AC covers AB, BC");
        String res = this.dump(sc.getSplits(), regions);
        this.checkDepths(sc.getSplits(), regions, 2, 2, 0);
        Assert.assertEquals((Object)"A:\t[A, B]\t[A, C]\t\nB:\t[A, C]\t[B, C]\t\nC:\t\n", (Object)res);
    }

    @Test
    public void testSplitCalculatorOverEndpoint() {
        SimpleRange a = new SimpleRange(Bytes.toBytes("A"), Bytes.toBytes("B"));
        SimpleRange b = new SimpleRange(Bytes.toBytes("B"), Bytes.toBytes("C"));
        SimpleRange c = new SimpleRange(Bytes.toBytes("B"), Bytes.toBytes("D"));
        RegionSplitCalculator<SimpleRange> sc = new RegionSplitCalculator<SimpleRange>(this.cmp);
        sc.add(a);
        sc.add(b);
        sc.add(c);
        Multimap<byte[], SimpleRange> regions = sc.calcCoverage();
        LOG.info("AB, BD covers BC");
        String res = this.dump(sc.getSplits(), regions);
        this.checkDepths(sc.getSplits(), regions, 1, 2, 1, 0);
        Assert.assertEquals((Object)"A:\t[A, B]\t\nB:\t[B, C]\t[B, D]\t\nC:\t[B, D]\t\nD:\t\n", (Object)res);
    }

    @Test
    public void testSplitCalculatorHoles() {
        SimpleRange a = new SimpleRange(Bytes.toBytes("A"), Bytes.toBytes("B"));
        SimpleRange b = new SimpleRange(Bytes.toBytes("B"), Bytes.toBytes("C"));
        SimpleRange c = new SimpleRange(Bytes.toBytes("E"), Bytes.toBytes("F"));
        RegionSplitCalculator<SimpleRange> sc = new RegionSplitCalculator<SimpleRange>(this.cmp);
        sc.add(a);
        sc.add(b);
        sc.add(c);
        Multimap<byte[], SimpleRange> regions = sc.calcCoverage();
        LOG.info("Hole between C and E");
        String res = this.dump(sc.getSplits(), regions);
        this.checkDepths(sc.getSplits(), regions, 1, 1, 0, 1, 0);
        Assert.assertEquals((Object)"A:\t[A, B]\t\nB:\t[B, C]\t\nC:\t\nE:\t[E, F]\t\nF:\t\n", (Object)res);
    }

    @Test
    public void testSplitCalculatorOverreach() {
        SimpleRange a = new SimpleRange(Bytes.toBytes("A"), Bytes.toBytes("C"));
        SimpleRange b = new SimpleRange(Bytes.toBytes("B"), Bytes.toBytes("D"));
        RegionSplitCalculator<SimpleRange> sc = new RegionSplitCalculator<SimpleRange>(this.cmp);
        sc.add(a);
        sc.add(b);
        Multimap<byte[], SimpleRange> regions = sc.calcCoverage();
        LOG.info("AC and BD overlap but share no start/end keys");
        String res = this.dump(sc.getSplits(), regions);
        this.checkDepths(sc.getSplits(), regions, 1, 2, 1, 0);
        Assert.assertEquals((Object)"A:\t[A, C]\t\nB:\t[A, C]\t[B, D]\t\nC:\t[B, D]\t\nD:\t\n", (Object)res);
    }

    @Test
    public void testSplitCalculatorFloor() {
        SimpleRange a = new SimpleRange(Bytes.toBytes("A"), Bytes.toBytes("C"));
        SimpleRange b = new SimpleRange(Bytes.toBytes("A"), Bytes.toBytes("B"));
        RegionSplitCalculator<SimpleRange> sc = new RegionSplitCalculator<SimpleRange>(this.cmp);
        sc.add(a);
        sc.add(b);
        Multimap<byte[], SimpleRange> regions = sc.calcCoverage();
        LOG.info("AC and AB overlap in the beginning");
        String res = this.dump(sc.getSplits(), regions);
        this.checkDepths(sc.getSplits(), regions, 2, 1, 0);
        Assert.assertEquals((Object)"A:\t[A, B]\t[A, C]\t\nB:\t[A, C]\t\nC:\t\n", (Object)res);
    }

    @Test
    public void testSplitCalculatorCeil() {
        SimpleRange a = new SimpleRange(Bytes.toBytes("A"), Bytes.toBytes("C"));
        SimpleRange b = new SimpleRange(Bytes.toBytes("B"), Bytes.toBytes("C"));
        RegionSplitCalculator<SimpleRange> sc = new RegionSplitCalculator<SimpleRange>(this.cmp);
        sc.add(a);
        sc.add(b);
        Multimap<byte[], SimpleRange> regions = sc.calcCoverage();
        LOG.info("AC and BC overlap in the end");
        String res = this.dump(sc.getSplits(), regions);
        this.checkDepths(sc.getSplits(), regions, 1, 2, 0);
        Assert.assertEquals((Object)"A:\t[A, C]\t\nB:\t[A, C]\t[B, C]\t\nC:\t\n", (Object)res);
    }

    @Test
    public void testSplitCalculatorEq() {
        SimpleRange a = new SimpleRange(Bytes.toBytes("A"), Bytes.toBytes("C"));
        SimpleRange b = new SimpleRange(Bytes.toBytes("A"), Bytes.toBytes("C"));
        LOG.info(a.tiebreaker + " - " + b.tiebreaker);
        RegionSplitCalculator<SimpleRange> sc = new RegionSplitCalculator<SimpleRange>(this.cmp);
        sc.add(a);
        sc.add(b);
        Multimap<byte[], SimpleRange> regions = sc.calcCoverage();
        LOG.info("AC and AC overlap completely");
        String res = this.dump(sc.getSplits(), regions);
        this.checkDepths(sc.getSplits(), regions, 2, 0);
        Assert.assertEquals((Object)"A:\t[A, C]\t[A, C]\t\nC:\t\n", (Object)res);
    }

    @Test
    public void testSplitCalculatorBackwards() {
        SimpleRange a = new SimpleRange(Bytes.toBytes("C"), Bytes.toBytes("A"));
        RegionSplitCalculator<SimpleRange> sc = new RegionSplitCalculator<SimpleRange>(this.cmp);
        sc.add(a);
        Multimap<byte[], SimpleRange> regions = sc.calcCoverage();
        LOG.info("CA is backwards");
        String res = this.dump(sc.getSplits(), regions);
        this.checkDepths(sc.getSplits(), regions, new Integer[0]);
        Assert.assertEquals((Object)"", (Object)res);
    }

    @Test
    public void testComplex() {
        RegionSplitCalculator<SimpleRange> sc = new RegionSplitCalculator<SimpleRange>(this.cmp);
        sc.add(new SimpleRange(Bytes.toBytes("A"), Bytes.toBytes("Am")));
        sc.add(new SimpleRange(Bytes.toBytes("A"), Bytes.toBytes("C")));
        sc.add(new SimpleRange(Bytes.toBytes("Am"), Bytes.toBytes("C")));
        sc.add(new SimpleRange(Bytes.toBytes("D"), Bytes.toBytes("E")));
        sc.add(new SimpleRange(Bytes.toBytes("F"), Bytes.toBytes("G")));
        sc.add(new SimpleRange(Bytes.toBytes("B"), Bytes.toBytes("E")));
        sc.add(new SimpleRange(Bytes.toBytes("H"), Bytes.toBytes("I")));
        sc.add(new SimpleRange(Bytes.toBytes("A"), Bytes.toBytes("B")));
        Multimap<byte[], SimpleRange> regions = sc.calcCoverage();
        LOG.info("Something fairly complex");
        String res = this.dump(sc.getSplits(), regions);
        this.checkDepths(sc.getSplits(), regions, 3, 3, 3, 1, 2, 0, 1, 0, 1, 0);
        Assert.assertEquals((Object)"A:\t[A, Am]\t[A, B]\t[A, C]\t\nAm:\t[A, B]\t[A, C]\t[Am, C]\t\nB:\t[A, C]\t[Am, C]\t[B, E]\t\nC:\t[B, E]\t\nD:\t[B, E]\t[D, E]\t\nE:\t\nF:\t[F, G]\t\nG:\t\nH:\t[H, I]\t\nI:\t\n", (Object)res);
    }

    @Test
    public void testBeginEndMarker() {
        RegionSplitCalculator<SimpleRange> sc = new RegionSplitCalculator<SimpleRange>(this.cmp);
        sc.add(new SimpleRange(Bytes.toBytes(""), Bytes.toBytes("A")));
        sc.add(new SimpleRange(Bytes.toBytes("A"), Bytes.toBytes("B")));
        sc.add(new SimpleRange(Bytes.toBytes("B"), Bytes.toBytes("")));
        Multimap<byte[], SimpleRange> regions = sc.calcCoverage();
        LOG.info("Special cases -- empty");
        String res = this.dump(sc.getSplits(), regions);
        this.checkDepths(sc.getSplits(), regions, 1, 1, 1, 0);
        Assert.assertEquals((Object)":\t[, A]\t\nA:\t[A, B]\t\nB:\t[B, ]\t\nnull:\t\n", (Object)res);
    }

    @Test
    public void testBigRanges() {
        SimpleRange ai = new SimpleRange(Bytes.toBytes("A"), Bytes.toBytes("I"));
        SimpleRange ae = new SimpleRange(Bytes.toBytes("A"), Bytes.toBytes("E"));
        SimpleRange ac = new SimpleRange(Bytes.toBytes("A"), Bytes.toBytes("C"));
        ArrayList<SimpleRange> bigOverlap = new ArrayList<SimpleRange>();
        bigOverlap.add(new SimpleRange(Bytes.toBytes("A"), Bytes.toBytes("E")));
        bigOverlap.add(new SimpleRange(Bytes.toBytes("A"), Bytes.toBytes("C")));
        bigOverlap.add(new SimpleRange(Bytes.toBytes("A"), Bytes.toBytes("B")));
        bigOverlap.add(new SimpleRange(Bytes.toBytes("B"), Bytes.toBytes("C")));
        bigOverlap.add(new SimpleRange(Bytes.toBytes("E"), Bytes.toBytes("H")));
        bigOverlap.add(ai);
        bigOverlap.add(ae);
        bigOverlap.add(ac);
        List bigRanges = RegionSplitCalculator.findBigRanges(bigOverlap, 1);
        Assert.assertEquals((long)1L, (long)bigRanges.size());
        Assert.assertEquals((Object)ai, bigRanges.get(0));
        bigRanges = RegionSplitCalculator.findBigRanges(bigOverlap, 3);
        Assert.assertEquals((long)3L, (long)bigRanges.size());
        Assert.assertEquals((Object)ai, bigRanges.get(0));
        SimpleRange r1 = (SimpleRange)bigRanges.get(1);
        SimpleRange r2 = (SimpleRange)bigRanges.get(2);
        Assert.assertEquals((Object)"A", (Object)Bytes.toString(r1.start));
        Assert.assertEquals((Object)"A", (Object)Bytes.toString(r2.start));
        String r1e = Bytes.toString(r1.end);
        String r2e = Bytes.toString(r2.end);
        Assert.assertTrue((r1e.equals("C") && r2e.equals("E") || r1e.equals("E") && r2e.equals("C") ? 1 : 0) != 0);
    }

    static class SimpleRange
    implements KeyRange {
        byte[] start;
        byte[] end;
        UUID tiebreaker;

        SimpleRange(byte[] start, byte[] end) {
            this.start = start;
            this.end = end;
            this.tiebreaker = UUID.randomUUID();
        }

        @Override
        public byte[] getStartKey() {
            return this.start;
        }

        @Override
        public byte[] getEndKey() {
            return this.end;
        }

        public String toString() {
            return "[" + Bytes.toString(this.start) + ", " + Bytes.toString(this.end) + "]";
        }
    }
}

