/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.regionserver;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.Cell;
import org.apache.hadoop.hbase.KeyValue;
import org.apache.hadoop.hbase.classification.InterfaceAudience;
import org.apache.hadoop.hbase.regionserver.StoreFile;
import org.apache.hadoop.hbase.regionserver.StoreScanner;
import org.apache.hadoop.hbase.regionserver.StripeStoreFileManager;
import org.apache.hadoop.hbase.regionserver.compactions.Compactor;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hive.org.apache.commons.logging.Log;
import org.apache.hive.org.apache.commons.logging.LogFactory;

@InterfaceAudience.Private
public abstract class StripeMultiFileWriter
implements Compactor.CellSink {
    private static final Log LOG = LogFactory.getLog(StripeMultiFileWriter.class);
    protected WriterFactory writerFactory;
    protected KeyValue.KVComparator comparator;
    protected List<StoreFile.Writer> existingWriters;
    protected List<byte[]> boundaries;
    protected StoreScanner sourceScanner;
    private boolean doWriteStripeMetadata = true;

    public void init(StoreScanner sourceScanner, WriterFactory factory, KeyValue.KVComparator comparator) throws IOException {
        this.writerFactory = factory;
        this.sourceScanner = sourceScanner;
        this.comparator = comparator;
    }

    public void setNoStripeMetadata() {
        this.doWriteStripeMetadata = false;
    }

    public List<Path> commitWriters(long maxSeqId, boolean isMajor) throws IOException {
        assert (this.existingWriters != null);
        this.commitWritersInternal();
        assert (this.boundaries.size() == this.existingWriters.size() + 1);
        LOG.debug((this.doWriteStripeMetadata ? "W" : "Not w") + "riting out metadata for " + this.existingWriters.size() + " writers");
        ArrayList<Path> paths = new ArrayList<Path>();
        for (int i = 0; i < this.existingWriters.size(); ++i) {
            StoreFile.Writer writer = this.existingWriters.get(i);
            if (writer == null) continue;
            if (this.doWriteStripeMetadata) {
                writer.appendFileInfo(StripeStoreFileManager.STRIPE_START_KEY, this.boundaries.get(i));
                writer.appendFileInfo(StripeStoreFileManager.STRIPE_END_KEY, this.boundaries.get(i + 1));
            }
            writer.appendMetadata(maxSeqId, isMajor);
            paths.add(writer.getPath());
            writer.close();
        }
        this.existingWriters = null;
        return paths;
    }

    public List<Path> abortWriters() {
        assert (this.existingWriters != null);
        ArrayList<Path> paths = new ArrayList<Path>();
        for (StoreFile.Writer writer : this.existingWriters) {
            try {
                paths.add(writer.getPath());
                writer.close();
            }
            catch (Exception ex) {
                LOG.error("Failed to close the writer after an unfinished compaction.", ex);
            }
        }
        this.existingWriters = null;
        return paths;
    }

    protected void sanityCheckLeft(byte[] left, byte[] row, int rowOffset, int rowLength) throws IOException {
        if (StripeStoreFileManager.OPEN_KEY != left && this.comparator.compareRows(row, rowOffset, rowLength, left, 0, left.length) < 0) {
            String error = "The first row is lower than the left boundary of [" + Bytes.toString(left) + "]: [" + Bytes.toString(row, rowOffset, rowLength) + "]";
            LOG.error(error);
            throw new IOException(error);
        }
    }

    protected void sanityCheckRight(byte[] right, byte[] row, int rowOffset, int rowLength) throws IOException {
        if (StripeStoreFileManager.OPEN_KEY != right && this.comparator.compareRows(row, rowOffset, rowLength, right, 0, right.length) >= 0) {
            String error = "The last row is higher or equal than the right boundary of [" + Bytes.toString(right) + "]: [" + Bytes.toString(row, rowOffset, rowLength) + "]";
            LOG.error(error);
            throw new IOException(error);
        }
    }

    protected abstract void commitWritersInternal() throws IOException;

    public static class SizeMultiWriter
    extends StripeMultiFileWriter {
        private int targetCount;
        private long targetCells;
        private byte[] left;
        private byte[] right;
        private Cell lastCell;
        private StoreFile.Writer currentWriter;
        protected byte[] lastRowInCurrentWriter = null;
        private long cellsInCurrentWriter = 0L;
        private long cellsSeen = 0L;
        private long cellsSeenInPrevious = 0L;

        public SizeMultiWriter(int targetCount, long targetKvs, byte[] left, byte[] right) {
            this.targetCount = targetCount;
            this.targetCells = targetKvs;
            this.left = left;
            this.right = right;
            int preallocate = Math.min(this.targetCount, 64);
            this.existingWriters = new ArrayList(preallocate);
            this.boundaries = new ArrayList(preallocate + 1);
        }

        @Override
        public void append(Cell cell) throws IOException {
            boolean doCreateWriter = false;
            if (this.currentWriter == null) {
                this.sanityCheckLeft(this.left, cell.getRowArray(), cell.getRowOffset(), cell.getRowLength());
                doCreateWriter = true;
            } else if (this.lastRowInCurrentWriter != null && !this.comparator.matchingRows(cell.getRowArray(), cell.getRowOffset(), cell.getRowLength(), this.lastRowInCurrentWriter, 0, this.lastRowInCurrentWriter.length)) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Stopping to use a writer after [" + Bytes.toString(this.lastRowInCurrentWriter) + "] row; wrote out " + this.cellsInCurrentWriter + " kvs");
                }
                this.lastRowInCurrentWriter = null;
                this.cellsInCurrentWriter = 0L;
                this.cellsSeenInPrevious += this.cellsSeen;
                doCreateWriter = true;
            }
            if (doCreateWriter) {
                byte[] boundary;
                byte[] byArray = boundary = this.existingWriters.isEmpty() ? this.left : cell.getRow();
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Creating new writer starting at [" + Bytes.toString(boundary) + "]");
                }
                this.currentWriter = this.writerFactory.createWriter();
                this.boundaries.add(boundary);
                this.existingWriters.add(this.currentWriter);
            }
            this.currentWriter.append(cell);
            this.lastCell = cell;
            ++this.cellsInCurrentWriter;
            this.cellsSeen = this.cellsInCurrentWriter;
            if (this.sourceScanner != null) {
                this.cellsSeen = Math.max(this.cellsSeen, this.sourceScanner.getEstimatedNumberOfKvsScanned() - this.cellsSeenInPrevious);
            }
            if (this.lastRowInCurrentWriter == null && this.existingWriters.size() < this.targetCount && this.cellsSeen >= this.targetCells) {
                this.lastRowInCurrentWriter = cell.getRow();
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Preparing to start a new writer after [" + Bytes.toString(this.lastRowInCurrentWriter) + "] row; observed " + this.cellsSeen + " kvs and wrote out " + this.cellsInCurrentWriter + " kvs");
                }
            }
        }

        @Override
        protected void commitWritersInternal() throws IOException {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Stopping with " + this.cellsInCurrentWriter + " kvs in last writer" + (this.sourceScanner == null ? "" : "; observed estimated " + this.sourceScanner.getEstimatedNumberOfKvsScanned() + " KVs total"));
            }
            if (this.lastCell != null) {
                this.sanityCheckRight(this.right, this.lastCell.getRowArray(), this.lastCell.getRowOffset(), this.lastCell.getRowLength());
            }
            if (this.existingWriters.isEmpty() && 1 == this.targetCount) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Merge expired stripes into one, create an empty file to preserve metadata.");
                }
                this.boundaries.add(this.left);
                this.existingWriters.add(this.writerFactory.createWriter());
            }
            this.boundaries.add(this.right);
        }
    }

    public static class BoundaryMultiWriter
    extends StripeMultiFileWriter {
        private StoreFile.Writer currentWriter;
        private byte[] currentWriterEndKey;
        private Cell lastCell;
        private long cellsInCurrentWriter = 0L;
        private int majorRangeFromIndex = -1;
        private int majorRangeToIndex = -1;
        private boolean hasAnyWriter = false;

        public BoundaryMultiWriter(List<byte[]> targetBoundaries, byte[] majorRangeFrom, byte[] majorRangeTo) throws IOException {
            this.boundaries = targetBoundaries;
            this.existingWriters = new ArrayList(this.boundaries.size() - 1);
            assert (majorRangeFrom == null == (majorRangeTo == null));
            if (majorRangeFrom != null) {
                this.majorRangeFromIndex = majorRangeFrom == StripeStoreFileManager.OPEN_KEY ? 0 : Collections.binarySearch(this.boundaries, majorRangeFrom, Bytes.BYTES_COMPARATOR);
                int n = this.majorRangeToIndex = majorRangeTo == StripeStoreFileManager.OPEN_KEY ? this.boundaries.size() : Collections.binarySearch(this.boundaries, majorRangeTo, Bytes.BYTES_COMPARATOR);
                if (this.majorRangeFromIndex < 0 || this.majorRangeToIndex < 0) {
                    throw new IOException("Major range does not match writer boundaries: [" + Bytes.toString(majorRangeFrom) + "] [" + Bytes.toString(majorRangeTo) + "]; from " + this.majorRangeFromIndex + " to " + this.majorRangeToIndex);
                }
            }
        }

        @Override
        public void append(Cell cell) throws IOException {
            if (this.currentWriter == null && this.existingWriters.isEmpty()) {
                this.sanityCheckLeft((byte[])this.boundaries.get(0), cell.getRowArray(), cell.getRowOffset(), cell.getRowLength());
            }
            this.prepareWriterFor(cell);
            this.currentWriter.append(cell);
            this.lastCell = cell;
            ++this.cellsInCurrentWriter;
        }

        private boolean isCellAfterCurrentWriter(Cell cell) {
            return this.currentWriterEndKey != StripeStoreFileManager.OPEN_KEY && this.comparator.compareRows(cell.getRowArray(), cell.getRowOffset(), cell.getRowLength(), this.currentWriterEndKey, 0, this.currentWriterEndKey.length) >= 0;
        }

        @Override
        protected void commitWritersInternal() throws IOException {
            this.stopUsingCurrentWriter();
            while (this.existingWriters.size() < this.boundaries.size() - 1) {
                this.createEmptyWriter();
            }
            if (this.lastCell != null) {
                this.sanityCheckRight((byte[])this.boundaries.get(this.boundaries.size() - 1), this.lastCell.getRowArray(), this.lastCell.getRowOffset(), this.lastCell.getRowLength());
            }
        }

        private void prepareWriterFor(Cell cell) throws IOException {
            if (this.currentWriter != null && !this.isCellAfterCurrentWriter(cell)) {
                return;
            }
            this.stopUsingCurrentWriter();
            while (this.isCellAfterCurrentWriter(cell)) {
                this.checkCanCreateWriter();
                this.createEmptyWriter();
            }
            this.checkCanCreateWriter();
            this.hasAnyWriter = true;
            this.currentWriter = this.writerFactory.createWriter();
            this.existingWriters.add(this.currentWriter);
        }

        private void createEmptyWriter() throws IOException {
            int index = this.existingWriters.size();
            boolean isInMajorRange = index >= this.majorRangeFromIndex && index < this.majorRangeToIndex;
            boolean isLastWriter = !this.hasAnyWriter && index == this.boundaries.size() - 2;
            boolean needEmptyFile = isInMajorRange || isLastWriter;
            this.existingWriters.add(needEmptyFile ? this.writerFactory.createWriter() : null);
            this.hasAnyWriter |= needEmptyFile;
            this.currentWriterEndKey = this.existingWriters.size() + 1 == this.boundaries.size() ? null : (byte[])this.boundaries.get(this.existingWriters.size() + 1);
        }

        private void checkCanCreateWriter() throws IOException {
            int maxWriterCount = this.boundaries.size() - 1;
            assert (this.existingWriters.size() <= maxWriterCount);
            if (this.existingWriters.size() >= maxWriterCount) {
                throw new IOException("Cannot create any more writers (created " + this.existingWriters.size() + " out of " + maxWriterCount + " - row might be out of range of all valid writers");
            }
        }

        private void stopUsingCurrentWriter() {
            if (this.currentWriter != null) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Stopping to use a writer after [" + Bytes.toString(this.currentWriterEndKey) + "] row; wrote out " + this.cellsInCurrentWriter + " kvs");
                }
                this.cellsInCurrentWriter = 0L;
            }
            this.currentWriter = null;
            this.currentWriterEndKey = this.existingWriters.size() + 1 == this.boundaries.size() ? null : (byte[])this.boundaries.get(this.existingWriters.size() + 1);
        }
    }

    public static interface WriterFactory {
        public StoreFile.Writer createWriter() throws IOException;
    }
}

