/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.master;

import java.io.IOException;
import java.util.List;
import java.util.Random;
import java.util.concurrent.Callable;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import org.apache.hadoop.hbase.ChoreService;
import org.apache.hadoop.hbase.HBaseTestingUtility;
import org.apache.hadoop.hbase.HColumnDescriptor;
import org.apache.hadoop.hbase.HRegionInfo;
import org.apache.hadoop.hbase.HTableDescriptor;
import org.apache.hadoop.hbase.NotServingRegionException;
import org.apache.hadoop.hbase.ScheduledChore;
import org.apache.hadoop.hbase.ServerName;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.TableNotDisabledException;
import org.apache.hadoop.hbase.Waiter;
import org.apache.hadoop.hbase.client.HBaseAdmin;
import org.apache.hadoop.hbase.coprocessor.BaseMasterObserver;
import org.apache.hadoop.hbase.coprocessor.MasterCoprocessorEnvironment;
import org.apache.hadoop.hbase.coprocessor.ObserverContext;
import org.apache.hadoop.hbase.master.HMaster;
import org.apache.hadoop.hbase.master.TableLockManager;
import org.apache.hadoop.hbase.regionserver.HRegion;
import org.apache.hadoop.hbase.testclassification.LargeTests;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.LoadTestTool;
import org.apache.hadoop.hbase.util.StoppableImplementation;
import org.apache.hadoop.hbase.util.Threads;
import org.apache.hadoop.hbase.zookeeper.ZKUtil;
import org.apache.hadoop.hbase.zookeeper.ZooKeeperWatcher;
import org.apache.hive.org.apache.commons.logging.Log;
import org.apache.hive.org.apache.commons.logging.LogFactory;
import org.junit.After;
import org.junit.Assert;
import org.junit.Test;
import org.junit.experimental.categories.Category;

@Category(value={LargeTests.class})
public class TestTableLockManager {
    private static final Log LOG = LogFactory.getLog(TestTableLockManager.class);
    private static final TableName TABLE_NAME = TableName.valueOf("TestTableLevelLocks");
    private static final byte[] FAMILY = Bytes.toBytes("f1");
    private static final byte[] NEW_FAMILY = Bytes.toBytes("f2");
    private final HBaseTestingUtility TEST_UTIL = new HBaseTestingUtility();
    private static final CountDownLatch deleteColumn = new CountDownLatch(1);
    private static final CountDownLatch addColumn = new CountDownLatch(1);

    public void prepareMiniCluster() throws Exception {
        this.TEST_UTIL.getConfiguration().setBoolean("hbase.online.schema.update.enable", true);
        this.TEST_UTIL.startMiniCluster(2);
        this.TEST_UTIL.createTable(TABLE_NAME, FAMILY);
    }

    public void prepareMiniZkCluster() throws Exception {
        this.TEST_UTIL.startMiniZKCluster(1, new int[0]);
    }

    @After
    public void tearDown() throws Exception {
        this.TEST_UTIL.shutdownMiniCluster();
    }

    @Test(timeout=600000L)
    public void testAlterAndDisable() throws Exception {
        this.prepareMiniCluster();
        HMaster master = this.TEST_UTIL.getHBaseCluster().getMaster();
        master.getMasterCoprocessorHost().load(TestAlterAndDisableMasterObserver.class, 0, this.TEST_UTIL.getConfiguration());
        ExecutorService executor = Executors.newFixedThreadPool(2);
        Future<Object> alterTableFuture = executor.submit(new Callable<Object>(){

            @Override
            public Object call() throws Exception {
                HBaseAdmin admin = TestTableLockManager.this.TEST_UTIL.getHBaseAdmin();
                admin.addColumn(TABLE_NAME, new HColumnDescriptor(NEW_FAMILY));
                LOG.info("Added new column family");
                HTableDescriptor tableDesc = admin.getTableDescriptor(TABLE_NAME);
                Assert.assertTrue((boolean)tableDesc.getFamiliesKeys().contains(NEW_FAMILY));
                return null;
            }
        });
        Future<Object> disableTableFuture = executor.submit(new Callable<Object>(){

            @Override
            public Object call() throws Exception {
                HBaseAdmin admin = TestTableLockManager.this.TEST_UTIL.getHBaseAdmin();
                admin.disableTable(TABLE_NAME);
                Assert.assertTrue((boolean)admin.isTableDisabled(TABLE_NAME));
                admin.deleteTable(TABLE_NAME);
                Assert.assertFalse((boolean)admin.tableExists(TABLE_NAME));
                return null;
            }
        });
        try {
            disableTableFuture.get();
            alterTableFuture.get();
        }
        catch (ExecutionException e) {
            if (e.getCause() instanceof AssertionError) {
                throw (AssertionError)((Object)e.getCause());
            }
            throw e;
        }
    }

    @Test(timeout=600000L)
    public void testDelete() throws Exception {
        this.prepareMiniCluster();
        HBaseAdmin admin = this.TEST_UTIL.getHBaseAdmin();
        admin.disableTable(TABLE_NAME);
        admin.deleteTable(TABLE_NAME);
        final ZooKeeperWatcher zkWatcher = this.TEST_UTIL.getZooKeeperWatcher();
        final String znode = ZKUtil.joinZNode(zkWatcher.tableLockZNode, TABLE_NAME.getNameAsString());
        this.TEST_UTIL.waitFor(5000L, new Waiter.Predicate<Exception>(){

            @Override
            public boolean evaluate() throws Exception {
                int ver = ZKUtil.checkExists(zkWatcher, znode);
                return ver < 0;
            }
        });
        int ver = ZKUtil.checkExists(zkWatcher, ZKUtil.joinZNode(zkWatcher.tableLockZNode, TABLE_NAME.getNameAsString()));
        Assert.assertTrue((String)("Unexpected znode version " + ver), (ver < 0 ? 1 : 0) != 0);
    }

    @Test(timeout=600000L)
    public void testReapAllTableLocks() throws Exception {
        this.prepareMiniZkCluster();
        ServerName serverName = ServerName.valueOf("localhost:10000", 0L);
        final TableLockManager lockManager = TableLockManager.createTableLockManager(this.TEST_UTIL.getConfiguration(), this.TEST_UTIL.getZooKeeperWatcher(), serverName);
        String[] tables = new String[]{"table1", "table2", "table3", "table4"};
        ExecutorService executor = Executors.newFixedThreadPool(6);
        final CountDownLatch writeLocksObtained = new CountDownLatch(4);
        final CountDownLatch writeLocksAttempted = new CountDownLatch(10);
        for (int i = 0; i < tables.length; ++i) {
            final String table = tables[i];
            for (int j = 0; j < i + 1; ++j) {
                executor.submit(new Callable<Void>(){

                    @Override
                    public Void call() throws Exception {
                        writeLocksAttempted.countDown();
                        lockManager.writeLock(TableName.valueOf(table), "testReapAllTableLocks").acquire();
                        writeLocksObtained.countDown();
                        return null;
                    }
                });
            }
        }
        writeLocksObtained.await();
        writeLocksAttempted.await();
        lockManager.reapWriteLocks();
        this.TEST_UTIL.getConfiguration().setInt("hbase.table.write.lock.timeout.ms", 0);
        TableLockManager zeroTimeoutLockManager = TableLockManager.createTableLockManager(this.TEST_UTIL.getConfiguration(), this.TEST_UTIL.getZooKeeperWatcher(), serverName);
        zeroTimeoutLockManager.writeLock(TableName.valueOf(tables[tables.length - 1]), "zero timeout").acquire();
        executor.shutdownNow();
    }

    @Test(timeout=600000L)
    public void testTableReadLock() throws Exception {
        this.prepareMiniCluster();
        LoadTestTool loadTool = new LoadTestTool();
        loadTool.setConf(this.TEST_UTIL.getConfiguration());
        int numKeys = 10000;
        final TableName tableName = TableName.valueOf("testTableReadLock");
        final HBaseAdmin admin = this.TEST_UTIL.getHBaseAdmin();
        final HTableDescriptor desc = new HTableDescriptor(tableName);
        final byte[] family = Bytes.toBytes("test_cf");
        desc.addFamily(new HColumnDescriptor(family));
        admin.createTable(desc);
        int ret = loadTool.run(new String[]{"-tn", tableName.getNameAsString(), "-write", String.format("%d:%d:%d", 1, 10, 10), "-num_keys", String.valueOf(numKeys), "-skip_init"});
        if (0 != ret) {
            String errorMsg = "Load failed with error code " + ret;
            LOG.error(errorMsg);
            Assert.fail((String)errorMsg);
        }
        int familyValues = admin.getTableDescriptor(tableName).getFamily(family).getValues().size();
        StoppableImplementation stopper = new StoppableImplementation();
        ChoreService choreService = new ChoreService("TEST_SERVER_NAME");
        ScheduledChore alterThread = new ScheduledChore("Alter Chore", stopper, 10000){

            @Override
            protected void chore() {
                Random random = new Random();
                try {
                    HTableDescriptor htd = admin.getTableDescriptor(tableName);
                    String val = String.valueOf(random.nextInt());
                    htd.getFamily(family).setValue(val, val);
                    desc.getFamily(family).setValue(val, val);
                    admin.modifyTable(tableName, htd);
                }
                catch (Exception ex) {
                    LOG.warn("Caught exception", ex);
                    Assert.fail((String)ex.getMessage());
                }
            }
        };
        ScheduledChore splitThread = new ScheduledChore("Split thread", stopper, 5000){

            @Override
            public void chore() {
                try {
                    HRegion region = TestTableLockManager.this.TEST_UTIL.getSplittableRegion(tableName, -1);
                    if (region != null) {
                        byte[] regionName = region.getRegionInfo().getRegionName();
                        admin.flushRegion(regionName);
                        admin.compactRegion(regionName);
                        admin.splitRegion(regionName);
                    } else {
                        LOG.warn("Could not find suitable region for the table.  Possibly the region got closed and the attempts got over before the region could have got reassigned.");
                    }
                }
                catch (NotServingRegionException nsre) {
                    LOG.warn("Caught exception", nsre);
                }
                catch (Exception ex) {
                    LOG.warn("Caught exception", ex);
                    Assert.fail((String)ex.getMessage());
                }
            }
        };
        choreService.scheduleChore(alterThread);
        choreService.scheduleChore(splitThread);
        this.TEST_UTIL.waitTableEnabled(tableName);
        while (true) {
            List<HRegionInfo> regions = admin.getTableRegions(tableName);
            LOG.info(String.format("Table #regions: %d regions: %s:", regions.size(), regions));
            Assert.assertEquals((Object)admin.getTableDescriptor(tableName), (Object)desc);
            for (HRegion region : this.TEST_UTIL.getMiniHBaseCluster().getRegions(tableName)) {
                Assert.assertEquals((Object)desc, (Object)region.getTableDesc());
            }
            if (regions.size() >= 5) break;
            Threads.sleep(1000L);
        }
        stopper.stop("test finished");
        int newFamilyValues = admin.getTableDescriptor(tableName).getFamily(family).getValues().size();
        LOG.info(String.format("Altered the table %d times", newFamilyValues - familyValues));
        Assert.assertTrue((newFamilyValues > familyValues ? 1 : 0) != 0);
        ret = loadTool.run(new String[]{"-tn", tableName.getNameAsString(), "-read", "100:10", "-num_keys", String.valueOf(numKeys), "-skip_init"});
        if (0 != ret) {
            String errorMsg = "Verify failed with error code " + ret;
            LOG.error(errorMsg);
            Assert.fail((String)errorMsg);
        }
        admin.close();
        choreService.shutdown();
    }

    public static class TestAlterAndDisableMasterObserver
    extends BaseMasterObserver {
        @Override
        public void preAddColumnHandler(ObserverContext<MasterCoprocessorEnvironment> ctx, TableName tableName, HColumnDescriptor column) throws IOException {
            LOG.debug("addColumn called");
            addColumn.countDown();
        }

        @Override
        public void postAddColumnHandler(ObserverContext<MasterCoprocessorEnvironment> ctx, TableName tableName, HColumnDescriptor column) throws IOException {
            Threads.sleep(6000L);
            try {
                ctx.getEnvironment().getMasterServices().checkTableModifiable(tableName);
            }
            catch (TableNotDisabledException expected) {
                return;
            }
            catch (IOException ex) {
                // empty catch block
            }
            Assert.fail((String)"was expecting the table to be enabled");
        }

        @Override
        public void preDisableTable(ObserverContext<MasterCoprocessorEnvironment> ctx, TableName tableName) throws IOException {
            try {
                LOG.debug("Waiting for addColumn to be processed first");
                addColumn.await();
                LOG.debug("addColumn started, we can continue");
            }
            catch (InterruptedException ex) {
                LOG.warn("Sleep interrupted while waiting for addColumn countdown");
            }
        }

        @Override
        public void postDisableTableHandler(ObserverContext<MasterCoprocessorEnvironment> ctx, TableName tableName) throws IOException {
            Threads.sleep(3000L);
        }
    }

    public static class TestLockTimeoutExceptionMasterObserver
    extends BaseMasterObserver {
        @Override
        public void preDeleteColumnHandler(ObserverContext<MasterCoprocessorEnvironment> ctx, TableName tableName, byte[] c) throws IOException {
            deleteColumn.countDown();
        }

        @Override
        public void postDeleteColumnHandler(ObserverContext<MasterCoprocessorEnvironment> ctx, TableName tableName, byte[] c) throws IOException {
            Threads.sleep(10000L);
        }

        @Override
        public void preAddColumnHandler(ObserverContext<MasterCoprocessorEnvironment> ctx, TableName tableName, HColumnDescriptor column) throws IOException {
            Assert.fail((String)"Add column should have timeouted out for acquiring the table lock");
        }
    }
}

