/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hive.metastore.tools;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ImmutableMap;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.OutputStream;
import java.io.PrintStream;
import java.net.URI;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.GnuParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.OptionBuilder;
import org.apache.commons.cli.OptionGroup;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.apache.commons.io.output.NullOutputStream;
import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hive.metastore.HiveMetaException;
import org.apache.hadoop.hive.metastore.IMetaStoreSchemaInfo;
import org.apache.hadoop.hive.metastore.MetaStoreSchemaInfoFactory;
import org.apache.hadoop.hive.metastore.TableType;
import org.apache.hadoop.hive.metastore.conf.MetastoreConf;
import org.apache.hadoop.hive.metastore.tools.HiveSchemaHelper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import sqlline.SqlLine;

public class MetastoreSchemaTool {
    private static final Logger LOG = LoggerFactory.getLogger(MetastoreSchemaTool.class);
    private static final String PASSWD_MASK = "[passwd stripped]";
    @VisibleForTesting
    public static String homeDir;
    private String userName = null;
    private String passWord = null;
    private boolean dryRun = false;
    private boolean verbose = false;
    private String dbOpts = null;
    private String url = null;
    private String driver = null;
    private URI[] validationServers = null;
    private String hiveUser;
    private String hivePasswd;
    private String hiveDb;
    private final Configuration conf;
    private final String dbType;
    private final IMetaStoreSchemaInfo metaStoreSchemaInfo;
    private boolean needsQuotedIdentifier;

    private static String findHomeDir() {
        homeDir = homeDir == null ? System.getenv("METASTORE_HOME") : homeDir;
        return homeDir == null ? System.getenv("HIVE_HOME") : homeDir;
    }

    private MetastoreSchemaTool(String dbType) throws HiveMetaException {
        this(MetastoreSchemaTool.findHomeDir(), MetastoreConf.newMetastoreConf(), dbType);
    }

    MetastoreSchemaTool(String metastoreHome, Configuration conf, String dbType) throws HiveMetaException {
        if (metastoreHome == null || metastoreHome.isEmpty()) {
            throw new HiveMetaException("No Metastore home directory provided");
        }
        this.conf = conf;
        this.dbType = dbType;
        this.needsQuotedIdentifier = this.getDbCommandParser(dbType).needsQuotedIdentifier();
        this.metaStoreSchemaInfo = MetaStoreSchemaInfoFactory.get(conf, metastoreHome, dbType);
    }

    Configuration getConf() {
        return this.conf;
    }

    void setUrl(String url) {
        this.url = url;
    }

    void setDriver(String driver) {
        this.driver = driver;
    }

    void setUserName(String userName) {
        this.userName = userName;
    }

    void setPassWord(String passWord) {
        this.passWord = passWord;
    }

    void setDryRun(boolean dryRun) {
        this.dryRun = dryRun;
    }

    void setVerbose(boolean verbose) {
        this.verbose = verbose;
    }

    private void setDbOpts(String dbOpts) {
        this.dbOpts = dbOpts;
    }

    private void setValidationServers(String servers) {
        if (StringUtils.isNotEmpty(servers)) {
            String[] strServers = servers.split(",");
            this.validationServers = new URI[strServers.length];
            for (int i = 0; i < this.validationServers.length; ++i) {
                this.validationServers[i] = new Path(strServers[i]).toUri();
            }
        }
    }

    private void setHiveUser(String hiveUser) {
        this.hiveUser = hiveUser;
    }

    private void setHivePasswd(String hivePasswd) {
        this.hivePasswd = hivePasswd;
    }

    private void setHiveDb(String hiveDb) {
        this.hiveDb = hiveDb;
    }

    private static int usage(Options cmdLineOptions) {
        HelpFormatter formatter = new HelpFormatter();
        formatter.printHelp("schemaTool", cmdLineOptions);
        return 1;
    }

    Connection getConnectionToMetastore(boolean printInfo) throws HiveMetaException {
        return HiveSchemaHelper.getConnectionToMetastore(this.userName, this.passWord, this.url, this.driver, printInfo, this.conf, null);
    }

    private HiveSchemaHelper.NestedScriptParser getDbCommandParser(String dbType) {
        return HiveSchemaHelper.getDbCommandParser(dbType, this.dbOpts, this.userName, this.passWord, this.conf, null, true);
    }

    private void showInfo() throws HiveMetaException {
        String hiveVersion = this.metaStoreSchemaInfo.getHiveSchemaVersion();
        String dbVersion = this.metaStoreSchemaInfo.getMetaStoreSchemaVersion(this.getConnectionInfo(true));
        System.out.println("Hive distribution version:\t " + hiveVersion);
        System.out.println("Metastore schema version:\t " + dbVersion);
        this.assertCompatibleVersion(hiveVersion, dbVersion);
    }

    boolean validateLocations(Connection conn, URI[] defaultServers) throws HiveMetaException {
        System.out.println("Validating DFS locations");
        boolean rtn = this.checkMetaStoreDBLocation(conn, defaultServers);
        rtn = this.checkMetaStoreTableLocation(conn, defaultServers) && rtn;
        rtn = this.checkMetaStorePartitionLocation(conn, defaultServers) && rtn;
        rtn = this.checkMetaStoreSkewedColumnsLocation(conn, defaultServers) && rtn;
        System.out.println((rtn ? "Succeeded" : "Failed") + " in DFS location validation.");
        return rtn;
    }

    private String getNameOrID(ResultSet res, int nameInx, int idInx) throws SQLException {
        String itemName = res.getString(nameInx);
        return itemName == null || itemName.isEmpty() ? "ID: " + res.getString(idInx) : "Name: " + itemName;
    }

    private boolean checkMetaStoreDBLocation(Connection conn, URI[] defaultServers) throws HiveMetaException {
        boolean isValid = true;
        int numOfInvalid = 0;
        String dbLoc = this.needsQuotedIdentifier ? "select dbt.\"DB_ID\", dbt.\"NAME\", dbt.\"DB_LOCATION_URI\" from \"DBS\" dbt order by dbt.\"DB_ID\" " : "select dbt.DB_ID, dbt.NAME, dbt.DB_LOCATION_URI from DBS dbt order by dbt.DB_ID";
        try (Statement stmt = conn.createStatement();
             ResultSet res = stmt.executeQuery(dbLoc);){
            while (res.next()) {
                String locValue = res.getString(3);
                String dbName = this.getNameOrID(res, 2, 1);
                if (this.checkLocation("Database " + dbName, locValue, defaultServers)) continue;
                ++numOfInvalid;
            }
        }
        catch (SQLException e) {
            throw new HiveMetaException("Failed to get DB Location Info.", e);
        }
        if (numOfInvalid > 0) {
            isValid = false;
        }
        return isValid;
    }

    private boolean checkMetaStoreTableLocation(Connection conn, URI[] defaultServers) throws HiveMetaException {
        boolean isValid = true;
        int numOfInvalid = 0;
        String tabIDRange = this.needsQuotedIdentifier ? "select max(\"TBL_ID\"), min(\"TBL_ID\") from \"TBLS\" " : "select max(TBL_ID), min(TBL_ID) from TBLS";
        String tabLoc = this.needsQuotedIdentifier ? "select tbl.\"TBL_ID\", tbl.\"TBL_NAME\", sd.\"LOCATION\", dbt.\"DB_ID\", dbt.\"NAME\" from \"TBLS\" tbl inner join \"SDS\" sd on tbl.\"SD_ID\" = sd.\"SD_ID\" and tbl.\"TBL_TYPE\" != '" + TableType.VIRTUAL_VIEW + "' and tbl.\"TBL_ID\" >= ? and tbl.\"TBL_ID\"<= ? inner join \"DBS\" dbt on tbl.\"DB_ID\" = dbt.\"DB_ID\" order by tbl.\"TBL_ID\" " : "select tbl.TBL_ID, tbl.TBL_NAME, sd.LOCATION, dbt.DB_ID, dbt.NAME from TBLS tbl join SDS sd on tbl.SD_ID = sd.SD_ID and tbl.TBL_TYPE !='" + TableType.VIRTUAL_VIEW + "' and tbl.TBL_ID >= ? and tbl.TBL_ID <= ?  inner join DBS dbt on tbl.DB_ID = dbt.DB_ID order by tbl.TBL_ID";
        long maxID = 0L;
        long minID = 0L;
        long rtnSize = 2000L;
        try {
            Statement stmt = conn.createStatement();
            ResultSet res = stmt.executeQuery(tabIDRange);
            if (res.next()) {
                maxID = res.getLong(1);
                minID = res.getLong(2);
            }
            res.close();
            stmt.close();
            PreparedStatement pStmt = conn.prepareStatement(tabLoc);
            while (minID <= maxID) {
                pStmt.setLong(1, minID);
                pStmt.setLong(2, minID + rtnSize);
                res = pStmt.executeQuery();
                while (res.next()) {
                    String locValue = res.getString(3);
                    String entity = "Database " + this.getNameOrID(res, 5, 4) + ", Table " + this.getNameOrID(res, 2, 1);
                    if (this.checkLocation(entity, locValue, defaultServers)) continue;
                    ++numOfInvalid;
                }
                res.close();
                minID += rtnSize + 1L;
            }
            pStmt.close();
        }
        catch (SQLException e) {
            throw new HiveMetaException("Failed to get Table Location Info.", e);
        }
        if (numOfInvalid > 0) {
            isValid = false;
        }
        return isValid;
    }

    private boolean checkMetaStorePartitionLocation(Connection conn, URI[] defaultServers) throws HiveMetaException {
        boolean isValid = true;
        int numOfInvalid = 0;
        String partIDRange = this.needsQuotedIdentifier ? "select max(\"PART_ID\"), min(\"PART_ID\") from \"PARTITIONS\" " : "select max(PART_ID), min(PART_ID) from PARTITIONS";
        String partLoc = this.needsQuotedIdentifier ? "select pt.\"PART_ID\", pt.\"PART_NAME\", sd.\"LOCATION\", tbl.\"TBL_ID\", tbl.\"TBL_NAME\",dbt.\"DB_ID\", dbt.\"NAME\" from \"PARTITIONS\" pt inner join \"SDS\" sd on pt.\"SD_ID\" = sd.\"SD_ID\" and pt.\"PART_ID\" >= ? and pt.\"PART_ID\"<= ?  inner join \"TBLS\" tbl on pt.\"TBL_ID\" = tbl.\"TBL_ID\" inner join \"DBS\" dbt on tbl.\"DB_ID\" = dbt.\"DB_ID\" order by tbl.\"TBL_ID\" " : "select pt.PART_ID, pt.PART_NAME, sd.LOCATION, tbl.TBL_ID, tbl.TBL_NAME, dbt.DB_ID, dbt.NAME from PARTITIONS pt inner join SDS sd on pt.SD_ID = sd.SD_ID and pt.PART_ID >= ? and pt.PART_ID <= ?  inner join TBLS tbl on tbl.TBL_ID = pt.TBL_ID inner join DBS dbt on tbl.DB_ID = dbt.DB_ID order by tbl.TBL_ID ";
        long maxID = 0L;
        long minID = 0L;
        long rtnSize = 2000L;
        try {
            Statement stmt = conn.createStatement();
            ResultSet res = stmt.executeQuery(partIDRange);
            if (res.next()) {
                maxID = res.getLong(1);
                minID = res.getLong(2);
            }
            res.close();
            stmt.close();
            PreparedStatement pStmt = conn.prepareStatement(partLoc);
            while (minID <= maxID) {
                pStmt.setLong(1, minID);
                pStmt.setLong(2, minID + rtnSize);
                res = pStmt.executeQuery();
                while (res.next()) {
                    String locValue = res.getString(3);
                    String entity = "Database " + this.getNameOrID(res, 7, 6) + ", Table " + this.getNameOrID(res, 5, 4) + ", Partition " + this.getNameOrID(res, 2, 1);
                    if (this.checkLocation(entity, locValue, defaultServers)) continue;
                    ++numOfInvalid;
                }
                res.close();
                minID += rtnSize + 1L;
            }
            pStmt.close();
        }
        catch (SQLException e) {
            throw new HiveMetaException("Failed to get Partiton Location Info.", e);
        }
        if (numOfInvalid > 0) {
            isValid = false;
        }
        return isValid;
    }

    private boolean checkMetaStoreSkewedColumnsLocation(Connection conn, URI[] defaultServers) throws HiveMetaException {
        boolean isValid = true;
        int numOfInvalid = 0;
        String skewedColIDRange = this.needsQuotedIdentifier ? "select max(\"STRING_LIST_ID_KID\"), min(\"STRING_LIST_ID_KID\") from \"SKEWED_COL_VALUE_LOC_MAP\" " : "select max(STRING_LIST_ID_KID), min(STRING_LIST_ID_KID) from SKEWED_COL_VALUE_LOC_MAP";
        String skewedColLoc = this.needsQuotedIdentifier ? "select t.\"TBL_NAME\", t.\"TBL_ID\", sk.\"STRING_LIST_ID_KID\", sk.\"LOCATION\", db.\"NAME\", db.\"DB_ID\"  from \"TBLS\" t, \"SDS\" s, \"DBS\" db, \"SKEWED_COL_VALUE_LOC_MAP\" sk where sk.\"SD_ID\" = s.\"SD_ID\" and s.\"SD_ID\" = t.\"SD_ID\" and t.\"DB_ID\" = db.\"DB_ID\" and sk.\"STRING_LIST_ID_KID\" >= ? and sk.\"STRING_LIST_ID_KID\" <= ? order by t.\"TBL_ID\" " : "select t.TBL_NAME, t.TBL_ID, sk.STRING_LIST_ID_KID, sk.LOCATION, db.NAME, db.DB_ID from TBLS t, SDS s, DBS db, SKEWED_COL_VALUE_LOC_MAP sk where sk.SD_ID = s.SD_ID and s.SD_ID = t.SD_ID and t.DB_ID = db.DB_ID and sk.STRING_LIST_ID_KID >= ? and sk.STRING_LIST_ID_KID <= ? order by t.TBL_ID ";
        long maxID = 0L;
        long minID = 0L;
        long rtnSize = 2000L;
        try {
            Statement stmt = conn.createStatement();
            ResultSet res = stmt.executeQuery(skewedColIDRange);
            if (res.next()) {
                maxID = res.getLong(1);
                minID = res.getLong(2);
            }
            res.close();
            stmt.close();
            PreparedStatement pStmt = conn.prepareStatement(skewedColLoc);
            while (minID <= maxID) {
                pStmt.setLong(1, minID);
                pStmt.setLong(2, minID + rtnSize);
                res = pStmt.executeQuery();
                while (res.next()) {
                    String locValue = res.getString(4);
                    String entity = "Database " + this.getNameOrID(res, 5, 6) + ", Table " + this.getNameOrID(res, 1, 2) + ", String list " + res.getString(3);
                    if (this.checkLocation(entity, locValue, defaultServers)) continue;
                    ++numOfInvalid;
                }
                res.close();
                minID += rtnSize + 1L;
            }
            pStmt.close();
        }
        catch (SQLException e) {
            throw new HiveMetaException("Failed to get skewed columns location info.", e);
        }
        if (numOfInvalid > 0) {
            isValid = false;
        }
        return isValid;
    }

    private boolean checkLocation(String entity, String entityLocation, URI[] defaultServers) {
        boolean isValid = true;
        if (entityLocation == null) {
            MetastoreSchemaTool.logAndPrintToError(entity + ", Error: empty location");
            isValid = false;
        } else {
            try {
                URI currentUri = new Path(entityLocation).toUri();
                String scheme = currentUri.getScheme();
                String path = currentUri.getPath();
                if (StringUtils.isEmpty(scheme)) {
                    MetastoreSchemaTool.logAndPrintToError(entity + ", Location: " + entityLocation + ", Error: missing location scheme.");
                    isValid = false;
                } else if (StringUtils.isEmpty(path)) {
                    MetastoreSchemaTool.logAndPrintToError(entity + ", Location: " + entityLocation + ", Error: missing location path.");
                    isValid = false;
                } else if (ArrayUtils.isNotEmpty(defaultServers) && currentUri.getAuthority() != null) {
                    String authority = currentUri.getAuthority();
                    boolean matchServer = false;
                    for (URI server : defaultServers) {
                        if (!StringUtils.equalsIgnoreCase(server.getScheme(), scheme) || !StringUtils.equalsIgnoreCase(server.getAuthority(), authority)) continue;
                        matchServer = true;
                        break;
                    }
                    if (!matchServer) {
                        MetastoreSchemaTool.logAndPrintToError(entity + ", Location: " + entityLocation + ", Error: mismatched server.");
                        isValid = false;
                    }
                }
                if (isValid && StringUtils.containsOnly(path, "/")) {
                    MetastoreSchemaTool.logAndPrintToError(entity + ", Location: " + entityLocation + ", Warn: location set to root, not a recommended config.");
                }
            }
            catch (Exception pe) {
                MetastoreSchemaTool.logAndPrintToError(entity + ", Error: invalid location - " + pe.getMessage());
                isValid = false;
            }
        }
        return isValid;
    }

    private void testConnectionToMetastore() throws HiveMetaException {
        Connection conn = this.getConnectionToMetastore(true);
        try {
            conn.close();
        }
        catch (SQLException e) {
            throw new HiveMetaException("Failed to close metastore connection", e);
        }
    }

    @VisibleForTesting
    void verifySchemaVersion() throws HiveMetaException {
        if (this.dryRun) {
            return;
        }
        String newSchemaVersion = this.metaStoreSchemaInfo.getMetaStoreSchemaVersion(this.getConnectionInfo(false));
        this.assertCompatibleVersion(this.metaStoreSchemaInfo.getHiveSchemaVersion(), newSchemaVersion);
    }

    private void assertCompatibleVersion(String hiveSchemaVersion, String dbSchemaVersion) throws HiveMetaException {
        if (!this.metaStoreSchemaInfo.isVersionCompatible(hiveSchemaVersion, dbSchemaVersion)) {
            throw new HiveMetaException("Metastore schema version is not compatible. Hive Version: " + hiveSchemaVersion + ", Database Schema Version: " + dbSchemaVersion);
        }
    }

    void doUpgrade() throws HiveMetaException {
        String fromVersion = this.metaStoreSchemaInfo.getMetaStoreSchemaVersion(this.getConnectionInfo(false));
        if (fromVersion == null || fromVersion.isEmpty()) {
            throw new HiveMetaException("Schema version not stored in the metastore. Metastore schema is too old or corrupt. Try specifying the version manually");
        }
        this.doUpgrade(fromVersion);
    }

    private HiveSchemaHelper.MetaStoreConnectionInfo getConnectionInfo(boolean printInfo) {
        return new HiveSchemaHelper.MetaStoreConnectionInfo(this.userName, this.passWord, this.url, this.driver, printInfo, this.conf, this.dbType, this.hiveDb);
    }

    void doUpgrade(String fromSchemaVer) throws HiveMetaException {
        if (this.metaStoreSchemaInfo.getHiveSchemaVersion().equals(fromSchemaVer)) {
            System.out.println("No schema upgrade required from version " + fromSchemaVer);
            return;
        }
        List<String> upgradeScripts = this.metaStoreSchemaInfo.getUpgradeScripts(fromSchemaVer);
        this.testConnectionToMetastore();
        System.out.println("Starting upgrade metastore schema from version " + fromSchemaVer + " to " + this.metaStoreSchemaInfo.getHiveSchemaVersion());
        String scriptDir = this.metaStoreSchemaInfo.getMetaStoreScriptDir();
        try {
            for (String scriptFile : upgradeScripts) {
                System.out.println("Upgrade script " + scriptFile);
                if (this.dryRun) continue;
                this.runPreUpgrade(scriptDir, scriptFile);
                this.runSqlLine(scriptDir, scriptFile);
                System.out.println("Completed " + scriptFile);
            }
        }
        catch (IOException eIO) {
            throw new HiveMetaException("Upgrade FAILED! Metastore state would be inconsistent !!", eIO);
        }
        this.verifySchemaVersion();
    }

    void doInit() throws HiveMetaException {
        this.doInit(this.metaStoreSchemaInfo.getHiveSchemaVersion());
        this.verifySchemaVersion();
    }

    void doInit(String toVersion) throws HiveMetaException {
        this.testConnectionToMetastore();
        System.out.println("Starting metastore schema initialization to " + toVersion);
        String initScriptDir = this.metaStoreSchemaInfo.getMetaStoreScriptDir();
        String initScriptFile = this.metaStoreSchemaInfo.generateInitFileName(toVersion);
        try {
            System.out.println("Initialization script " + initScriptFile);
            if (!this.dryRun) {
                this.runSqlLine(initScriptDir, initScriptFile);
                System.out.println("Initialization script completed");
            }
        }
        catch (IOException e) {
            throw new HiveMetaException("Schema initialization FAILED! Metastore state would be inconsistent !!", e);
        }
    }

    private void doCreateUser() throws HiveMetaException {
        this.testConnectionToMetastore();
        System.out.println("Starting user creation");
        String scriptDir = this.metaStoreSchemaInfo.getMetaStoreScriptDir();
        String protoCreateFile = this.metaStoreSchemaInfo.getCreateUserScript();
        try {
            File createFile = this.subUserAndPassword(scriptDir, protoCreateFile);
            System.out.println("Creation script " + createFile.getAbsolutePath());
            if (!this.dryRun) {
                if ("oracle".equals(this.dbType)) {
                    this.oracleCreateUserHack(createFile);
                } else {
                    this.runSqlLine(createFile.getParent(), createFile.getName());
                }
                System.out.println("User creation completed");
            }
        }
        catch (IOException e) {
            throw new HiveMetaException("User creation FAILED! Metastore unusable !!", e);
        }
    }

    private File subUserAndPassword(String parent, String filename) throws IOException {
        File createFile = File.createTempFile("create-hive-user-" + this.dbType, ".sql");
        BufferedWriter writer = new BufferedWriter(new FileWriter(createFile));
        File proto = new File(parent, filename);
        BufferedReader reader = new BufferedReader(new FileReader(proto));
        reader.lines().map(s -> s.replace("_REPLACE_WITH_USER_", this.hiveUser).replace("_REPLACE_WITH_PASSWD_", this.hivePasswd).replace("_REPLACE_WITH_DB_", this.hiveDb)).forEach(s -> {
            try {
                writer.write((String)s);
                writer.newLine();
            }
            catch (IOException e) {
                throw new RuntimeException("Unable to write to tmp file ", e);
            }
        });
        reader.close();
        writer.close();
        return createFile;
    }

    private void oracleCreateUserHack(File createFile) throws HiveMetaException {
        LOG.debug("Found oracle, hacking our way through it rather than using SqlLine");
        try (BufferedReader reader = new BufferedReader(new FileReader(createFile));
             Connection conn = this.getConnectionToMetastore(false);
             Statement stmt = conn.createStatement();){
            reader.lines().forEach(s -> {
                assert (s.charAt(s.length() - 1) == ';');
                try {
                    stmt.execute(s.substring(0, s.length() - 1));
                }
                catch (SQLException e) {
                    LOG.error("statement <" + s.substring(0, s.length() - 2) + "> failed", (Throwable)e);
                    throw new RuntimeException(e);
                }
            });
        }
        catch (IOException e) {
            LOG.error("Caught IOException trying to read modified create user script " + createFile.getAbsolutePath(), (Throwable)e);
            throw new HiveMetaException(e);
        }
        catch (HiveMetaException e) {
            LOG.error("Failed to connect to RDBMS", (Throwable)e);
            throw e;
        }
        catch (SQLException e) {
            LOG.error("Got SQLException", (Throwable)e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private int doValidate() throws HiveMetaException {
        System.out.println("Starting metastore validation\n");
        Connection conn = this.getConnectionToMetastore(false);
        boolean success = true;
        try {
            if (this.validateSchemaVersions()) {
                System.out.println("[SUCCESS]\n");
            } else {
                success = false;
                System.out.println("[FAIL]\n");
            }
            if (this.validateSequences(conn)) {
                System.out.println("[SUCCESS]\n");
            } else {
                success = false;
                System.out.println("[FAIL]\n");
            }
            if (this.validateSchemaTables(conn)) {
                System.out.println("[SUCCESS]\n");
            } else {
                success = false;
                System.out.println("[FAIL]\n");
            }
            if (this.validateLocations(conn, this.validationServers)) {
                System.out.println("[SUCCESS]\n");
            } else {
                System.out.println("[WARN]\n");
            }
            if (this.validateColumnNullValues(conn)) {
                System.out.println("[SUCCESS]\n");
            } else {
                System.out.println("[WARN]\n");
            }
        }
        finally {
            if (conn != null) {
                try {
                    conn.close();
                }
                catch (SQLException sQLException) {}
            }
        }
        System.out.print("Done with metastore validation: ");
        if (!success) {
            System.out.println("[FAIL]");
            return 1;
        }
        System.out.println("[SUCCESS]");
        return 0;
    }

    boolean validateSequences(Connection conn) throws HiveMetaException {
        ImmutableMap<String, Pair<String, String>> seqNameToTable = new ImmutableMap.Builder<String, Pair<String, String>>().put("MDatabase", Pair.of("DBS", "DB_ID")).put("MRole", Pair.of("ROLES", "ROLE_ID")).put("MGlobalPrivilege", Pair.of("GLOBAL_PRIVS", "USER_GRANT_ID")).put("MTable", Pair.of("TBLS", "TBL_ID")).put("MStorageDescriptor", Pair.of("SDS", "SD_ID")).put("MSerDeInfo", Pair.of("SERDES", "SERDE_ID")).put("MColumnDescriptor", Pair.of("CDS", "CD_ID")).put("MTablePrivilege", Pair.of("TBL_PRIVS", "TBL_GRANT_ID")).put("MTableColumnStatistics", Pair.of("TAB_COL_STATS", "CS_ID")).put("MPartition", Pair.of("PARTITIONS", "PART_ID")).put("MPartitionColumnStatistics", Pair.of("PART_COL_STATS", "CS_ID")).put("MFunction", Pair.of("FUNCS", "FUNC_ID")).put("MIndex", Pair.of("IDXS", "INDEX_ID")).put("MStringList", Pair.of("SKEWED_STRING_LIST", "STRING_LIST_ID")).build();
        System.out.println("Validating sequence number for SEQUENCE_TABLE");
        boolean isValid = true;
        try {
            Statement stmt = conn.createStatement();
            for (String seqName : seqNameToTable.keySet()) {
                long maxId;
                String tableName = (String)((Pair)seqNameToTable.get(seqName)).getLeft();
                String tableKey = (String)((Pair)seqNameToTable.get(seqName)).getRight();
                String fullSequenceName = "org.apache.hadoop.hive.metastore.model." + seqName;
                String seqQuery = this.needsQuotedIdentifier ? "select t.\"NEXT_VAL\" from \"SEQUENCE_TABLE\" t WHERE t.\"SEQUENCE_NAME\"=? order by t.\"SEQUENCE_NAME\" " : "select t.NEXT_VAL from SEQUENCE_TABLE t WHERE t.SEQUENCE_NAME=? order by t.SEQUENCE_NAME ";
                String maxIdQuery = this.needsQuotedIdentifier ? "select max(\"" + tableKey + "\") from \"" + tableName + "\"" : "select max(" + tableKey + ") from " + tableName;
                ResultSet res = stmt.executeQuery(maxIdQuery);
                if (!res.next() || (maxId = res.getLong(1)) <= 0L) continue;
                PreparedStatement pStmt = conn.prepareStatement(seqQuery);
                pStmt.setString(1, fullSequenceName);
                ResultSet resSeq = pStmt.executeQuery();
                if (!resSeq.next()) {
                    isValid = false;
                    MetastoreSchemaTool.logAndPrintToError("Missing SEQUENCE_NAME " + seqName + " from SEQUENCE_TABLE");
                    continue;
                }
                if (resSeq.getLong(1) >= maxId) continue;
                isValid = false;
                MetastoreSchemaTool.logAndPrintToError("NEXT_VAL for " + seqName + " in SEQUENCE_TABLE < max(" + tableKey + ") in " + tableName);
            }
            System.out.println((isValid ? "Succeeded" : "Failed") + " in sequence number validation for SEQUENCE_TABLE.");
            return isValid;
        }
        catch (SQLException e) {
            throw new HiveMetaException("Failed to validate sequence number for SEQUENCE_TABLE", e);
        }
    }

    boolean validateSchemaVersions() throws HiveMetaException {
        System.out.println("Validating schema version");
        try {
            String newSchemaVersion = this.metaStoreSchemaInfo.getMetaStoreSchemaVersion(this.getConnectionInfo(false));
            this.assertCompatibleVersion(this.metaStoreSchemaInfo.getHiveSchemaVersion(), newSchemaVersion);
        }
        catch (HiveMetaException hme) {
            if (hme.getMessage().contains("Metastore schema version is not compatible") || hme.getMessage().contains("Multiple versions were found in metastore") || hme.getMessage().contains("Could not find version info in metastore VERSION table")) {
                MetastoreSchemaTool.logAndPrintToError(hme.getMessage());
                System.out.println("Failed in schema version validation.");
                return false;
            }
            throw hme;
        }
        System.out.println("Succeeded in schema version validation.");
        return true;
    }

    boolean validateSchemaTables(Connection conn) throws HiveMetaException {
        String version;
        ResultSet rs = null;
        ArrayList<String> dbTables = new ArrayList<String>();
        ArrayList<String> schemaTables = new ArrayList<String>();
        ArrayList<String> subScripts = new ArrayList<String>();
        System.out.println("Validating metastore schema tables");
        try {
            version = this.metaStoreSchemaInfo.getMetaStoreSchemaVersion(this.getConnectionInfo(false));
        }
        catch (HiveMetaException he) {
            MetastoreSchemaTool.logAndPrintToError("Failed to determine schema version from Hive Metastore DB. " + he.getMessage());
            System.out.println("Failed in schema table validation.");
            LOG.debug("Failed to determine schema version from Hive Metastore DB," + he.getMessage());
            return false;
        }
        Connection hmsConn = this.getConnectionToMetastore(false);
        LOG.debug("Validating tables in the schema for version " + version);
        try {
            DatabaseMetaData metadata = conn.getMetaData();
            String[] types = new String[]{"TABLE"};
            rs = metadata.getTables(null, hmsConn.getSchema(), "%", types);
            while (rs.next()) {
                String table = rs.getString("TABLE_NAME");
                dbTables.add(table.toLowerCase());
                LOG.debug("Found table " + table + " in HMS dbstore");
            }
        }
        catch (SQLException e) {
            throw new HiveMetaException("Failed to retrieve schema tables from Hive Metastore DB," + e.getMessage());
        }
        finally {
            if (rs != null) {
                try {
                    rs.close();
                }
                catch (SQLException sQLException) {}
            }
        }
        String baseDir = new File(this.metaStoreSchemaInfo.getMetaStoreScriptDir()).getParent();
        Object schemaFile = new File(this.metaStoreSchemaInfo.getMetaStoreScriptDir(), this.metaStoreSchemaInfo.generateInitFileName(version)).getPath();
        try {
            LOG.debug("Parsing schema script " + (String)schemaFile);
            subScripts.addAll(this.findCreateTable((String)schemaFile, schemaTables));
            while (subScripts.size() > 0) {
                schemaFile = baseDir + "/" + this.dbType + "/" + (String)subScripts.remove(0);
                LOG.debug("Parsing subscript " + (String)schemaFile);
                subScripts.addAll(this.findCreateTable((String)schemaFile, schemaTables));
            }
        }
        catch (Exception e) {
            MetastoreSchemaTool.logAndPrintToError("Exception in parsing schema file. Cause:" + e.getMessage());
            System.out.println("Failed in schema table validation.");
            return false;
        }
        LOG.debug("Schema tables:[ " + Arrays.toString(schemaTables.toArray()) + " ]");
        LOG.debug("DB tables:[ " + Arrays.toString(dbTables.toArray()) + " ]");
        schemaTables.removeAll(dbTables);
        if (schemaTables.size() > 0) {
            Collections.sort(schemaTables);
            MetastoreSchemaTool.logAndPrintToError("Table(s) [ " + Arrays.toString(schemaTables.toArray()) + " ] are missing from the metastore database schema.");
            System.out.println("Failed in schema table validation.");
            return false;
        }
        System.out.println("Succeeded in schema table validation.");
        return true;
    }

    private List<String> findCreateTable(String path, List<String> tableList) throws Exception {
        HiveSchemaHelper.NestedScriptParser sp = HiveSchemaHelper.getDbCommandParser(this.dbType, true);
        ArrayList<String> subs = new ArrayList<String>();
        int groupNo = 2;
        Pattern regexp = Pattern.compile("CREATE TABLE(\\s+IF NOT EXISTS)?\\s+(\\S+).*");
        if (!new File(path).exists()) {
            throw new Exception(path + " does not exist. Potentially incorrect version in the metastore VERSION table");
        }
        try (BufferedReader reader = new BufferedReader(new FileReader(path));){
            String line;
            while ((line = reader.readLine()) != null) {
                if (sp.isNestedScript(line)) {
                    String subScript = sp.getScriptName(line);
                    LOG.debug("Schema subscript " + subScript + " found");
                    subs.add(subScript);
                    continue;
                }
                line = line.replaceAll("( )+", " ");
                line = line.replaceAll("\\(", " ");
                line = line.replaceAll("IF NOT EXISTS ", "");
                line = line.replaceAll("`", "");
                line = line.replaceAll("'", "");
                Matcher matcher = regexp.matcher(line = line.replaceAll("\"", ""));
                if (!matcher.find()) continue;
                String table = matcher.group(groupNo);
                if (this.dbType.equals("derby")) {
                    table = table.replaceAll("APP\\.", "");
                }
                tableList.add(table.toLowerCase());
                LOG.debug("Found table " + table + " in the schema");
            }
        }
        catch (IOException ex) {
            throw new Exception(ex.getMessage());
        }
        return subs;
    }

    boolean validateColumnNullValues(Connection conn) throws HiveMetaException {
        System.out.println("Validating columns for incorrect NULL values.");
        boolean isValid = true;
        try {
            Statement stmt = conn.createStatement();
            String tblQuery = this.needsQuotedIdentifier ? "select t.* from \"TBLS\" t WHERE t.\"SD_ID\" IS NULL and (t.\"TBL_TYPE\"='" + TableType.EXTERNAL_TABLE + "' or t.\"TBL_TYPE\"='" + TableType.MANAGED_TABLE + "') order by t.\"TBL_ID\" " : "select t.* from TBLS t WHERE t.SD_ID IS NULL and (t.TBL_TYPE='" + TableType.EXTERNAL_TABLE + "' or t.TBL_TYPE='" + TableType.MANAGED_TABLE + "') order by t.TBL_ID ";
            ResultSet res = stmt.executeQuery(tblQuery);
            while (res.next()) {
                long tableId = res.getLong("TBL_ID");
                String tableName = res.getString("TBL_NAME");
                String tableType = res.getString("TBL_TYPE");
                isValid = false;
                MetastoreSchemaTool.logAndPrintToError("SD_ID in TBLS should not be NULL for Table Name=" + tableName + ", Table ID=" + tableId + ", Table Type=" + tableType);
            }
            System.out.println((isValid ? "Succeeded" : "Failed") + " in column validation for incorrect NULL values.");
            return isValid;
        }
        catch (SQLException e) {
            throw new HiveMetaException("Failed to validate columns for incorrect NULL values", e);
        }
    }

    private void runPreUpgrade(String scriptDir, String scriptFile) {
        String preUpgradeScript;
        File preUpgradeScriptFile;
        int i = 0;
        while ((preUpgradeScriptFile = new File(scriptDir, preUpgradeScript = this.metaStoreSchemaInfo.getPreUpgradeScriptName(i, scriptFile))).isFile()) {
            block3: {
                try {
                    this.runSqlLine(scriptDir, preUpgradeScript);
                    System.out.println("Completed " + preUpgradeScript);
                }
                catch (Exception e) {
                    MetastoreSchemaTool.logAndPrintToError("Warning in pre-upgrade script " + preUpgradeScript + ": " + e.getMessage());
                    if (!this.verbose) break block3;
                    e.printStackTrace();
                }
            }
            ++i;
        }
    }

    private void runSqlLine(String scriptDir, String scriptFile) throws IOException, HiveMetaException {
        this.runSqlLine(scriptDir + File.separatorChar + scriptFile);
    }

    void runSqlLine(String sqlScriptFile) throws IOException {
        CommandBuilder builder = new CommandBuilder(this.conf, this.url, this.driver, this.userName, this.passWord, sqlScriptFile);
        SqlLine sqlLine = new SqlLine();
        ByteArrayOutputStream outputForLog = null;
        if (!this.verbose) {
            Object out;
            if (LOG.isDebugEnabled()) {
                outputForLog = new ByteArrayOutputStream();
                out = outputForLog;
            } else {
                out = new NullOutputStream();
            }
            sqlLine.setOutputStream(new PrintStream((OutputStream)out));
            System.setProperty("sqlline.silent", "true");
        }
        LOG.info("Going to run command <" + builder.buildToLog() + ">");
        SqlLine.Status status = sqlLine.begin(builder.buildToRun(), null, false);
        if (LOG.isDebugEnabled() && outputForLog != null) {
            LOG.debug("Received following output from Sqlline:");
            LOG.debug(outputForLog.toString("UTF-8"));
        }
        if (status != SqlLine.Status.OK) {
            throw new IOException("Schema script failed, errorcode " + status);
        }
    }

    private static void initOptions(Options cmdLineOptions) {
        Option help = new Option("help", "print this message");
        Option upgradeOpt = new Option("upgradeSchema", "Schema upgrade");
        OptionBuilder.withArgName((String)"upgradeFrom");
        OptionBuilder.hasArg();
        OptionBuilder.withDescription((String)"Schema upgrade from a version");
        Option upgradeFromOpt = OptionBuilder.create((String)"upgradeSchemaFrom");
        Option initOpt = new Option("initSchema", "Schema initialization");
        OptionBuilder.withArgName((String)"initTo");
        OptionBuilder.hasArg();
        OptionBuilder.withDescription((String)"Schema initialization to a version");
        Option initToOpt = OptionBuilder.create((String)"initSchemaTo");
        Option infoOpt = new Option("info", "Show config and schema details");
        Option validateOpt = new Option("validate", "Validate the database");
        Option createUserOpt = new Option("createUser", "Create the Hive user, use admin user and password with this");
        OptionGroup optGroup = new OptionGroup();
        optGroup.addOption(upgradeOpt).addOption(initOpt).addOption(help).addOption(upgradeFromOpt).addOption(createUserOpt).addOption(initToOpt).addOption(infoOpt).addOption(validateOpt);
        optGroup.setRequired(true);
        OptionBuilder.withArgName((String)"user");
        OptionBuilder.hasArgs();
        OptionBuilder.withDescription((String)"Override config file user name");
        Option userNameOpt = OptionBuilder.create((String)"userName");
        OptionBuilder.withArgName((String)"password");
        OptionBuilder.hasArgs();
        OptionBuilder.withDescription((String)"Override config file password");
        Option passwdOpt = OptionBuilder.create((String)"passWord");
        OptionBuilder.hasArg();
        OptionBuilder.withDescription((String)"Hive user (for use with createUser)");
        Option hiveUserOpt = OptionBuilder.create((String)"hiveUser");
        OptionBuilder.hasArg();
        OptionBuilder.withDescription((String)"Hive password (for use with createUser)");
        Option hivePasswdOpt = OptionBuilder.create((String)"hivePassword");
        OptionBuilder.hasArg();
        OptionBuilder.withDescription((String)"Hive database (for use with createUser)");
        Option hiveDbOpt = OptionBuilder.create((String)"hiveDb");
        OptionBuilder.withArgName((String)"databaseType");
        OptionBuilder.hasArgs();
        OptionBuilder.withDescription((String)"Metastore database type");
        Option dbTypeOpt = OptionBuilder.create((String)"dbType");
        OptionBuilder.withArgName((String)"url");
        OptionBuilder.hasArgs();
        OptionBuilder.withDescription((String)"connection url to the database");
        Option urlOpt = OptionBuilder.create((String)"url");
        OptionBuilder.withArgName((String)"driver");
        OptionBuilder.hasArgs();
        OptionBuilder.withDescription((String)"driver name for connection");
        Option driverOpt = OptionBuilder.create((String)"driver");
        OptionBuilder.withArgName((String)"databaseOpts");
        OptionBuilder.hasArgs();
        OptionBuilder.withDescription((String)"Backend DB specific options");
        Option dbOpts = OptionBuilder.create((String)"dbOpts");
        Option dryRunOpt = new Option("dryRun", "list SQL scripts (no execute)");
        Option verboseOpt = new Option("verbose", "only print SQL statements");
        OptionBuilder.withArgName((String)"serverList");
        OptionBuilder.hasArgs();
        OptionBuilder.withDescription((String)"a comma-separated list of servers used in location validation in the format of scheme://authority (e.g. maprfs://localhost:8000)");
        Option serversOpt = OptionBuilder.create((String)"servers");
        cmdLineOptions.addOption(help);
        cmdLineOptions.addOption(dryRunOpt);
        cmdLineOptions.addOption(userNameOpt);
        cmdLineOptions.addOption(passwdOpt);
        cmdLineOptions.addOption(dbTypeOpt);
        cmdLineOptions.addOption(verboseOpt);
        cmdLineOptions.addOption(urlOpt);
        cmdLineOptions.addOption(driverOpt);
        cmdLineOptions.addOption(dbOpts);
        cmdLineOptions.addOption(serversOpt);
        cmdLineOptions.addOption(hiveUserOpt);
        cmdLineOptions.addOption(hivePasswdOpt);
        cmdLineOptions.addOption(hiveDbOpt);
        cmdLineOptions.addOptionGroup(optGroup);
    }

    private static void logAndPrintToError(String errmsg) {
        LOG.error(errmsg);
        System.err.println(errmsg);
    }

    public static void main(String[] args) {
        System.exit(MetastoreSchemaTool.run(args));
    }

    public static int run(String[] args) {
        block35: {
            String dbType;
            CommandLine line;
            LOG.debug("Going to run command: " + StringUtils.join((Object[])args, " "));
            GnuParser parser = new GnuParser();
            Options cmdLineOptions = new Options();
            MetastoreSchemaTool.initOptions(cmdLineOptions);
            try {
                line = parser.parse(cmdLineOptions, args);
            }
            catch (ParseException e) {
                MetastoreSchemaTool.logAndPrintToError("HiveSchemaTool:Parsing failed.  Reason: " + e.getLocalizedMessage());
                return MetastoreSchemaTool.usage(cmdLineOptions);
            }
            assert (line != null);
            if (line.hasOption("help")) {
                HelpFormatter formatter = new HelpFormatter();
                formatter.printHelp("schemaTool", cmdLineOptions);
                return 1;
            }
            if (line.hasOption("dbType")) {
                dbType = line.getOptionValue("dbType");
                if (!(dbType.equalsIgnoreCase("derby") || dbType.equalsIgnoreCase("mssql") || dbType.equalsIgnoreCase("mysql") || dbType.equalsIgnoreCase("postgres") || dbType.equalsIgnoreCase("oracle"))) {
                    MetastoreSchemaTool.logAndPrintToError("Unsupported dbType " + dbType);
                    return MetastoreSchemaTool.usage(cmdLineOptions);
                }
            } else {
                MetastoreSchemaTool.logAndPrintToError("no dbType supplied");
                return MetastoreSchemaTool.usage(cmdLineOptions);
            }
            System.setProperty(MetastoreConf.ConfVars.SCHEMA_VERIFICATION.toString(), "true");
            try {
                MetastoreSchemaTool schemaTool = new MetastoreSchemaTool(dbType);
                if (line.hasOption("userName")) {
                    schemaTool.setUserName(line.getOptionValue("userName"));
                } else {
                    schemaTool.setUserName(MetastoreConf.getVar(schemaTool.getConf(), MetastoreConf.ConfVars.CONNECTION_USER_NAME));
                }
                if (line.hasOption("passWord")) {
                    schemaTool.setPassWord(line.getOptionValue("passWord"));
                } else {
                    try {
                        schemaTool.setPassWord(MetastoreConf.getPassword(schemaTool.getConf(), MetastoreConf.ConfVars.PWD));
                    }
                    catch (IOException err) {
                        throw new HiveMetaException("Error getting metastore password", err);
                    }
                }
                if (line.hasOption("hiveUser")) {
                    schemaTool.setHiveUser(line.getOptionValue("hiveUser"));
                }
                if (line.hasOption("hivePassword")) {
                    schemaTool.setHivePasswd(line.getOptionValue("hivePassword"));
                }
                if (line.hasOption("hiveDb")) {
                    schemaTool.setHiveDb(line.getOptionValue("hiveDb"));
                }
                if (line.hasOption("url")) {
                    schemaTool.setUrl(line.getOptionValue("url"));
                }
                if (line.hasOption("driver")) {
                    schemaTool.setDriver(line.getOptionValue("driver"));
                }
                if (line.hasOption("dryRun")) {
                    schemaTool.setDryRun(true);
                }
                if (line.hasOption("verbose")) {
                    schemaTool.setVerbose(true);
                }
                if (line.hasOption("dbOpts")) {
                    schemaTool.setDbOpts(line.getOptionValue("dbOpts"));
                }
                if (line.hasOption("validate") && line.hasOption("servers")) {
                    schemaTool.setValidationServers(line.getOptionValue("servers"));
                }
                if (line.hasOption("info")) {
                    schemaTool.showInfo();
                    break block35;
                }
                if (line.hasOption("upgradeSchema")) {
                    schemaTool.doUpgrade();
                    break block35;
                }
                if (line.hasOption("upgradeSchemaFrom")) {
                    String schemaVer = line.getOptionValue("upgradeSchemaFrom");
                    schemaTool.doUpgrade(schemaVer);
                    break block35;
                }
                if (line.hasOption("initSchema")) {
                    schemaTool.doInit();
                    break block35;
                }
                if (line.hasOption("initSchemaTo")) {
                    String schemaVer = line.getOptionValue("initSchemaTo");
                    schemaTool.doInit(schemaVer);
                    break block35;
                }
                if (line.hasOption("validate")) {
                    return schemaTool.doValidate();
                }
                if (line.hasOption("createUser")) {
                    schemaTool.doCreateUser();
                    break block35;
                }
                MetastoreSchemaTool.logAndPrintToError("no valid option supplied");
                return MetastoreSchemaTool.usage(cmdLineOptions);
            }
            catch (HiveMetaException e) {
                MetastoreSchemaTool.logAndPrintToError(e.getMessage());
                if (e.getCause() != null) {
                    Throwable t = e.getCause();
                    MetastoreSchemaTool.logAndPrintToError("Underlying cause: " + t.getClass().getName() + " : " + t.getMessage());
                    if (e.getCause() instanceof SQLException) {
                        MetastoreSchemaTool.logAndPrintToError("SQL Error code: " + ((SQLException)t).getErrorCode());
                    }
                }
                if (line.hasOption("verbose")) {
                    e.printStackTrace();
                } else {
                    MetastoreSchemaTool.logAndPrintToError("Use --verbose for detailed stacktrace.");
                }
                MetastoreSchemaTool.logAndPrintToError("*** schemaTool failed ***");
                return 1;
            }
        }
        System.out.println("schemaTool completed");
        return 0;
    }

    static class CommandBuilder {
        private final Configuration conf;
        private final String userName;
        private final String password;
        private final String sqlScriptFile;
        private final String driver;
        private final String url;

        CommandBuilder(Configuration conf, String url, String driver, String userName, String password, String sqlScriptFile) {
            this.conf = conf;
            this.userName = userName;
            this.password = password;
            this.url = url;
            this.driver = driver;
            this.sqlScriptFile = sqlScriptFile;
        }

        String[] buildToRun() throws IOException {
            return this.argsWith(this.password);
        }

        String buildToLog() throws IOException {
            this.logScript();
            return StringUtils.join((Object[])this.argsWith(MetastoreSchemaTool.PASSWD_MASK), " ");
        }

        private String[] argsWith(String password) throws IOException {
            return new String[]{"-u", this.url == null ? MetastoreConf.getVar(this.conf, MetastoreConf.ConfVars.CONNECT_URL_KEY) : this.url, "-d", this.driver == null ? MetastoreConf.getVar(this.conf, MetastoreConf.ConfVars.CONNECTION_DRIVER) : this.driver, "-n", this.userName, "-p", password, "--isolation=TRANSACTION_READ_COMMITTED", "-f", this.sqlScriptFile};
        }

        private void logScript() throws IOException {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Going to invoke file that contains:");
                try (BufferedReader reader = new BufferedReader(new FileReader(this.sqlScriptFile));){
                    String line;
                    while ((line = reader.readLine()) != null) {
                        LOG.debug("script: " + line);
                    }
                }
            }
        }
    }
}

