/*
 * Decompiled with CFR 0.152.
 */
package org.apache.commons.compress.utils;

import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.nio.channels.ClosedChannelException;
import java.nio.channels.WritableByteChannel;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.commons.compress.utils.FixedLengthBlockOutputStream;
import org.hamcrest.Matcher;
import org.hamcrest.MatcherAssert;
import org.hamcrest.Matchers;
import org.hamcrest.core.IsInstanceOf;
import org.junit.Assert;
import org.junit.Test;

public class FixedLengthBlockOutputStreamTest {
    @Test
    public void testSmallWrite() throws IOException {
        this.testWriteAndPad(10240, "hello world!\n", false);
        this.testWriteAndPad(512, "hello world!\n", false);
        this.testWriteAndPad(11, "hello world!\n", false);
        this.testWriteAndPad(3, "hello world!\n", false);
    }

    @Test
    public void testSmallWriteToStream() throws IOException {
        this.testWriteAndPadToStream(10240, "hello world!\n", false);
        this.testWriteAndPadToStream(512, "hello world!\n", false);
        this.testWriteAndPadToStream(11, "hello world!\n", false);
        this.testWriteAndPadToStream(3, "hello     world!\n", false);
    }

    @Test
    public void testWriteSingleBytes() throws IOException {
        int blockSize = 4;
        MockWritableByteChannel mock = new MockWritableByteChannel(4, false);
        ByteArrayOutputStream bos = mock.bos;
        String text = "hello world avengers";
        byte[] msg = "hello world avengers".getBytes();
        int len = msg.length;
        try (FixedLengthBlockOutputStream out = new FixedLengthBlockOutputStream((WritableByteChannel)mock, 4);){
            for (int i = 0; i < len; ++i) {
                out.write((int)msg[i]);
            }
        }
        byte[] output = bos.toByteArray();
        this.validate(4, msg, output);
    }

    @Test
    public void testWriteBuf() throws IOException {
        String hwa = "hello world avengers";
        this.testBuf(4, "hello world avengers");
        this.testBuf(512, "hello world avengers");
        this.testBuf(10240, "hello world avengers");
        this.testBuf(11, "hello world avengershello world avengershello world avengers");
    }

    @Test
    public void testMultiWriteBuf() throws IOException {
        int i;
        int blockSize = 13;
        MockWritableByteChannel mock = new MockWritableByteChannel(13, false);
        String testString = "hello world";
        byte[] msg = "hello world".getBytes();
        int reps = 17;
        try (FixedLengthBlockOutputStream out = new FixedLengthBlockOutputStream((WritableByteChannel)mock, 13);){
            for (int i2 = 0; i2 < 17; ++i2) {
                ByteBuffer buf = this.getByteBuffer(msg);
                out.write(buf);
            }
        }
        ByteArrayOutputStream bos = mock.bos;
        double v = Math.ceil((double)(17 * msg.length) / 13.0) * 13.0;
        Assert.assertEquals((String)"wrong size", (long)((long)v), (long)bos.size());
        int strLen = msg.length * 17;
        byte[] output = bos.toByteArray();
        String l = new String(output, 0, strLen);
        StringBuilder buf = new StringBuilder(strLen);
        for (i = 0; i < 17; ++i) {
            buf.append("hello world");
        }
        Assert.assertEquals((Object)buf.toString(), (Object)l);
        for (i = strLen; i < output.length; ++i) {
            Assert.assertEquals((long)0L, (long)output[i]);
        }
    }

    @Test
    public void testPartialWritingThrowsException() {
        try {
            this.testWriteAndPad(512, "hello world!\n", true);
            Assert.fail((String)"Exception for partial write not thrown");
        }
        catch (IOException e) {
            String msg = e.getMessage();
            Assert.assertEquals((String)"exception message", (Object)"Failed to write 512 bytes atomically. Only wrote  511", (Object)msg);
        }
    }

    @Test
    public void testWriteFailsAfterFLClosedThrowsException() {
        FixedLengthBlockOutputStream out;
        try {
            out = this.getClosedFLBOS();
            out.write(1);
            Assert.fail((String)"expected Closed Channel Exception");
        }
        catch (IOException e) {
            MatcherAssert.assertThat((Object)e, (Matcher)IsInstanceOf.instanceOf(ClosedChannelException.class));
        }
        try {
            out = this.getClosedFLBOS();
            out.write(new byte[]{0, 1, 2, 3});
            Assert.fail((String)"expected Closed Channel Exception");
        }
        catch (IOException e) {
            MatcherAssert.assertThat((Object)e, (Matcher)IsInstanceOf.instanceOf(ClosedChannelException.class));
        }
        try {
            out = this.getClosedFLBOS();
            out.write(ByteBuffer.wrap(new byte[]{0, 1, 2, 3}));
            Assert.fail((String)"expected Closed Channel Exception");
        }
        catch (IOException e) {
            MatcherAssert.assertThat((Object)e, (Matcher)IsInstanceOf.instanceOf(ClosedChannelException.class));
        }
    }

    private FixedLengthBlockOutputStream getClosedFLBOS() throws IOException {
        int blockSize = 512;
        FixedLengthBlockOutputStream out = new FixedLengthBlockOutputStream((OutputStream)new MockOutputStream(512, false), 512);
        out.write(1);
        Assert.assertTrue((boolean)out.isOpen());
        out.close();
        Assert.assertFalse((boolean)out.isOpen());
        return out;
    }

    @Test
    public void testWriteFailsAfterDestClosedThrowsException() {
        int blockSize = 2;
        MockOutputStream mock = new MockOutputStream(2, false);
        FixedLengthBlockOutputStream out = new FixedLengthBlockOutputStream((OutputStream)mock, 2);
        try {
            out.write(1);
            Assert.assertTrue((boolean)out.isOpen());
            mock.close();
            out.write(1);
            Assert.fail((String)"expected IO Exception");
        }
        catch (IOException iOException) {
            // empty catch block
        }
        Assert.assertFalse((boolean)out.isOpen());
    }

    @Test
    public void testWithFileOutputStream() throws IOException {
        int i;
        Path tempFile = Files.createTempFile("xxx", "yyy", new FileAttribute[0]);
        Runtime.getRuntime().addShutdownHook(new Thread(() -> {
            try {
                Files.deleteIfExists(tempFile);
            }
            catch (IOException iOException) {
                // empty catch block
            }
        }));
        int blockSize = 512;
        int reps = 1000;
        OutputStream os = Files.newOutputStream(tempFile.toFile().toPath(), new OpenOption[0]);
        try (FixedLengthBlockOutputStream out = new FixedLengthBlockOutputStream(os, 512);){
            DataOutputStream dos = new DataOutputStream((OutputStream)out);
            for (int i2 = 0; i2 < 1000; ++i2) {
                dos.writeInt(i2);
            }
        }
        long expectedDataSize = 4000L;
        long expectedFileSize = (long)Math.ceil(7.8125) * 512L;
        Assert.assertEquals((String)"file size", (long)expectedFileSize, (long)Files.size(tempFile));
        DataInputStream din = new DataInputStream(Files.newInputStream(tempFile, new OpenOption[0]));
        for (i = 0; i < 1000; ++i) {
            Assert.assertEquals((String)"file int", (long)i, (long)din.readInt());
        }
        i = 0;
        while ((long)i < expectedFileSize - 4000L) {
            Assert.assertEquals((long)0L, (long)din.read());
            ++i;
        }
        Assert.assertEquals((long)-1L, (long)din.read());
    }

    private void testBuf(int blockSize, String text) throws IOException {
        MockWritableByteChannel mock = new MockWritableByteChannel(blockSize, false);
        ByteArrayOutputStream bos = mock.bos;
        byte[] msg = text.getBytes();
        ByteBuffer buf = this.getByteBuffer(msg);
        try (FixedLengthBlockOutputStream out = new FixedLengthBlockOutputStream((WritableByteChannel)mock, blockSize);){
            out.write(buf);
        }
        double v = Math.ceil((double)msg.length / (double)blockSize) * (double)blockSize;
        Assert.assertEquals((String)"wrong size", (long)((long)v), (long)bos.size());
        byte[] output = bos.toByteArray();
        String l = new String(output, 0, msg.length);
        Assert.assertEquals((Object)text, (Object)l);
        for (int i = msg.length; i < bos.size(); ++i) {
            Assert.assertEquals((String)String.format("output[%d]", i), (long)0L, (long)output[i]);
        }
    }

    private ByteBuffer getByteBuffer(byte[] msg) {
        int len = msg.length;
        ByteBuffer buf = ByteBuffer.allocate(len);
        buf.put(msg);
        buf.flip();
        return buf;
    }

    private void testWriteAndPad(int blockSize, String text, boolean doPartialWrite) throws IOException {
        MockWritableByteChannel mock = new MockWritableByteChannel(blockSize, doPartialWrite);
        byte[] msg = text.getBytes(StandardCharsets.US_ASCII);
        ByteArrayOutputStream bos = mock.bos;
        try (FixedLengthBlockOutputStream out = new FixedLengthBlockOutputStream((WritableByteChannel)mock, blockSize);){
            out.write(msg);
            Assert.assertEquals((String)"no partial write", (long)(msg.length / blockSize * blockSize), (long)bos.size());
        }
        this.validate(blockSize, msg, bos.toByteArray());
    }

    private void testWriteAndPadToStream(int blockSize, String text, boolean doPartialWrite) throws IOException {
        MockOutputStream mock = new MockOutputStream(blockSize, doPartialWrite);
        byte[] msg = text.getBytes(StandardCharsets.US_ASCII);
        ByteArrayOutputStream bos = mock.bos;
        try (FixedLengthBlockOutputStream out = new FixedLengthBlockOutputStream((OutputStream)mock, blockSize);){
            out.write(msg);
            Assert.assertEquals((String)"no partial write", (long)(msg.length / blockSize * blockSize), (long)bos.size());
        }
        this.validate(blockSize, msg, bos.toByteArray());
    }

    private void validate(int blockSize, byte[] expectedBytes, byte[] actualBytes) {
        double v = Math.ceil((double)expectedBytes.length / (double)blockSize) * (double)blockSize;
        Assert.assertEquals((String)"wrong size", (long)((long)v), (long)actualBytes.length);
        FixedLengthBlockOutputStreamTest.assertContainsAtOffset("output", expectedBytes, 0, actualBytes);
        for (int i = expectedBytes.length; i < actualBytes.length; ++i) {
            Assert.assertEquals((String)String.format("output[%d]", i), (long)0L, (long)actualBytes[i]);
        }
    }

    private static void assertContainsAtOffset(String msg, byte[] expected, int offset, byte[] actual) {
        MatcherAssert.assertThat((Object)actual.length, (Matcher)Matchers.greaterThanOrEqualTo((Comparable)Integer.valueOf(offset + expected.length)));
        for (int i = 0; i < expected.length; ++i) {
            Assert.assertEquals((String)String.format("%s ([%d])", msg, i), (long)expected[i], (long)actual[i + offset]);
        }
    }

    private static class MockWritableByteChannel
    implements WritableByteChannel {
        final ByteArrayOutputStream bos = new ByteArrayOutputStream();
        private final int requiredWriteSize;
        private final boolean doPartialWrite;
        final AtomicBoolean closed = new AtomicBoolean();

        private MockWritableByteChannel(int requiredWriteSize, boolean doPartialWrite) {
            this.requiredWriteSize = requiredWriteSize;
            this.doPartialWrite = doPartialWrite;
        }

        @Override
        public int write(ByteBuffer src) throws IOException {
            Assert.assertEquals((String)"write size", (long)this.requiredWriteSize, (long)src.remaining());
            if (this.doPartialWrite) {
                src.limit(src.limit() - 1);
            }
            int bytesOut = src.remaining();
            while (src.hasRemaining()) {
                this.bos.write(src.get());
            }
            return bytesOut;
        }

        @Override
        public boolean isOpen() {
            return !this.closed.get();
        }

        @Override
        public void close() throws IOException {
            this.closed.compareAndSet(false, true);
        }
    }

    private static class MockOutputStream
    extends OutputStream {
        final ByteArrayOutputStream bos = new ByteArrayOutputStream();
        private final int requiredWriteSize;
        private final boolean doPartialWrite;
        private final AtomicBoolean closed = new AtomicBoolean();

        private MockOutputStream(int requiredWriteSize, boolean doPartialWrite) {
            this.requiredWriteSize = requiredWriteSize;
            this.doPartialWrite = doPartialWrite;
        }

        @Override
        public void write(byte[] b, int off, int len) throws IOException {
            this.checkIsOpen();
            Assert.assertEquals((String)"write size", (long)this.requiredWriteSize, (long)len);
            if (this.doPartialWrite) {
                --len;
            }
            this.bos.write(b, off, len);
        }

        private void checkIsOpen() throws IOException {
            if (this.closed.get()) {
                throw new IOException("Closed");
            }
        }

        @Override
        public void write(int b) throws IOException {
            this.checkIsOpen();
            Assert.assertEquals((String)"write size", (long)this.requiredWriteSize, (long)1L);
            this.bos.write(b);
        }

        @Override
        public void close() throws IOException {
            if (this.closed.compareAndSet(false, true)) {
                this.bos.close();
            }
        }
    }
}

