/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.yarn.server.nodemanager.containermanager.linux.resources;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.math.BigInteger;
import java.nio.charset.StandardCharsets;
import java.util.function.Function;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.hadoop.classification.VisibleForTesting;
import org.apache.hadoop.util.CpuTimeTracker;
import org.apache.hadoop.util.Shell;
import org.apache.hadoop.util.SysInfoLinux;
import org.apache.hadoop.yarn.exceptions.YarnException;
import org.apache.hadoop.yarn.server.nodemanager.containermanager.linux.resources.CGroupsHandler;
import org.apache.hadoop.yarn.server.nodemanager.containermanager.linux.resources.ResourceHandlerModule;
import org.apache.hadoop.yarn.util.Clock;
import org.apache.hadoop.yarn.util.ResourceCalculatorProcessTree;
import org.apache.hadoop.yarn.util.SystemClock;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CGroupsResourceCalculator
extends ResourceCalculatorProcessTree {
    protected static final Logger LOG = LoggerFactory.getLogger(CGroupsResourceCalculator.class);
    private static final String PROCFS = "/proc";
    static final String CGROUP = "cgroup";
    static final String CPU_STAT = "cpuacct.stat";
    static final String MEM_STAT = "memory.usage_in_bytes";
    static final String MEMSW_STAT = "memory.memsw.usage_in_bytes";
    private static final String USER = "user ";
    private static final String SYSTEM = "system ";
    private static final Pattern CGROUP_FILE_FORMAT = Pattern.compile("^(\\d+):([^:]+):/(.*)$");
    private final String procfsDir;
    private CGroupsHandler cGroupsHandler;
    private String pid;
    private File cpuStat;
    private File memStat;
    private File memswStat;
    private BigInteger processTotalJiffies;
    private long processPhysicalMemory;
    private long processVirtualMemory;
    private final long jiffyLengthMs;
    private final CpuTimeTracker cpuTimeTracker;
    private Clock clock;

    public CGroupsResourceCalculator() throws YarnException {
        this(null, PROCFS, ResourceHandlerModule.getCGroupsHandler(), (Clock)SystemClock.getInstance(), SysInfoLinux.JIFFY_LENGTH_IN_MILLIS);
    }

    public CGroupsResourceCalculator(String pid) {
        this(pid, PROCFS, ResourceHandlerModule.getCGroupsHandler(), (Clock)SystemClock.getInstance(), SysInfoLinux.JIFFY_LENGTH_IN_MILLIS);
    }

    @VisibleForTesting
    CGroupsResourceCalculator(String pid, String procfsDir, CGroupsHandler cGroupsHandler, Clock clock, long jiffyLengthMs0) {
        super(pid);
        this.procfsDir = procfsDir;
        this.cGroupsHandler = cGroupsHandler;
        this.pid = pid != null && pid.equals("0") ? "1" : pid;
        this.jiffyLengthMs = jiffyLengthMs0;
        this.cpuTimeTracker = new CpuTimeTracker(this.jiffyLengthMs);
        this.clock = clock;
        this.processTotalJiffies = BigInteger.ZERO;
        this.processPhysicalMemory = -1L;
        this.processVirtualMemory = -1L;
    }

    public void initialize() throws YarnException {
        if (!CGroupsResourceCalculator.isAvailable()) {
            throw new YarnException("CGroupsResourceCalculator is not available");
        }
        this.setCGroupFilePaths();
    }

    public float getCpuUsagePercent() {
        LOG.debug("Process {} jiffies:{}", (Object)this.pid, (Object)this.processTotalJiffies);
        return this.cpuTimeTracker.getCpuTrackerUsagePercent();
    }

    public long getCumulativeCpuTime() {
        if (this.jiffyLengthMs < 0L) {
            return -1L;
        }
        return this.processTotalJiffies.longValue() * this.jiffyLengthMs;
    }

    public long getRssMemorySize(int olderThanAge) {
        if (olderThanAge > 1) {
            return -1L;
        }
        return this.processPhysicalMemory;
    }

    public long getVirtualMemorySize(int olderThanAge) {
        if (olderThanAge > 1) {
            return -1L;
        }
        return this.processVirtualMemory;
    }

    public void updateProcessTree() {
        try {
            this.processTotalJiffies = this.readTotalProcessJiffies();
            this.cpuTimeTracker.updateElapsedJiffies(this.processTotalJiffies, this.clock.getTime());
        }
        catch (YarnException e) {
            LOG.warn("Failed to parse " + this.pid, (Throwable)e);
        }
        this.processPhysicalMemory = this.getMemorySize(this.memStat);
        if (this.memswStat.exists()) {
            this.processVirtualMemory = this.getMemorySize(this.memswStat);
        } else {
            LOG.debug("Swap cgroups monitoring is not compiled into the kernel {}", (Object)this.memswStat.getAbsolutePath());
        }
    }

    public String getProcessTreeDump() {
        return this.pid;
    }

    public boolean checkPidPgrpidForMatch() {
        return true;
    }

    public static boolean isAvailable() {
        try {
            if (!Shell.LINUX) {
                LOG.info("CGroupsResourceCalculator currently is supported only on Linux.");
                return false;
            }
            if (ResourceHandlerModule.getCGroupsHandler() == null || ResourceHandlerModule.getCpuResourceHandler() == null || ResourceHandlerModule.getMemoryResourceHandler() == null) {
                LOG.info("CGroupsResourceCalculator requires enabling CGroupscpu and memory");
                return false;
            }
        }
        catch (SecurityException se) {
            LOG.warn("Failed to get Operating System name. " + se);
            return false;
        }
        return true;
    }

    private long getMemorySize(File cgroupUsageFile) {
        long[] mem = new long[1];
        try {
            this.processFile(cgroupUsageFile, line -> {
                mem[0] = Long.parseLong(line);
                return Result.Exit;
            });
            return mem[0];
        }
        catch (YarnException e) {
            LOG.warn("Failed to parse cgroups " + this.memswStat, (Throwable)e);
            return -1L;
        }
    }

    private BigInteger readTotalProcessJiffies() throws YarnException {
        BigInteger[] totalCPUTimeJiffies = new BigInteger[]{BigInteger.ZERO};
        this.processFile(this.cpuStat, line -> {
            if (line.startsWith(USER)) {
                totalCPUTimeJiffies[0] = totalCPUTimeJiffies[0].add(new BigInteger(line.substring(USER.length())));
            }
            if (line.startsWith(SYSTEM)) {
                totalCPUTimeJiffies[0] = totalCPUTimeJiffies[0].add(new BigInteger(line.substring(SYSTEM.length())));
            }
            return Result.Continue;
        });
        return totalCPUTimeJiffies[0];
    }

    private String getCGroupRelativePath(CGroupsHandler.CGroupController controller) throws YarnException {
        if (this.pid == null) {
            return this.cGroupsHandler.getRelativePathForCGroup("");
        }
        return this.getCGroupRelativePathForPid(controller);
    }

    private String getCGroupRelativePathForPid(CGroupsHandler.CGroupController controller) throws YarnException {
        File pidCgroupFile = new File(new File(this.procfsDir, this.pid), CGROUP);
        String[] result = new String[1];
        this.processFile(pidCgroupFile, line -> {
            Matcher m = CGROUP_FILE_FORMAT.matcher((CharSequence)line);
            boolean mat = m.find();
            if (mat) {
                if (m.group(2).contains(controller.getName())) {
                    String cgroupPath = m.group(3);
                    if (cgroupPath != null) {
                        String cgroup = new File(cgroupPath).toPath().getFileName().toString();
                        result[0] = this.cGroupsHandler.getRelativePathForCGroup(cgroup);
                    } else {
                        LOG.warn("Invalid cgroup path for " + pidCgroupFile);
                    }
                    return Result.Exit;
                }
            } else {
                LOG.warn("Unexpected: cgroup file is not in the expected format for process with pid " + this.pid);
            }
            return Result.Continue;
        });
        if (result[0] == null) {
            throw new YarnException(controller.getName() + " CGroup for pid " + this.pid + " not found " + pidCgroupFile);
        }
        return result[0];
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private void processFile(File file, Function<String, Result> processLine) throws YarnException {
        try (InputStreamReader fReader = new InputStreamReader((InputStream)new FileInputStream(file), StandardCharsets.UTF_8);
             BufferedReader in = new BufferedReader(fReader);){
            String str;
            while ((str = in.readLine()) != null) {
                Result result = processLine.apply(str);
                if (result == Result.Exit) {
                    return;
                }
            }
            return;
        }
        catch (IOException f) {
            throw new YarnException("The process vanished in the interim " + this.pid, (Throwable)f);
        }
    }

    void setCGroupFilePaths() throws YarnException {
        if (this.cGroupsHandler == null) {
            throw new YarnException("CGroups handler is not initialized");
        }
        File cpuDir = new File(this.cGroupsHandler.getControllerPath(CGroupsHandler.CGroupController.CPUACCT), this.getCGroupRelativePath(CGroupsHandler.CGroupController.CPUACCT));
        File memDir = new File(this.cGroupsHandler.getControllerPath(CGroupsHandler.CGroupController.MEMORY), this.getCGroupRelativePath(CGroupsHandler.CGroupController.MEMORY));
        this.cpuStat = new File(cpuDir, CPU_STAT);
        this.memStat = new File(memDir, MEM_STAT);
        this.memswStat = new File(memDir, MEMSW_STAT);
    }

    static enum Result {
        Continue,
        Exit;

    }
}

