/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.yarn.util.resource;

import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import org.apache.commons.io.FileUtils;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.yarn.api.protocolrecords.ResourceTypes;
import org.apache.hadoop.yarn.api.records.Resource;
import org.apache.hadoop.yarn.api.records.ResourceInformation;
import org.apache.hadoop.yarn.api.records.ResourceTypeInfo;
import org.apache.hadoop.yarn.conf.YarnConfiguration;
import org.apache.hadoop.yarn.exceptions.YarnRuntimeException;
import org.apache.hadoop.yarn.util.resource.ResourceUtils;
import org.apache.hadoop.yarn.util.resource.Resources;
import org.assertj.core.api.Assertions;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TestResourceUtils {
    private static final Logger LOG = LoggerFactory.getLogger(TestResourceUtils.class);
    private File nodeResourcesFile;
    private File resourceTypesFile;
    @Rule
    public ExpectedException expexted = ExpectedException.none();

    @Before
    public void setup() {
        ResourceUtils.resetResourceTypes();
    }

    @After
    public void teardown() {
        if (this.nodeResourcesFile != null && this.nodeResourcesFile.exists()) {
            this.nodeResourcesFile.delete();
        }
        if (this.resourceTypesFile != null && this.resourceTypesFile.exists()) {
            this.resourceTypesFile.delete();
        }
    }

    public static String setupResourceTypes(Configuration conf, String filename) throws Exception {
        File source = new File(conf.getClassLoader().getResource(filename).getFile());
        File dest = new File(source.getParent(), "resource-types.xml");
        FileUtils.copyFile((File)source, (File)dest);
        try {
            ResourceUtils.getResourceTypes();
        }
        catch (Exception e) {
            if (!dest.delete()) {
                LOG.error("Could not delete {}", (Object)dest);
            }
            throw e;
        }
        return dest.getAbsolutePath();
    }

    private Map<String, ResourceInformation> setupResourceTypesInternal(Configuration conf, String srcFileName) throws IOException {
        URL srcFileUrl = conf.getClassLoader().getResource(srcFileName);
        if (srcFileUrl == null) {
            throw new IllegalArgumentException("Source file does not exist: " + srcFileName);
        }
        File source = new File(srcFileUrl.getFile());
        File dest = new File(source.getParent(), "resource-types.xml");
        FileUtils.copyFile((File)source, (File)dest);
        this.resourceTypesFile = dest;
        return ResourceUtils.getResourceTypes();
    }

    private Map<String, ResourceInformation> setupNodeResources(Configuration conf, String srcFileName) throws IOException {
        URL srcFileUrl = conf.getClassLoader().getResource(srcFileName);
        if (srcFileUrl == null) {
            throw new IllegalArgumentException("Source file does not exist: " + srcFileName);
        }
        File source = new File(srcFileUrl.getFile());
        File dest = new File(source.getParent(), "node-resources.xml");
        FileUtils.copyFile((File)source, (File)dest);
        this.nodeResourcesFile = dest;
        return ResourceUtils.getNodeResourceInformation((Configuration)conf);
    }

    private void testMemoryAndVcores(Map<String, ResourceInformation> res) {
        String memory = ResourceInformation.MEMORY_MB.getName();
        String vcores = ResourceInformation.VCORES.getName();
        Assert.assertTrue((String)"Resource 'memory' missing", (boolean)res.containsKey(memory));
        Assert.assertEquals((String)"'memory' units incorrect", (Object)ResourceInformation.MEMORY_MB.getUnits(), (Object)res.get(memory).getUnits());
        Assert.assertEquals((String)"'memory' types incorrect", (Object)ResourceInformation.MEMORY_MB.getResourceType(), (Object)res.get(memory).getResourceType());
        Assert.assertTrue((String)"Resource 'vcores' missing", (boolean)res.containsKey(vcores));
        Assert.assertEquals((String)"'vcores' units incorrect", (Object)ResourceInformation.VCORES.getUnits(), (Object)res.get(vcores).getUnits());
        Assert.assertEquals((String)"'vcores' type incorrect", (Object)ResourceInformation.VCORES.getResourceType(), (Object)res.get(vcores).getResourceType());
    }

    @Test
    public void testGetResourceTypes() {
        Map res = ResourceUtils.getResourceTypes();
        Assert.assertEquals((long)2L, (long)res.size());
        this.testMemoryAndVcores(res);
    }

    @Test
    public void testGetResourceTypesConfigs() throws Exception {
        ResourceFileInformation[] tests;
        YarnConfiguration conf = new YarnConfiguration();
        ResourceFileInformation testFile1 = new ResourceFileInformation("resource-types-1.xml", 2);
        ResourceFileInformation testFile2 = new ResourceFileInformation("resource-types-2.xml", 3);
        testFile2.resourceNameUnitsMap.put("resource1", "G");
        ResourceFileInformation testFile3 = new ResourceFileInformation("resource-types-3.xml", 3);
        testFile3.resourceNameUnitsMap.put("resource2", "");
        ResourceFileInformation testFile4 = new ResourceFileInformation("resource-types-4.xml", 5);
        testFile4.resourceNameUnitsMap.put("resource1", "G");
        testFile4.resourceNameUnitsMap.put("resource2", "m");
        testFile4.resourceNameUnitsMap.put("yarn.io/gpu", "");
        for (ResourceFileInformation testInformation : tests = new ResourceFileInformation[]{testFile1, testFile2, testFile3, testFile4}) {
            ResourceUtils.resetResourceTypes();
            Map<String, ResourceInformation> res = this.setupResourceTypesInternal((Configuration)conf, testInformation.filename);
            this.testMemoryAndVcores(res);
            Assert.assertEquals((long)testInformation.resourceCount, (long)res.size());
            for (Map.Entry<String, String> entry : testInformation.resourceNameUnitsMap.entrySet()) {
                String resourceName = entry.getKey();
                Assert.assertTrue((String)("Missing key " + resourceName), (boolean)res.containsKey(resourceName));
                Assert.assertEquals((Object)entry.getValue(), (Object)res.get(resourceName).getUnits());
            }
        }
    }

    @Test
    public void testGetRequestedResourcesFromConfig() {
        Configuration conf = new Configuration();
        String propertyPrefix = "mapreduce.mapper.proper.rt.";
        String[] expectedKeys = new String[]{"yarn.io/gpu", "yarn.io/fpga", "yarn.io/anything_without_a_dot", "regular_rt", "regular_rt/with_slash"};
        String[] invalidKeys = new String[]{propertyPrefix + "too.many_parts", propertyPrefix + "yarn.notio/gpu", "incorrect.prefix.yarn.io/gpu", propertyPrefix + "yarn.io/", propertyPrefix};
        for (String s : expectedKeys) {
            conf.set(propertyPrefix + s, "42");
        }
        for (String s : invalidKeys) {
            conf.set(s, "24");
        }
        List properList = ResourceUtils.getRequestedResourcesFromConfig((Configuration)conf, (String)propertyPrefix);
        HashSet<String> expectedSet = new HashSet<String>(Arrays.asList(expectedKeys));
        Assert.assertEquals((long)properList.size(), (long)expectedKeys.length);
        properList.forEach(item -> Assert.assertTrue((boolean)expectedSet.contains(item.getName())));
    }

    @Test
    public void testGetResourceTypesConfigErrors() throws IOException {
        String[] resourceFiles;
        YarnConfiguration conf = new YarnConfiguration();
        for (String resourceFile : resourceFiles = new String[]{"resource-types-error-1.xml", "resource-types-error-2.xml", "resource-types-error-3.xml", "resource-types-error-4.xml"}) {
            ResourceUtils.resetResourceTypes();
            try {
                this.setupResourceTypesInternal((Configuration)conf, resourceFile);
                Assert.fail((String)("Expected error with file " + resourceFile));
            }
            catch (IllegalArgumentException | YarnRuntimeException throwable) {
                // empty catch block
            }
        }
    }

    @Test
    public void testInitializeResourcesMap() {
        String[][][] allTests;
        String[] empty = new String[]{"", ""};
        String[] res1 = new String[]{"resource1", "m"};
        String[] res2 = new String[]{"resource2", "G"};
        String[][] test1 = new String[][]{empty};
        String[][] test2 = new String[][]{res1};
        String[][] test3 = new String[][]{res2};
        String[][] test4 = new String[][]{res1, res2};
        for (String[][] test : allTests = new String[][][]{test1, test2, test3, test4}) {
            YarnConfiguration conf = new YarnConfiguration();
            String resSt = "";
            for (String[] resources : test) {
                resSt = resSt + resources[0] + ",";
            }
            resSt = resSt.substring(0, resSt.length() - 1);
            conf.set("yarn.resource-types", resSt);
            for (String[] resources : test) {
                String name = "yarn.resource-types." + resources[0] + ".units";
                conf.set(name, resources[1]);
            }
            Map ret = ResourceUtils.resetResourceTypes((Configuration)conf);
            int len = 3;
            if (test == test1) {
                len = 2;
            } else if (test == test4) {
                len = 4;
            }
            Assert.assertEquals((long)len, (long)ret.size());
            for (String[] resources : test) {
                if (resources[0].length() == 0) continue;
                Assert.assertTrue((boolean)ret.containsKey(resources[0]));
                ResourceInformation resInfo = (ResourceInformation)ret.get(resources[0]);
                Assert.assertEquals((Object)resources[1], (Object)resInfo.getUnits());
                Assert.assertEquals((Object)ResourceTypes.COUNTABLE, (Object)resInfo.getResourceType());
            }
            Assert.assertTrue((boolean)ret.containsKey("memory-mb"));
            ResourceInformation memInfo = (ResourceInformation)ret.get("memory-mb");
            Assert.assertEquals((Object)"Mi", (Object)memInfo.getUnits());
            Assert.assertEquals((Object)ResourceTypes.COUNTABLE, (Object)memInfo.getResourceType());
            Assert.assertTrue((boolean)ret.containsKey("vcores"));
            ResourceInformation vcoresInfo = (ResourceInformation)ret.get("vcores");
            Assert.assertEquals((Object)"", (Object)vcoresInfo.getUnits());
            Assert.assertEquals((Object)ResourceTypes.COUNTABLE, (Object)vcoresInfo.getResourceType());
        }
    }

    @Test
    public void testInitializeResourcesMapErrors() {
        String[][][] allTests;
        String[] mem1 = new String[]{"memory-mb", ""};
        String[] vcores1 = new String[]{"vcores", "M"};
        String[] mem2 = new String[]{"memory-mb", "m"};
        String[] vcores2 = new String[]{"vcores", "G"};
        String[] mem3 = new String[]{"memory", ""};
        String[][] test1 = new String[][]{mem1, vcores1};
        String[][] test2 = new String[][]{mem2, vcores2};
        String[][] test3 = new String[][]{mem3};
        for (String[][] test : allTests = new String[][][]{test1, test2, test3}) {
            YarnConfiguration conf = new YarnConfiguration();
            String resSt = "";
            for (String[] resources : test) {
                resSt = resSt + resources[0] + ",";
            }
            resSt = resSt.substring(0, resSt.length() - 1);
            conf.set("yarn.resource-types", resSt);
            for (String[] resources : test) {
                String name = "yarn.resource-types." + resources[0] + ".units";
                conf.set(name, resources[1]);
            }
            try {
                ResourceUtils.initializeResourcesMap((Configuration)conf);
                Assert.fail((String)"resource map initialization should fail");
            }
            catch (Exception exception) {
                // empty catch block
            }
        }
    }

    @Test
    public void testGetResourceInformation() throws Exception {
        YarnConfiguration conf = new YarnConfiguration();
        HashMap<String, Resource> testRun = new HashMap<String, Resource>();
        this.setupResourceTypesInternal((Configuration)conf, "resource-types-4.xml");
        Resource test3Resources = Resource.newInstance((int)0, (int)0);
        test3Resources.setResourceInformation("resource1", ResourceInformation.newInstance((String)"resource1", (String)"Gi", (long)5L));
        test3Resources.setResourceInformation("resource2", ResourceInformation.newInstance((String)"resource2", (String)"m", (long)2L));
        test3Resources.setResourceInformation("yarn.io/gpu", ResourceInformation.newInstance((String)"yarn.io/gpu", (String)"", (long)1L));
        testRun.put("node-resources-2.xml", test3Resources);
        for (Map.Entry entry : testRun.entrySet()) {
            String resourceFile = (String)entry.getKey();
            ResourceUtils.resetNodeResources();
            Map<String, ResourceInformation> actual = this.setupNodeResources((Configuration)conf, resourceFile);
            Assert.assertEquals((long)actual.size(), (long)((Resource)entry.getValue()).getResources().length);
            for (ResourceInformation resInfo : ((Resource)entry.getValue()).getResources()) {
                Assert.assertEquals((Object)resInfo, (Object)actual.get(resInfo.getName()));
            }
        }
    }

    @Test
    public void testGetNodeResourcesConfigErrors() throws Exception {
        String[] invalidNodeResFiles;
        YarnConfiguration conf = new YarnConfiguration();
        this.setupResourceTypesInternal((Configuration)conf, "resource-types-4.xml");
        for (String resourceFile : invalidNodeResFiles = new String[]{"node-resources-error-1.xml"}) {
            ResourceUtils.resetNodeResources();
            try {
                this.setupNodeResources((Configuration)conf, resourceFile);
                Assert.fail((String)("Expected error with file " + resourceFile));
            }
            catch (YarnRuntimeException yarnRuntimeException) {
                // empty catch block
            }
        }
    }

    @Test
    public void testGetNodeResourcesRedefineFpgaErrors() throws Exception {
        YarnConfiguration conf = new YarnConfiguration();
        this.expexted.expect(YarnRuntimeException.class);
        this.expexted.expectMessage("Defined mandatory resource type=yarn.io/fpga");
        this.setupResourceTypesInternal((Configuration)conf, "resource-types-error-redefine-fpga-unit.xml");
    }

    @Test
    public void testGetNodeResourcesRedefineGpuErrors() throws Exception {
        YarnConfiguration conf = new YarnConfiguration();
        this.expexted.expect(YarnRuntimeException.class);
        this.expexted.expectMessage("Defined mandatory resource type=yarn.io/gpu");
        this.setupResourceTypesInternal((Configuration)conf, "resource-types-error-redefine-gpu-unit.xml");
    }

    @Test
    public void testResourceNameFormatValidation() {
        String[] validNames = new String[]{"yarn.io/gpu", "gpu", "g_1_2", "123.io/gpu", "prefix/resource_1", "a___-3", "a....b"};
        String[] invalidNames = new String[]{"asd/resource/-name", "prefix/-resource_1", "prefix/0123resource", "0123resource", "-resource_1", "........abc"};
        for (String validName : validNames) {
            ResourceUtils.validateNameOfResourceNameAndThrowException((String)validName);
        }
        for (String invalidName : invalidNames) {
            try {
                ResourceUtils.validateNameOfResourceNameAndThrowException((String)invalidName);
                Assert.fail((String)("Expected to fail name check, the name=" + invalidName + " is illegal."));
            }
            catch (YarnRuntimeException yarnRuntimeException) {
                // empty catch block
            }
        }
    }

    @Test
    public void testGetResourceInformationWithDiffUnits() throws Exception {
        YarnConfiguration conf = new YarnConfiguration();
        HashMap<String, Resource> testRun = new HashMap<String, Resource>();
        this.setupResourceTypesInternal((Configuration)conf, "resource-types-4.xml");
        Resource test3Resources = Resource.newInstance((int)0, (int)0);
        test3Resources.setResourceInformation("resource1", ResourceInformation.newInstance((String)"resource1", (String)"T", (long)5L));
        test3Resources.setResourceInformation("resource2", ResourceInformation.newInstance((String)"resource2", (String)"M", (long)2L));
        test3Resources.setResourceInformation("yarn.io/gpu", ResourceInformation.newInstance((String)"yarn.io/gpu", (String)"", (long)1L));
        testRun.put("node-resources-3.xml", test3Resources);
        for (Map.Entry entry : testRun.entrySet()) {
            String resourceFile = (String)entry.getKey();
            ResourceUtils.resetNodeResources();
            Map<String, ResourceInformation> actual = this.setupNodeResources((Configuration)conf, resourceFile);
            Assert.assertEquals((long)actual.size(), (long)((Resource)entry.getValue()).getResources().length);
            for (ResourceInformation resInfo : ((Resource)entry.getValue()).getResources()) {
                Assert.assertEquals((Object)resInfo, (Object)actual.get(resInfo.getName()));
            }
        }
    }

    @Test
    public void testResourceUnitParsing() throws Exception {
        Resource res = ResourceUtils.createResourceFromString((String)"memory=20g,vcores=3", (List)ResourceUtils.getResourcesTypeInfo());
        Assert.assertEquals((Object)Resources.createResource((int)20480, (int)3), (Object)res);
        res = ResourceUtils.createResourceFromString((String)"memory=20G,vcores=3", (List)ResourceUtils.getResourcesTypeInfo());
        Assert.assertEquals((Object)Resources.createResource((int)20480, (int)3), (Object)res);
        res = ResourceUtils.createResourceFromString((String)"memory=20M,vcores=3", (List)ResourceUtils.getResourcesTypeInfo());
        Assert.assertEquals((Object)Resources.createResource((int)20, (int)3), (Object)res);
        res = ResourceUtils.createResourceFromString((String)"memory=20m,vcores=3", (List)ResourceUtils.getResourcesTypeInfo());
        Assert.assertEquals((Object)Resources.createResource((int)20, (int)3), (Object)res);
        res = ResourceUtils.createResourceFromString((String)"memory-mb=20,vcores=3", (List)ResourceUtils.getResourcesTypeInfo());
        Assert.assertEquals((Object)Resources.createResource((int)20, (int)3), (Object)res);
        res = ResourceUtils.createResourceFromString((String)"memory-mb=20m,vcores=3", (List)ResourceUtils.getResourcesTypeInfo());
        Assert.assertEquals((Object)Resources.createResource((int)20, (int)3), (Object)res);
        res = ResourceUtils.createResourceFromString((String)"memory-mb=20G,vcores=3", (List)ResourceUtils.getResourcesTypeInfo());
        Assert.assertEquals((Object)Resources.createResource((int)20480, (int)3), (Object)res);
        res = ResourceUtils.createResourceFromString((String)"memory=20,vcores=3", (List)ResourceUtils.getResourcesTypeInfo());
        Assert.assertEquals((Object)Resources.createResource((int)0, (int)3), (Object)res);
        ArrayList<ResourceTypeInfo> resTypes = new ArrayList<ResourceTypeInfo>(ResourceUtils.getResourcesTypeInfo());
        resTypes.add(ResourceTypeInfo.newInstance((String)"yarn.io/gpu", (String)""));
        ResourceUtils.reinitializeResources(resTypes);
        res = ResourceUtils.createResourceFromString((String)"memory=2G,vcores=3,gpu=0", resTypes);
        Assert.assertEquals((long)2048L, (long)res.getMemorySize());
        Assert.assertEquals((long)0L, (long)res.getResourceValue("yarn.io/gpu"));
        res = ResourceUtils.createResourceFromString((String)"memory=2G,vcores=3,gpu=3", resTypes);
        Assert.assertEquals((long)2048L, (long)res.getMemorySize());
        Assert.assertEquals((long)3L, (long)res.getResourceValue("yarn.io/gpu"));
        res = ResourceUtils.createResourceFromString((String)"memory=2G,vcores=3", resTypes);
        Assert.assertEquals((long)2048L, (long)res.getMemorySize());
        Assert.assertEquals((long)0L, (long)res.getResourceValue("yarn.io/gpu"));
        res = ResourceUtils.createResourceFromString((String)"memory=2G,vcores=3,yarn.io/gpu=0", resTypes);
        Assert.assertEquals((long)2048L, (long)res.getMemorySize());
        Assert.assertEquals((long)0L, (long)res.getResourceValue("yarn.io/gpu"));
        res = ResourceUtils.createResourceFromString((String)"memory=2G,vcores=3,yarn.io/gpu=3", resTypes);
        Assert.assertEquals((long)2048L, (long)res.getMemorySize());
        Assert.assertEquals((long)3L, (long)res.getResourceValue("yarn.io/gpu"));
    }

    @Test
    public void testMultipleOpsForResourcesWithTags() throws Exception {
        YarnConfiguration conf = new YarnConfiguration();
        TestResourceUtils.setupResourceTypes((Configuration)conf, "resource-types-6.xml");
        Resource resourceA = Resource.newInstance((int)2, (int)4);
        Resource resourceB = Resource.newInstance((int)3, (int)6);
        resourceA.setResourceInformation("resource1", ResourceInformation.newInstance((String)"resource1", (String)"T", (long)5L));
        resourceA.setResourceInformation("resource2", ResourceInformation.newInstance((String)"resource2", (String)"M", (long)2L));
        resourceA.setResourceInformation("yarn.io/gpu", ResourceInformation.newInstance((String)"yarn.io/gpu", (String)"", (long)1L));
        resourceA.setResourceInformation("yarn.io/test-volume", ResourceInformation.newInstance((String)"yarn.io/test-volume", (String)"", (long)2L));
        resourceB.setResourceInformation("resource1", ResourceInformation.newInstance((String)"resource1", (String)"T", (long)3L));
        resourceB.setResourceInformation("resource2", ResourceInformation.newInstance((String)"resource2", (String)"M", (long)4L));
        resourceB.setResourceInformation("yarn.io/gpu", ResourceInformation.newInstance((String)"yarn.io/gpu", (String)"", (long)2L));
        resourceB.setResourceInformation("yarn.io/test-volume", ResourceInformation.newInstance((String)"yarn.io/test-volume", (String)"", (long)3L));
        Resource addedResource = Resources.add((Resource)resourceA, (Resource)resourceB);
        Assertions.assertThat((long)addedResource.getMemorySize()).isEqualTo(5L);
        Assertions.assertThat((int)addedResource.getVirtualCores()).isEqualTo(10);
        Assertions.assertThat((long)addedResource.getResourceInformation("resource1").getValue()).isEqualTo(8L);
        Assertions.assertThat((long)addedResource.getResourceInformation("yarn.io/test-volume").getValue()).isEqualTo(2L);
        Resource mulResource = Resources.multiplyAndRoundDown((Resource)resourceA, (double)3.0);
        Assertions.assertThat((long)mulResource.getMemorySize()).isEqualTo(6L);
        Assertions.assertThat((int)mulResource.getVirtualCores()).isEqualTo(12);
        Assertions.assertThat((long)mulResource.getResourceInformation("resource1").getValue()).isEqualTo(15L);
        Assertions.assertThat((long)mulResource.getResourceInformation("yarn.io/test-volume").getValue()).isEqualTo(2L);
    }

    static class ResourceFileInformation {
        String filename;
        int resourceCount;
        Map<String, String> resourceNameUnitsMap;

        ResourceFileInformation(String name, int count) {
            this.filename = name;
            this.resourceCount = count;
            this.resourceNameUnitsMap = new HashMap<String, String>();
        }
    }
}

