/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.qjournal.client;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Maps;
import com.google.common.util.concurrent.ListenableFuture;
import java.io.IOException;
import java.net.URL;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeoutException;
import org.apache.hadoop.hdfs.qjournal.client.AsyncLogger;
import org.apache.hadoop.hdfs.qjournal.client.QuorumCall;
import org.apache.hadoop.hdfs.qjournal.protocol.QJournalProtocolProtos;
import org.apache.hadoop.hdfs.server.common.StorageInfo;
import org.apache.hadoop.hdfs.server.protocol.NamespaceInfo;
import org.apache.hadoop.hdfs.server.protocol.RemoteEditLogManifest;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class AsyncLoggerSet {
    static final Logger LOG = LoggerFactory.getLogger(AsyncLoggerSet.class);
    private final List<AsyncLogger> loggers;
    private static final long INVALID_EPOCH = -1L;
    private long myEpoch = -1L;

    public AsyncLoggerSet(List<AsyncLogger> loggers) {
        this.loggers = ImmutableList.copyOf(loggers);
    }

    void setEpoch(long e) {
        Preconditions.checkState((!this.isEpochEstablished() ? 1 : 0) != 0, (String)"Epoch already established: epoch=%s", (long)this.myEpoch);
        this.myEpoch = e;
        for (AsyncLogger l : this.loggers) {
            l.setEpoch(e);
        }
    }

    public void setCommittedTxId(long txid) {
        for (AsyncLogger logger : this.loggers) {
            logger.setCommittedTxId(txid);
        }
    }

    boolean isEpochEstablished() {
        return this.myEpoch != -1L;
    }

    long getEpoch() {
        Preconditions.checkState((this.myEpoch != -1L ? 1 : 0) != 0, (Object)"No epoch created yet");
        return this.myEpoch;
    }

    void close() {
        for (AsyncLogger logger : this.loggers) {
            logger.close();
        }
    }

    void purgeLogsOlderThan(long minTxIdToKeep) {
        for (AsyncLogger logger : this.loggers) {
            logger.purgeLogsOlderThan(minTxIdToKeep);
        }
    }

    <V> Map<AsyncLogger, V> waitForWriteQuorum(QuorumCall<AsyncLogger, V> q, int timeoutMs, String operationName) throws IOException {
        int majority = this.getMajoritySize();
        try {
            q.waitFor(this.loggers.size(), majority, majority, timeoutMs, operationName);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new IOException("Interrupted waiting " + timeoutMs + "ms for a quorum of nodes to respond.");
        }
        catch (TimeoutException e) {
            throw new IOException("Timed out waiting " + timeoutMs + "ms for a quorum of nodes to respond.");
        }
        if (q.countSuccesses() < majority) {
            q.rethrowException("Got too many exceptions to achieve quorum size " + this.getMajorityString());
        }
        return q.getResults();
    }

    int getMajoritySize() {
        return this.loggers.size() / 2 + 1;
    }

    String getMajorityString() {
        return this.getMajoritySize() + "/" + this.loggers.size();
    }

    int size() {
        return this.loggers.size();
    }

    public String toString() {
        return "[" + Joiner.on((String)", ").join(this.loggers) + "]";
    }

    void appendReport(StringBuilder sb) {
        int len = this.loggers.size();
        for (int i = 0; i < len; ++i) {
            AsyncLogger l = this.loggers.get(i);
            if (i != 0) {
                sb.append(", ");
            }
            sb.append(l).append(" (");
            l.appendReport(sb);
            sb.append(")");
        }
    }

    @VisibleForTesting
    List<AsyncLogger> getLoggersForTests() {
        return this.loggers;
    }

    public QuorumCall<AsyncLogger, QJournalProtocolProtos.GetJournalStateResponseProto> getJournalState() {
        HashMap calls = Maps.newHashMap();
        for (AsyncLogger logger : this.loggers) {
            calls.put(logger, logger.getJournalState());
        }
        return QuorumCall.create(calls);
    }

    public QuorumCall<AsyncLogger, Boolean> isFormatted() {
        HashMap calls = Maps.newHashMap();
        for (AsyncLogger logger : this.loggers) {
            calls.put(logger, logger.isFormatted());
        }
        return QuorumCall.create(calls);
    }

    public QuorumCall<AsyncLogger, QJournalProtocolProtos.NewEpochResponseProto> newEpoch(NamespaceInfo nsInfo, long epoch) {
        HashMap calls = Maps.newHashMap();
        for (AsyncLogger logger : this.loggers) {
            calls.put(logger, logger.newEpoch(epoch));
        }
        return QuorumCall.create(calls);
    }

    public QuorumCall<AsyncLogger, Void> startLogSegment(long txid, int layoutVersion) {
        HashMap calls = Maps.newHashMap();
        for (AsyncLogger logger : this.loggers) {
            calls.put(logger, logger.startLogSegment(txid, layoutVersion));
        }
        return QuorumCall.create(calls);
    }

    public QuorumCall<AsyncLogger, Void> finalizeLogSegment(long firstTxId, long lastTxId) {
        HashMap calls = Maps.newHashMap();
        for (AsyncLogger logger : this.loggers) {
            calls.put(logger, logger.finalizeLogSegment(firstTxId, lastTxId));
        }
        return QuorumCall.create(calls);
    }

    public QuorumCall<AsyncLogger, Void> sendEdits(long segmentTxId, long firstTxnId, int numTxns, byte[] data) {
        HashMap calls = Maps.newHashMap();
        for (AsyncLogger logger : this.loggers) {
            ListenableFuture<Void> future = logger.sendEdits(segmentTxId, firstTxnId, numTxns, data);
            calls.put(logger, future);
        }
        return QuorumCall.create(calls);
    }

    public QuorumCall<AsyncLogger, RemoteEditLogManifest> getEditLogManifest(long fromTxnId, boolean inProgressOk) {
        HashMap calls = Maps.newHashMap();
        for (AsyncLogger logger : this.loggers) {
            ListenableFuture<RemoteEditLogManifest> future = logger.getEditLogManifest(fromTxnId, inProgressOk);
            calls.put(logger, future);
        }
        return QuorumCall.create(calls);
    }

    QuorumCall<AsyncLogger, QJournalProtocolProtos.PrepareRecoveryResponseProto> prepareRecovery(long segmentTxId) {
        HashMap calls = Maps.newHashMap();
        for (AsyncLogger logger : this.loggers) {
            ListenableFuture<QJournalProtocolProtos.PrepareRecoveryResponseProto> future = logger.prepareRecovery(segmentTxId);
            calls.put(logger, future);
        }
        return QuorumCall.create(calls);
    }

    QuorumCall<AsyncLogger, Void> acceptRecovery(QJournalProtocolProtos.SegmentStateProto log, URL fromURL) {
        HashMap calls = Maps.newHashMap();
        for (AsyncLogger logger : this.loggers) {
            ListenableFuture<Void> future = logger.acceptRecovery(log, fromURL);
            calls.put(logger, future);
        }
        return QuorumCall.create(calls);
    }

    QuorumCall<AsyncLogger, Void> format(NamespaceInfo nsInfo) {
        HashMap calls = Maps.newHashMap();
        for (AsyncLogger logger : this.loggers) {
            ListenableFuture<Void> future = logger.format(nsInfo);
            calls.put(logger, future);
        }
        return QuorumCall.create(calls);
    }

    public QuorumCall<AsyncLogger, Void> discardSegments(long startTxId) {
        HashMap calls = Maps.newHashMap();
        for (AsyncLogger logger : this.loggers) {
            ListenableFuture<Void> future = logger.discardSegments(startTxId);
            calls.put(logger, future);
        }
        return QuorumCall.create(calls);
    }

    QuorumCall<AsyncLogger, Void> doPreUpgrade() {
        HashMap calls = Maps.newHashMap();
        for (AsyncLogger logger : this.loggers) {
            ListenableFuture<Void> future = logger.doPreUpgrade();
            calls.put(logger, future);
        }
        return QuorumCall.create(calls);
    }

    public QuorumCall<AsyncLogger, Void> doUpgrade(StorageInfo sInfo) {
        HashMap calls = Maps.newHashMap();
        for (AsyncLogger logger : this.loggers) {
            ListenableFuture<Void> future = logger.doUpgrade(sInfo);
            calls.put(logger, future);
        }
        return QuorumCall.create(calls);
    }

    public QuorumCall<AsyncLogger, Void> doFinalize() {
        HashMap calls = Maps.newHashMap();
        for (AsyncLogger logger : this.loggers) {
            ListenableFuture<Void> future = logger.doFinalize();
            calls.put(logger, future);
        }
        return QuorumCall.create(calls);
    }

    public QuorumCall<AsyncLogger, Boolean> canRollBack(StorageInfo storage, StorageInfo prevStorage, int targetLayoutVersion) {
        HashMap calls = Maps.newHashMap();
        for (AsyncLogger logger : this.loggers) {
            ListenableFuture<Boolean> future = logger.canRollBack(storage, prevStorage, targetLayoutVersion);
            calls.put(logger, future);
        }
        return QuorumCall.create(calls);
    }

    public QuorumCall<AsyncLogger, Void> doRollback() {
        HashMap calls = Maps.newHashMap();
        for (AsyncLogger logger : this.loggers) {
            ListenableFuture<Void> future = logger.doRollback();
            calls.put(logger, future);
        }
        return QuorumCall.create(calls);
    }

    public QuorumCall<AsyncLogger, Long> getJournalCTime() {
        HashMap calls = Maps.newHashMap();
        for (AsyncLogger logger : this.loggers) {
            ListenableFuture<Long> future = logger.getJournalCTime();
            calls.put(logger, future);
        }
        return QuorumCall.create(calls);
    }
}

