/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.util;

import com.google.common.base.Preconditions;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.Map;
import java.util.Queue;
import org.apache.hadoop.HadoopIllegalArgumentException;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.util.Time;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
public abstract class ByteArrayManager {
    static final Logger LOG = LoggerFactory.getLogger(ByteArrayManager.class);
    private static final ThreadLocal<StringBuilder> debugMessage = new ThreadLocal<StringBuilder>(){

        @Override
        protected StringBuilder initialValue() {
            return new StringBuilder();
        }
    };
    static final int MIN_ARRAY_LENGTH = 32;
    static final byte[] EMPTY_BYTE_ARRAY = new byte[0];

    private static void logDebugMessage() {
        StringBuilder b = debugMessage.get();
        LOG.debug("{}", (Object)b);
        b.setLength(0);
    }

    public static int leastPowerOfTwo(int n) {
        if (n <= 0) {
            throw new HadoopIllegalArgumentException("n = " + n + " <= 0");
        }
        int highestOne = Integer.highestOneBit(n);
        if (highestOne == n) {
            return n;
        }
        int roundUp = highestOne << 1;
        if (roundUp < 0) {
            long overflow = (long)highestOne << 1;
            throw new ArithmeticException("Overflow: for n = " + n + ", the least power of two (the least integer x with x >= n and x a power of two) = " + overflow + " > Integer.MAX_VALUE = 2147483647");
        }
        return roundUp;
    }

    public abstract byte[] newByteArray(int var1) throws InterruptedException;

    public abstract int release(byte[] var1);

    public static ByteArrayManager newInstance(Conf conf) {
        return conf == null ? new NewByteArrayWithoutLimit() : new Impl(conf);
    }

    static class Impl
    extends ByteArrayManager {
        private final Conf conf;
        private final CounterMap counters;
        private final ManagerMap managers;

        Impl(Conf conf) {
            this.conf = conf;
            this.counters = new CounterMap(conf.countResetTimePeriodMs);
            this.managers = new ManagerMap(conf.countLimit);
        }

        @Override
        public byte[] newByteArray(int arrayLength) throws InterruptedException {
            byte[] array;
            Preconditions.checkArgument((arrayLength >= 0 ? 1 : 0) != 0);
            if (LOG.isDebugEnabled()) {
                debugMessage.get().append("allocate(").append(arrayLength).append(")");
            }
            if (arrayLength == 0) {
                array = EMPTY_BYTE_ARRAY;
            } else {
                int powerOfTwo = arrayLength <= 32 ? 32 : Impl.leastPowerOfTwo(arrayLength);
                long count = this.counters.get(powerOfTwo, true).increment();
                boolean aboveThreshold = count > (long)this.conf.countThreshold;
                FixedLengthManager manager = this.managers.get(powerOfTwo, aboveThreshold);
                if (LOG.isDebugEnabled()) {
                    debugMessage.get().append(": count=").append(count).append(aboveThreshold ? ", aboveThreshold" : ", belowThreshold");
                }
                byte[] byArray = array = manager != null ? manager.allocate() : new byte[powerOfTwo];
            }
            if (LOG.isDebugEnabled()) {
                debugMessage.get().append(", return byte[").append(array.length).append("]");
                ByteArrayManager.logDebugMessage();
            }
            return array;
        }

        @Override
        public int release(byte[] array) {
            int freeQueueSize;
            Preconditions.checkNotNull((Object)array);
            if (LOG.isDebugEnabled()) {
                debugMessage.get().append("recycle: array.length=").append(array.length);
            }
            if (array.length == 0) {
                freeQueueSize = -1;
            } else {
                FixedLengthManager manager = this.managers.get(array.length, false);
                int n = freeQueueSize = manager == null ? -1 : manager.recycle(array);
            }
            if (LOG.isDebugEnabled()) {
                debugMessage.get().append(", freeQueueSize=").append(freeQueueSize);
                ByteArrayManager.logDebugMessage();
            }
            return freeQueueSize;
        }

        CounterMap getCounters() {
            return this.counters;
        }

        ManagerMap getManagers() {
            return this.managers;
        }
    }

    static class NewByteArrayWithoutLimit
    extends ByteArrayManager {
        NewByteArrayWithoutLimit() {
        }

        @Override
        public byte[] newByteArray(int size) throws InterruptedException {
            return new byte[size];
        }

        @Override
        public int release(byte[] array) {
            return 0;
        }
    }

    public static class Conf {
        private final int countThreshold;
        private final int countLimit;
        private final long countResetTimePeriodMs;

        public Conf(int countThreshold, int countLimit, long countResetTimePeriodMs) {
            this.countThreshold = countThreshold;
            this.countLimit = countLimit;
            this.countResetTimePeriodMs = countResetTimePeriodMs;
        }
    }

    static class ManagerMap {
        private final int countLimit;
        private final Map<Integer, FixedLengthManager> map = new HashMap<Integer, FixedLengthManager>();

        ManagerMap(int countLimit) {
            this.countLimit = countLimit;
        }

        synchronized FixedLengthManager get(Integer arrayLength, boolean createIfNotExist) {
            FixedLengthManager manager = this.map.get(arrayLength);
            if (manager == null && createIfNotExist) {
                manager = new FixedLengthManager(arrayLength, this.countLimit);
                this.map.put(arrayLength, manager);
            }
            return manager;
        }

        synchronized void clear() {
            this.map.clear();
        }
    }

    static class FixedLengthManager {
        private final int byteArrayLength;
        private final int maxAllocated;
        private final Queue<byte[]> freeQueue = new LinkedList<byte[]>();
        private int numAllocated = 0;

        FixedLengthManager(int arrayLength, int maxAllocated) {
            this.byteArrayLength = arrayLength;
            this.maxAllocated = maxAllocated;
        }

        synchronized byte[] allocate() throws InterruptedException {
            if (LOG.isDebugEnabled()) {
                debugMessage.get().append(", ").append(this);
            }
            while (this.numAllocated >= this.maxAllocated) {
                if (LOG.isDebugEnabled()) {
                    debugMessage.get().append(": wait ...");
                    ByteArrayManager.logDebugMessage();
                }
                this.wait();
                if (!LOG.isDebugEnabled()) continue;
                debugMessage.get().append("wake up: ").append(this);
            }
            ++this.numAllocated;
            byte[] array = this.freeQueue.poll();
            if (LOG.isDebugEnabled()) {
                debugMessage.get().append(", recycled? ").append(array != null);
            }
            return array != null ? array : new byte[this.byteArrayLength];
        }

        synchronized int recycle(byte[] array) {
            Preconditions.checkNotNull((Object)array);
            Preconditions.checkArgument((array.length == this.byteArrayLength ? 1 : 0) != 0);
            if (LOG.isDebugEnabled()) {
                debugMessage.get().append(", ").append(this);
            }
            this.notify();
            --this.numAllocated;
            if (this.numAllocated < 0) {
                this.numAllocated = 0;
            }
            if (this.freeQueue.size() < this.maxAllocated - this.numAllocated) {
                if (LOG.isDebugEnabled()) {
                    debugMessage.get().append(", freeQueue.offer");
                }
                this.freeQueue.offer(array);
            }
            return this.freeQueue.size();
        }

        public synchronized String toString() {
            return "[" + this.byteArrayLength + ": " + this.numAllocated + "/" + this.maxAllocated + ", free=" + this.freeQueue.size() + "]";
        }
    }

    static class CounterMap {
        private final long countResetTimePeriodMs;
        private final Map<Integer, Counter> map = new HashMap<Integer, Counter>();

        private CounterMap(long countResetTimePeriodMs) {
            this.countResetTimePeriodMs = countResetTimePeriodMs;
        }

        synchronized Counter get(Integer key, boolean createIfNotExist) {
            Counter count = this.map.get(key);
            if (count == null && createIfNotExist) {
                count = new Counter(this.countResetTimePeriodMs);
                this.map.put(key, count);
            }
            return count;
        }

        synchronized void clear() {
            this.map.clear();
        }
    }

    static class Counter {
        private final long countResetTimePeriodMs;
        private long count = 0L;
        private long timestamp = Time.monotonicNow();

        Counter(long countResetTimePeriodMs) {
            this.countResetTimePeriodMs = countResetTimePeriodMs;
        }

        synchronized long getCount() {
            return this.count;
        }

        synchronized long increment() {
            long now = Time.monotonicNow();
            if (now - this.timestamp > this.countResetTimePeriodMs) {
                this.count = 0L;
            }
            this.timestamp = now;
            return ++this.count;
        }
    }
}

