/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs;

import com.google.common.annotations.VisibleForTesting;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.lang.reflect.Constructor;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.StringTokenizer;
import java.util.concurrent.ConcurrentHashMap;
import org.apache.hadoop.HadoopIllegalArgumentException;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.BlockLocation;
import org.apache.hadoop.fs.CreateFlag;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileAlreadyExistsException;
import org.apache.hadoop.fs.FileChecksum;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.FsServerDefaults;
import org.apache.hadoop.fs.FsStatus;
import org.apache.hadoop.fs.InvalidPathException;
import org.apache.hadoop.fs.LocatedFileStatus;
import org.apache.hadoop.fs.Options;
import org.apache.hadoop.fs.ParentNotDirectoryException;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.RemoteIterator;
import org.apache.hadoop.fs.UnresolvedLinkException;
import org.apache.hadoop.fs.UnsupportedFileSystemException;
import org.apache.hadoop.fs.XAttrSetFlag;
import org.apache.hadoop.fs.permission.AclEntry;
import org.apache.hadoop.fs.permission.AclStatus;
import org.apache.hadoop.fs.permission.FsAction;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.security.AccessControlException;
import org.apache.hadoop.security.SecurityUtil;
import org.apache.hadoop.security.token.Token;
import org.apache.hadoop.util.Progressable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Public
@InterfaceStability.Evolving
public abstract class AbstractFileSystem {
    static final Logger LOG = LoggerFactory.getLogger(AbstractFileSystem.class);
    private static final Map<URI, FileSystem.Statistics> STATISTICS_TABLE = new HashMap<URI, FileSystem.Statistics>();
    private static final Map<Class<?>, Constructor<?>> CONSTRUCTOR_CACHE = new ConcurrentHashMap();
    private static final Class<?>[] URI_CONFIG_ARGS = new Class[]{URI.class, Configuration.class};
    protected FileSystem.Statistics statistics;
    @VisibleForTesting
    static final String NO_ABSTRACT_FS_ERROR = "No AbstractFileSystem configured for scheme";
    private final URI myUri;

    public FileSystem.Statistics getStatistics() {
        return this.statistics;
    }

    public boolean isValidName(String src) {
        StringTokenizer tokens = new StringTokenizer(src, "/");
        while (tokens.hasMoreTokens()) {
            String element = tokens.nextToken();
            if (!element.equals("..") && !element.equals(".") && element.indexOf(":") < 0) continue;
            return false;
        }
        return true;
    }

    static <T> T newInstance(Class<T> theClass, URI uri, Configuration conf) {
        Object result;
        try {
            Constructor<Object> meth = CONSTRUCTOR_CACHE.get(theClass);
            if (meth == null) {
                meth = theClass.getDeclaredConstructor(URI_CONFIG_ARGS);
                meth.setAccessible(true);
                CONSTRUCTOR_CACHE.put(theClass, meth);
            }
            result = meth.newInstance(uri, conf);
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
        return (T)result;
    }

    public static AbstractFileSystem createFileSystem(URI uri, Configuration conf) throws UnsupportedFileSystemException {
        String fsImplConf = String.format("fs.AbstractFileSystem.%s.impl", uri.getScheme());
        Class<?> clazz = conf.getClass(fsImplConf, null);
        if (clazz == null) {
            throw new UnsupportedFileSystemException(String.format("%s=null: %s: %s", fsImplConf, NO_ABSTRACT_FS_ERROR, uri.getScheme()));
        }
        return (AbstractFileSystem)AbstractFileSystem.newInstance(clazz, uri, conf);
    }

    protected static synchronized FileSystem.Statistics getStatistics(URI uri) {
        String scheme = uri.getScheme();
        if (scheme == null) {
            throw new IllegalArgumentException("Scheme not defined in the uri: " + uri);
        }
        URI baseUri = AbstractFileSystem.getBaseUri(uri);
        FileSystem.Statistics result = STATISTICS_TABLE.get(baseUri);
        if (result == null) {
            result = new FileSystem.Statistics(scheme);
            STATISTICS_TABLE.put(baseUri, result);
        }
        return result;
    }

    private static URI getBaseUri(URI uri) {
        String scheme = uri.getScheme();
        String authority = uri.getAuthority();
        String baseUriString = scheme + "://";
        baseUriString = authority != null ? baseUriString + authority : baseUriString + "/";
        return URI.create(baseUriString);
    }

    public static synchronized void clearStatistics() {
        for (FileSystem.Statistics stat : STATISTICS_TABLE.values()) {
            stat.reset();
        }
    }

    public static synchronized void printStatistics() {
        for (Map.Entry<URI, FileSystem.Statistics> pair : STATISTICS_TABLE.entrySet()) {
            System.out.println("  FileSystem " + pair.getKey().getScheme() + "://" + pair.getKey().getAuthority() + ": " + pair.getValue());
        }
    }

    protected static synchronized Map<URI, FileSystem.Statistics> getAllStatistics() {
        HashMap<URI, FileSystem.Statistics> statsMap = new HashMap<URI, FileSystem.Statistics>(STATISTICS_TABLE.size());
        for (Map.Entry<URI, FileSystem.Statistics> pair : STATISTICS_TABLE.entrySet()) {
            URI key = pair.getKey();
            FileSystem.Statistics value = pair.getValue();
            FileSystem.Statistics newStatsObj = new FileSystem.Statistics(value);
            statsMap.put(URI.create(key.toString()), newStatsObj);
        }
        return statsMap;
    }

    public static AbstractFileSystem get(URI uri, Configuration conf) throws UnsupportedFileSystemException {
        return AbstractFileSystem.createFileSystem(uri, conf);
    }

    public AbstractFileSystem(URI uri, String supportedScheme, boolean authorityNeeded, int defaultPort) throws URISyntaxException {
        this.myUri = this.getUri(uri, supportedScheme, authorityNeeded, defaultPort);
        this.statistics = AbstractFileSystem.getStatistics(uri);
    }

    public void checkScheme(URI uri, String supportedScheme) {
        String scheme = uri.getScheme();
        if (scheme == null) {
            throw new HadoopIllegalArgumentException("Uri without scheme: " + uri);
        }
        if (!scheme.equals(supportedScheme)) {
            throw new HadoopIllegalArgumentException("Uri scheme " + uri + " does not match the scheme " + supportedScheme);
        }
    }

    private URI getUri(URI uri, String supportedScheme, boolean authorityNeeded, int defaultPort) throws URISyntaxException {
        this.checkScheme(uri, supportedScheme);
        if (defaultPort < 0 && authorityNeeded) {
            throw new HadoopIllegalArgumentException("FileSystem implementation error -  default port " + defaultPort + " is not valid");
        }
        String authority = uri.getAuthority();
        if (authority == null) {
            if (authorityNeeded) {
                throw new HadoopIllegalArgumentException("Uri without authority: " + uri);
            }
            return new URI(supportedScheme + ":///");
        }
        int port = uri.getPort();
        int n = port = port == -1 ? defaultPort : port;
        if (port == -1) {
            return new URI(supportedScheme, authority, "/", null);
        }
        return new URI(supportedScheme + "://" + uri.getHost() + ":" + port);
    }

    public abstract int getUriDefaultPort();

    public URI getUri() {
        return this.myUri;
    }

    public void checkPath(Path path) {
        URI uri = path.toUri();
        String thatScheme = uri.getScheme();
        String thatAuthority = uri.getAuthority();
        if (thatScheme == null) {
            if (thatAuthority == null) {
                if (path.isUriPathAbsolute()) {
                    return;
                }
                throw new InvalidPathException("relative paths not allowed:" + path);
            }
            throw new InvalidPathException("Path without scheme with non-null authority:" + path);
        }
        String thisScheme = this.getUri().getScheme();
        String thisHost = this.getUri().getHost();
        String thatHost = uri.getHost();
        if (!thisScheme.equalsIgnoreCase(thatScheme) || thisHost != null && !thisHost.equalsIgnoreCase(thatHost) || thisHost == null && thatHost != null) {
            throw new InvalidPathException("Wrong FS: " + path + ", expected: " + this.getUri());
        }
        int thisPort = this.getUri().getPort();
        int thatPort = uri.getPort();
        if (thatPort == -1) {
            thatPort = this.getUriDefaultPort();
        }
        if (thisPort != thatPort) {
            throw new InvalidPathException("Wrong FS: " + path + ", expected: " + this.getUri());
        }
    }

    public String getUriPath(Path p) {
        this.checkPath(p);
        String s = p.toUri().getPath();
        if (!this.isValidName(s)) {
            throw new InvalidPathException("Path part " + s + " from URI " + p + " is not a valid filename.");
        }
        return s;
    }

    public Path makeQualified(Path path) {
        this.checkPath(path);
        return path.makeQualified(this.getUri(), null);
    }

    public Path getInitialWorkingDirectory() {
        return null;
    }

    public Path getHomeDirectory() {
        return new Path("/user/" + System.getProperty("user.name")).makeQualified(this.getUri(), null);
    }

    public abstract FsServerDefaults getServerDefaults() throws IOException;

    public Path resolvePath(Path p) throws FileNotFoundException, UnresolvedLinkException, AccessControlException, IOException {
        this.checkPath(p);
        return this.getFileStatus(p).getPath();
    }

    public final FSDataOutputStream create(Path f, EnumSet<CreateFlag> createFlag, Options.CreateOpts ... opts) throws AccessControlException, FileAlreadyExistsException, FileNotFoundException, ParentNotDirectoryException, UnsupportedFileSystemException, UnresolvedLinkException, IOException {
        this.checkPath(f);
        int bufferSize = -1;
        short replication = -1;
        long blockSize = -1L;
        int bytesPerChecksum = -1;
        Options.ChecksumOpt checksumOpt = null;
        FsPermission permission = null;
        Progressable progress = null;
        Boolean createParent = null;
        for (Options.CreateOpts iOpt : opts) {
            if (Options.CreateOpts.BlockSize.class.isInstance(iOpt)) {
                if (blockSize != -1L) {
                    throw new HadoopIllegalArgumentException("BlockSize option is set multiple times");
                }
                blockSize = ((Options.CreateOpts.BlockSize)iOpt).getValue();
                continue;
            }
            if (Options.CreateOpts.BufferSize.class.isInstance(iOpt)) {
                if (bufferSize != -1) {
                    throw new HadoopIllegalArgumentException("BufferSize option is set multiple times");
                }
                bufferSize = ((Options.CreateOpts.BufferSize)iOpt).getValue();
                continue;
            }
            if (Options.CreateOpts.ReplicationFactor.class.isInstance(iOpt)) {
                if (replication != -1) {
                    throw new HadoopIllegalArgumentException("ReplicationFactor option is set multiple times");
                }
                replication = ((Options.CreateOpts.ReplicationFactor)iOpt).getValue();
                continue;
            }
            if (Options.CreateOpts.BytesPerChecksum.class.isInstance(iOpt)) {
                if (bytesPerChecksum != -1) {
                    throw new HadoopIllegalArgumentException("BytesPerChecksum option is set multiple times");
                }
                bytesPerChecksum = ((Options.CreateOpts.BytesPerChecksum)iOpt).getValue();
                continue;
            }
            if (Options.CreateOpts.ChecksumParam.class.isInstance(iOpt)) {
                if (checksumOpt != null) {
                    throw new HadoopIllegalArgumentException("CreateChecksumType option is set multiple times");
                }
                checksumOpt = ((Options.CreateOpts.ChecksumParam)iOpt).getValue();
                continue;
            }
            if (Options.CreateOpts.Perms.class.isInstance(iOpt)) {
                if (permission != null) {
                    throw new HadoopIllegalArgumentException("Perms option is set multiple times");
                }
                permission = ((Options.CreateOpts.Perms)iOpt).getValue();
                continue;
            }
            if (Options.CreateOpts.Progress.class.isInstance(iOpt)) {
                if (progress != null) {
                    throw new HadoopIllegalArgumentException("Progress option is set multiple times");
                }
                progress = ((Options.CreateOpts.Progress)iOpt).getValue();
                continue;
            }
            if (Options.CreateOpts.CreateParent.class.isInstance(iOpt)) {
                if (createParent != null) {
                    throw new HadoopIllegalArgumentException("CreateParent option is set multiple times");
                }
                createParent = ((Options.CreateOpts.CreateParent)iOpt).getValue();
                continue;
            }
            throw new HadoopIllegalArgumentException("Unkown CreateOpts of type " + iOpt.getClass().getName());
        }
        if (permission == null) {
            throw new HadoopIllegalArgumentException("no permission supplied");
        }
        FsServerDefaults ssDef = this.getServerDefaults();
        if (ssDef.getBlockSize() % (long)ssDef.getBytesPerChecksum() != 0L) {
            throw new IOException("Internal error: default blockSize is not a multiple of default bytesPerChecksum ");
        }
        if (blockSize == -1L) {
            blockSize = ssDef.getBlockSize();
        }
        Options.ChecksumOpt defaultOpt = new Options.ChecksumOpt(ssDef.getChecksumType(), ssDef.getBytesPerChecksum());
        checksumOpt = Options.ChecksumOpt.processChecksumOpt(defaultOpt, checksumOpt, bytesPerChecksum);
        if (bufferSize == -1) {
            bufferSize = ssDef.getFileBufferSize();
        }
        if (replication == -1) {
            replication = ssDef.getReplication();
        }
        if (createParent == null) {
            createParent = false;
        }
        if (blockSize % (long)bytesPerChecksum != 0L) {
            throw new HadoopIllegalArgumentException("blockSize should be a multiple of checksumsize");
        }
        return this.createInternal(f, createFlag, permission, bufferSize, replication, blockSize, progress, checksumOpt, createParent);
    }

    public abstract FSDataOutputStream createInternal(Path var1, EnumSet<CreateFlag> var2, FsPermission var3, int var4, short var5, long var6, Progressable var8, Options.ChecksumOpt var9, boolean var10) throws AccessControlException, FileAlreadyExistsException, FileNotFoundException, ParentNotDirectoryException, UnsupportedFileSystemException, UnresolvedLinkException, IOException;

    public abstract void mkdir(Path var1, FsPermission var2, boolean var3) throws AccessControlException, FileAlreadyExistsException, FileNotFoundException, UnresolvedLinkException, IOException;

    public abstract boolean delete(Path var1, boolean var2) throws AccessControlException, FileNotFoundException, UnresolvedLinkException, IOException;

    public FSDataInputStream open(Path f) throws AccessControlException, FileNotFoundException, UnresolvedLinkException, IOException {
        return this.open(f, this.getServerDefaults().getFileBufferSize());
    }

    public abstract FSDataInputStream open(Path var1, int var2) throws AccessControlException, FileNotFoundException, UnresolvedLinkException, IOException;

    public boolean truncate(Path f, long newLength) throws AccessControlException, FileNotFoundException, UnresolvedLinkException, IOException {
        throw new UnsupportedOperationException(this.getClass().getSimpleName() + " doesn't support truncate");
    }

    public abstract boolean setReplication(Path var1, short var2) throws AccessControlException, FileNotFoundException, UnresolvedLinkException, IOException;

    public final void rename(Path src, Path dst, Options.Rename ... options) throws AccessControlException, FileAlreadyExistsException, FileNotFoundException, ParentNotDirectoryException, UnresolvedLinkException, IOException {
        boolean overwrite = false;
        if (null != options) {
            for (Options.Rename option : options) {
                if (option != Options.Rename.OVERWRITE) continue;
                overwrite = true;
            }
        }
        this.renameInternal(src, dst, overwrite);
    }

    public abstract void renameInternal(Path var1, Path var2) throws AccessControlException, FileAlreadyExistsException, FileNotFoundException, ParentNotDirectoryException, UnresolvedLinkException, IOException;

    public void renameInternal(Path src, Path dst, boolean overwrite) throws AccessControlException, FileAlreadyExistsException, FileNotFoundException, ParentNotDirectoryException, UnresolvedLinkException, IOException {
        FileStatus dstStatus;
        FileStatus srcStatus = this.getFileLinkStatus(src);
        try {
            dstStatus = this.getFileLinkStatus(dst);
        }
        catch (IOException e) {
            dstStatus = null;
        }
        if (dstStatus != null) {
            RemoteIterator<FileStatus> list;
            if (dst.equals(src)) {
                throw new FileAlreadyExistsException("The source " + src + " and destination " + dst + " are the same");
            }
            if (srcStatus.isSymlink() && dst.equals(srcStatus.getSymlink())) {
                throw new FileAlreadyExistsException("Cannot rename symlink " + src + " to its target " + dst);
            }
            if (srcStatus.isDirectory() != dstStatus.isDirectory()) {
                throw new IOException("Source " + src + " and destination " + dst + " must both be directories");
            }
            if (!overwrite) {
                throw new FileAlreadyExistsException("Rename destination " + dst + " already exists.");
            }
            if (dstStatus.isDirectory() && (list = this.listStatusIterator(dst)) != null && list.hasNext()) {
                throw new IOException("Rename cannot overwrite non empty destination directory " + dst);
            }
            this.delete(dst, false);
        } else {
            Path parent = dst.getParent();
            FileStatus parentStatus = this.getFileStatus(parent);
            if (parentStatus.isFile()) {
                throw new ParentNotDirectoryException("Rename destination parent " + parent + " is a file.");
            }
        }
        this.renameInternal(src, dst);
    }

    public boolean supportsSymlinks() {
        return false;
    }

    public void createSymlink(Path target, Path link, boolean createParent) throws IOException, UnresolvedLinkException {
        throw new IOException("File system does not support symlinks");
    }

    public Path getLinkTarget(Path f) throws IOException {
        throw new AssertionError((Object)("Implementation Error: " + this.getClass() + " that threw an UnresolvedLinkException, causing this method to be called, needs to override this method."));
    }

    public abstract void setPermission(Path var1, FsPermission var2) throws AccessControlException, FileNotFoundException, UnresolvedLinkException, IOException;

    public abstract void setOwner(Path var1, String var2, String var3) throws AccessControlException, FileNotFoundException, UnresolvedLinkException, IOException;

    public abstract void setTimes(Path var1, long var2, long var4) throws AccessControlException, FileNotFoundException, UnresolvedLinkException, IOException;

    public abstract FileChecksum getFileChecksum(Path var1) throws AccessControlException, FileNotFoundException, UnresolvedLinkException, IOException;

    public abstract FileStatus getFileStatus(Path var1) throws AccessControlException, FileNotFoundException, UnresolvedLinkException, IOException;

    @InterfaceAudience.LimitedPrivate(value={"HDFS", "Hive"})
    public void access(Path path, FsAction mode) throws AccessControlException, FileNotFoundException, UnresolvedLinkException, IOException {
        FileSystem.checkAccessPermissions(this.getFileStatus(path), mode);
    }

    public FileStatus getFileLinkStatus(Path f) throws AccessControlException, FileNotFoundException, UnsupportedFileSystemException, IOException {
        return this.getFileStatus(f);
    }

    public abstract BlockLocation[] getFileBlockLocations(Path var1, long var2, long var4) throws AccessControlException, FileNotFoundException, UnresolvedLinkException, IOException;

    public FsStatus getFsStatus(Path f) throws AccessControlException, FileNotFoundException, UnresolvedLinkException, IOException {
        return this.getFsStatus();
    }

    public abstract FsStatus getFsStatus() throws AccessControlException, FileNotFoundException, IOException;

    public RemoteIterator<FileStatus> listStatusIterator(final Path f) throws AccessControlException, FileNotFoundException, UnresolvedLinkException, IOException {
        return new RemoteIterator<FileStatus>(){
            private int i = 0;
            private FileStatus[] statusList = AbstractFileSystem.this.listStatus(f);

            @Override
            public boolean hasNext() {
                return this.i < this.statusList.length;
            }

            @Override
            public FileStatus next() {
                if (!this.hasNext()) {
                    throw new NoSuchElementException();
                }
                return this.statusList[this.i++];
            }
        };
    }

    public RemoteIterator<LocatedFileStatus> listLocatedStatus(final Path f) throws AccessControlException, FileNotFoundException, UnresolvedLinkException, IOException {
        return new RemoteIterator<LocatedFileStatus>(){
            private RemoteIterator<FileStatus> itor;
            {
                this.itor = AbstractFileSystem.this.listStatusIterator(f);
            }

            @Override
            public boolean hasNext() throws IOException {
                return this.itor.hasNext();
            }

            @Override
            public LocatedFileStatus next() throws IOException {
                if (!this.hasNext()) {
                    throw new NoSuchElementException("No more entry in " + f);
                }
                FileStatus result = this.itor.next();
                BlockLocation[] locs = null;
                if (result.isFile()) {
                    locs = AbstractFileSystem.this.getFileBlockLocations(result.getPath(), 0L, result.getLen());
                }
                return new LocatedFileStatus(result, locs);
            }
        };
    }

    public abstract FileStatus[] listStatus(Path var1) throws AccessControlException, FileNotFoundException, UnresolvedLinkException, IOException;

    public RemoteIterator<Path> listCorruptFileBlocks(Path path) throws IOException {
        throw new UnsupportedOperationException(this.getClass().getCanonicalName() + " does not support listCorruptFileBlocks");
    }

    public abstract void setVerifyChecksum(boolean var1) throws AccessControlException, IOException;

    public String getCanonicalServiceName() {
        return SecurityUtil.buildDTServiceName(this.getUri(), this.getUriDefaultPort());
    }

    @InterfaceAudience.LimitedPrivate(value={"HDFS", "MapReduce"})
    public List<Token<?>> getDelegationTokens(String renewer) throws IOException {
        return new ArrayList(0);
    }

    public void modifyAclEntries(Path path, List<AclEntry> aclSpec) throws IOException {
        throw new UnsupportedOperationException(this.getClass().getSimpleName() + " doesn't support modifyAclEntries");
    }

    public void removeAclEntries(Path path, List<AclEntry> aclSpec) throws IOException {
        throw new UnsupportedOperationException(this.getClass().getSimpleName() + " doesn't support removeAclEntries");
    }

    public void removeDefaultAcl(Path path) throws IOException {
        throw new UnsupportedOperationException(this.getClass().getSimpleName() + " doesn't support removeDefaultAcl");
    }

    public void removeAcl(Path path) throws IOException {
        throw new UnsupportedOperationException(this.getClass().getSimpleName() + " doesn't support removeAcl");
    }

    public void setAcl(Path path, List<AclEntry> aclSpec) throws IOException {
        throw new UnsupportedOperationException(this.getClass().getSimpleName() + " doesn't support setAcl");
    }

    public AclStatus getAclStatus(Path path) throws IOException {
        throw new UnsupportedOperationException(this.getClass().getSimpleName() + " doesn't support getAclStatus");
    }

    public void setXAttr(Path path, String name, byte[] value) throws IOException {
        this.setXAttr(path, name, value, EnumSet.of(XAttrSetFlag.CREATE, XAttrSetFlag.REPLACE));
    }

    public void setXAttr(Path path, String name, byte[] value, EnumSet<XAttrSetFlag> flag) throws IOException {
        throw new UnsupportedOperationException(this.getClass().getSimpleName() + " doesn't support setXAttr");
    }

    public byte[] getXAttr(Path path, String name) throws IOException {
        throw new UnsupportedOperationException(this.getClass().getSimpleName() + " doesn't support getXAttr");
    }

    public Map<String, byte[]> getXAttrs(Path path) throws IOException {
        throw new UnsupportedOperationException(this.getClass().getSimpleName() + " doesn't support getXAttrs");
    }

    public Map<String, byte[]> getXAttrs(Path path, List<String> names) throws IOException {
        throw new UnsupportedOperationException(this.getClass().getSimpleName() + " doesn't support getXAttrs");
    }

    public List<String> listXAttrs(Path path) throws IOException {
        throw new UnsupportedOperationException(this.getClass().getSimpleName() + " doesn't support listXAttrs");
    }

    public void removeXAttr(Path path, String name) throws IOException {
        throw new UnsupportedOperationException(this.getClass().getSimpleName() + " doesn't support removeXAttr");
    }

    public int hashCode() {
        return this.myUri.hashCode();
    }

    public boolean equals(Object other) {
        if (other == null || !(other instanceof AbstractFileSystem)) {
            return false;
        }
        return this.myUri.equals(((AbstractFileSystem)other).myUri);
    }
}

