/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.datanode;

import com.google.common.base.Joiner;
import com.google.common.collect.Lists;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.concurrent.BrokenBarrierException;
import java.util.concurrent.CyclicBarrier;
import java.util.concurrent.TimeoutException;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.conf.ReconfigurationException;
import org.apache.hadoop.fs.BlockLocation;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hdfs.BlockMissingException;
import org.apache.hadoop.hdfs.DFSTestUtil;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.hadoop.hdfs.MiniDFSNNTopology;
import org.apache.hadoop.hdfs.protocol.BlockListAsLongs;
import org.apache.hadoop.hdfs.protocol.ExtendedBlock;
import org.apache.hadoop.hdfs.protocolPB.DatanodeProtocolClientSideTranslatorPB;
import org.apache.hadoop.hdfs.server.common.Storage;
import org.apache.hadoop.hdfs.server.datanode.BlockPoolSliceStorage;
import org.apache.hadoop.hdfs.server.datanode.DataNode;
import org.apache.hadoop.hdfs.server.datanode.DataNodeTestUtils;
import org.apache.hadoop.hdfs.server.datanode.DataStorage;
import org.apache.hadoop.hdfs.server.datanode.StorageLocation;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.FsDatasetSpi;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.FsVolumeSpi;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.FsDatasetTestUtil;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.FsVolumeImpl;
import org.apache.hadoop.hdfs.server.protocol.BlockReportContext;
import org.apache.hadoop.hdfs.server.protocol.DatanodeRegistration;
import org.apache.hadoop.hdfs.server.protocol.DatanodeStorage;
import org.apache.hadoop.hdfs.server.protocol.StorageBlockReport;
import org.apache.hadoop.test.GenericTestUtils;
import org.hamcrest.CoreMatchers;
import org.hamcrest.Matcher;
import org.hamcrest.core.Is;
import org.junit.After;
import org.junit.Assert;
import org.junit.Test;
import org.mockito.Matchers;
import org.mockito.Mockito;
import org.mockito.invocation.InvocationOnMock;
import org.mockito.stubbing.Answer;
import org.mockito.verification.VerificationMode;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TestDataNodeHotSwapVolumes {
    private static final Logger LOG = LoggerFactory.getLogger(TestDataNodeHotSwapVolumes.class);
    private static final int BLOCK_SIZE = 512;
    private MiniDFSCluster cluster;

    @After
    public void tearDown() {
        this.shutdown();
    }

    private void startDFSCluster(int numNameNodes, int numDataNodes) throws IOException {
        this.shutdown();
        Configuration conf = new Configuration();
        conf.setLong("dfs.blocksize", 512L);
        conf.setInt("dfs.heartbeat.interval", 1);
        conf.setInt("dfs.df.interval", 1000);
        conf.setInt("dfs.namenode.heartbeat.recheck-interval", 1000);
        conf.setInt("dfs.datanode.failed.volumes.tolerated", 1);
        MiniDFSNNTopology nnTopology = MiniDFSNNTopology.simpleFederatedTopology(numNameNodes);
        this.cluster = new MiniDFSCluster.Builder(conf).nnTopology(nnTopology).numDataNodes(numDataNodes).build();
        this.cluster.waitActive();
    }

    private void shutdown() {
        if (this.cluster != null) {
            this.cluster.shutdown();
            this.cluster = null;
        }
    }

    private void createFile(Path path, int numBlocks) throws IOException, InterruptedException, TimeoutException {
        boolean replicateFactor = true;
        this.createFile(path, numBlocks, (short)1);
    }

    private void createFile(Path path, int numBlocks, short replicateFactor) throws IOException, InterruptedException, TimeoutException {
        this.createFile(0, path, numBlocks, replicateFactor);
    }

    private void createFile(int fsIdx, Path path, int numBlocks) throws IOException, InterruptedException, TimeoutException {
        boolean replicateFactor = true;
        this.createFile(fsIdx, path, numBlocks, (short)1);
    }

    private void createFile(int fsIdx, Path path, int numBlocks, short replicateFactor) throws IOException, TimeoutException, InterruptedException {
        boolean seed = false;
        DistributedFileSystem fs = this.cluster.getFileSystem(fsIdx);
        DFSTestUtil.createFile((FileSystem)fs, path, 512 * numBlocks, replicateFactor, 0L);
        DFSTestUtil.waitReplication((FileSystem)fs, path, replicateFactor);
    }

    private static void verifyFileLength(FileSystem fs, Path path, int numBlocks) throws IOException {
        FileStatus status = fs.getFileStatus(path);
        Assert.assertEquals((long)(numBlocks * 512), (long)status.getLen());
    }

    private static int getNumReplicas(FileSystem fs, Path file, int blockIdx) throws IOException {
        BlockLocation[] locs = fs.getFileBlockLocations(file, 0L, Long.MAX_VALUE);
        return locs.length < blockIdx + 1 ? 0 : locs[blockIdx].getNames().length;
    }

    private static void waitReplication(FileSystem fs, Path file, int blockIdx, int numReplicas) throws IOException, TimeoutException, InterruptedException {
        for (int attempts = 50; attempts > 0; --attempts) {
            int actualReplicas = TestDataNodeHotSwapVolumes.getNumReplicas(fs, file, blockIdx);
            if (actualReplicas == numReplicas) {
                return;
            }
            System.out.printf("Block %d of file %s has %d replicas (desired %d).\n", blockIdx, file.toString(), actualReplicas, numReplicas);
            Thread.sleep(100L);
        }
        throw new TimeoutException("Timed out waiting the " + blockIdx + "-th block of " + file + " to have " + numReplicas + " replicas.");
    }

    private static List<String> getDataDirs(DataNode datanode) {
        return new ArrayList<String>(datanode.getConf().getTrimmedStringCollection("dfs.datanode.data.dir"));
    }

    private static void triggerDeleteReport(DataNode datanode) throws IOException {
        datanode.scheduleAllBlockReport(0L);
        DataNodeTestUtils.triggerDeletionReport(datanode);
    }

    @Test
    public void testParseChangedVolumes() throws IOException {
        this.startDFSCluster(1, 1);
        DataNode dn = this.cluster.getDataNodes().get(0);
        Configuration conf = dn.getConf();
        String oldPaths = conf.get("dfs.datanode.data.dir");
        ArrayList<StorageLocation> oldLocations = new ArrayList<StorageLocation>();
        for (String path : oldPaths.split(",")) {
            oldLocations.add(StorageLocation.parse((String)path));
        }
        Assert.assertFalse((boolean)oldLocations.isEmpty());
        String newPaths = ((StorageLocation)oldLocations.get(0)).getFile().getAbsolutePath() + ",/foo/path1,/foo/path2";
        DataNode.ChangedVolumes changedVolumes = dn.parseChangedVolumes(newPaths);
        List newVolumes = changedVolumes.newLocations;
        Assert.assertEquals((long)2L, (long)newVolumes.size());
        Assert.assertEquals((Object)new File("/foo/path1").getAbsolutePath(), (Object)((StorageLocation)newVolumes.get(0)).getFile().getAbsolutePath());
        Assert.assertEquals((Object)new File("/foo/path2").getAbsolutePath(), (Object)((StorageLocation)newVolumes.get(1)).getFile().getAbsolutePath());
        List removedVolumes = changedVolumes.deactivateLocations;
        Assert.assertEquals((long)1L, (long)removedVolumes.size());
        Assert.assertEquals((Object)((StorageLocation)oldLocations.get(1)).getFile(), (Object)((StorageLocation)removedVolumes.get(0)).getFile());
        Assert.assertEquals((long)1L, (long)changedVolumes.unchangedLocations.size());
        Assert.assertEquals((Object)((StorageLocation)oldLocations.get(0)).getFile(), (Object)((StorageLocation)changedVolumes.unchangedLocations.get(0)).getFile());
    }

    @Test
    public void testParseChangedVolumesFailures() throws IOException {
        this.startDFSCluster(1, 1);
        DataNode dn = this.cluster.getDataNodes().get(0);
        try {
            dn.parseChangedVolumes("");
            Assert.fail((String)"Should throw IOException: empty inputs.");
        }
        catch (IOException e) {
            GenericTestUtils.assertExceptionContains((String)"No directory is specified.", (Throwable)e);
        }
    }

    private void addVolumes(int numNewVolumes) throws ReconfigurationException, IOException {
        File volumeDir;
        File dataDir = new File(this.cluster.getDataDirectory());
        DataNode dn = this.cluster.getDataNodes().get(0);
        Configuration conf = dn.getConf();
        String oldDataDir = conf.get("dfs.datanode.data.dir");
        ArrayList<File> newVolumeDirs = new ArrayList<File>();
        StringBuilder newDataDirBuf = new StringBuilder(oldDataDir);
        int startIdx = oldDataDir.split(",").length + 1;
        while ((volumeDir = new File(dataDir, "data" + startIdx)).exists()) {
            ++startIdx;
        }
        for (int i = startIdx; i < startIdx + numNewVolumes; ++i) {
            File volumeDir2 = new File(dataDir, "data" + String.valueOf(i));
            newVolumeDirs.add(volumeDir2);
            volumeDir2.mkdirs();
            newDataDirBuf.append(",");
            newDataDirBuf.append(StorageLocation.parse((String)volumeDir2.toString()).toString());
        }
        String newDataDir = newDataDirBuf.toString();
        dn.reconfigurePropertyImpl("dfs.datanode.data.dir", newDataDir);
        String[] effectiveDataDirs = conf.get("dfs.datanode.data.dir").split(",");
        String[] expectDataDirs = newDataDir.split(",");
        Assert.assertEquals((long)expectDataDirs.length, (long)effectiveDataDirs.length);
        for (int i = 0; i < expectDataDirs.length; ++i) {
            StorageLocation expectLocation = StorageLocation.parse((String)expectDataDirs[i]);
            StorageLocation effectiveLocation = StorageLocation.parse((String)effectiveDataDirs[i]);
            Assert.assertEquals((Object)expectLocation.getStorageType(), (Object)effectiveLocation.getStorageType());
            Assert.assertEquals((Object)expectLocation.getFile().getCanonicalFile(), (Object)effectiveLocation.getFile().getCanonicalFile());
        }
        for (File volumeDir3 : newVolumeDirs) {
            File curDir = new File(volumeDir3, "current");
            Assert.assertTrue((boolean)curDir.exists());
            Assert.assertTrue((boolean)curDir.isDirectory());
        }
    }

    private List<List<Integer>> getNumBlocksReport(int namesystemIdx) {
        ArrayList<List<Integer>> results = new ArrayList<List<Integer>>();
        String bpid = this.cluster.getNamesystem(namesystemIdx).getBlockPoolId();
        List<Map<DatanodeStorage, BlockListAsLongs>> blockReports = this.cluster.getAllBlockReports(bpid);
        for (Map<DatanodeStorage, BlockListAsLongs> datanodeReport : blockReports) {
            ArrayList<Integer> numBlocksPerDN = new ArrayList<Integer>();
            for (BlockListAsLongs blocks : datanodeReport.values()) {
                numBlocksPerDN.add(blocks.getNumberOfBlocks());
            }
            results.add(numBlocksPerDN);
        }
        return results;
    }

    @Test(timeout=60000L)
    public void testAddOneNewVolume() throws IOException, ReconfigurationException, InterruptedException, TimeoutException {
        this.startDFSCluster(1, 1);
        String bpid = this.cluster.getNamesystem().getBlockPoolId();
        int numBlocks = 10;
        this.addVolumes(1);
        Path testFile = new Path("/test");
        this.createFile(testFile, 10);
        List<Map<DatanodeStorage, BlockListAsLongs>> blockReports = this.cluster.getAllBlockReports(bpid);
        Assert.assertEquals((long)1L, (long)blockReports.size());
        Assert.assertEquals((long)3L, (long)blockReports.get(0).size());
        int minNumBlocks = Integer.MAX_VALUE;
        int maxNumBlocks = Integer.MIN_VALUE;
        for (BlockListAsLongs blockList : blockReports.get(0).values()) {
            minNumBlocks = Math.min(minNumBlocks, blockList.getNumberOfBlocks());
            maxNumBlocks = Math.max(maxNumBlocks, blockList.getNumberOfBlocks());
        }
        Assert.assertTrue((Math.abs(maxNumBlocks - maxNumBlocks) <= 1 ? 1 : 0) != 0);
        TestDataNodeHotSwapVolumes.verifyFileLength((FileSystem)this.cluster.getFileSystem(), testFile, 10);
    }

    @Test(timeout=60000L)
    public void testAddVolumesDuringWrite() throws IOException, InterruptedException, TimeoutException, ReconfigurationException {
        this.startDFSCluster(1, 1);
        String bpid = this.cluster.getNamesystem().getBlockPoolId();
        Path testFile = new Path("/test");
        this.createFile(testFile, 4);
        this.addVolumes(2);
        DFSTestUtil.appendFile((FileSystem)this.cluster.getFileSystem(), testFile, 4096);
        TestDataNodeHotSwapVolumes.verifyFileLength((FileSystem)this.cluster.getFileSystem(), testFile, 12);
        List<Integer> expectedNumBlocks = Arrays.asList(2, 2, 4, 4);
        List<Map<DatanodeStorage, BlockListAsLongs>> blockReports = this.cluster.getAllBlockReports(bpid);
        Assert.assertEquals((long)1L, (long)blockReports.size());
        Assert.assertEquals((long)4L, (long)blockReports.get(0).size());
        Map<DatanodeStorage, BlockListAsLongs> dnReport = blockReports.get(0);
        ArrayList<Integer> actualNumBlocks = new ArrayList<Integer>();
        for (BlockListAsLongs blockList : dnReport.values()) {
            actualNumBlocks.add(blockList.getNumberOfBlocks());
        }
        Collections.sort(actualNumBlocks);
        Assert.assertEquals(expectedNumBlocks, actualNumBlocks);
    }

    @Test(timeout=60000L)
    public void testAddVolumesToFederationNN() throws IOException, TimeoutException, InterruptedException, ReconfigurationException {
        int numNameNodes = 2;
        boolean numDataNodes = true;
        this.startDFSCluster(2, 1);
        Path testFile = new Path("/test");
        this.createFile(0, testFile, 4);
        this.createFile(1, testFile, 4);
        int numNewVolumes = 2;
        this.addVolumes(2);
        DFSTestUtil.appendFile((FileSystem)this.cluster.getFileSystem(0), testFile, 4096);
        List<List<Integer>> actualNumBlocks = this.getNumBlocksReport(0);
        Assert.assertEquals((long)this.cluster.getDataNodes().size(), (long)actualNumBlocks.size());
        List<Integer> blocksOnFirstDN = actualNumBlocks.get(0);
        Collections.sort(blocksOnFirstDN);
        Assert.assertEquals(Arrays.asList(2, 2, 4, 4), blocksOnFirstDN);
        actualNumBlocks = this.getNumBlocksReport(1);
        Assert.assertEquals((long)4L, (long)actualNumBlocks.get(0).size());
        Assert.assertEquals((long)2L, (long)Collections.frequency((Collection)actualNumBlocks.get(0), 0));
    }

    @Test(timeout=60000L)
    public void testRemoveOneVolume() throws ReconfigurationException, InterruptedException, TimeoutException, IOException {
        this.startDFSCluster(1, 1);
        boolean replFactor = true;
        Path testFile = new Path("/test");
        this.createFile(testFile, 10, (short)1);
        DataNode dn = this.cluster.getDataNodes().get(0);
        List<String> oldDirs = TestDataNodeHotSwapVolumes.getDataDirs(dn);
        String newDirs = (String)oldDirs.iterator().next();
        dn.reconfigurePropertyImpl("dfs.datanode.data.dir", newDirs);
        TestDataNodeHotSwapVolumes.assertFileLocksReleased(new ArrayList<String>(oldDirs).subList(1, oldDirs.size()));
        dn.scheduleAllBlockReport(0L);
        try {
            DFSTestUtil.readFile((FileSystem)this.cluster.getFileSystem(), testFile);
            Assert.fail((String)"Expect to throw BlockMissingException.");
        }
        catch (BlockMissingException e) {
            GenericTestUtils.assertExceptionContains((String)"Could not obtain block", (Throwable)e);
        }
        Path newFile = new Path("/newFile");
        this.createFile(newFile, 6);
        String bpid = this.cluster.getNamesystem().getBlockPoolId();
        List<Map<DatanodeStorage, BlockListAsLongs>> blockReports = this.cluster.getAllBlockReports(bpid);
        Assert.assertEquals((long)1L, (long)blockReports.size());
        BlockListAsLongs blocksForVolume1 = blockReports.get(0).values().iterator().next();
        Assert.assertEquals((long)11L, (long)blocksForVolume1.getNumberOfBlocks());
    }

    @Test(timeout=60000L)
    public void testReplicatingAfterRemoveVolume() throws InterruptedException, TimeoutException, IOException, ReconfigurationException {
        this.startDFSCluster(1, 2);
        DistributedFileSystem fs = this.cluster.getFileSystem();
        int replFactor = 2;
        Path testFile = new Path("/test");
        this.createFile(testFile, 4, (short)2);
        DataNode dn = this.cluster.getDataNodes().get(0);
        List<String> oldDirs = TestDataNodeHotSwapVolumes.getDataDirs(dn);
        String newDirs = (String)oldDirs.iterator().next();
        dn.reconfigurePropertyImpl("dfs.datanode.data.dir", newDirs);
        TestDataNodeHotSwapVolumes.assertFileLocksReleased(new ArrayList<String>(oldDirs).subList(1, oldDirs.size()));
        TestDataNodeHotSwapVolumes.triggerDeleteReport(dn);
        TestDataNodeHotSwapVolumes.waitReplication((FileSystem)fs, testFile, 1, 1);
        DFSTestUtil.waitReplication((FileSystem)fs, testFile, (short)2);
    }

    @Test
    public void testAddVolumeFailures() throws IOException {
        this.startDFSCluster(1, 1);
        String dataDir = this.cluster.getDataDirectory();
        DataNode dn = this.cluster.getDataNodes().get(0);
        ArrayList newDirs = Lists.newArrayList();
        int NUM_NEW_DIRS = 4;
        for (int i = 0; i < 4; ++i) {
            File newVolume = new File(dataDir, "new_vol" + i);
            newDirs.add(newVolume.toString());
            if (i % 2 != 0) continue;
            newVolume.createNewFile();
        }
        String newValue = dn.getConf().get("dfs.datanode.data.dir") + "," + Joiner.on((String)",").join((Iterable)newDirs);
        try {
            dn.reconfigurePropertyImpl("dfs.datanode.data.dir", newValue);
            Assert.fail((String)"Expect to throw IOException.");
        }
        catch (ReconfigurationException e) {
            String errorMessage = e.getCause().getMessage();
            String[] messages = errorMessage.split("\\r?\\n");
            Assert.assertEquals((long)2L, (long)messages.length);
            Assert.assertThat((Object)messages[0], (Matcher)CoreMatchers.containsString((String)"new_vol0"));
            Assert.assertThat((Object)messages[1], (Matcher)CoreMatchers.containsString((String)"new_vol2"));
        }
        FsDatasetSpi dataset = dn.getFSDataset();
        for (FsVolumeSpi volume : dataset.getVolumes()) {
            Assert.assertThat((Object)volume.getBasePath(), (Matcher)Is.is((Matcher)CoreMatchers.not((Matcher)CoreMatchers.anyOf((Matcher)Is.is((Object)((String)newDirs.get(0))), (Matcher)Is.is((Object)((String)newDirs.get(2)))))));
        }
        DataStorage storage = dn.getStorage();
        for (int i = 0; i < storage.getNumStorageDirs(); ++i) {
            Storage.StorageDirectory sd = storage.getStorageDir(i);
            Assert.assertThat((Object)sd.getRoot().toString(), (Matcher)Is.is((Matcher)CoreMatchers.not((Matcher)CoreMatchers.anyOf((Matcher)Is.is((Object)((String)newDirs.get(0))), (Matcher)Is.is((Object)((String)newDirs.get(2)))))));
        }
        String[] effectiveVolumes = dn.getConf().get("dfs.datanode.data.dir").split(",");
        Assert.assertEquals((long)4L, (long)effectiveVolumes.length);
        for (String ev : effectiveVolumes) {
            Assert.assertThat((Object)StorageLocation.parse((String)ev).getFile().getCanonicalPath(), (Matcher)Is.is((Matcher)CoreMatchers.not((Matcher)CoreMatchers.anyOf((Matcher)Is.is((Object)((String)newDirs.get(0))), (Matcher)Is.is((Object)((String)newDirs.get(2)))))));
        }
    }

    private static void assertFileLocksReleased(Collection<String> dirs) throws IOException {
        for (String dir : dirs) {
            try {
                FsDatasetTestUtil.assertFileLockReleased(dir);
            }
            catch (IOException e) {
                LOG.warn("{}", (Throwable)e);
            }
        }
    }

    @Test(timeout=180000L)
    public void testRemoveVolumeBeingWritten() throws InterruptedException, TimeoutException, ReconfigurationException, IOException, BrokenBarrierException {
        for (int i = 0; i < 3; ++i) {
            this.testRemoveVolumeBeingWrittenForDatanode(i);
        }
    }

    private void testRemoveVolumeBeingWrittenForDatanode(int dataNodeIdx) throws IOException, ReconfigurationException, TimeoutException, InterruptedException, BrokenBarrierException {
        this.startDFSCluster(1, 3);
        int REPLICATION = 3;
        final DataNode dn = this.cluster.getDataNodes().get(dataNodeIdx);
        DistributedFileSystem fs = this.cluster.getFileSystem();
        Path testFile = new Path("/test");
        long lastTimeDiskErrorCheck = dn.getLastDiskErrorCheck();
        FSDataOutputStream out = fs.create(testFile, (short)3);
        Random rb = new Random(0L);
        byte[] writeBuf = new byte[256];
        rb.nextBytes(writeBuf);
        out.write(writeBuf);
        out.hflush();
        final FsDatasetSpi data = dn.data;
        dn.data = (FsDatasetSpi)Mockito.spy((Object)data);
        ((FsDatasetSpi)Mockito.doAnswer((Answer)new Answer<Object>(){

            public Object answer(InvocationOnMock invocation) throws IOException, InterruptedException {
                Thread.sleep(1000L);
                data.finalizeBlock((ExtendedBlock)invocation.getArguments()[0], ((Boolean)invocation.getArguments()[1]).booleanValue());
                return null;
            }
        }).when((Object)dn.data)).finalizeBlock((ExtendedBlock)Matchers.any(ExtendedBlock.class), Mockito.anyBoolean());
        final CyclicBarrier barrier = new CyclicBarrier(2);
        List<String> oldDirs = TestDataNodeHotSwapVolumes.getDataDirs(dn);
        final String newDirs = oldDirs.get(1);
        final ArrayList exceptions = new ArrayList();
        Thread reconfigThread = new Thread(){

            @Override
            public void run() {
                try {
                    barrier.await();
                    dn.reconfigurePropertyImpl("dfs.datanode.data.dir", newDirs);
                }
                catch (InterruptedException | BrokenBarrierException | ReconfigurationException e) {
                    exceptions.add(e);
                }
            }
        };
        reconfigThread.start();
        barrier.await();
        rb.nextBytes(writeBuf);
        out.write(writeBuf);
        out.hflush();
        out.close();
        reconfigThread.join();
        DFSTestUtil.waitReplication((FileSystem)fs, testFile, (short)3);
        byte[] content = DFSTestUtil.readFileBuffer((FileSystem)fs, testFile);
        Assert.assertEquals((long)512L, (long)content.length);
        Assert.assertEquals((long)lastTimeDiskErrorCheck, (long)dn.getLastDiskErrorCheck());
        if (!exceptions.isEmpty()) {
            throw new IOException(((Exception)exceptions.get(0)).getCause());
        }
    }

    @Test(timeout=60000L)
    public void testAddBackRemovedVolume() throws IOException, TimeoutException, InterruptedException, ReconfigurationException {
        this.startDFSCluster(1, 2);
        this.createFile(new Path("/test"), 32);
        DataNode dn = this.cluster.getDataNodes().get(0);
        Configuration conf = dn.getConf();
        String oldDataDir = conf.get("dfs.datanode.data.dir");
        String keepDataDir = oldDataDir.split(",")[0];
        String removeDataDir = oldDataDir.split(",")[1];
        dn.reconfigurePropertyImpl("dfs.datanode.data.dir", keepDataDir);
        for (int i = 0; i < this.cluster.getNumNameNodes(); ++i) {
            String bpid = this.cluster.getNamesystem(i).getBlockPoolId();
            BlockPoolSliceStorage bpsStorage = dn.getStorage().getBPStorage(bpid);
            for (int j = 0; j < bpsStorage.getNumStorageDirs(); ++j) {
                Storage.StorageDirectory sd = bpsStorage.getStorageDir(j);
                Assert.assertFalse((boolean)sd.getRoot().getAbsolutePath().startsWith(new File(removeDataDir).getAbsolutePath()));
            }
            Assert.assertEquals((long)dn.getStorage().getBPStorage(bpid).getNumStorageDirs(), (long)1L);
        }
        dn.reconfigurePropertyImpl("dfs.datanode.data.dir", oldDataDir);
    }

    private FsVolumeImpl getVolume(DataNode dn, File basePath) {
        for (FsVolumeSpi vol : dn.getFSDataset().getVolumes()) {
            if (!vol.getBasePath().equals(basePath.getPath())) continue;
            return (FsVolumeImpl)vol;
        }
        return null;
    }

    @Test(timeout=60000L)
    public void testDirectlyReloadAfterCheckDiskError() throws IOException, TimeoutException, InterruptedException, ReconfigurationException {
        this.startDFSCluster(1, 2);
        this.createFile(new Path("/test"), 32, (short)2);
        DataNode dn = this.cluster.getDataNodes().get(0);
        String oldDataDir = dn.getConf().get("dfs.datanode.data.dir");
        File dirToFail = new File(this.cluster.getDataDirectory(), "data1");
        FsVolumeImpl failedVolume = this.getVolume(dn, dirToFail);
        Assert.assertTrue((String)("No FsVolume was found for " + dirToFail), (failedVolume != null ? 1 : 0) != 0);
        long used = failedVolume.getDfsUsed();
        DataNodeTestUtils.injectDataDirFailure(dirToFail);
        long lastDiskErrorCheck = dn.getLastDiskErrorCheck();
        dn.checkDiskErrorAsync();
        while (dn.getLastDiskErrorCheck() == lastDiskErrorCheck) {
            Thread.sleep(100L);
        }
        this.createFile(new Path("/test1"), 32, (short)2);
        Assert.assertEquals((long)used, (long)failedVolume.getDfsUsed());
        DataNodeTestUtils.restoreDataDirFromFailure(dirToFail);
        dn.reconfigurePropertyImpl("dfs.datanode.data.dir", oldDataDir);
        this.createFile(new Path("/test2"), 32, (short)2);
        FsVolumeImpl restoredVolume = this.getVolume(dn, dirToFail);
        Assert.assertTrue((restoredVolume != null ? 1 : 0) != 0);
        Assert.assertTrue((restoredVolume != failedVolume ? 1 : 0) != 0);
        Assert.assertTrue((restoredVolume.getDfsUsed() > used ? 1 : 0) != 0);
    }

    @Test(timeout=100000L)
    public void testFullBlockReportAfterRemovingVolumes() throws IOException, ReconfigurationException {
        Configuration conf = new Configuration();
        conf.setLong("dfs.blocksize", 512L);
        conf.setLong("dfs.blockreport.intervalMsec", 10800000L);
        conf.setLong("dfs.heartbeat.interval", 1080L);
        this.cluster = new MiniDFSCluster.Builder(conf).numDataNodes(2).build();
        this.cluster.waitActive();
        DataNode dn = this.cluster.getDataNodes().get(0);
        DatanodeProtocolClientSideTranslatorPB spy = DataNodeTestUtils.spyOnBposToNN(dn, this.cluster.getNameNode());
        File dataDirToKeep = new File(this.cluster.getDataDirectory(), "data1");
        dn.reconfigurePropertyImpl("dfs.datanode.data.dir", dataDirToKeep.toString());
        ((DatanodeProtocolClientSideTranslatorPB)Mockito.verify((Object)spy, (VerificationMode)Mockito.timeout((int)60000).times(1))).blockReport((DatanodeRegistration)Matchers.any(DatanodeRegistration.class), Matchers.anyString(), (StorageBlockReport[])Matchers.any(StorageBlockReport[].class), (BlockReportContext)Matchers.any(BlockReportContext.class));
    }
}

