/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.blockmanagement;

import java.util.HashMap;
import java.util.concurrent.ThreadLocalRandom;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hdfs.server.blockmanagement.DatanodeDescriptor;
import org.apache.hadoop.thirdparty.com.google.common.base.Preconditions;
import org.apache.hadoop.util.Time;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class BlockReportLeaseManager {
    static final Logger LOG = LoggerFactory.getLogger(BlockReportLeaseManager.class);
    private final NodeData deferredHead = NodeData.ListHead("deferredHead");
    private final NodeData pendingHead = NodeData.ListHead("pendingHead");
    private final HashMap<String, NodeData> nodes = new HashMap();
    private int numPending = 0;
    private final int maxPending;
    private final long leaseExpiryMs;
    private long nextId = ThreadLocalRandom.current().nextLong();

    BlockReportLeaseManager(Configuration conf) {
        this(conf.getInt("dfs.namenode.max.full.block.report.leases", 6), conf.getLong("dfs.namenode.full.block.report.lease.length.ms", 300000L));
    }

    BlockReportLeaseManager(int maxPending, long leaseExpiryMs) {
        Preconditions.checkArgument((maxPending >= 1 ? 1 : 0) != 0, (Object)"Cannot set the maximum number of block report leases to a value less than 1.");
        this.maxPending = maxPending;
        Preconditions.checkArgument((leaseExpiryMs >= 1L ? 1 : 0) != 0, (Object)"Cannot set full block report lease expiry period to a value less than 1.");
        this.leaseExpiryMs = leaseExpiryMs;
    }

    private long getNextId() {
        return ++this.nextId == 0L ? (this.nextId = this.nextId + 1L) : this.nextId;
    }

    public synchronized void register(DatanodeDescriptor dn) {
        this.registerNode(dn);
    }

    private synchronized NodeData registerNode(DatanodeDescriptor dn) {
        if (this.nodes.containsKey(dn.getDatanodeUuid())) {
            LOG.info("Can't register DN {} because it is already registered.", (Object)dn.getDatanodeUuid());
            return null;
        }
        NodeData node = new NodeData(dn.getDatanodeUuid());
        this.deferredHead.addToBeginning(node);
        this.nodes.put(dn.getDatanodeUuid(), node);
        LOG.info("Registered DN {} ({}).", (Object)dn.getDatanodeUuid(), (Object)dn.getXferAddr());
        return node;
    }

    private synchronized void remove(NodeData node) {
        if (node.leaseId != 0L) {
            --this.numPending;
            node.leaseId = 0L;
            node.leaseTimeMs = 0L;
        }
        node.removeSelf();
    }

    public synchronized void unregister(DatanodeDescriptor dn) {
        NodeData node = this.nodes.remove(dn.getDatanodeUuid());
        if (node == null) {
            LOG.info("Can't unregister DN {} because it is not currently registered.", (Object)dn.getDatanodeUuid());
            return;
        }
        this.remove(node);
    }

    public synchronized long requestLease(DatanodeDescriptor dn) {
        NodeData node = this.nodes.get(dn.getDatanodeUuid());
        if (node == null) {
            LOG.warn("DN {} ({}) requested a lease even though it wasn't yet registered.  Registering now.", (Object)dn.getDatanodeUuid(), (Object)dn.getXferAddr());
            node = this.registerNode(dn);
        }
        if (node.leaseId != 0L) {
            LOG.debug("Removing existing BR lease 0x{} for DN {} in order to issue a new one.", (Object)Long.toHexString(node.leaseId), (Object)dn.getDatanodeUuid());
        }
        this.remove(node);
        long monotonicNowMs = Time.monotonicNow();
        this.pruneExpiredPending(monotonicNowMs);
        if (this.numPending >= this.maxPending) {
            if (LOG.isDebugEnabled()) {
                StringBuilder allLeases = new StringBuilder();
                String prefix = "";
                NodeData cur = this.pendingHead.next;
                while (cur != this.pendingHead) {
                    allLeases.append(prefix).append(cur.datanodeUuid);
                    prefix = ", ";
                    cur = cur.next;
                }
                LOG.debug("Can't create a new BR lease for DN {}, because numPending equals maxPending at {}.  Current leases: {}", new Object[]{dn.getDatanodeUuid(), this.numPending, allLeases.toString()});
            }
            return 0L;
        }
        ++this.numPending;
        node.leaseId = this.getNextId();
        node.leaseTimeMs = monotonicNowMs;
        this.pendingHead.addToEnd(node);
        if (LOG.isDebugEnabled()) {
            LOG.debug("Created a new BR lease 0x{} for DN {}.  numPending = {}", new Object[]{Long.toHexString(node.leaseId), dn.getDatanodeUuid(), this.numPending});
        }
        return node.leaseId;
    }

    private synchronized boolean pruneIfExpired(long monotonicNowMs, NodeData node) {
        if (monotonicNowMs < node.leaseTimeMs + this.leaseExpiryMs) {
            return false;
        }
        LOG.info("Removing expired block report lease 0x{} for DN {}.", (Object)Long.toHexString(node.leaseId), (Object)node.datanodeUuid);
        Preconditions.checkState((node.leaseId != 0L ? 1 : 0) != 0);
        this.remove(node);
        this.deferredHead.addToBeginning(node);
        return true;
    }

    private synchronized void pruneExpiredPending(long monotonicNowMs) {
        NodeData cur = this.pendingHead.next;
        while (cur != this.pendingHead) {
            NodeData next = cur.next;
            if (!this.pruneIfExpired(monotonicNowMs, cur)) {
                return;
            }
            cur = next;
        }
        LOG.trace("No entries remaining in the pending list.");
    }

    public synchronized boolean checkLease(DatanodeDescriptor dn, long monotonicNowMs, long id) {
        if (id == 0L) {
            LOG.debug("Datanode {} is using BR lease id 0x0 to bypass rate-limiting.", (Object)dn.getDatanodeUuid());
            return true;
        }
        NodeData node = this.nodes.get(dn.getDatanodeUuid());
        if (node == null) {
            LOG.info("BR lease 0x{} is not valid for unknown datanode {}", (Object)Long.toHexString(id), (Object)dn.getDatanodeUuid());
            return false;
        }
        if (node.leaseId == 0L) {
            LOG.warn("BR lease 0x{} is not valid for DN {}, because the DN is not in the pending set.", (Object)Long.toHexString(id), (Object)dn.getDatanodeUuid());
            return false;
        }
        if (this.pruneIfExpired(monotonicNowMs, node)) {
            LOG.warn("BR lease 0x{} is not valid for DN {}, because the lease has expired.", (Object)Long.toHexString(id), (Object)dn.getDatanodeUuid());
            return false;
        }
        if (id != node.leaseId) {
            LOG.warn("BR lease 0x{} is not valid for DN {}.  Expected BR lease 0x{}.", new Object[]{Long.toHexString(id), dn.getDatanodeUuid(), Long.toHexString(node.leaseId)});
            return false;
        }
        if (LOG.isTraceEnabled()) {
            LOG.trace("BR lease 0x{} is valid for DN {}.", (Object)Long.toHexString(id), (Object)dn.getDatanodeUuid());
        }
        return true;
    }

    public synchronized long removeLease(DatanodeDescriptor dn) {
        NodeData node = this.nodes.get(dn.getDatanodeUuid());
        if (node == null) {
            LOG.info("Can't remove lease for unknown datanode {}", (Object)dn.getDatanodeUuid());
            return 0L;
        }
        long id = node.leaseId;
        if (id == 0L) {
            LOG.debug("DN {} has no lease to remove.", (Object)dn.getDatanodeUuid());
            return 0L;
        }
        this.remove(node);
        this.deferredHead.addToEnd(node);
        if (LOG.isTraceEnabled()) {
            LOG.trace("Removed BR lease 0x{} for DN {}.  numPending = {}", new Object[]{Long.toHexString(id), dn.getDatanodeUuid(), this.numPending});
        }
        return id;
    }

    private static class NodeData {
        final String datanodeUuid;
        long leaseId;
        long leaseTimeMs;
        NodeData prev;
        NodeData next;

        static NodeData ListHead(String name) {
            NodeData node;
            node.next = node = new NodeData(name);
            node.prev = node;
            return node;
        }

        NodeData(String datanodeUuid) {
            this.datanodeUuid = datanodeUuid;
        }

        void removeSelf() {
            if (this.prev != null) {
                this.prev.next = this.next;
            }
            if (this.next != null) {
                this.next.prev = this.prev;
            }
            this.next = null;
            this.prev = null;
        }

        void addToEnd(NodeData node) {
            Preconditions.checkState((node.next == null ? 1 : 0) != 0);
            Preconditions.checkState((node.prev == null ? 1 : 0) != 0);
            node.prev = this.prev;
            node.next = this;
            this.prev.next = node;
            this.prev = node;
        }

        void addToBeginning(NodeData node) {
            Preconditions.checkState((node.next == null ? 1 : 0) != 0);
            Preconditions.checkState((node.prev == null ? 1 : 0) != 0);
            node.next = this.next;
            node.prev = this;
            this.next.prev = node;
            this.next = node;
        }
    }
}

