/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.yarn.server.volume;

import java.io.IOException;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.shaded.com.google.gson.JsonArray;
import org.apache.hadoop.util.BaseMapRUtil;
import org.apache.hadoop.util.MaprShellCommandExecutor;
import org.apache.hadoop.util.RMVolumeShardingUtil;
import org.apache.hadoop.util.Shell;
import org.apache.hadoop.yarn.conf.YarnDefaultProperties;
import org.apache.hadoop.yarn.server.api.ConfigurableAuxiliaryService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class VolumeManager
extends ConfigurableAuxiliaryService {
    public static final String VOLUME_NAME = "name";
    public static final String VOLUME_PATH = "path";
    public static final String VOLUME_MOUNTED = "mounted";
    public static final String HS_VOLUME_NAME = "mapr.historyserver.volume";
    public static final String RM_VOLUME_NAME = "mapr.resourcemanager.volume";
    protected Logger LOG = LoggerFactory.getLogger(VolumeManager.class);
    protected String volumeLogfilePath;
    protected String volumeMode;
    protected String mountPath;
    protected Path yarnDir;
    protected boolean newVolumePathSupportEnabled;
    protected FileSystem fs;
    private static final long timestamp = System.currentTimeMillis();

    public VolumeManager(String name) {
        super(name);
    }

    public void serviceInit(Configuration conf) throws Exception {
        this.fs = FileSystem.get((Configuration)conf);
        this.newVolumePathSupportEnabled = conf.getBoolean("yarn.resourcemanager.dir.new-volume-path-support.enabled", false);
        this.volumeLogfilePath = RMVolumeShardingUtil.getPathToVolumeLog();
        this.yarnDir = new Path(conf.get("yarn.dir", YarnDefaultProperties.DEFAULT_YARN_DIR));
    }

    public abstract void createVolumes(Configuration var1) throws Exception;

    protected void createVolume(String volumeNumber) throws Exception {
        String pathToVolumeScript = RMVolumeShardingUtil.getPathToVolumeCreateScript();
        int argsCount = !volumeNumber.equals("") ? 6 : 5;
        String[] args = new String[argsCount];
        args[0] = pathToVolumeScript;
        args[1] = BaseMapRUtil.getMapRHostName();
        args[2] = this.mountPath;
        args[3] = this.mountPath;
        args[4] = this.volumeMode;
        if (argsCount > 5) {
            args[5] = volumeNumber;
        }
        String volumePath = argsCount > 5 ? args[2] + "/" + args[5] : args[2];
        HashMap<String, String> env = new HashMap<String, String>();
        env.put("MAPR_MAPREDUCE_MODE", "yarn");
        Shell.ShellCommandExecutor shexec = new Shell.ShellCommandExecutor(args, null, env);
        if (this.LOG.isInfoEnabled()) {
            this.LOG.info("Checking for volume. If volume not present command will create and mount it. Command invoked is : " + shexec.toString());
        }
        int numAttempts = 3;
        for (int i = 0; i < numAttempts; ++i) {
            try {
                shexec.execute();
                break;
            }
            catch (IOException ioe) {
                if (i == numAttempts - 1) {
                    int exitCode = shexec.getExitCode();
                    if (exitCode != 0) {
                        this.LOG.error("Failed to create and mount volume at " + volumePath + ". Please see logs at " + this.volumeLogfilePath);
                        this.LOG.error("Command ran " + shexec.toString());
                        this.LOG.error("Command output " + shexec.getOutput());
                    }
                    throw ioe;
                }
                Thread.sleep(100L);
                if (!this.LOG.isInfoEnabled()) continue;
                this.LOG.info("Retrying check for volume ... ");
                continue;
            }
        }
        if (this.LOG.isInfoEnabled()) {
            this.LOG.info("Sucessfully created volume and mounted at " + volumePath);
        }
    }

    protected void waitForYarnPathCreated() throws Exception {
        int timeout = 200;
        Path yarnParentDir = this.yarnDir.getParent();
        if (!this.fs.exists(yarnParentDir)) {
            this.LOG.info("Waiting for Yarn parent directory creation(" + yarnParentDir.toString() + ") before mounting RM volumes");
        }
        for (int i = 0; i < timeout && !this.fs.exists(yarnParentDir); ++i) {
            TimeUnit.SECONDS.sleep(3L);
        }
        if (!this.fs.exists(yarnParentDir)) {
            throw new RuntimeException("Yarn parent directory is not found: " + yarnParentDir.toString());
        }
        this.verifyYarnDirPermissions(this.yarnDir);
        if (this.newVolumePathSupportEnabled) {
            this.fs.mkdirs(this.yarnDir);
        }
    }

    protected void createDir(String pathName, FsPermission perm) throws IOException {
        if (this.LOG.isInfoEnabled()) {
            this.LOG.info("Creating dir: " + pathName + " with permission: " + perm);
        }
        FileSystem.mkdirs((FileSystem)this.fs, (Path)new Path(pathName), (FsPermission)perm);
    }

    protected void verifyYarnDirPermissions(Path yarnDir) throws Exception {
        Path clusterDirectory = yarnDir.getParent();
        Path parentClusterDirectory = clusterDirectory.getParent();
        if (this.fs.exists(parentClusterDirectory)) {
            FileStatus parentClusterDirectoryFileStatus = this.fs.getFileStatus(parentClusterDirectory);
            if (this.fs.exists(clusterDirectory)) {
                this.copyOwnerAndPermission(parentClusterDirectoryFileStatus, clusterDirectory);
            }
            if (this.fs.exists(yarnDir)) {
                this.copyOwnerAndPermission(parentClusterDirectoryFileStatus, yarnDir);
            }
        }
    }

    protected void copyPermissionsIfNeeded(FileStatus srcStatus, Path dst) throws Exception {
        Path src = srcStatus.getPath();
        if (srcStatus.isDirectory()) {
            List<FileStatus> contents = Arrays.asList(this.fs.listStatus(src));
            for (FileStatus innerFile : contents) {
                this.copyPermissionsIfNeeded(innerFile, new Path(dst, innerFile.getPath().getName()));
            }
        }
        this.copyOwnerAndPermission(srcStatus, dst);
    }

    protected void copyOwnerAndPermission(FileStatus srcStatus, Path dst) throws Exception {
        FileStatus dstFileStatus = this.fs.getFileStatus(dst);
        FsPermission srcFilePermission = srcStatus.getPermission();
        String srcFileOwner = srcStatus.getOwner();
        String srcFileGroup = srcStatus.getGroup();
        if (!dstFileStatus.getPermission().equals((Object)srcFilePermission)) {
            this.fs.setPermission(dst, srcFilePermission);
        }
        if (!dstFileStatus.getOwner().equals(srcFileOwner) || !dstFileStatus.getGroup().equals(srcFileGroup)) {
            this.fs.setOwner(dst, srcFileOwner, srcFileGroup);
        }
    }

    protected Map<String, String> getVolumeInfo(String volumeName) throws IOException {
        MaprShellCommandExecutor executor = new MaprShellCommandExecutor();
        String[] volumeListCommand = new String[]{"volume", "list"};
        HashMap<String, String> volumeListParams = new HashMap<String, String>();
        volumeListParams.put("columns", "volumename,mountdir,mounted");
        volumeListParams.put("filter", "[n==" + volumeName + "]");
        JsonArray result = executor.execute(volumeListCommand, volumeListParams, false);
        if (result != null && result.size() > 0) {
            String name = result.get(0).getAsJsonObject().get("volumename").getAsString();
            String path = result.get(0).getAsJsonObject().get("mountdir").getAsString();
            String mounted = result.get(0).getAsJsonObject().get(VOLUME_MOUNTED).getAsString();
            HashMap<String, String> volumeInfo = new HashMap<String, String>();
            volumeInfo.put(VOLUME_NAME, name);
            volumeInfo.put(VOLUME_PATH, path);
            volumeInfo.put(VOLUME_MOUNTED, mounted);
            return volumeInfo;
        }
        return null;
    }

    protected boolean isVolumeMounted(String volumeName) throws IOException {
        Map<String, String> volumeInfo = this.getVolumeInfo(volumeName);
        if (volumeInfo != null) {
            return Integer.parseInt(volumeInfo.get(VOLUME_MOUNTED)) == 1;
        }
        return false;
    }

    protected void mountVolume(String volumeName, String volumeMountPath) throws IOException {
        MaprShellCommandExecutor executor = new MaprShellCommandExecutor();
        String[] command = new String[]{"volume", "mount"};
        HashMap<String, String> params = new HashMap<String, String>();
        params.put(VOLUME_NAME, volumeName);
        params.put(VOLUME_PATH, volumeMountPath);
        executor.execute(command, params, false);
    }

    protected void unmountVolume(String volumeName) throws IOException {
        MaprShellCommandExecutor executor = new MaprShellCommandExecutor();
        String[] command = new String[]{"volume", "unmount"};
        HashMap<String, String> params = new HashMap<String, String>();
        params.put(VOLUME_NAME, volumeName);
        executor.execute(command, params, false);
    }

    protected void lockVolume(String volumeName) throws Exception {
        this.waitForYarnPathCreated();
        Path yarnParentDir = this.yarnDir.getParent();
        Path lockFile = new Path(yarnParentDir, volumeName + "_lock");
        Path lockFileID = new Path(lockFile, BaseMapRUtil.getMapRHostName() + "_" + timestamp);
        if (!this.fs.exists(lockFile)) {
            this.LOG.debug("Volume creation locked by current instance, lockfile is " + lockFileID.toString());
            this.fs.mkdirs(lockFileID);
        } else {
            this.LOG.info("Volume creation (" + volumeName + ") is performed by another VolumeManager instance, waiting for completion, lockfile is " + lockFile.toString());
            this.waitForLockRelease(lockFile);
            this.lockVolume(volumeName);
        }
    }

    protected void unlockVolume(String volumeName) throws IOException {
        Path yarnParentDir = this.yarnDir.getParent();
        Path lockFile = new Path(yarnParentDir, volumeName + "_lock");
        Path lockFileID = new Path(lockFile, BaseMapRUtil.getMapRHostName() + "_" + timestamp);
        if (this.fs.exists(lockFileID)) {
            this.fs.delete(lockFile, true);
        }
    }

    protected void waitForLockRelease(Path lockFile) throws Exception {
        int timeout = 200;
        for (int i = 0; i < timeout && this.fs.exists(lockFile); ++i) {
            TimeUnit.SECONDS.sleep(3L);
        }
        if (this.fs.exists(lockFile)) {
            long creationTime = this.fs.getFileStatus(lockFile).getModificationTime();
            long currentMillis = System.currentTimeMillis();
            if (currentMillis - creationTime > 590000L) {
                this.fs.delete(lockFile, true);
            } else {
                throw new RuntimeException("Volume creation lock wait timeout");
            }
        }
    }
}

