/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.yarn.server.resourcemanager.resource;

import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.shaded.com.fasterxml.jackson.databind.ObjectMapper;
import org.apache.hadoop.thirdparty.com.google.common.annotations.VisibleForTesting;
import org.apache.hadoop.yarn.api.records.Resource;
import org.apache.hadoop.yarn.api.records.ResourceInformation;
import org.apache.hadoop.yarn.exceptions.YARNFeatureNotEnabledException;
import org.apache.hadoop.yarn.exceptions.YarnException;
import org.apache.hadoop.yarn.server.resourcemanager.resource.ResourceProfilesManager;
import org.apache.hadoop.yarn.util.resource.ResourceUtils;
import org.apache.hadoop.yarn.util.resource.Resources;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ResourceProfilesManagerImpl
implements ResourceProfilesManager {
    private static final Logger LOG = LoggerFactory.getLogger(ResourceProfilesManagerImpl.class);
    private final Map<String, Resource> profiles = new ConcurrentHashMap<String, Resource>();
    private Configuration conf;
    private boolean profileEnabled = false;
    private static final String MEMORY = ResourceInformation.MEMORY_MB.getName();
    private static final String VCORES = ResourceInformation.VCORES.getName();
    public static final String DEFAULT_PROFILE = "default";
    public static final String MINIMUM_PROFILE = "minimum";
    public static final String MAXIMUM_PROFILE = "maximum";
    protected final ReentrantReadWriteLock.ReadLock readLock;
    protected final ReentrantReadWriteLock.WriteLock writeLock;
    private static final String[] MANDATORY_PROFILES = new String[]{"default", "minimum", "maximum"};
    private static final String FEATURE_NOT_ENABLED_MSG = "Resource profile is not enabled, please enable resource profile feature before using its functions. (by setting yarn.resourcemanager.resource-profiles.enabled to true)";

    public ResourceProfilesManagerImpl() {
        ReentrantReadWriteLock lock = new ReentrantReadWriteLock();
        this.readLock = lock.readLock();
        this.writeLock = lock.writeLock();
    }

    @Override
    public void init(Configuration config) throws IOException {
        this.conf = config;
        this.loadProfiles();
    }

    private void loadProfiles() throws IOException {
        URL tmp;
        String sourceFile;
        this.profileEnabled = this.conf.getBoolean("yarn.resourcemanager.resource-profiles.enabled", false);
        if (!this.profileEnabled) {
            return;
        }
        String resourcesFile = sourceFile = this.conf.get("yarn.resourcemanager.resource-profiles.source-file", "resource-profiles.json");
        ClassLoader classLoader = Thread.currentThread().getContextClassLoader();
        if (classLoader == null) {
            classLoader = ResourceProfilesManagerImpl.class.getClassLoader();
        }
        if (classLoader != null && (tmp = classLoader.getResource(sourceFile)) != null) {
            resourcesFile = tmp.getPath();
        }
        ObjectMapper mapper = new ObjectMapper();
        Map data = (Map)mapper.readValue(new File(resourcesFile), Map.class);
        for (Map.Entry entry : data.entrySet()) {
            String profileName = entry.getKey().toString();
            if (profileName.isEmpty()) {
                throw new IOException("Name of resource profile cannot be an empty string");
            }
            if (profileName.equals(MINIMUM_PROFILE) || profileName.equals(MAXIMUM_PROFILE)) {
                throw new IOException(String.format("profile={%s, %s} is should not be specified inside %s, they will be loaded from resource-types.xml", MINIMUM_PROFILE, MAXIMUM_PROFILE, sourceFile));
            }
            if (!(entry.getValue() instanceof Map)) continue;
            Map profileInfo = (Map)entry.getValue();
            if (!profileInfo.containsKey(MEMORY) || !profileInfo.containsKey(VCORES)) {
                throw new IOException("Illegal resource profile definition; profile '" + profileName + "' must contain '" + MEMORY + "' and '" + VCORES + "'");
            }
            Resource resource = this.parseResource(profileInfo);
            this.profiles.put(profileName, resource);
            LOG.info("Added profile '" + profileName + "' with resources: " + resource);
        }
        this.profiles.put(MINIMUM_PROFILE, ResourceUtils.getResourceTypesMinimumAllocation());
        this.profiles.put(MAXIMUM_PROFILE, ResourceUtils.getResourceTypesMaximumAllocation());
        for (String profile : MANDATORY_PROFILES) {
            if (this.profiles.containsKey(profile)) continue;
            throw new IOException("Mandatory profile missing '" + profile + "' missing. " + Arrays.toString(MANDATORY_PROFILES) + " must be present");
        }
        LOG.info("Loaded profiles: " + this.profiles.keySet());
    }

    private Resource parseResource(Map profileInfo) throws IOException {
        Resource resource = Resource.newInstance((int)0, (int)0);
        Iterator iterator = profileInfo.entrySet().iterator();
        Map resourceTypes = ResourceUtils.getResourceTypes();
        while (iterator.hasNext()) {
            Map.Entry resourceEntry = iterator.next();
            String resourceName = resourceEntry.getKey().toString();
            ResourceInformation resourceValue = this.fromString(resourceName, resourceEntry.getValue().toString());
            if (resourceName.equals(MEMORY)) {
                resource.setMemorySize(resourceValue.getValue());
                continue;
            }
            if (resourceName.equals(VCORES)) {
                resource.setVirtualCores(Long.valueOf(resourceValue.getValue()).intValue());
                continue;
            }
            if (resourceTypes.containsKey(resourceName)) {
                resource.setResourceInformation(resourceName, resourceValue);
                continue;
            }
            throw new IOException("Unrecognized resource type '" + resourceName + "'. Recognized resource types are '" + resourceTypes.keySet() + "'");
        }
        return resource;
    }

    private void checkAndThrowExceptionWhenFeatureDisabled() throws YARNFeatureNotEnabledException {
        if (!this.profileEnabled) {
            throw new YARNFeatureNotEnabledException(FEATURE_NOT_ENABLED_MSG);
        }
    }

    @Override
    public Resource getProfile(String profile) throws YarnException {
        this.checkAndThrowExceptionWhenFeatureDisabled();
        if (profile == null) {
            throw new YarnException("Profile name cannot be null");
        }
        Resource profileRes = this.profiles.get(profile);
        if (profileRes == null) {
            throw new YarnException("Resource profile '" + profile + "' not found");
        }
        return Resources.clone((Resource)profileRes);
    }

    @Override
    public Map<String, Resource> getResourceProfiles() throws YARNFeatureNotEnabledException {
        this.checkAndThrowExceptionWhenFeatureDisabled();
        return Collections.unmodifiableMap(this.profiles);
    }

    @Override
    @VisibleForTesting
    public void reloadProfiles() throws IOException {
        this.profiles.clear();
        this.loadProfiles();
    }

    @Override
    public Resource getDefaultProfile() throws YarnException {
        return this.getProfile(DEFAULT_PROFILE);
    }

    @Override
    public Resource getMinimumProfile() throws YarnException {
        return this.getProfile(MINIMUM_PROFILE);
    }

    @Override
    public Resource getMaximumProfile() throws YarnException {
        return this.getProfile(MAXIMUM_PROFILE);
    }

    private ResourceInformation fromString(String name, String value) {
        String units = ResourceUtils.getUnits((String)value);
        Long resourceValue = Long.valueOf(value.substring(0, value.length() - units.length()));
        return ResourceInformation.newInstance((String)name, (String)units, (long)resourceValue);
    }
}

