/**
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.liance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License. See accompanying LICENSE file.
 */
package org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.security.authentication.util;

import java.util.Properties;
import org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.javax.servlet.ServletContext;
import org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.javax.servlet.ServletException;
import org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.security.authentication.server.AuthenticationFilter;
import org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.security.authentication.server.CookieSignatureSecretFactory;

import org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.slf4j.Logger;
import org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.slf4j.LoggerFactory;

/**
 * MapR implementation of SignerSecretProvider abstract class.
 */
@InterfaceStability.Unstable
@InterfaceAudience.Private
public class MapRSignerSecretProvider extends SignerSecretProvider {

    private static Logger LOG = LoggerFactory.getLogger(MapRSignerSecretProvider.class);

    private byte[] secret;
    private byte[][] secrets;

    public MapRSignerSecretProvider() {}

    @Override
    public void init(Properties config, ServletContext servletContext,
                     long tokenValidity) throws Exception {
        String signatureSecret = config.getProperty(
                AuthenticationFilter.SIGNATURE_SECRET, "org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.mapr.security.maprauth.MaprSignatureSecretFactory");

        Class<?> signatureFactoryClass = Thread.currentThread()
                .getContextClassLoader().loadClass(signatureSecret);

        Object signatureFactory = signatureFactoryClass.newInstance();
        if (signatureFactory instanceof CookieSignatureSecretFactory) {
            signatureSecret = ((CookieSignatureSecretFactory) signatureFactory)
                    .getSignatureSecret();
        } else {
            throw new ServletException("The cookie signature secret factory class should implement "
                    + CookieSignatureSecretFactory.class.getName() + " interface");
        }

        // MapR signature secret can only generated by a mapr user. When a service
        // is not running as mapr user, we fall back to random secret.
        if (signatureSecret == null) {
            LOG.info("Fall back to random secret");
            initUsingRandomSecret(config, servletContext, tokenValidity);
        } else {
            secret = signatureSecret.getBytes();
            secrets = new byte[][]{secret};
        }
    }

    private void initUsingRandomSecret(Properties config, ServletContext ctx,
                                       long validity) throws Exception {

        SignerSecretProvider provider = new RandomSignerSecretProvider();
        provider.init(config, ctx, validity);
        secret = provider.getCurrentSecret();
        secrets = provider.getAllSecrets();
    }

    @Override
    public byte[] getCurrentSecret() {
        return secret;
    }

    @Override
    public byte[][] getAllSecrets() {
        return secrets;
    }
}