/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.runtime.util.config.memory;

import java.util.Map;
import java.util.function.Consumer;
import org.apache.flink.configuration.ConfigOption;
import org.apache.flink.configuration.Configuration;
import org.apache.flink.configuration.ConfigurationUtils;
import org.apache.flink.configuration.IllegalConfigurationException;
import org.apache.flink.configuration.MemorySize;
import org.apache.flink.core.testutils.CommonTestUtils;
import org.apache.flink.runtime.util.config.memory.ProcessMemoryOptions;
import org.apache.flink.runtime.util.config.memory.ProcessMemorySpec;
import org.apache.flink.runtime.util.config.memory.ProcessMemoryUtils;
import org.apache.flink.util.Preconditions;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class ProcessMemoryUtilsTestBase<T extends ProcessMemorySpec> {
    protected final Logger log = LoggerFactory.getLogger(this.getClass());
    private static Map<String, String> oldEnvVariables;
    private final ProcessMemoryOptions options;
    private final ConfigOption<MemorySize> newOptionForLegacyHeapOption;

    protected ProcessMemoryUtilsTestBase(ProcessMemoryOptions options, ConfigOption<MemorySize> newOptionForLegacyHeapOption) {
        this.options = (ProcessMemoryOptions)Preconditions.checkNotNull((Object)options);
        this.newOptionForLegacyHeapOption = (ConfigOption)Preconditions.checkNotNull(newOptionForLegacyHeapOption);
    }

    @BeforeEach
    void setup() {
        oldEnvVariables = System.getenv();
    }

    @AfterEach
    void teardown() {
        if (oldEnvVariables != null) {
            CommonTestUtils.setEnv(oldEnvVariables, (boolean)true);
        }
    }

    @Test
    void testGenerateJvmParameters() {
        JvmArgTestingProcessMemorySpec spec = JvmArgTestingProcessMemorySpec.generate();
        String jvmParamsStr = ProcessMemoryUtils.generateJvmParametersStr((ProcessMemorySpec)spec, (boolean)true);
        Map configs = ConfigurationUtils.parseJvmArgString((String)jvmParamsStr);
        Assertions.assertThat((Map)configs).hasSize(4);
        Assertions.assertThat((Comparable)MemorySize.parse((String)((String)configs.get("-Xmx")))).isEqualTo((Object)spec.getJvmHeapMemorySize());
        Assertions.assertThat((Comparable)MemorySize.parse((String)((String)configs.get("-Xms")))).isEqualTo((Object)spec.getJvmHeapMemorySize());
        Assertions.assertThat((Comparable)MemorySize.parse((String)((String)configs.get("-XX:MaxMetaspaceSize=")))).isEqualTo((Object)spec.getJvmMetaspaceSize());
        Assertions.assertThat((Comparable)MemorySize.parse((String)((String)configs.get("-XX:MaxDirectMemorySize=")))).isEqualTo((Object)spec.getJvmDirectMemorySize());
    }

    @Test
    void testGenerateJvmParametersWithoutDirectMemoryLimit() {
        JvmArgTestingProcessMemorySpec spec = JvmArgTestingProcessMemorySpec.generate();
        String jvmParamsStr = ProcessMemoryUtils.generateJvmParametersStr((ProcessMemorySpec)spec, (boolean)false);
        Map configs = ConfigurationUtils.parseJvmArgString((String)jvmParamsStr);
        Assertions.assertThat((Map)configs).hasSize(3);
        Assertions.assertThat((Comparable)MemorySize.parse((String)((String)configs.get("-Xmx")))).isEqualTo((Object)spec.getJvmHeapMemorySize());
        Assertions.assertThat((Comparable)MemorySize.parse((String)((String)configs.get("-Xms")))).isEqualTo((Object)spec.getJvmHeapMemorySize());
        Assertions.assertThat((Comparable)MemorySize.parse((String)((String)configs.get("-XX:MaxMetaspaceSize=")))).isEqualTo((Object)spec.getJvmMetaspaceSize());
        Assertions.assertThat((boolean)configs.containsKey("-XX:MaxDirectMemorySize=")).isFalse();
    }

    @Test
    void testConfigTotalFlinkMemory() {
        MemorySize totalFlinkMemorySize = MemorySize.parse((String)"1g");
        Configuration conf = new Configuration();
        conf.set(this.options.getTotalFlinkMemoryOption(), (Object)totalFlinkMemorySize);
        T processSpec = this.processSpecFromConfig(conf);
        Assertions.assertThat((Comparable)processSpec.getTotalFlinkMemorySize()).isEqualTo((Object)totalFlinkMemorySize);
    }

    @Test
    void testConfigTotalProcessMemorySize() {
        MemorySize totalProcessMemorySize = MemorySize.parse((String)"2g");
        Configuration conf = new Configuration();
        conf.set(this.options.getTotalProcessMemoryOption(), (Object)totalProcessMemorySize);
        T processSpec = this.processSpecFromConfig(conf);
        Assertions.assertThat((Comparable)processSpec.getTotalProcessMemorySize()).isEqualTo((Object)totalProcessMemorySize);
    }

    @Test
    void testExceptionShouldContainRequiredConfigOptions() {
        try {
            this.processSpecFromConfig(new Configuration());
        }
        catch (IllegalConfigurationException e) {
            this.options.getRequiredFineGrainedOptions().forEach(option -> Assertions.assertThat((Throwable)e).hasMessageContaining(option.key()));
            Assertions.assertThat((Throwable)e).hasMessageContaining(this.options.getTotalFlinkMemoryOption().key()).hasMessageContaining(this.options.getTotalProcessMemoryOption().key());
        }
    }

    @Test
    void testDerivedTotalProcessMemoryGreaterThanConfiguredFailureWithFineGrainedOptions() {
        Configuration conf = this.getConfigurationWithJvmMetaspaceAndTotalFlinkMemory(100L, 200L);
        MemorySize totalFlinkMemorySize = MemorySize.ofMebiBytes((long)150L);
        this.configWithFineGrainedOptions(conf, totalFlinkMemorySize);
        this.validateFail(conf);
    }

    @Test
    void testDerivedTotalProcessMemoryGreaterThanConfiguredFailureWithTotalFlinkMemory() {
        Configuration conf = this.getConfigurationWithJvmMetaspaceAndTotalFlinkMemory(100L, 200L);
        MemorySize totalFlinkMemorySize = MemorySize.ofMebiBytes((long)150L);
        conf.set(this.options.getTotalFlinkMemoryOption(), (Object)totalFlinkMemorySize);
        this.validateFail(conf);
    }

    private Configuration getConfigurationWithJvmMetaspaceAndTotalFlinkMemory(long jvmMetaspaceSizeMb, long totalProcessMemorySizeMb) {
        MemorySize jvmMetaspaceSize = MemorySize.ofMebiBytes((long)jvmMetaspaceSizeMb);
        MemorySize totalProcessMemorySize = MemorySize.ofMebiBytes((long)totalProcessMemorySizeMb);
        Configuration conf = new Configuration();
        conf.set(this.options.getJvmOptions().getJvmMetaspaceOption(), (Object)jvmMetaspaceSize);
        conf.set(this.options.getTotalProcessMemoryOption(), (Object)totalProcessMemorySize);
        return conf;
    }

    @Test
    void testConfigJvmMetaspaceSize() {
        MemorySize jvmMetaspaceSize = MemorySize.parse((String)"50m");
        Configuration conf = new Configuration();
        conf.set(this.options.getJvmOptions().getJvmMetaspaceOption(), (Object)jvmMetaspaceSize);
        this.validateInAllConfigurations(conf, processSpec -> Assertions.assertThat((Comparable)processSpec.getJvmMetaspaceSize()).isEqualTo((Object)jvmMetaspaceSize));
    }

    @Test
    void testConfigJvmOverheadRange() {
        MemorySize minSize = MemorySize.parse((String)"50m");
        MemorySize maxSize = MemorySize.parse((String)"200m");
        Configuration conf = new Configuration();
        conf.set(this.options.getJvmOptions().getJvmOverheadMax(), (Object)maxSize);
        conf.set(this.options.getJvmOptions().getJvmOverheadMin(), (Object)minSize);
        this.validateInAllConfigurations(conf, JobManagerProcessSpec -> {
            Assertions.assertThat((long)JobManagerProcessSpec.getJvmOverheadSize().getBytes()).isGreaterThanOrEqualTo(minSize.getBytes());
            Assertions.assertThat((long)JobManagerProcessSpec.getJvmOverheadSize().getBytes()).isLessThanOrEqualTo(maxSize.getBytes());
        });
    }

    @Test
    void testConfigJvmOverheadRangeFailure() {
        MemorySize minSize = MemorySize.parse((String)"200m");
        MemorySize maxSize = MemorySize.parse((String)"50m");
        Configuration conf = new Configuration();
        conf.set(this.options.getJvmOptions().getJvmOverheadMax(), (Object)maxSize);
        conf.set(this.options.getJvmOptions().getJvmOverheadMin(), (Object)minSize);
        this.validateFailInAllConfigurations(conf);
    }

    @Test
    void testConfigJvmOverheadFraction() {
        MemorySize minSize = MemorySize.ZERO;
        MemorySize maxSize = MemorySize.parse((String)"1t");
        float fraction = 0.2f;
        Configuration conf = new Configuration();
        conf.set(this.options.getJvmOptions().getJvmOverheadMax(), (Object)maxSize);
        conf.set(this.options.getJvmOptions().getJvmOverheadMin(), (Object)minSize);
        conf.set(this.options.getJvmOptions().getJvmOverheadFraction(), (Object)Float.valueOf(fraction));
        this.validateInAllConfigurations(conf, jobManagerProcessSpec -> Assertions.assertThat((Comparable)jobManagerProcessSpec.getJvmOverheadSize()).isEqualTo((Object)jobManagerProcessSpec.getTotalProcessMemorySize().multiply((double)fraction)));
    }

    @Test
    void testConfigJvmOverheadFractionFailureNegative() {
        Configuration conf = new Configuration();
        conf.set(this.options.getJvmOptions().getJvmOverheadFraction(), (Object)Float.valueOf(-0.1f));
        this.validateFailInAllConfigurations(conf);
    }

    @Test
    void testConfigJvmOverheadFractionFailureNoLessThanOne() {
        Configuration conf = new Configuration();
        conf.set(this.options.getJvmOptions().getJvmOverheadFraction(), (Object)Float.valueOf(1.0f));
        this.validateFailInAllConfigurations(conf);
    }

    @Test
    void testConfigJvmOverheadDeriveFromProcessAndFlinkMemorySize() {
        Configuration conf = new Configuration();
        conf.set(this.options.getTotalProcessMemoryOption(), (Object)MemorySize.parse((String)"1000m"));
        conf.set(this.options.getTotalFlinkMemoryOption(), (Object)MemorySize.parse((String)"800m"));
        conf.set(this.options.getJvmOptions().getJvmMetaspaceOption(), (Object)MemorySize.parse((String)"100m"));
        conf.set(this.options.getJvmOptions().getJvmOverheadMin(), (Object)MemorySize.parse((String)"50m"));
        conf.set(this.options.getJvmOptions().getJvmOverheadMax(), (Object)MemorySize.parse((String)"200m"));
        conf.set(this.options.getJvmOptions().getJvmOverheadFraction(), (Object)Float.valueOf(0.5f));
        T jobManagerProcessSpec = this.processSpecFromConfig(conf);
        Assertions.assertThat((Comparable)jobManagerProcessSpec.getJvmOverheadSize()).isEqualTo((Object)MemorySize.parse((String)"100m"));
    }

    @Test
    void testConfigJvmOverheadDeriveFromProcessAndFlinkMemorySizeFailure() {
        Configuration conf = new Configuration();
        conf.set(this.options.getTotalProcessMemoryOption(), (Object)MemorySize.parse((String)"1000m"));
        conf.set(this.options.getTotalFlinkMemoryOption(), (Object)MemorySize.parse((String)"800m"));
        conf.set(this.options.getJvmOptions().getJvmMetaspaceOption(), (Object)MemorySize.parse((String)"100m"));
        conf.set(this.options.getJvmOptions().getJvmOverheadMin(), (Object)MemorySize.parse((String)"150m"));
        conf.set(this.options.getJvmOptions().getJvmOverheadMax(), (Object)MemorySize.parse((String)"200m"));
        conf.set(this.options.getJvmOptions().getJvmOverheadFraction(), (Object)Float.valueOf(0.5f));
        this.validateFail(conf);
    }

    @Test
    void testConfigTotalProcessMemoryAddUpFailure() {
        MemorySize totalProcessMemory = MemorySize.parse((String)"699m");
        MemorySize totalFlinkMemory = MemorySize.parse((String)"500m");
        MemorySize jvmMetaspace = MemorySize.parse((String)"100m");
        MemorySize jvmOverhead = MemorySize.parse((String)"100m");
        Configuration conf = new Configuration();
        conf.set(this.options.getTotalProcessMemoryOption(), (Object)totalProcessMemory);
        conf.set(this.options.getTotalFlinkMemoryOption(), (Object)totalFlinkMemory);
        conf.set(this.options.getJvmOptions().getJvmMetaspaceOption(), (Object)jvmMetaspace);
        conf.set(this.options.getJvmOptions().getJvmOverheadMin(), (Object)jvmOverhead);
        conf.set(this.options.getJvmOptions().getJvmOverheadMax(), (Object)jvmOverhead);
        this.validateFail(conf);
    }

    protected abstract void validateInAllConfigurations(Configuration var1, Consumer<T> var2);

    protected abstract void validateFailInAllConfigurations(Configuration var1);

    protected abstract void validateFail(Configuration var1);

    protected abstract T processSpecFromConfig(Configuration var1);

    protected abstract void configWithFineGrainedOptions(Configuration var1, MemorySize var2);

    protected ConfigOption<MemorySize> getNewOptionForLegacyHeapOption() {
        return this.newOptionForLegacyHeapOption;
    }

    private static class JvmArgTestingProcessMemorySpec
    implements ProcessMemorySpec {
        private static final long serialVersionUID = 2863985135320165745L;
        private final MemorySize heap;
        private final MemorySize directMemory;
        private final MemorySize metaspace;

        private JvmArgTestingProcessMemorySpec(MemorySize heap, MemorySize directMemory, MemorySize metaspace) {
            this.heap = heap;
            this.directMemory = directMemory;
            this.metaspace = metaspace;
        }

        public MemorySize getJvmHeapMemorySize() {
            return this.heap;
        }

        public MemorySize getJvmDirectMemorySize() {
            return this.directMemory;
        }

        public MemorySize getJvmMetaspaceSize() {
            return this.metaspace;
        }

        public MemorySize getJvmOverheadSize() {
            throw new UnsupportedOperationException();
        }

        public MemorySize getTotalFlinkMemorySize() {
            throw new UnsupportedOperationException();
        }

        public MemorySize getTotalProcessMemorySize() {
            throw new UnsupportedOperationException();
        }

        public static JvmArgTestingProcessMemorySpec generate() {
            return new JvmArgTestingProcessMemorySpec(MemorySize.ofMebiBytes((long)1L), MemorySize.ofMebiBytes((long)2L), MemorySize.ofMebiBytes((long)3L));
        }
    }
}

