/*
 * Decompiled with CFR 0.152.
 */
package org.apache.drill.jdbc;

import java.math.BigDecimal;
import java.sql.Clob;
import java.sql.Connection;
import java.sql.Date;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.SQLFeatureNotSupportedException;
import java.sql.SQLTimeoutException;
import java.sql.Statement;
import java.sql.Timestamp;
import java.util.List;
import java.util.Properties;
import java.util.Random;
import java.util.concurrent.TimeUnit;
import org.apache.drill.categories.JdbcTest;
import org.apache.drill.exec.ExecConstants;
import org.apache.drill.exec.physical.impl.ScreenCreator;
import org.apache.drill.exec.testing.Controls;
import org.apache.drill.jdbc.Driver;
import org.apache.drill.jdbc.JdbcTestBase;
import org.apache.drill.jdbc.SqlTimeoutException;
import org.apache.drill.shaded.guava.com.google.common.collect.ImmutableList;
import org.hamcrest.CoreMatchers;
import org.hamcrest.Matcher;
import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.BeforeClass;
import org.junit.Ignore;
import org.junit.Test;
import org.junit.experimental.categories.Category;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Category(value={JdbcTest.class})
public class PreparedStatementTest
extends JdbcTestBase {
    private static final Logger logger = LoggerFactory.getLogger(PreparedStatementTest.class);
    private static final Random RANDOMIZER = new Random(20150304L);
    private static final String SYS_VERSION_SQL = "select * from sys.version";
    private static final String SYS_RANDOM_SQL = "SELECT cast(random() as varchar) as myStr FROM (VALUES(1)) union SELECT cast(random() as varchar) as myStr FROM (VALUES(1)) union SELECT cast(random() as varchar) as myStr FROM (VALUES(1)) ";
    private static final Matcher<String> PARAMETERS_NOT_SUPPORTED_MSG_MATCHER = CoreMatchers.allOf((Matcher)CoreMatchers.containsString((String)"arameter"), (Matcher)CoreMatchers.containsString((String)"not"), (Matcher)CoreMatchers.containsString((String)"support"));
    private static Connection connection;

    @BeforeClass
    public static void setUpConnection() throws SQLException {
        Driver.load();
        Properties properties = new Properties();
        properties.setProperty(ExecConstants.bootDefaultFor((String)"prepare.statement.create_timeout_ms"), "30000");
        connection = DriverManager.getConnection("jdbc:drill:zk=local", properties);
        try (Statement stmt = connection.createStatement();){
            stmt.execute(String.format("alter session set `%s` = true", "planner.enable_decimal_data_type"));
        }
    }

    @AfterClass
    public static void tearDownConnection() throws SQLException {
        if (connection != null) {
            try (Statement stmt = connection.createStatement();){
                stmt.execute(String.format("alter session set `%s` = false", "planner.enable_decimal_data_type"));
            }
        }
        connection.close();
    }

    @Test
    public void testExecuteQueryBasicCaseWorks() throws SQLException {
        try (PreparedStatement stmt = connection.prepareStatement("VALUES 11");
             ResultSet rs = stmt.executeQuery();){
            Assert.assertThat((String)"Unexpected column count", (Object)rs.getMetaData().getColumnCount(), (Matcher)CoreMatchers.equalTo((Object)1));
            Assert.assertTrue((String)"No expected first row", (boolean)rs.next());
            Assert.assertThat((Object)rs.getInt(1), (Matcher)CoreMatchers.equalTo((Object)11));
            Assert.assertFalse((String)"Unexpected second row", (boolean)rs.next());
        }
    }

    @Test
    public void testQueryMetadataInPreparedStatement() throws SQLException {
        try (PreparedStatement stmt = connection.prepareStatement("SELECT cast(1 as INTEGER ) as int_field, cast(12384729 as BIGINT ) as bigint_field, cast('varchar_value' as varchar(50)) as varchar_field, timestamp '2008-2-23 10:00:20.123' as ts_field, date '2008-2-23' as date_field, cast('99999912399.4567' as decimal(18, 5)) as decimal_field FROM sys.version");){
            ImmutableList exp = ImmutableList.of((Object)new ExpectedColumnResult("int_field", 4, 0, 11, 0, 0, true, Integer.class.getName()), (Object)new ExpectedColumnResult("bigint_field", -5, 0, 20, 0, 0, true, Long.class.getName()), (Object)new ExpectedColumnResult("varchar_field", 12, 0, 50, 50, 0, false, String.class.getName()), (Object)new ExpectedColumnResult("ts_field", 93, 0, 19, 0, 0, false, Timestamp.class.getName()), (Object)new ExpectedColumnResult("date_field", 91, 0, 10, 0, 0, false, Date.class.getName()), (Object)new ExpectedColumnResult("decimal_field", 3, 0, 20, 18, 5, true, BigDecimal.class.getName()));
            ResultSetMetaData prepareMetadata = stmt.getMetaData();
            PreparedStatementTest.verifyMetadata(prepareMetadata, (List<ExpectedColumnResult>)exp);
            try (ResultSet rs = stmt.executeQuery();){
                ResultSetMetaData executeMetadata = rs.getMetaData();
                PreparedStatementTest.verifyMetadata(executeMetadata, (List<ExpectedColumnResult>)exp);
                Assert.assertTrue((String)"No expected first row", (boolean)rs.next());
                Assert.assertThat((Object)rs.getInt(1), (Matcher)CoreMatchers.equalTo((Object)1));
                Assert.assertThat((Object)rs.getLong(2), (Matcher)CoreMatchers.equalTo((Object)12384729L));
                Assert.assertThat((Object)rs.getString(3), (Matcher)CoreMatchers.equalTo((Object)"varchar_value"));
                Assert.assertThat((Object)rs.getTimestamp(4), (Matcher)CoreMatchers.equalTo((Object)Timestamp.valueOf("2008-2-23 10:00:20.123")));
                Assert.assertThat((Object)rs.getDate(5), (Matcher)CoreMatchers.equalTo((Object)Date.valueOf("2008-2-23")));
                Assert.assertThat((Object)rs.getBigDecimal(6), (Matcher)CoreMatchers.equalTo((Object)new BigDecimal("99999912399.45670")));
                Assert.assertFalse((String)"Unexpected second row", (boolean)rs.next());
            }
        }
    }

    private static void verifyMetadata(ResultSetMetaData act, List<ExpectedColumnResult> exp) throws SQLException {
        Assert.assertEquals((long)exp.size(), (long)act.getColumnCount());
        int i = 0;
        for (ExpectedColumnResult e : exp) {
            Assert.assertTrue((String)("Failed to find the expected column metadata. Expected " + e + ". Was: " + PreparedStatementTest.toString(act, ++i)), (boolean)e.isEqualsTo(act, i));
        }
    }

    private static String toString(ResultSetMetaData metadata, int colNum) throws SQLException {
        return "ResultSetMetaData(" + colNum + ")[columnName='" + metadata.getColumnName(colNum) + '\'' + ", type='" + metadata.getColumnType(colNum) + '\'' + ", nullable=" + metadata.isNullable(colNum) + ", displaySize=" + metadata.getColumnDisplaySize(colNum) + ", precision=" + metadata.getPrecision(colNum) + ", scale=" + metadata.getScale(colNum) + ", signed=" + metadata.isSigned(colNum) + ", className='" + metadata.getColumnClassName(colNum) + '\'' + ']';
    }

    @Test
    public void testDefaultGetQueryTimeout() throws SQLException {
        try (PreparedStatement stmt = connection.prepareStatement(SYS_VERSION_SQL);){
            int timeoutValue = stmt.getQueryTimeout();
            Assert.assertEquals((long)0L, (long)timeoutValue);
        }
    }

    @Test
    public void testInvalidSetQueryTimeout() throws SQLException {
        try (PreparedStatement stmt = connection.prepareStatement(SYS_VERSION_SQL);){
            int valueToSet = -10;
            try {
                stmt.setQueryTimeout(valueToSet);
            }
            catch (SQLException e) {
                Assert.assertThat((Object)e.getMessage(), (Matcher)CoreMatchers.containsString((String)"illegal timeout value"));
            }
        }
    }

    @Test
    public void testValidSetQueryTimeout() throws SQLException {
        try (PreparedStatement stmt = connection.prepareStatement(SYS_VERSION_SQL);){
            int valueToSet = RANDOMIZER.nextInt(59) + 1;
            logger.info("Setting timeout as {} seconds", (Object)valueToSet);
            stmt.setQueryTimeout(valueToSet);
            Assert.assertEquals((long)valueToSet, (long)stmt.getQueryTimeout());
        }
    }

    @Test
    public void testSetQueryTimeoutAsZero() throws SQLException {
        try (PreparedStatement stmt = connection.prepareStatement(SYS_RANDOM_SQL);){
            stmt.setQueryTimeout(0);
            stmt.executeQuery();
            ResultSet rs = stmt.getResultSet();
            int rowCount = 0;
            while (rs.next()) {
                rs.getBytes(1);
                ++rowCount;
            }
            Assert.assertEquals((long)3L, (long)rowCount);
        }
    }

    @Test
    public void testClientTriggeredQueryTimeout() throws Exception {
        int timeoutDuration = 3;
        int rowsCounted = 0;
        try (PreparedStatement stmt = connection.prepareStatement(SYS_RANDOM_SQL);){
            stmt.setQueryTimeout(timeoutDuration);
            logger.info("Set a timeout of {} seconds", (Object)stmt.getQueryTimeout());
            ResultSet rs = stmt.executeQuery();
            try {
                while (rs.next()) {
                    rs.getString(1);
                    ++rowsCounted;
                    try {
                        Thread.sleep(TimeUnit.SECONDS.toMillis(timeoutDuration + 1));
                    }
                    catch (InterruptedException interruptedException) {
                        // empty catch block
                    }
                    logger.info("Paused for {} seconds", (Object)(timeoutDuration + 1));
                }
            }
            catch (SQLTimeoutException sqlEx) {
                logger.info("Counted " + rowsCounted + " rows before hitting timeout");
                if (stmt != null) {
                    stmt.close();
                }
                return;
            }
        }
        throw new Exception("Failed to trigger timeout of " + timeoutDuration + " sec");
    }

    @Ignore(value="Pause Injection appears broken for PreparedStatement")
    @Test(expected=SqlTimeoutException.class)
    public void testServerTriggeredQueryTimeout() throws Exception {
        int timeoutDuration = 2;
        long serverPause = timeoutDuration + 2;
        int cleanupPause = 3;
        String controls = Controls.newBuilder().addTimedPause(ScreenCreator.class, "sending-data", 0, TimeUnit.SECONDS.toMillis(serverPause)).build();
        try (Connection exclusiveConnection = new Driver().connect("jdbc:drill:zk=local", null);){
            try (Statement stmt = exclusiveConnection.createStatement();){
                Assert.assertThat((Object)stmt.execute(String.format("ALTER session SET `%s` = '%s'", "drill.exec.testing.controls", controls)), (Matcher)CoreMatchers.equalTo((Object)true));
            }
            try (PreparedStatement pStmt = exclusiveConnection.prepareStatement(SYS_RANDOM_SQL);){
                pStmt.setQueryTimeout(timeoutDuration);
                logger.info("Set a timeout of {} seconds", (Object)pStmt.getQueryTimeout());
                ResultSet rs = pStmt.executeQuery();
                while (rs.next()) {
                    rs.getBytes(1);
                }
            }
            catch (SQLTimeoutException sqlEx) {
                logger.info("SQLTimeoutException thrown: {}", (Object)sqlEx.getMessage());
                throw (SqlTimeoutException)sqlEx;
            }
            finally {
                try {
                    Thread.sleep(TimeUnit.SECONDS.toMillis(cleanupPause));
                }
                catch (InterruptedException interruptedException) {}
            }
        }
    }

    @Test
    public void testNonTriggeredQueryTimeout() throws SQLException {
        try (PreparedStatement stmt = connection.prepareStatement(SYS_VERSION_SQL);){
            stmt.setQueryTimeout(60);
            stmt.executeQuery();
            ResultSet rs = stmt.getResultSet();
            int rowCount = 0;
            while (rs.next()) {
                rs.getBytes(1);
                ++rowCount;
            }
            Assert.assertEquals((long)1L, (long)rowCount);
        }
    }

    @Test(expected=SQLException.class)
    public void testSqlQueryWithParamNotSupported() throws SQLException {
        try {
            connection.prepareStatement("VALUES ?, ?");
        }
        catch (SQLException e) {
            Assert.assertThat((String)"Check whether params.-unsupported wording changed or checks changed.", (Object)e.toString(), (Matcher)CoreMatchers.containsString((String)"Illegal use of dynamic parameter"));
            throw e;
        }
    }

    @Test(expected=SQLFeatureNotSupportedException.class)
    public void testParamSettingWhenNoParametersIndexSaysUnsupported() throws SQLException {
        try (PreparedStatement prepStmt = connection.prepareStatement("VALUES 1");){
            try {
                prepStmt.setBytes(4, null);
            }
            catch (SQLFeatureNotSupportedException e) {
                Assert.assertThat((String)"Check whether params.-unsupported wording changed or checks changed.", (Object)e.toString(), PARAMETERS_NOT_SUPPORTED_MSG_MATCHER);
                throw e;
            }
        }
    }

    @Test(expected=SQLFeatureNotSupportedException.class)
    public void testParamSettingWhenUnsupportedTypeSaysUnsupported() throws SQLException {
        try (PreparedStatement prepStmt = connection.prepareStatement("VALUES 1");){
            try {
                prepStmt.setClob(2, (Clob)null);
            }
            catch (SQLFeatureNotSupportedException e) {
                Assert.assertThat((String)"Check whether params.-unsupported wording changed or checks changed.", (Object)e.toString(), PARAMETERS_NOT_SUPPORTED_MSG_MATCHER);
                throw e;
            }
        }
    }

    private static class ExpectedColumnResult {
        final String columnName;
        final int type;
        final int nullable;
        final int displaySize;
        final int precision;
        final int scale;
        final boolean signed;
        final String className;

        ExpectedColumnResult(String columnName, int type, int nullable, int displaySize, int precision, int scale, boolean signed, String className) {
            this.columnName = columnName;
            this.type = type;
            this.nullable = nullable;
            this.displaySize = displaySize;
            this.precision = precision;
            this.scale = scale;
            this.signed = signed;
            this.className = className;
        }

        boolean isEqualsTo(ResultSetMetaData metadata, int colNum) throws SQLException {
            return metadata.getCatalogName(colNum).equals("DRILL") && metadata.getSchemaName(colNum).isEmpty() && metadata.getTableName(colNum).isEmpty() && metadata.getColumnName(colNum).equals(this.columnName) && metadata.getColumnLabel(colNum).equals(this.columnName) && metadata.getColumnType(colNum) == this.type && metadata.isNullable(colNum) == this.nullable && metadata.getScale(colNum) == this.scale && metadata.isSigned(colNum) == this.signed && metadata.getColumnDisplaySize(colNum) == this.displaySize && metadata.getColumnClassName(colNum).equals(this.className) && metadata.isSearchable(colNum) && !metadata.isAutoIncrement(colNum) && !metadata.isCaseSensitive(colNum) && metadata.isReadOnly(colNum) && !metadata.isWritable(colNum) && !metadata.isDefinitelyWritable(colNum) && !metadata.isCurrency(colNum);
        }

        public String toString() {
            return "ExpectedColumnResult[columnName='" + this.columnName + '\'' + ", type='" + this.type + '\'' + ", nullable=" + this.nullable + ", displaySize=" + this.displaySize + ", precision=" + this.precision + ", scale=" + this.scale + ", signed=" + this.signed + ", className='" + this.className + '\'' + ']';
        }
    }
}

