/*
 * Decompiled with CFR 0.152.
 */
package org.apache.drill.exec.store.msaccess;

import com.healthmarketscience.jackcess.Column;
import com.healthmarketscience.jackcess.DataType;
import com.healthmarketscience.jackcess.Database;
import com.healthmarketscience.jackcess.DatabaseBuilder;
import com.healthmarketscience.jackcess.Row;
import com.healthmarketscience.jackcess.Table;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.math.BigDecimal;
import java.nio.file.Files;
import java.nio.file.StandardCopyOption;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import org.apache.commons.lang3.StringUtils;
import org.apache.drill.common.AutoCloseables;
import org.apache.drill.common.exceptions.CustomErrorContext;
import org.apache.drill.common.exceptions.UserException;
import org.apache.drill.common.types.TypeProtos;
import org.apache.drill.exec.physical.impl.scan.v3.FixedReceiver;
import org.apache.drill.exec.physical.impl.scan.v3.ManagedReader;
import org.apache.drill.exec.physical.impl.scan.v3.file.FileDescrip;
import org.apache.drill.exec.physical.impl.scan.v3.file.FileSchemaNegotiator;
import org.apache.drill.exec.physical.resultSet.ResultSetLoader;
import org.apache.drill.exec.physical.resultSet.RowSetLoader;
import org.apache.drill.exec.record.metadata.SchemaBuilder;
import org.apache.drill.exec.record.metadata.TupleMetadata;
import org.apache.drill.exec.store.msaccess.MSAccessFormatConfig;
import org.apache.drill.exec.vector.accessor.ArrayWriter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MSAccessBatchReader
implements ManagedReader {
    private static final Logger logger = LoggerFactory.getLogger(MSAccessBatchReader.class);
    private final FileDescrip file;
    private final CustomErrorContext errorContext;
    private final RowSetLoader rowWriter;
    private final File tempDir;
    private final List<MSAccessColumn> columnList;
    private final MSAccessFormatConfig config;
    private final boolean metadataOnly;
    private File tempFile;
    private Set<String> tableList;
    private Iterator<Row> rowIterator;
    private Iterator<String> tableIterator;
    private InputStream fsStream;
    private Table table;
    private Database db;

    public MSAccessBatchReader(FileSchemaNegotiator negotiator, File tempDir, MSAccessFormatConfig config) {
        this.tempDir = tempDir;
        this.columnList = new ArrayList<MSAccessColumn>();
        this.config = config;
        this.file = negotiator.file();
        this.errorContext = negotiator.parentErrorContext();
        this.metadataOnly = StringUtils.isEmpty((CharSequence)config.getTableName());
        this.openFile();
        this.buildSchema(negotiator);
        if (this.metadataOnly) {
            this.tableIterator = this.tableList.iterator();
        } else {
            this.rowIterator = this.table.iterator();
        }
        ResultSetLoader loader = negotiator.build();
        this.rowWriter = loader.writer();
    }

    private void buildSchema(FileSchemaNegotiator negotiator) {
        SchemaBuilder schemaBuilder = new SchemaBuilder();
        if (this.metadataOnly) {
            TupleMetadata metadataSchema = this.buildMetadataSchema(schemaBuilder);
            negotiator.tableSchema(metadataSchema, true);
        } else {
            TupleMetadata derivedSchema = this.buildSchemaFromTable(schemaBuilder, this.config.getTableName());
            if (negotiator.providedSchema() != null) {
                TupleMetadata mergeSchemas = FixedReceiver.Builder.mergeSchemas((TupleMetadata)negotiator.providedSchema(), (TupleMetadata)derivedSchema);
                negotiator.tableSchema(mergeSchemas, true);
            } else {
                negotiator.tableSchema(derivedSchema, true);
            }
        }
    }

    private TupleMetadata buildMetadataSchema(SchemaBuilder builder) {
        builder.add("table", TypeProtos.MinorType.VARCHAR);
        builder.add("created_date", TypeProtos.MinorType.TIMESTAMP);
        builder.add("updated_date", TypeProtos.MinorType.TIMESTAMP);
        builder.add("row_count", TypeProtos.MinorType.INT);
        builder.add("col_count", TypeProtos.MinorType.INT);
        builder.addArray("columns", TypeProtos.MinorType.VARCHAR);
        return builder.buildSchema();
    }

    private TupleMetadata buildSchemaFromTable(SchemaBuilder builder, String tableName) {
        try {
            this.table = this.db.getTable(tableName);
        }
        catch (IOException e) {
            this.deleteTempFile();
            throw UserException.dataReadError((Throwable)e).message("Table " + this.config.getTableName() + " not found. " + e.getMessage(), new Object[0]).addContext(this.errorContext).build(logger);
        }
        List columns = this.table.getColumns();
        for (Column column : columns) {
            TypeProtos.MinorType drillDataType;
            String columnName = column.getName();
            DataType dataType = column.getType();
            switch (dataType) {
                case BOOLEAN: {
                    builder.addNullable(columnName, TypeProtos.MinorType.BIT);
                    drillDataType = TypeProtos.MinorType.BIT;
                    break;
                }
                case BYTE: {
                    builder.addNullable(columnName, TypeProtos.MinorType.TINYINT);
                    drillDataType = TypeProtos.MinorType.TINYINT;
                    break;
                }
                case INT: {
                    builder.addNullable(columnName, TypeProtos.MinorType.SMALLINT);
                    drillDataType = TypeProtos.MinorType.SMALLINT;
                    break;
                }
                case LONG: {
                    builder.addNullable(columnName, TypeProtos.MinorType.INT);
                    drillDataType = TypeProtos.MinorType.INT;
                    break;
                }
                case BIG_INT: 
                case COMPLEX_TYPE: {
                    builder.addNullable(columnName, TypeProtos.MinorType.BIGINT);
                    drillDataType = TypeProtos.MinorType.BIGINT;
                    break;
                }
                case FLOAT: {
                    builder.addNullable(columnName, TypeProtos.MinorType.FLOAT4);
                    drillDataType = TypeProtos.MinorType.FLOAT4;
                    break;
                }
                case DOUBLE: {
                    builder.addNullable(columnName, TypeProtos.MinorType.FLOAT8);
                    drillDataType = TypeProtos.MinorType.FLOAT8;
                    break;
                }
                case MEMO: 
                case TEXT: 
                case GUID: {
                    builder.addNullable(columnName, TypeProtos.MinorType.VARCHAR);
                    drillDataType = TypeProtos.MinorType.VARCHAR;
                    break;
                }
                case MONEY: 
                case NUMERIC: {
                    builder.addNullable(columnName, TypeProtos.MinorType.VARDECIMAL);
                    drillDataType = TypeProtos.MinorType.VARDECIMAL;
                    break;
                }
                case OLE: 
                case BINARY: 
                case UNSUPPORTED_VARLEN: 
                case UNSUPPORTED_FIXEDLEN: 
                case UNKNOWN_0D: 
                case UNKNOWN_11: {
                    builder.addNullable(columnName, TypeProtos.MinorType.VARBINARY);
                    drillDataType = TypeProtos.MinorType.VARBINARY;
                    break;
                }
                case EXT_DATE_TIME: 
                case SHORT_DATE_TIME: {
                    builder.addNullable(columnName, TypeProtos.MinorType.TIMESTAMP);
                    drillDataType = TypeProtos.MinorType.TIMESTAMP;
                    break;
                }
                default: {
                    this.deleteTempFile();
                    throw UserException.dataReadError().message(dataType.name() + " is not supported.", new Object[0]).build(logger);
                }
            }
            this.columnList.add(new MSAccessColumn(columnName, drillDataType));
        }
        return builder.buildSchema();
    }

    public boolean next() {
        while (!this.rowWriter.isFull()) {
            if (this.metadataOnly) {
                if (this.tableIterator.hasNext()) {
                    try {
                        this.processMetadataRow(this.tableIterator.next());
                        continue;
                    }
                    catch (IOException e) {
                        this.deleteTempFile();
                        throw UserException.dataReadError((Throwable)e).message("Error retrieving metadata for table: " + e.getMessage(), new Object[0]).addContext(this.errorContext).build(logger);
                    }
                }
                return false;
            }
            if (this.rowIterator.hasNext()) {
                this.processRow(this.rowIterator.next());
                continue;
            }
            return false;
        }
        return true;
    }

    private void openFile() {
        try {
            this.fsStream = this.file.fileSystem().openPossiblyCompressedStream(this.file.split().getPath());
            this.db = DatabaseBuilder.open((File)this.convertInputStreamToFile(this.fsStream));
            this.tableList = this.db.getTableNames();
        }
        catch (IOException e) {
            this.deleteTempFile();
            throw UserException.dataReadError((Throwable)e).message("Error reading MS Access file: " + e.getMessage(), new Object[0]).addContext(this.errorContext).build(logger);
        }
    }

    private void processMetadataRow(String tableName) throws IOException {
        Table table;
        try {
            table = this.db.getTable(tableName);
        }
        catch (IOException e) {
            this.deleteTempFile();
            throw UserException.dataReadError((Throwable)e).message("Error retrieving metadata for table " + tableName + ": " + e.getMessage(), new Object[0]).addContext(this.errorContext).build(logger);
        }
        this.rowWriter.start();
        this.rowWriter.scalar("table").setString(tableName);
        LocalDateTime createdDate = table.getCreatedDate();
        this.rowWriter.scalar("created_date").setTimestamp(createdDate.toInstant(ZoneOffset.UTC));
        LocalDateTime updatedDate = table.getCreatedDate();
        this.rowWriter.scalar("updated_date").setTimestamp(updatedDate.toInstant(ZoneOffset.UTC));
        this.rowWriter.scalar("row_count").setInt(table.getRowCount());
        this.rowWriter.scalar("col_count").setInt(table.getColumnCount());
        ArrayWriter arrayWriter = this.rowWriter.array("columns");
        for (Column column : table.getColumns()) {
            arrayWriter.scalar().setString(column.getName());
        }
        arrayWriter.save();
        this.rowWriter.save();
    }

    private void processRow(Row next) {
        this.rowWriter.start();
        for (MSAccessColumn col : this.columnList) {
            switch (col.dataType) {
                case BIT: {
                    Boolean boolValue = next.getBoolean(col.columnName);
                    this.rowWriter.scalar(col.columnName).setBoolean(boolValue.booleanValue());
                    break;
                }
                case SMALLINT: {
                    Short shortValue = next.getShort(col.columnName);
                    this.rowWriter.scalar(col.columnName).setInt((int)shortValue.shortValue());
                    break;
                }
                case TINYINT: {
                    Byte byteValue = next.getByte(col.columnName);
                    this.rowWriter.scalar(col.columnName).setInt((int)byteValue.byteValue());
                    break;
                }
                case BIGINT: 
                case INT: {
                    Integer intValue = next.getInt(col.columnName);
                    if (intValue == null) break;
                    this.rowWriter.scalar(col.columnName).setInt(intValue.intValue());
                    break;
                }
                case FLOAT4: {
                    Float floatValue = next.getFloat(col.columnName);
                    if (floatValue == null) break;
                    this.rowWriter.scalar(col.columnName).setFloat(floatValue.floatValue());
                    break;
                }
                case FLOAT8: {
                    Double doubleValue = next.getDouble(col.columnName);
                    this.rowWriter.scalar(col.columnName).setDouble(doubleValue.doubleValue());
                    break;
                }
                case VARDECIMAL: {
                    BigDecimal bigDecimal = next.getBigDecimal(col.columnName);
                    this.rowWriter.scalar(col.columnName).setDecimal(bigDecimal);
                    break;
                }
                case VARCHAR: {
                    String stringValue = next.getString(col.columnName);
                    if (!StringUtils.isNotEmpty((CharSequence)stringValue)) break;
                    this.rowWriter.scalar(col.columnName).setString(stringValue);
                    break;
                }
                case TIMESTAMP: {
                    LocalDateTime tsValue = next.getLocalDateTime(col.columnName);
                    if (tsValue == null) break;
                    this.rowWriter.scalar(col.columnName).setTimestamp(tsValue.toInstant(ZoneOffset.UTC));
                    break;
                }
                case VARBINARY: {
                    byte[] byteValueArray = next.getBytes(col.columnName);
                    this.rowWriter.scalar(col.columnName).setBytes(byteValueArray, byteValueArray.length);
                }
            }
        }
        this.rowWriter.save();
    }

    public void close() {
        AutoCloseables.closeSilently((AutoCloseable[])new AutoCloseable[]{this.db});
        AutoCloseables.closeSilently((AutoCloseable[])new AutoCloseable[]{this.fsStream});
        this.deleteTempFile();
    }

    private File convertInputStreamToFile(InputStream stream) {
        String tempFileName = this.tempDir.getPath() + "/~" + this.file.filePath().getName();
        this.tempFile = new File(tempFileName);
        try {
            Files.copy(stream, this.tempFile.toPath(), StandardCopyOption.REPLACE_EXISTING);
        }
        catch (Exception e) {
            if (this.tempFile.exists() && !this.tempFile.delete()) {
                logger.warn("{} not deleted.", (Object)this.tempFile.getName());
            }
            throw UserException.dataWriteError((Throwable)e).message("Failed to create temp HDF5 file: %s", new Object[]{this.file.filePath()}).addContext(e.getMessage()).build(logger);
        }
        AutoCloseables.closeSilently((AutoCloseable[])new AutoCloseable[]{stream});
        return this.tempFile;
    }

    private void deleteTempFile() {
        if (this.tempFile != null) {
            if (!this.tempFile.delete()) {
                logger.warn("{} file not deleted.", (Object)this.tempFile.getName());
            }
            this.tempFile = null;
        }
    }

    private static class MSAccessColumn {
        private final String columnName;
        private final TypeProtos.MinorType dataType;

        public MSAccessColumn(String columnName, TypeProtos.MinorType dataType) {
            this.columnName = columnName;
            this.dataType = dataType;
        }
    }
}

