/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iceberg.actions;

import java.io.Closeable;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.iceberg.RewriteFiles;
import org.apache.iceberg.Table;
import org.apache.iceberg.actions.RewriteFileGroup;
import org.apache.iceberg.exceptions.CommitStateUnknownException;
import org.apache.iceberg.relocated.com.google.common.base.Preconditions;
import org.apache.iceberg.relocated.com.google.common.collect.Lists;
import org.apache.iceberg.relocated.com.google.common.collect.Queues;
import org.apache.iceberg.relocated.com.google.common.collect.Sets;
import org.apache.iceberg.relocated.com.google.common.util.concurrent.ThreadFactoryBuilder;
import org.apache.iceberg.util.Tasks;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RewriteDataFilesCommitManager {
    private static final Logger LOG = LoggerFactory.getLogger(RewriteDataFilesCommitManager.class);
    private final Table table;
    private final long startingSnapshotId;

    public RewriteDataFilesCommitManager(Table table) {
        this(table, table.currentSnapshot().snapshotId());
    }

    public RewriteDataFilesCommitManager(Table table, long startingSnapshotId) {
        this.table = table;
        this.startingSnapshotId = startingSnapshotId;
    }

    public void commitFileGroups(Set<RewriteFileGroup> fileGroups) {
        HashSet rewrittenDataFiles = Sets.newHashSet();
        HashSet addedDataFiles = Sets.newHashSet();
        for (RewriteFileGroup group : fileGroups) {
            rewrittenDataFiles = Sets.union((Set)rewrittenDataFiles, group.rewrittenFiles());
            addedDataFiles = Sets.union((Set)addedDataFiles, group.addedFiles());
        }
        RewriteFiles rewrite = this.table.newRewrite().validateFromSnapshot(this.startingSnapshotId).rewriteFiles((Set)rewrittenDataFiles, (Set)addedDataFiles);
        rewrite.commit();
    }

    public void abortFileGroup(RewriteFileGroup fileGroup) {
        Preconditions.checkState((fileGroup.addedFiles() != null ? 1 : 0) != 0, (Object)"Cannot abort a fileGroup that was not rewritten");
        Tasks.foreach(fileGroup.addedFiles()).noRetry().suppressFailureWhenFinished().onFailure((dataFile, exc) -> LOG.warn("Failed to delete: {}", (Object)dataFile.path(), (Object)exc)).run(dataFile -> this.table.io().deleteFile(dataFile.path().toString()));
    }

    public void commitOrClean(Set<RewriteFileGroup> rewriteGroups) {
        try {
            this.commitFileGroups(rewriteGroups);
        }
        catch (CommitStateUnknownException e) {
            LOG.error("Commit state unknown for {}, cannot clean up files because they may have been committed successfully.", rewriteGroups, (Object)e);
            throw e;
        }
        catch (Exception e) {
            LOG.error("Cannot commit groups {}, attempting to clean up written files", rewriteGroups, (Object)e);
            rewriteGroups.forEach(this::abortFileGroup);
            throw e;
        }
    }

    public CommitService service(int rewritesPerCommit) {
        return new CommitService(rewritesPerCommit);
    }

    public class CommitService
    implements Closeable {
        private final ExecutorService committerService;
        private final ConcurrentLinkedQueue<RewriteFileGroup> completedRewrites;
        private final List<RewriteFileGroup> committedRewrites;
        private final int rewritesPerCommit;
        private final AtomicBoolean running = new AtomicBoolean(false);

        CommitService(int rewritesPerCommit) {
            LOG.info("Creating commit service for table {} with {} groups per commit", (Object)RewriteDataFilesCommitManager.this.table, (Object)rewritesPerCommit);
            this.rewritesPerCommit = rewritesPerCommit;
            this.committerService = Executors.newSingleThreadExecutor(new ThreadFactoryBuilder().setNameFormat("Committer-Service").build());
            this.completedRewrites = Queues.newConcurrentLinkedQueue();
            this.committedRewrites = Lists.newArrayList();
        }

        public void start() {
            Preconditions.checkState((boolean)this.running.compareAndSet(false, true), (Object)"Rewrite Commit service already started");
            LOG.info("Starting commit service for {}", (Object)RewriteDataFilesCommitManager.this.table);
            this.committerService.execute(() -> {
                while (this.running.get() || this.completedRewrites.size() > 0) {
                    try {
                        if (this.completedRewrites.size() == 0) {
                            Thread.sleep(100L);
                        }
                    }
                    catch (InterruptedException e) {
                        Thread.currentThread().interrupt();
                        throw new RuntimeException("Interrupted while processing commits", e);
                    }
                    if (this.completedRewrites.size() < this.rewritesPerCommit && (this.running.get() || this.completedRewrites.size() <= 0)) continue;
                    HashSet batch = Sets.newHashSetWithExpectedSize((int)this.rewritesPerCommit);
                    for (int i = 0; i < this.rewritesPerCommit && !this.completedRewrites.isEmpty(); ++i) {
                        batch.add(this.completedRewrites.poll());
                    }
                    try {
                        RewriteDataFilesCommitManager.this.commitOrClean(batch);
                        this.committedRewrites.addAll(batch);
                    }
                    catch (Exception e) {
                        LOG.error("Failure during rewrite commit process, partial progress enabled. Ignoring", (Throwable)e);
                    }
                }
            });
        }

        public void offer(RewriteFileGroup group) {
            LOG.debug("Offered to commit service: {}", (Object)group);
            Preconditions.checkState((boolean)this.running.get(), (Object)"Cannot add rewrites to a service which has already been closed");
            this.completedRewrites.add(group);
        }

        public List<RewriteFileGroup> results() {
            Preconditions.checkState((boolean)this.committerService.isShutdown(), (Object)"Cannot get results from a service which has not been closed");
            return this.committedRewrites;
        }

        @Override
        public void close() {
            Preconditions.checkState((boolean)this.running.compareAndSet(true, false), (Object)"Cannot close already closed RewriteService");
            LOG.info("Closing commit service for {}", (Object)RewriteDataFilesCommitManager.this.table);
            this.committerService.shutdown();
            try {
                if (!this.committerService.awaitTermination(10L, TimeUnit.MINUTES)) {
                    LOG.warn("Commit operation did not complete within 10 minutes of the files being written. This may mean that changes were not successfully committed to the the Iceberg table.");
                }
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new RuntimeException("Cannot complete commit for rewrite, commit service interrupted", e);
            }
            Preconditions.checkState((boolean)this.completedRewrites.isEmpty(), (Object)"File groups offered after service was closed, they were not successfully committed.");
        }
    }
}

