/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.snapshot;

import java.io.IOException;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.HBaseTestingUtility;
import org.apache.hadoop.hbase.HRegionInfo;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.TableNotFoundException;
import org.apache.hadoop.hbase.client.Admin;
import org.apache.hadoop.hbase.client.HBaseAdmin;
import org.apache.hadoop.hbase.client.Table;
import org.apache.hadoop.hbase.master.HMaster;
import org.apache.hadoop.hbase.protobuf.generated.HBaseProtos;
import org.apache.hadoop.hbase.regionserver.ConstantSizeRegionSplitPolicy;
import org.apache.hadoop.hbase.snapshot.SnapshotCreationException;
import org.apache.hadoop.hbase.snapshot.SnapshotTestingUtils;
import org.apache.hadoop.hbase.testclassification.LargeTests;
import org.apache.hadoop.hbase.util.Bytes;
import org.junit.After;
import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;
import org.junit.experimental.categories.Category;

@Category(value={LargeTests.class})
public class TestFlushSnapshotFromClient {
    private static final Log LOG = LogFactory.getLog(TestFlushSnapshotFromClient.class);
    protected static final HBaseTestingUtility UTIL = new HBaseTestingUtility();
    private static final int NUM_RS = 2;
    private static final byte[] TEST_FAM = Bytes.toBytes("fam");
    private static final TableName TABLE_NAME = TableName.valueOf("test");
    private final int DEFAULT_NUM_ROWS = 100;

    @BeforeClass
    public static void setupCluster() throws Exception {
        TestFlushSnapshotFromClient.setupConf(UTIL.getConfiguration());
        UTIL.startMiniCluster(2);
    }

    protected static void setupConf(Configuration conf) {
        conf.setInt("hbase.regionsever.info.port", -1);
        conf.setInt("hbase.hregion.memstore.flush.size", 25000);
        conf.setInt("hbase.hstore.compaction.min", 10);
        conf.setInt("hbase.hstore.compactionThreshold", 10);
        conf.setInt("hbase.hstore.blockingStoreFiles", 12);
        conf.setBoolean("hbase.snapshot.enabled", true);
        conf.set("hbase.regionserver.region.split.policy", ConstantSizeRegionSplitPolicy.class.getName());
    }

    @Before
    public void setup() throws Exception {
        SnapshotTestingUtils.createTable(UTIL, TABLE_NAME, new byte[][]{TEST_FAM});
    }

    @After
    public void tearDown() throws Exception {
        UTIL.deleteTable(TABLE_NAME);
        SnapshotTestingUtils.deleteAllSnapshots(UTIL.getHBaseAdmin());
        SnapshotTestingUtils.deleteArchiveDirectory(UTIL);
    }

    @AfterClass
    public static void cleanupTest() throws Exception {
        try {
            UTIL.shutdownMiniCluster();
        }
        catch (Exception e) {
            LOG.warn((Object)"failure shutting down cluster", (Throwable)e);
        }
    }

    @Test(timeout=300000L)
    public void testFlushTableSnapshot() throws Exception {
        HBaseAdmin admin = UTIL.getHBaseAdmin();
        SnapshotTestingUtils.assertNoSnapshots(admin);
        SnapshotTestingUtils.loadData(UTIL, TABLE_NAME, 100, (byte[][])new byte[][]{TEST_FAM});
        LOG.debug((Object)"FS state before snapshot:");
        UTIL.getHBaseCluster().getMaster().getMasterFileSystem().logFileSystemState(LOG);
        String snapshotString = "offlineTableSnapshot";
        byte[] snapshot = Bytes.toBytes(snapshotString);
        admin.snapshot(snapshotString, TABLE_NAME, HBaseProtos.SnapshotDescription.Type.FLUSH);
        LOG.debug((Object)"Snapshot completed.");
        List<HBaseProtos.SnapshotDescription> snapshots = SnapshotTestingUtils.assertOneSnapshotThatMatches((Admin)admin, snapshot, TABLE_NAME);
        LOG.debug((Object)"FS state after snapshot:");
        UTIL.getHBaseCluster().getMaster().getMasterFileSystem().logFileSystemState(LOG);
        SnapshotTestingUtils.confirmSnapshotValid(UTIL, snapshots.get(0), TABLE_NAME, TEST_FAM);
    }

    @Test(timeout=30000L)
    public void testSkipFlushTableSnapshot() throws Exception {
        HBaseAdmin admin = UTIL.getHBaseAdmin();
        SnapshotTestingUtils.assertNoSnapshots(admin);
        try (Table table = UTIL.getConnection().getTable(TABLE_NAME);){
            UTIL.loadTable(table, TEST_FAM);
        }
        LOG.debug((Object)"FS state before snapshot:");
        UTIL.getHBaseCluster().getMaster().getMasterFileSystem().logFileSystemState(LOG);
        String snapshotString = "skipFlushTableSnapshot";
        byte[] snapshot = Bytes.toBytes(snapshotString);
        admin.snapshot(snapshotString, TABLE_NAME, HBaseProtos.SnapshotDescription.Type.SKIPFLUSH);
        LOG.debug((Object)"Snapshot completed.");
        List<HBaseProtos.SnapshotDescription> snapshots = SnapshotTestingUtils.assertOneSnapshotThatMatches((Admin)admin, snapshot, TABLE_NAME);
        LOG.debug((Object)"FS state after snapshot:");
        UTIL.getHBaseCluster().getMaster().getMasterFileSystem().logFileSystemState(LOG);
        SnapshotTestingUtils.confirmSnapshotValid(UTIL, snapshots.get(0), TABLE_NAME, TEST_FAM);
        admin.deleteSnapshot(snapshot);
        snapshots = admin.listSnapshots();
        SnapshotTestingUtils.assertNoSnapshots(admin);
    }

    @Test(timeout=300000L)
    public void testFlushTableSnapshotWithProcedure() throws Exception {
        HBaseAdmin admin = UTIL.getHBaseAdmin();
        SnapshotTestingUtils.assertNoSnapshots(admin);
        SnapshotTestingUtils.loadData(UTIL, TABLE_NAME, 100, (byte[][])new byte[][]{TEST_FAM});
        LOG.debug((Object)"FS state before snapshot:");
        UTIL.getHBaseCluster().getMaster().getMasterFileSystem().logFileSystemState(LOG);
        String snapshotString = "offlineTableSnapshot";
        byte[] snapshot = Bytes.toBytes(snapshotString);
        HashMap<String, String> props = new HashMap<String, String>();
        props.put("table", TABLE_NAME.getNameAsString());
        admin.execProcedure("online-snapshot", snapshotString, props);
        LOG.debug((Object)"Snapshot completed.");
        List<HBaseProtos.SnapshotDescription> snapshots = SnapshotTestingUtils.assertOneSnapshotThatMatches((Admin)admin, snapshot, TABLE_NAME);
        LOG.debug((Object)"FS state after snapshot:");
        UTIL.getHBaseCluster().getMaster().getMasterFileSystem().logFileSystemState(LOG);
        SnapshotTestingUtils.confirmSnapshotValid(UTIL, snapshots.get(0), TABLE_NAME, TEST_FAM);
    }

    @Test(timeout=300000L)
    public void testSnapshotFailsOnNonExistantTable() throws Exception {
        HBaseAdmin admin = UTIL.getHBaseAdmin();
        SnapshotTestingUtils.assertNoSnapshots(admin);
        TableName tableName = TableName.valueOf("_not_a_table");
        boolean fail = false;
        do {
            try {
                admin.getTableDescriptor(tableName);
                fail = true;
                LOG.error((Object)("Table:" + tableName + " already exists, checking a new name"));
                tableName = TableName.valueOf(tableName + "!");
            }
            catch (TableNotFoundException e) {
                fail = false;
            }
        } while (fail);
        try {
            admin.snapshot("fail", tableName, HBaseProtos.SnapshotDescription.Type.FLUSH);
            Assert.fail((String)"Snapshot succeeded even though there is not table.");
        }
        catch (SnapshotCreationException e) {
            LOG.info((Object)("Correctly failed to snapshot a non-existant table:" + e.getMessage()));
        }
    }

    @Test(timeout=300000L)
    public void testAsyncFlushSnapshot() throws Exception {
        HBaseAdmin admin = UTIL.getHBaseAdmin();
        HBaseProtos.SnapshotDescription snapshot = HBaseProtos.SnapshotDescription.newBuilder().setName("asyncSnapshot").setTable(TABLE_NAME.getNameAsString()).setType(HBaseProtos.SnapshotDescription.Type.FLUSH).build();
        admin.takeSnapshotAsync(snapshot);
        HMaster master = UTIL.getMiniHBaseCluster().getMaster();
        SnapshotTestingUtils.waitForSnapshotToComplete(master, snapshot, 200L);
        LOG.info((Object)" === Async Snapshot Completed ===");
        UTIL.getHBaseCluster().getMaster().getMasterFileSystem().logFileSystemState(LOG);
        SnapshotTestingUtils.assertOneSnapshotThatMatches(admin, snapshot);
    }

    @Test(timeout=300000L)
    public void testSnapshotStateAfterMerge() throws Exception {
        int numRows = 100;
        HBaseAdmin admin = UTIL.getHBaseAdmin();
        SnapshotTestingUtils.assertNoSnapshots(admin);
        SnapshotTestingUtils.loadData(UTIL, TABLE_NAME, numRows, (byte[][])new byte[][]{TEST_FAM});
        String snapshotBeforeMergeName = "snapshotBeforeMerge";
        admin.snapshot(snapshotBeforeMergeName, TABLE_NAME, HBaseProtos.SnapshotDescription.Type.FLUSH);
        TableName cloneBeforeMergeName = TableName.valueOf("cloneBeforeMerge");
        admin.cloneSnapshot(snapshotBeforeMergeName, cloneBeforeMergeName);
        SnapshotTestingUtils.waitForTableToBeOnline(UTIL, cloneBeforeMergeName);
        List<HRegionInfo> regions = admin.getTableRegions(TABLE_NAME);
        Collections.sort(regions, new Comparator<HRegionInfo>(){

            @Override
            public int compare(HRegionInfo r1, HRegionInfo r2) {
                return Bytes.compareTo(r1.getStartKey(), r2.getStartKey());
            }
        });
        int numRegions = admin.getTableRegions(TABLE_NAME).size();
        int numRegionsAfterMerge = numRegions - 2;
        admin.mergeRegions(regions.get(1).getEncodedNameAsBytes(), regions.get(2).getEncodedNameAsBytes(), true);
        admin.mergeRegions(regions.get(5).getEncodedNameAsBytes(), regions.get(6).getEncodedNameAsBytes(), true);
        this.waitRegionsAfterMerge(numRegionsAfterMerge);
        Assert.assertEquals((long)numRegionsAfterMerge, (long)admin.getTableRegions(TABLE_NAME).size());
        TableName cloneAfterMergeName = TableName.valueOf("cloneAfterMerge");
        admin.cloneSnapshot(snapshotBeforeMergeName, cloneAfterMergeName);
        SnapshotTestingUtils.waitForTableToBeOnline(UTIL, cloneAfterMergeName);
        SnapshotTestingUtils.verifyRowCount(UTIL, TABLE_NAME, numRows);
        SnapshotTestingUtils.verifyRowCount(UTIL, cloneBeforeMergeName, numRows);
        SnapshotTestingUtils.verifyRowCount(UTIL, cloneAfterMergeName, numRows);
        UTIL.deleteTable(cloneAfterMergeName);
        UTIL.deleteTable(cloneBeforeMergeName);
    }

    @Test(timeout=300000L)
    public void testTakeSnapshotAfterMerge() throws Exception {
        int numRows = 100;
        HBaseAdmin admin = UTIL.getHBaseAdmin();
        SnapshotTestingUtils.assertNoSnapshots(admin);
        SnapshotTestingUtils.loadData(UTIL, TABLE_NAME, numRows, (byte[][])new byte[][]{TEST_FAM});
        List<HRegionInfo> regions = admin.getTableRegions(TABLE_NAME);
        Collections.sort(regions, new Comparator<HRegionInfo>(){

            @Override
            public int compare(HRegionInfo r1, HRegionInfo r2) {
                return Bytes.compareTo(r1.getStartKey(), r2.getStartKey());
            }
        });
        int numRegions = admin.getTableRegions(TABLE_NAME).size();
        int numRegionsAfterMerge = numRegions - 2;
        admin.mergeRegions(regions.get(1).getEncodedNameAsBytes(), regions.get(2).getEncodedNameAsBytes(), true);
        admin.mergeRegions(regions.get(5).getEncodedNameAsBytes(), regions.get(6).getEncodedNameAsBytes(), true);
        this.waitRegionsAfterMerge(numRegionsAfterMerge);
        Assert.assertEquals((long)numRegionsAfterMerge, (long)admin.getTableRegions(TABLE_NAME).size());
        String snapshotName = "snapshotAfterMerge";
        SnapshotTestingUtils.snapshot(admin, snapshotName, TABLE_NAME.getNameAsString(), HBaseProtos.SnapshotDescription.Type.FLUSH, 3);
        TableName cloneName = TableName.valueOf("cloneMerge");
        admin.cloneSnapshot(snapshotName, cloneName);
        SnapshotTestingUtils.waitForTableToBeOnline(UTIL, cloneName);
        SnapshotTestingUtils.verifyRowCount(UTIL, TABLE_NAME, numRows);
        SnapshotTestingUtils.verifyRowCount(UTIL, cloneName, numRows);
        UTIL.deleteTable(cloneName);
    }

    @Test(timeout=300000L)
    public void testFlushCreateListDestroy() throws Exception {
        LOG.debug((Object)"------- Starting Snapshot test -------------");
        HBaseAdmin admin = UTIL.getHBaseAdmin();
        SnapshotTestingUtils.assertNoSnapshots(admin);
        SnapshotTestingUtils.loadData(UTIL, TABLE_NAME, 100, (byte[][])new byte[][]{TEST_FAM});
        String snapshotName = "flushSnapshotCreateListDestroy";
        FileSystem fs = UTIL.getHBaseCluster().getMaster().getMasterFileSystem().getFileSystem();
        Path rootDir = UTIL.getHBaseCluster().getMaster().getMasterFileSystem().getRootDir();
        SnapshotTestingUtils.createSnapshotAndValidate(admin, TABLE_NAME, Bytes.toString(TEST_FAM), snapshotName, rootDir, fs, true);
    }

    private void waitRegionsAfterMerge(long numRegionsAfterMerge) throws IOException, InterruptedException {
        HBaseAdmin admin = UTIL.getHBaseAdmin();
        long startTime = System.currentTimeMillis();
        while ((long)admin.getTableRegions(TABLE_NAME).size() != numRegionsAfterMerge && System.currentTimeMillis() - startTime <= 15000L) {
            Thread.sleep(100L);
        }
        SnapshotTestingUtils.waitForTableToBeOnline(UTIL, TABLE_NAME);
    }
}

