/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.master.cleaner;

import drill.shaded.hbase.guava.com.google.common.annotations.VisibleForTesting;
import drill.shaded.hbase.guava.com.google.common.base.Preconditions;
import drill.shaded.hbase.guava.com.google.common.collect.ImmutableSet;
import drill.shaded.hbase.guava.com.google.common.collect.Iterables;
import drill.shaded.hbase.guava.com.google.common.collect.Lists;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathIsNotEmptyDirectoryException;
import org.apache.hadoop.hbase.ScheduledChore;
import org.apache.hadoop.hbase.Stoppable;
import org.apache.hadoop.hbase.classification.InterfaceAudience;
import org.apache.hadoop.hbase.master.cleaner.DirScanPool;
import org.apache.hadoop.hbase.master.cleaner.FileCleanerDelegate;
import org.apache.hadoop.ipc.RemoteException;

@InterfaceAudience.Private
public abstract class CleanerChore<T extends FileCleanerDelegate>
extends ScheduledChore {
    private static final Log LOG = LogFactory.getLog((String)CleanerChore.class.getName());
    private static final int AVAIL_PROCESSORS = Runtime.getRuntime().availableProcessors();
    public static final String CHORE_POOL_SIZE = "hbase.cleaner.scan.dir.concurrent.size";
    static final String DEFAULT_CHORE_POOL_SIZE = "0.25";
    private final DirScanPool pool;
    protected final FileSystem fs;
    private final Path oldFileDir;
    private final Configuration conf;
    protected List<T> cleanersChain;
    protected Map<String, Object> params;
    private AtomicBoolean enabled = new AtomicBoolean(true);

    public CleanerChore(String name, int sleepPeriod, Stoppable s, Configuration conf, FileSystem fs, Path oldFileDir, String confKey, DirScanPool pool) {
        this(name, sleepPeriod, s, conf, fs, oldFileDir, confKey, pool, null);
    }

    public CleanerChore(String name, int sleepPeriod, Stoppable s, Configuration conf, FileSystem fs, Path oldFileDir, String confKey, DirScanPool pool, Map<String, Object> params) {
        super(name, s, sleepPeriod);
        Preconditions.checkNotNull(pool, "Chore's pool can not be null");
        this.pool = pool;
        this.fs = fs;
        this.oldFileDir = oldFileDir;
        this.conf = conf;
        this.params = params;
        this.initCleanerChain(confKey);
    }

    static int calculatePoolSize(String poolSize) {
        if (poolSize.matches("[1-9][0-9]*")) {
            int size = Math.min(Integer.parseInt(poolSize), AVAIL_PROCESSORS);
            if (size == AVAIL_PROCESSORS) {
                LOG.warn((Object)("Use full core processors to scan dir, size=" + size));
            }
            return size;
        }
        if (poolSize.matches("0.[0-9]+|1.0")) {
            int computedThreads = (int)((double)AVAIL_PROCESSORS * Double.valueOf(poolSize));
            if (computedThreads < 1) {
                LOG.debug((Object)("Computed " + computedThreads + " threads for CleanerChore, using 1 instead"));
                return 1;
            }
            return computedThreads;
        }
        LOG.error((Object)("Unrecognized value: " + poolSize + " for " + CHORE_POOL_SIZE + ", use default config: " + DEFAULT_CHORE_POOL_SIZE + " instead."));
        return CleanerChore.calculatePoolSize(DEFAULT_CHORE_POOL_SIZE);
    }

    protected abstract boolean validate(Path var1);

    private void initCleanerChain(String confKey) {
        this.cleanersChain = new LinkedList<T>();
        String[] logCleaners = this.conf.getStrings(confKey);
        if (logCleaners != null) {
            for (String className : logCleaners) {
                T logCleaner = this.newFileCleaner(className, this.conf);
                if (logCleaner == null) continue;
                LOG.info((Object)("initialize cleaner=" + className));
                this.cleanersChain.add(logCleaner);
            }
        }
    }

    private T newFileCleaner(String className, Configuration conf) {
        try {
            Class<FileCleanerDelegate> c = Class.forName(className).asSubclass(FileCleanerDelegate.class);
            FileCleanerDelegate cleaner = c.getDeclaredConstructor(new Class[0]).newInstance(new Object[0]);
            cleaner.setConf(conf);
            cleaner.init(this.params);
            return (T)cleaner;
        }
        catch (Exception e) {
            LOG.warn((Object)("Can NOT create CleanerDelegate: " + className), (Throwable)e);
            return null;
        }
    }

    @Override
    protected void chore() {
        if (this.getEnabled()) {
            try {
                this.pool.latchCountUp();
                if (this.runCleaner()) {
                    if (LOG.isTraceEnabled()) {
                        LOG.trace((Object)("Cleaned all WALs under " + this.oldFileDir));
                    }
                } else if (LOG.isTraceEnabled()) {
                    LOG.trace((Object)("WALs outstanding under " + this.oldFileDir));
                }
            }
            finally {
                this.pool.latchCountDown();
            }
            this.pool.tryUpdatePoolSize((long)(0.8 * (double)this.getTimeUnit().toMillis(this.getPeriod())));
        } else {
            LOG.trace((Object)"Cleaner chore disabled! Not cleaning.");
        }
    }

    public boolean runCleaner() {
        try {
            final AsyncResult result = new AsyncResult();
            this.pool.execute(new Runnable(){

                @Override
                public void run() {
                    CleanerChore.this.traverseAndDelete(CleanerChore.this.oldFileDir, true, result);
                }
            });
            return (Boolean)result.get();
        }
        catch (Exception e) {
            LOG.info((Object)("Failed to traverse and delete paths under the dir: " + this.oldFileDir), (Throwable)e);
            return false;
        }
    }

    private boolean checkAndDeleteFiles(List<FileStatus> files) {
        if (files == null) {
            return true;
        }
        ArrayList<FileStatus> validFiles = Lists.newArrayListWithCapacity(files.size());
        ArrayList<FileStatus> invalidFiles = Lists.newArrayList();
        for (FileStatus fileStatus : files) {
            if (this.validate(fileStatus.getPath())) {
                validFiles.add(fileStatus);
                continue;
            }
            LOG.warn((Object)("Found a wrongly formatted file: " + fileStatus.getPath() + " - will delete it."));
            invalidFiles.add(fileStatus);
        }
        Iterable<FileStatus> deletableValidFiles = validFiles;
        for (FileCleanerDelegate cleaner : this.cleanersChain) {
            if (cleaner.isStopped() || this.getStopper().isStopped()) {
                LOG.warn((Object)("A file cleaner" + this.getName() + " is stopped, won't delete any more files in:" + this.oldFileDir));
                return false;
            }
            Iterable<FileStatus> filteredFiles = cleaner.getDeletableFiles(deletableValidFiles);
            if (LOG.isTraceEnabled()) {
                ImmutableSet<FileStatus> filteredFileSet = ImmutableSet.copyOf(filteredFiles);
                for (FileStatus file : deletableValidFiles) {
                    if (filteredFileSet.contains(file)) continue;
                    LOG.trace((Object)(file.getPath() + " is not deletable according to:" + cleaner));
                }
            }
            deletableValidFiles = filteredFiles;
        }
        Iterable<FileStatus> iterable = Iterables.concat(invalidFiles, deletableValidFiles);
        return this.deleteFiles(iterable) == files.size();
    }

    protected int deleteFiles(Iterable<FileStatus> filesToDelete) {
        int deletedFileCount = 0;
        for (FileStatus file : filesToDelete) {
            Path filePath = file.getPath();
            LOG.trace((Object)("Removing " + file + " from archive"));
            try {
                LOG.info((Object)("Deleting: " + filePath));
                boolean success = this.fs.delete(filePath, false);
                if (success) {
                    ++deletedFileCount;
                    continue;
                }
                LOG.warn((Object)("Attempted to delete:" + filePath + ", but couldn't. Run cleaner chain and attempt to delete on next pass."));
            }
            catch (IOException e) {
                e = e instanceof RemoteException ? ((RemoteException)((Object)e)).unwrapRemoteException() : e;
                LOG.warn((Object)("Error while deleting: " + filePath), (Throwable)e);
            }
        }
        return deletedFileCount;
    }

    @Override
    public synchronized void cleanup() {
        for (FileCleanerDelegate lc : this.cleanersChain) {
            try {
                lc.stop("Exiting");
            }
            catch (Throwable t) {
                LOG.warn((Object)"Stopping", t);
            }
        }
    }

    @VisibleForTesting
    int getChorePoolSize() {
        return this.pool.getSize();
    }

    public boolean setEnabled(boolean enabled) {
        return this.enabled.getAndSet(enabled);
    }

    public boolean getEnabled() {
        return this.enabled.get();
    }

    private void traverseAndDelete(final Path dir, final boolean root, final AsyncResult<Boolean> result) {
        block7: {
            try {
                boolean allFilesDeleted;
                final Action<Boolean> curDirDeletion = new Action<Boolean>(){

                    @Override
                    public Boolean act() throws IOException {
                        LOG.info((Object)("Deleting: " + dir));
                        return CleanerChore.this.fs.delete(dir, false);
                    }
                };
                List<FileStatus> allPaths = Arrays.asList(this.fs.listStatus(dir));
                ArrayList<FileStatus> subDirs = new ArrayList<FileStatus>();
                final ArrayList<FileStatus> files = new ArrayList<FileStatus>();
                for (FileStatus status : allPaths) {
                    if (status.isDirectory()) {
                        subDirs.add(status);
                        continue;
                    }
                    if (!status.isFile()) continue;
                    files.add(status);
                }
                boolean bl = allFilesDeleted = files.isEmpty() || this.deleteAction(new Action<Boolean>(){

                    @Override
                    public Boolean act() throws IOException {
                        return CleanerChore.this.checkAndDeleteFiles(files);
                    }
                }, "files", dir);
                if (subDirs.isEmpty()) {
                    boolean deleted = allFilesDeleted;
                    if (allFilesDeleted && !root) {
                        deleted = this.deleteAction(curDirDeletion, "dir", dir);
                    }
                    result.set(deleted);
                    return;
                }
                final AtomicInteger remain = new AtomicInteger(subDirs.size());
                Callback<Boolean> callback = new Callback<Boolean>(){
                    private volatile boolean allSubDirDeleted = true;

                    @Override
                    public void run(Boolean subDirDeleted) {
                        this.allSubDirDeleted &= subDirDeleted.booleanValue();
                        if (remain.decrementAndGet() == 0) {
                            boolean deleted;
                            boolean bl = deleted = allFilesDeleted && this.allSubDirDeleted;
                            if (deleted && !root) {
                                deleted = CleanerChore.this.deleteAction(curDirDeletion, "dir", dir);
                            }
                            result.set(deleted);
                        }
                    }
                };
                Iterator iterator = subDirs.iterator();
                while (iterator.hasNext()) {
                    FileStatus subDir;
                    final FileStatus finalSubDir = subDir = (FileStatus)iterator.next();
                    final AsyncResult<Boolean> asyncResult = new AsyncResult<Boolean>(callback);
                    this.pool.execute(new Runnable(){

                        @Override
                        public void run() {
                            CleanerChore.this.traverseAndDelete(finalSubDir.getPath(), false, asyncResult);
                        }
                    });
                }
            }
            catch (Exception e) {
                result.set(false);
                if (!LOG.isDebugEnabled()) break block7;
                LOG.debug((Object)("Failed to traverse and delete the path=" + dir + ", root=" + root), (Throwable)e);
            }
        }
    }

    private boolean deleteAction(Action<Boolean> deletion, String type, Path dir) {
        boolean deleted;
        try {
            if (LOG.isTraceEnabled()) {
                LOG.trace((Object)("Start deleting " + type + " under " + dir));
            }
            deleted = deletion.act();
        }
        catch (PathIsNotEmptyDirectoryException exception) {
            if (LOG.isDebugEnabled()) {
                LOG.debug((Object)("Couldn't delete '" + dir + "' yet because it isn't empty w/exception."), (Throwable)exception);
            }
            deleted = false;
        }
        catch (IOException ioe) {
            LOG.info((Object)("Could not delete " + type + " under " + dir + ". might be transient; we'll retry. if it keeps happening, use following exception when asking on mailing list."), (Throwable)ioe);
            deleted = false;
        }
        catch (Exception e) {
            LOG.info((Object)"unexpected exception: ", (Throwable)e);
            deleted = false;
        }
        if (LOG.isTraceEnabled()) {
            LOG.trace((Object)("Finish deleting " + type + " under " + dir + ", deleted=" + deleted));
        }
        return deleted;
    }

    private final class AsyncResult<T> {
        private Callback<T> callback;
        private T result;
        private boolean resultSet = false;

        AsyncResult(Callback<T> callback) {
            this.callback = callback;
        }

        AsyncResult() {
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        void set(T result) {
            AsyncResult asyncResult = this;
            synchronized (asyncResult) {
                this.result = result;
                if (this.callback != null) {
                    this.callback.run(result);
                }
                this.resultSet = true;
                this.notifyAll();
            }
        }

        synchronized T get() throws Exception {
            while (!this.resultSet) {
                this.wait();
            }
            return this.result;
        }
    }

    private static interface Callback<T> {
        public void run(T var1);
    }

    private static interface Action<T> {
        public T act() throws IOException;
    }
}

