/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.io.hfile;

import drill.shaded.hbase.guava.com.google.common.annotations.VisibleForTesting;
import drill.shaded.hbase.guava.com.google.common.base.Preconditions;
import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutput;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.Cell;
import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.classification.InterfaceAudience;
import org.apache.hadoop.hbase.fs.HFileSystem;
import org.apache.hadoop.hbase.io.ByteBufferInputStream;
import org.apache.hadoop.hbase.io.FSDataInputStreamWrapper;
import org.apache.hadoop.hbase.io.encoding.DataBlockEncoding;
import org.apache.hadoop.hbase.io.encoding.HFileBlockDecodingContext;
import org.apache.hadoop.hbase.io.encoding.HFileBlockDefaultDecodingContext;
import org.apache.hadoop.hbase.io.encoding.HFileBlockDefaultEncodingContext;
import org.apache.hadoop.hbase.io.encoding.HFileBlockEncodingContext;
import org.apache.hadoop.hbase.io.hfile.BlockType;
import org.apache.hadoop.hbase.io.hfile.CacheConfig;
import org.apache.hadoop.hbase.io.hfile.Cacheable;
import org.apache.hadoop.hbase.io.hfile.CacheableDeserializer;
import org.apache.hadoop.hbase.io.hfile.CacheableDeserializerIdManager;
import org.apache.hadoop.hbase.io.hfile.ChecksumUtil;
import org.apache.hadoop.hbase.io.hfile.HFile;
import org.apache.hadoop.hbase.io.hfile.HFileContext;
import org.apache.hadoop.hbase.io.hfile.HFileContextBuilder;
import org.apache.hadoop.hbase.io.hfile.HFileDataBlockEncoder;
import org.apache.hadoop.hbase.io.hfile.NoOpDataBlockEncoder;
import org.apache.hadoop.hbase.util.ByteBufferUtils;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.ChecksumType;
import org.apache.hadoop.hbase.util.ClassSize;
import org.apache.hadoop.io.IOUtils;

@InterfaceAudience.Private
public class HFileBlock
implements Cacheable {
    static final int CHECKSUM_VERIFICATION_NUM_IO_THRESHOLD = 3;
    public static final boolean FILL_HEADER = true;
    public static final boolean DONT_FILL_HEADER = false;
    public static final int ENCODED_HEADER_SIZE = 35;
    static final byte[] DUMMY_HEADER_NO_CHECKSUM = new byte[24];
    public static final int BYTE_BUFFER_HEAP_SIZE = (int)ClassSize.estimateBase(ByteBuffer.wrap(new byte[0], 0, 0).getClass(), false);
    public static final int EXTRA_SERIALIZATION_SPACE = 13;
    static final int CHECKSUM_SIZE = 4;
    static final CacheableDeserializer<Cacheable> blockDeserializer = new CacheableDeserializer<Cacheable>(){

        @Override
        public HFileBlock deserialize(ByteBuffer buf, boolean reuse) throws IOException {
            ByteBuffer newByteBuffer;
            buf.limit(buf.limit() - 13).rewind();
            if (reuse) {
                newByteBuffer = buf.slice();
            } else {
                newByteBuffer = ByteBuffer.allocate(buf.limit());
                newByteBuffer.put(buf);
            }
            buf.position(buf.limit());
            buf.limit(buf.limit() + 13);
            boolean usesChecksum = buf.get() == 1;
            HFileBlock hFileBlock = new HFileBlock(newByteBuffer, usesChecksum);
            hFileBlock.offset = buf.getLong();
            hFileBlock.nextBlockOnDiskSizeWithHeader = buf.getInt();
            if (hFileBlock.hasNextBlockHeader()) {
                hFileBlock.buf.limit(hFileBlock.buf.limit() - hFileBlock.headerSize());
            }
            return hFileBlock;
        }

        @Override
        public int getDeserialiserIdentifier() {
            return deserializerIdentifier;
        }

        @Override
        public HFileBlock deserialize(ByteBuffer b) throws IOException {
            return this.deserialize(b, false);
        }
    };
    private static final int deserializerIdentifier = CacheableDeserializerIdManager.registerDeserializer(blockDeserializer);
    private BlockType blockType;
    private int onDiskSizeWithoutHeader;
    private final int uncompressedSizeWithoutHeader;
    private final long prevBlockOffset;
    private final int onDiskDataSizeWithHeader;
    private ByteBuffer buf;
    private HFileContext fileContext;
    private long offset = -1L;
    private int nextBlockOnDiskSizeWithHeader = -1;

    HFileBlock(BlockType blockType, int onDiskSizeWithoutHeader, int uncompressedSizeWithoutHeader, long prevBlockOffset, ByteBuffer buf, boolean fillHeader, long offset, int onDiskDataSizeWithHeader, HFileContext fileContext) {
        this.blockType = blockType;
        this.onDiskSizeWithoutHeader = onDiskSizeWithoutHeader;
        this.uncompressedSizeWithoutHeader = uncompressedSizeWithoutHeader;
        this.prevBlockOffset = prevBlockOffset;
        this.buf = buf;
        this.offset = offset;
        this.onDiskDataSizeWithHeader = onDiskDataSizeWithHeader;
        this.fileContext = fileContext;
        if (fillHeader) {
            this.overwriteHeader();
        }
        this.buf.rewind();
    }

    HFileBlock(HFileBlock that) {
        this.blockType = that.blockType;
        this.onDiskSizeWithoutHeader = that.onDiskSizeWithoutHeader;
        this.uncompressedSizeWithoutHeader = that.uncompressedSizeWithoutHeader;
        this.prevBlockOffset = that.prevBlockOffset;
        this.buf = that.buf.duplicate();
        this.offset = that.offset;
        this.onDiskDataSizeWithHeader = that.onDiskDataSizeWithHeader;
        this.fileContext = that.fileContext;
        this.nextBlockOnDiskSizeWithHeader = that.nextBlockOnDiskSizeWithHeader;
    }

    HFileBlock(ByteBuffer b, boolean usesHBaseChecksum) throws IOException {
        b.rewind();
        this.blockType = BlockType.read(b);
        this.onDiskSizeWithoutHeader = b.getInt(Header.ON_DISK_SIZE_WITHOUT_HEADER_INDEX);
        this.uncompressedSizeWithoutHeader = b.getInt(Header.UNCOMPRESSED_SIZE_WITHOUT_HEADER_INDEX);
        this.prevBlockOffset = b.getLong(Header.PREV_BLOCK_OFFSET_INDEX);
        HFileContextBuilder contextBuilder = new HFileContextBuilder();
        contextBuilder.withHBaseCheckSum(usesHBaseChecksum);
        if (usesHBaseChecksum) {
            contextBuilder.withChecksumType(ChecksumType.codeToType(b.get(Header.CHECKSUM_TYPE_INDEX)));
            contextBuilder.withBytesPerCheckSum(b.getInt(Header.BYTES_PER_CHECKSUM_INDEX));
            this.onDiskDataSizeWithHeader = b.getInt(Header.ON_DISK_DATA_SIZE_WITH_HEADER_INDEX);
        } else {
            contextBuilder.withChecksumType(ChecksumType.NULL);
            contextBuilder.withBytesPerCheckSum(0);
            this.onDiskDataSizeWithHeader = this.onDiskSizeWithoutHeader + 24;
        }
        this.fileContext = contextBuilder.build();
        this.buf = b;
        this.buf.rewind();
    }

    @Override
    public BlockType getBlockType() {
        return this.blockType;
    }

    public short getDataBlockEncodingId() {
        if (this.blockType != BlockType.ENCODED_DATA) {
            throw new IllegalArgumentException("Querying encoder ID of a block of type other than " + (Object)((Object)BlockType.ENCODED_DATA) + ": " + (Object)((Object)this.blockType));
        }
        return this.buf.getShort(this.headerSize());
    }

    public int getOnDiskSizeWithHeader() {
        return this.onDiskSizeWithoutHeader + this.headerSize();
    }

    public int getOnDiskSizeWithoutHeader() {
        return this.onDiskSizeWithoutHeader;
    }

    public int getUncompressedSizeWithoutHeader() {
        return this.uncompressedSizeWithoutHeader;
    }

    public long getPrevBlockOffset() {
        return this.prevBlockOffset;
    }

    private void overwriteHeader() {
        this.buf.rewind();
        this.blockType.write(this.buf);
        this.buf.putInt(this.onDiskSizeWithoutHeader);
        this.buf.putInt(this.uncompressedSizeWithoutHeader);
        this.buf.putLong(this.prevBlockOffset);
        if (this.fileContext.isUseHBaseChecksum()) {
            this.buf.put(this.fileContext.getChecksumType().getCode());
            this.buf.putInt(this.fileContext.getBytesPerChecksum());
            this.buf.putInt(this.onDiskDataSizeWithHeader);
        }
    }

    public ByteBuffer getBufferWithoutHeader() {
        ByteBuffer dup = this.buf.duplicate();
        dup.position(this.headerSize());
        dup.limit(this.buf.limit() - this.totalChecksumBytes());
        return dup.slice();
    }

    public ByteBuffer getBufferReadOnly() {
        ByteBuffer dup = this.buf.duplicate();
        dup.limit(this.buf.limit() - this.totalChecksumBytes());
        return dup.slice();
    }

    public ByteBuffer getBufferReadOnlyWithHeader() {
        ByteBuffer dup = this.buf.duplicate();
        return dup.slice();
    }

    ByteBuffer getBufferWithHeader() {
        ByteBuffer dupBuf = this.buf.duplicate();
        dupBuf.rewind();
        return dupBuf;
    }

    private void sanityCheckAssertion(long valueFromBuf, long valueFromField, String fieldName) throws IOException {
        if (valueFromBuf != valueFromField) {
            throw new AssertionError((Object)(fieldName + " in the buffer (" + valueFromBuf + ") is different from that in the field (" + valueFromField + ")"));
        }
    }

    private void sanityCheckAssertion(BlockType valueFromBuf, BlockType valueFromField) throws IOException {
        if (valueFromBuf != valueFromField) {
            throw new IOException("Block type stored in the buffer: " + (Object)((Object)valueFromBuf) + ", block type field: " + (Object)((Object)valueFromField));
        }
    }

    void sanityCheck() throws IOException {
        this.buf.rewind();
        this.sanityCheckAssertion(BlockType.read(this.buf), this.blockType);
        this.sanityCheckAssertion(this.buf.getInt(), this.onDiskSizeWithoutHeader, "onDiskSizeWithoutHeader");
        this.sanityCheckAssertion(this.buf.getInt(), this.uncompressedSizeWithoutHeader, "uncompressedSizeWithoutHeader");
        this.sanityCheckAssertion(this.buf.getLong(), this.prevBlockOffset, "prevBlocKOffset");
        if (this.fileContext.isUseHBaseChecksum()) {
            this.sanityCheckAssertion(this.buf.get(), this.fileContext.getChecksumType().getCode(), "checksumType");
            this.sanityCheckAssertion(this.buf.getInt(), this.fileContext.getBytesPerChecksum(), "bytesPerChecksum");
            this.sanityCheckAssertion(this.buf.getInt(), this.onDiskDataSizeWithHeader, "onDiskDataSizeWithHeader");
        }
        int cksumBytes = this.totalChecksumBytes();
        int expectedBufLimit = this.onDiskDataSizeWithHeader + cksumBytes;
        if (this.buf.limit() != expectedBufLimit) {
            throw new AssertionError((Object)("Expected buffer limit " + expectedBufLimit + ", got " + this.buf.limit()));
        }
        int hdrSize = this.headerSize();
        if (this.buf.capacity() != expectedBufLimit && this.buf.capacity() != expectedBufLimit + hdrSize) {
            throw new AssertionError((Object)("Invalid buffer capacity: " + this.buf.capacity() + ", expected " + expectedBufLimit + " or " + (expectedBufLimit + hdrSize)));
        }
    }

    public String toString() {
        StringBuilder sb = new StringBuilder().append("HFileBlock [").append(" fileOffset=").append(this.offset).append(" headerSize()=").append(this.headerSize()).append(" blockType=").append((Object)this.blockType).append(" onDiskSizeWithoutHeader=").append(this.onDiskSizeWithoutHeader).append(" uncompressedSizeWithoutHeader=").append(this.uncompressedSizeWithoutHeader).append(" prevBlockOffset=").append(this.prevBlockOffset).append(" isUseHBaseChecksum()=").append(this.fileContext.isUseHBaseChecksum());
        if (this.fileContext.isUseHBaseChecksum()) {
            sb.append(" checksumType=").append((Object)ChecksumType.codeToType(this.buf.get(24))).append(" bytesPerChecksum=").append(this.buf.getInt(25)).append(" onDiskDataSizeWithHeader=").append(this.onDiskDataSizeWithHeader);
        } else {
            sb.append(" onDiskDataSizeWithHeader=").append(this.onDiskDataSizeWithHeader).append("(").append(this.onDiskSizeWithoutHeader).append("+").append(24).append(")");
        }
        String dataBegin = null;
        if (this.buf.hasArray()) {
            dataBegin = Bytes.toStringBinary(this.buf.array(), this.buf.arrayOffset() + this.headerSize(), Math.min(32, this.buf.limit() - this.buf.arrayOffset() - this.headerSize()));
        } else {
            ByteBuffer bufWithoutHeader = this.getBufferWithoutHeader();
            byte[] dataBeginBytes = new byte[Math.min(32, bufWithoutHeader.limit() - bufWithoutHeader.position())];
            bufWithoutHeader.get(dataBeginBytes);
            dataBegin = Bytes.toStringBinary(dataBeginBytes);
        }
        sb.append(" getOnDiskSizeWithHeader()=").append(this.getOnDiskSizeWithHeader()).append(" totalChecksumBytes()=").append(this.totalChecksumBytes()).append(" isUnpacked()=").append(this.isUnpacked()).append(" buf=[ ").append(this.buf).append(" ]").append(" dataBeginsWith=").append(dataBegin).append(" fileContext=").append(this.fileContext).append(" ]");
        return sb.toString();
    }

    private static void validateOnDiskSizeWithoutHeader(int expectedOnDiskSizeWithoutHeader, int actualOnDiskSizeWithoutHeader, ByteBuffer buf, long offset) throws IOException {
        if (actualOnDiskSizeWithoutHeader != expectedOnDiskSizeWithoutHeader) {
            ByteBuffer bufReadOnly = buf.asReadOnlyBuffer();
            String dataBegin = null;
            byte[] dataBeginBytes = new byte[Math.min(32, bufReadOnly.limit() - bufReadOnly.position())];
            bufReadOnly.get(dataBeginBytes);
            dataBegin = Bytes.toStringBinary(dataBeginBytes);
            String blockInfoMsg = "Block offset: " + offset + ", data starts with: " + dataBegin;
            throw new IOException("On-disk size without header provided is " + expectedOnDiskSizeWithoutHeader + ", but block header contains " + actualOnDiskSizeWithoutHeader + ". " + blockInfoMsg);
        }
    }

    HFileBlock unpack(HFileContext fileContext, FSReader reader) throws IOException {
        if (!fileContext.isCompressedOrEncrypted()) {
            return this;
        }
        HFileBlock unpacked = new HFileBlock(this);
        unpacked.allocateBuffer();
        HFileBlockDecodingContext ctx = this.blockType == BlockType.ENCODED_DATA ? reader.getBlockDecodingContext() : reader.getDefaultBlockDecodingContext();
        ByteBuffer dup = this.buf.duplicate();
        dup.position(this.headerSize());
        dup = dup.slice();
        ctx.prepareDecoding(unpacked.getOnDiskSizeWithoutHeader(), unpacked.getUncompressedSizeWithoutHeader(), unpacked.getBufferWithoutHeader(), dup);
        if (unpacked.hasNextBlockHeader()) {
            ByteBuffer inDup = this.buf.duplicate();
            inDup.limit(inDup.limit() + this.headerSize());
            ByteBuffer outDup = unpacked.buf.duplicate();
            outDup.limit(outDup.limit() + unpacked.headerSize());
            ByteBufferUtils.copyFromBufferToBuffer(outDup, inDup, this.onDiskDataSizeWithHeader, unpacked.headerSize() + unpacked.uncompressedSizeWithoutHeader + unpacked.totalChecksumBytes(), unpacked.headerSize());
        }
        return unpacked;
    }

    private boolean hasNextBlockHeader() {
        return this.nextBlockOnDiskSizeWithHeader > 0;
    }

    private void allocateBuffer() {
        int cksumBytes = this.totalChecksumBytes();
        int headerSize = this.headerSize();
        int capacityNeeded = headerSize + this.uncompressedSizeWithoutHeader + cksumBytes + (this.hasNextBlockHeader() ? headerSize : 0);
        ByteBuffer newBuf = ByteBuffer.allocate(capacityNeeded);
        ByteBuffer dup = this.buf.duplicate();
        dup.position(0);
        dup.get(newBuf.array(), newBuf.arrayOffset(), headerSize);
        this.buf = newBuf;
        this.buf.limit(headerSize + this.uncompressedSizeWithoutHeader + cksumBytes);
    }

    public boolean isUnpacked() {
        int cksumBytes = this.totalChecksumBytes();
        int headerSize = this.headerSize();
        int expectedCapacity = headerSize + this.uncompressedSizeWithoutHeader + cksumBytes;
        int bufCapacity = this.buf.capacity();
        return bufCapacity == expectedCapacity || bufCapacity == expectedCapacity + headerSize;
    }

    public static void verifyUncompressed(ByteBuffer buf, boolean useHBaseChecksum) throws IOException {
        int onDiskSizeWithoutHeader = buf.getInt(Header.ON_DISK_SIZE_WITHOUT_HEADER_INDEX);
        int uncompressedSizeWithoutHeader = buf.getInt(Header.UNCOMPRESSED_SIZE_WITHOUT_HEADER_INDEX);
        int checksumBytes = 0;
        if (useHBaseChecksum) {
            int onDiskDataSizeWithHeader = buf.getInt(Header.ON_DISK_DATA_SIZE_WITH_HEADER_INDEX);
            checksumBytes = (int)ChecksumUtil.numBytes(onDiskDataSizeWithHeader, buf.getInt(Header.BYTES_PER_CHECKSUM_INDEX));
        }
        if (onDiskSizeWithoutHeader != uncompressedSizeWithoutHeader + checksumBytes) {
            throw new IOException("Using no compression but onDiskSizeWithoutHeader=" + onDiskSizeWithoutHeader + ", uncompressedSizeWithoutHeader=" + uncompressedSizeWithoutHeader + ", numChecksumbytes=" + checksumBytes);
        }
    }

    public void expectType(BlockType expectedType) throws IOException {
        if (this.blockType != expectedType) {
            throw new IOException("Invalid block type: expected=" + (Object)((Object)expectedType) + ", actual=" + (Object)((Object)this.blockType));
        }
    }

    public long getOffset() {
        if (this.offset < 0L) {
            throw new IllegalStateException("HFile block offset not initialized properly");
        }
        return this.offset;
    }

    public DataInputStream getByteStream() {
        ByteBuffer dup = this.buf.duplicate();
        dup.position(this.headerSize());
        return new DataInputStream(new ByteBufferInputStream(dup));
    }

    @Override
    public long heapSize() {
        long size = ClassSize.align((long)(ClassSize.OBJECT + 3 * ClassSize.REFERENCE + 16 + 16) + this.fileContext.heapSize());
        if (this.buf != null) {
            size += (long)ClassSize.align(this.buf.capacity() + BYTE_BUFFER_HEAP_SIZE);
        }
        return ClassSize.align(size);
    }

    public static boolean readWithExtra(InputStream in, byte[] buf, int bufOffset, int necessaryLen, int extraLen) throws IOException {
        int bytesRemaining;
        int ret;
        for (bytesRemaining = necessaryLen + extraLen; bytesRemaining > 0 && ((ret = in.read(buf, bufOffset, bytesRemaining)) != -1 || bytesRemaining > extraLen); bytesRemaining -= ret) {
            if (ret < 0) {
                throw new IOException("Premature EOF from inputStream (read returned " + ret + ", was trying to read " + necessaryLen + " necessary bytes and " + extraLen + " extra bytes, successfully read " + (necessaryLen + extraLen - bytesRemaining));
            }
            bufOffset += ret;
        }
        return bytesRemaining <= 0;
    }

    @VisibleForTesting
    static boolean positionalReadWithExtra(FSDataInputStream in, long position, byte[] buf, int bufOffset, int necessaryLen, int extraLen) throws IOException {
        int bytesRead;
        int ret;
        int bytesRemaining = necessaryLen + extraLen;
        for (bytesRead = 0; bytesRead < necessaryLen; bytesRead += ret) {
            ret = in.read(position, buf, bufOffset, bytesRemaining);
            if (ret < 0) {
                throw new IOException("Premature EOF from inputStream (positional read returned " + ret + ", was trying to read " + necessaryLen + " necessary bytes and " + extraLen + " extra bytes, successfully read " + bytesRead);
            }
            position += (long)ret;
            bufOffset += ret;
            bytesRemaining -= ret;
        }
        return bytesRead != necessaryLen && bytesRemaining <= 0;
    }

    public int getNextBlockOnDiskSizeWithHeader() {
        return this.nextBlockOnDiskSizeWithHeader;
    }

    @Override
    public int getSerializedLength() {
        if (this.buf != null) {
            int extraSpace = this.hasNextBlockHeader() ? this.headerSize() : 0;
            return this.buf.limit() + extraSpace + 13;
        }
        return 0;
    }

    @Override
    public void serialize(ByteBuffer destination) {
        ByteBufferUtils.copyFromBufferToBuffer(destination, this.buf, 0, this.getSerializedLength() - 13);
        this.serializeExtraInfo(destination);
    }

    public void serializeExtraInfo(ByteBuffer destination) {
        destination.put(this.fileContext.isUseHBaseChecksum() ? (byte)1 : 0);
        destination.putLong(this.offset);
        destination.putInt(this.nextBlockOnDiskSizeWithHeader);
        destination.rewind();
    }

    @Override
    public CacheableDeserializer<Cacheable> getDeserializer() {
        return blockDeserializer;
    }

    public boolean equals(Object comparison) {
        if (this == comparison) {
            return true;
        }
        if (comparison == null) {
            return false;
        }
        if (comparison.getClass() != this.getClass()) {
            return false;
        }
        HFileBlock castedComparison = (HFileBlock)comparison;
        if (castedComparison.blockType != this.blockType) {
            return false;
        }
        if (castedComparison.nextBlockOnDiskSizeWithHeader != this.nextBlockOnDiskSizeWithHeader) {
            return false;
        }
        if (castedComparison.offset != this.offset) {
            return false;
        }
        if (castedComparison.onDiskSizeWithoutHeader != this.onDiskSizeWithoutHeader) {
            return false;
        }
        if (castedComparison.prevBlockOffset != this.prevBlockOffset) {
            return false;
        }
        if (castedComparison.uncompressedSizeWithoutHeader != this.uncompressedSizeWithoutHeader) {
            return false;
        }
        return ByteBufferUtils.compareTo(this.buf, 0, this.buf.limit(), castedComparison.buf, 0, castedComparison.buf.limit()) == 0;
    }

    public DataBlockEncoding getDataBlockEncoding() {
        if (this.blockType == BlockType.ENCODED_DATA) {
            return DataBlockEncoding.getEncodingById(this.getDataBlockEncodingId());
        }
        return DataBlockEncoding.NONE;
    }

    byte getChecksumType() {
        return this.fileContext.getChecksumType().getCode();
    }

    int getBytesPerChecksum() {
        return this.fileContext.getBytesPerChecksum();
    }

    int getOnDiskDataSizeWithHeader() {
        return this.onDiskDataSizeWithHeader;
    }

    int totalChecksumBytes() {
        return HFileBlock.totalChecksumBytes(this.fileContext, this.onDiskDataSizeWithHeader);
    }

    private static int totalChecksumBytes(HFileContext fileContext, int onDiskDataSizeWithHeader) {
        if (!fileContext.isUseHBaseChecksum() || fileContext.getBytesPerChecksum() == 0) {
            return 0;
        }
        return (int)ChecksumUtil.numBytes(onDiskDataSizeWithHeader, fileContext.getBytesPerChecksum());
    }

    public int headerSize() {
        return HFileBlock.headerSize(this.fileContext.isUseHBaseChecksum());
    }

    public static int headerSize(boolean usesHBaseChecksum) {
        if (usesHBaseChecksum) {
            return 33;
        }
        return 24;
    }

    public byte[] getDummyHeaderForVersion() {
        return HFileBlock.getDummyHeaderForVersion(this.fileContext.isUseHBaseChecksum());
    }

    private static byte[] getDummyHeaderForVersion(boolean usesHBaseChecksum) {
        if (usesHBaseChecksum) {
            return HConstants.HFILEBLOCK_DUMMY_HEADER;
        }
        return DUMMY_HEADER_NO_CHECKSUM;
    }

    public HFileContext getHFileContext() {
        return this.fileContext;
    }

    public static String toStringHeader(ByteBuffer buf) throws IOException {
        byte[] magicBuf = new byte[Math.min(buf.limit() - buf.position(), 8)];
        buf.get(magicBuf);
        int compressedBlockSizeNoHeader = buf.getInt();
        int uncompressedBlockSizeNoHeader = buf.getInt();
        long prevBlockOffset = buf.getLong();
        byte cksumtype = buf.get();
        long bytesPerChecksum = buf.getInt();
        long onDiskDataSizeWithHeader = buf.getInt();
        return " Header dump: magic: " + Bytes.toString(magicBuf) + " blockType " + magicBuf + " compressedBlockSizeNoHeader " + compressedBlockSizeNoHeader + " uncompressedBlockSizeNoHeader " + uncompressedBlockSizeNoHeader + " prevBlockOffset " + prevBlockOffset + " checksumType " + (Object)((Object)ChecksumType.codeToType(cksumtype)) + " bytesPerChecksum " + bytesPerChecksum + " onDiskDataSizeWithHeader " + onDiskDataSizeWithHeader;
    }

    static class FSReaderImpl
    extends AbstractFSReader {
        protected FSDataInputStreamWrapper streamWrapper;
        private HFileBlockDecodingContext encodedBlockDecodingCtx;
        private final HFileBlockDefaultDecodingContext defaultDecodingCtx;
        private ThreadLocal<PrefetchedHeader> prefetchedHeaderForThread = new ThreadLocal<PrefetchedHeader>(){

            @Override
            public PrefetchedHeader initialValue() {
                return new PrefetchedHeader();
            }
        };

        public FSReaderImpl(FSDataInputStreamWrapper stream, long fileSize, HFileSystem hfs, Path path, HFileContext fileContext) throws IOException {
            super(fileSize, hfs, path, fileContext);
            this.streamWrapper = stream;
            this.streamWrapper.prepareForBlockReader(!fileContext.isUseHBaseChecksum());
            this.defaultDecodingCtx = new HFileBlockDefaultDecodingContext(fileContext);
            this.encodedBlockDecodingCtx = this.defaultDecodingCtx;
        }

        FSReaderImpl(FSDataInputStream istream, long fileSize, HFileContext fileContext) throws IOException {
            this(new FSDataInputStreamWrapper(istream), fileSize, null, null, fileContext);
        }

        @Override
        public HFileBlock readBlockData(long offset, long onDiskSizeWithHeaderL, int uncompressedSize, boolean pread) throws IOException {
            boolean doVerificationThruHBaseChecksum = this.streamWrapper.shouldUseHBaseChecksum();
            FSDataInputStream is = this.streamWrapper.getStream(doVerificationThruHBaseChecksum);
            HFileBlock blk = this.readBlockDataInternal(is, offset, (int)onDiskSizeWithHeaderL, uncompressedSize, pread, doVerificationThruHBaseChecksum);
            if (blk == null) {
                HFile.LOG.warn((Object)("HBase checksum verification failed for file " + this.path + " at offset " + offset + " filesize " + this.fileSize + ". Retrying read with HDFS checksums turned on..."));
                if (!doVerificationThruHBaseChecksum) {
                    String msg = "HBase checksum verification failed for file " + this.path + " at offset " + offset + " filesize " + this.fileSize + " but this cannot happen because doVerify is " + doVerificationThruHBaseChecksum;
                    HFile.LOG.warn((Object)msg);
                    throw new IOException(msg);
                }
                HFile.checksumFailures.incrementAndGet();
                is = this.streamWrapper.fallbackToFsChecksum(3);
                doVerificationThruHBaseChecksum = false;
                blk = this.readBlockDataInternal(is, offset, (int)onDiskSizeWithHeaderL, uncompressedSize, pread, doVerificationThruHBaseChecksum);
                if (blk != null) {
                    HFile.LOG.warn((Object)("HDFS checksum verification suceeded for file " + this.path + " at offset " + offset + " filesize " + this.fileSize));
                }
            }
            if (blk == null && !doVerificationThruHBaseChecksum) {
                String msg = "readBlockData failed, possibly due to checksum verification failed for file " + this.path + " at offset " + offset + " filesize " + this.fileSize;
                HFile.LOG.warn((Object)msg);
                throw new IOException(msg);
            }
            this.streamWrapper.checksumOk();
            return blk;
        }

        protected HFileBlock readBlockDataInternal(FSDataInputStream is, long offset, int onDiskSizeWithHeader, int uncompressedSize, boolean pread, boolean verifyChecksum) throws IOException {
            if (offset < 0L) {
                throw new IOException("Invalid offset=" + offset + " trying to read block (onDiskSize=" + onDiskSizeWithHeader + ", uncompressedSize=" + uncompressedSize + ")");
            }
            if (uncompressedSize != -1) {
                throw new IOException("Version 2 block reader API does not need the uncompressed size parameter");
            }
            if (onDiskSizeWithHeader < this.hdrSize && onDiskSizeWithHeader != -1 || onDiskSizeWithHeader >= Integer.MAX_VALUE) {
                throw new IOException("Invalid onDisksize=" + onDiskSizeWithHeader + ": expected to be at least " + this.hdrSize + " and at most " + Integer.MAX_VALUE + ", or -1 (offset=" + offset + ", uncompressedSize=" + uncompressedSize + ")");
            }
            PrefetchedHeader prefetchedHeader = this.prefetchedHeaderForThread.get();
            ByteBuffer headerBuf = prefetchedHeader.offset == offset ? prefetchedHeader.buf : null;
            int nextBlockOnDiskSize = 0;
            byte[] onDiskBlock = null;
            if (onDiskSizeWithHeader > 0) {
                int preReadHeaderSize = headerBuf == null ? 0 : this.hdrSize;
                onDiskBlock = new byte[onDiskSizeWithHeader + this.hdrSize];
                nextBlockOnDiskSize = this.readAtOffset(is, onDiskBlock, preReadHeaderSize, onDiskSizeWithHeader - preReadHeaderSize, true, offset + (long)preReadHeaderSize, pread);
                if (headerBuf != null) {
                    assert (headerBuf.hasArray());
                    System.arraycopy(headerBuf.array(), headerBuf.arrayOffset(), onDiskBlock, 0, this.hdrSize);
                } else {
                    headerBuf = ByteBuffer.wrap(onDiskBlock, 0, this.hdrSize);
                }
                int expectedOnDiskSizeWithoutHeader = onDiskSizeWithHeader - this.hdrSize;
                int actualOnDiskSizeWithoutHeader = headerBuf.getInt(Header.ON_DISK_SIZE_WITHOUT_HEADER_INDEX);
                HFileBlock.validateOnDiskSizeWithoutHeader(expectedOnDiskSizeWithoutHeader, actualOnDiskSizeWithoutHeader, headerBuf, offset);
            } else {
                if (headerBuf == null) {
                    headerBuf = ByteBuffer.allocate(this.hdrSize);
                    this.readAtOffset(is, headerBuf.array(), headerBuf.arrayOffset(), this.hdrSize, false, offset, pread);
                }
                int onDiskSizeWithoutHeader = headerBuf.getInt(Header.ON_DISK_SIZE_WITHOUT_HEADER_INDEX);
                onDiskSizeWithHeader = onDiskSizeWithoutHeader + this.hdrSize;
                onDiskBlock = new byte[onDiskSizeWithHeader + this.hdrSize];
                System.arraycopy(headerBuf.array(), headerBuf.arrayOffset(), onDiskBlock, 0, this.hdrSize);
                nextBlockOnDiskSize = this.readAtOffset(is, onDiskBlock, this.hdrSize, onDiskSizeWithHeader - this.hdrSize, true, offset + (long)this.hdrSize, pread);
            }
            ByteBuffer onDiskBlockByteBuffer = ByteBuffer.wrap(onDiskBlock, 0, onDiskSizeWithHeader);
            if (!this.fileContext.isCompressedOrEncrypted()) {
                HFileBlock.verifyUncompressed(headerBuf, this.fileContext.isUseHBaseChecksum());
            }
            if (verifyChecksum && !this.validateChecksum(offset, onDiskBlockByteBuffer, this.hdrSize)) {
                return null;
            }
            HFileBlock b = new HFileBlock(onDiskBlockByteBuffer, this.fileContext.isUseHBaseChecksum());
            b.nextBlockOnDiskSizeWithHeader = nextBlockOnDiskSize;
            if (b.hasNextBlockHeader()) {
                prefetchedHeader.offset = offset + (long)b.getOnDiskSizeWithHeader();
                System.arraycopy(onDiskBlock, onDiskSizeWithHeader, prefetchedHeader.header, 0, this.hdrSize);
            }
            b.offset = offset;
            b.fileContext.setIncludesTags(this.fileContext.isIncludesTags());
            b.fileContext.setIncludesMvcc(this.fileContext.isIncludesMvcc());
            return b;
        }

        void setIncludesMemstoreTS(boolean includesMemstoreTS) {
            this.fileContext.setIncludesMvcc(includesMemstoreTS);
        }

        void setDataBlockEncoder(HFileDataBlockEncoder encoder) {
            this.encodedBlockDecodingCtx = encoder.newDataBlockDecodingContext(this.fileContext);
        }

        @Override
        public HFileBlockDecodingContext getBlockDecodingContext() {
            return this.encodedBlockDecodingCtx;
        }

        @Override
        public HFileBlockDecodingContext getDefaultBlockDecodingContext() {
            return this.defaultDecodingCtx;
        }

        protected boolean validateChecksum(long offset, ByteBuffer data, int hdrSize) throws IOException {
            if (!this.fileContext.isUseHBaseChecksum()) {
                return false;
            }
            return ChecksumUtil.validateChecksum(data, this.path, offset, hdrSize);
        }

        @Override
        public void closeStreams() throws IOException {
            this.streamWrapper.close();
        }

        @Override
        public void unbufferStream() {
            if (this.streamLock.tryLock()) {
                try {
                    this.streamWrapper.unbuffer();
                }
                finally {
                    this.streamLock.unlock();
                }
            }
        }

        public String toString() {
            return "hfs=" + (Object)((Object)this.hfs) + ", path=" + this.path + ", fileContext=" + this.fileContext;
        }
    }

    private static class PrefetchedHeader {
        long offset = -1L;
        byte[] header = new byte[33];
        final ByteBuffer buf = ByteBuffer.wrap(this.header, 0, 33);

        private PrefetchedHeader() {
        }
    }

    private static abstract class AbstractFSReader
    implements FSReader {
        protected long fileSize;
        protected final int hdrSize;
        protected HFileSystem hfs;
        protected Path path;
        protected final Lock streamLock = new ReentrantLock();
        public static final int DEFAULT_BUFFER_SIZE = 0x100000;
        protected HFileContext fileContext;

        public AbstractFSReader(long fileSize, HFileSystem hfs, Path path, HFileContext fileContext) throws IOException {
            this.fileSize = fileSize;
            this.hfs = hfs;
            this.path = path;
            this.fileContext = fileContext;
            this.hdrSize = HFileBlock.headerSize(fileContext.isUseHBaseChecksum());
        }

        @Override
        public BlockIterator blockRange(final long startOffset, final long endOffset) {
            final AbstractFSReader owner = this;
            return new BlockIterator(){
                private long offset;
                {
                    this.offset = startOffset;
                }

                @Override
                public HFileBlock nextBlock() throws IOException {
                    if (this.offset >= endOffset) {
                        return null;
                    }
                    HFileBlock b = this.readBlockData(this.offset, -1L, -1, false);
                    this.offset += (long)b.getOnDiskSizeWithHeader();
                    return b.unpack(fileContext, owner);
                }

                @Override
                public HFileBlock nextBlockWithBlockType(BlockType blockType) throws IOException {
                    HFileBlock blk = this.nextBlock();
                    if (blk.getBlockType() != blockType) {
                        throw new IOException("Expected block of type " + (Object)((Object)blockType) + " but found " + (Object)((Object)blk.getBlockType()));
                    }
                    return blk;
                }
            };
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        protected int readAtOffset(FSDataInputStream istream, byte[] dest, int destOffset, int size, boolean peekIntoNextBlock, long fileOffset, boolean pread) throws IOException {
            block11: {
                int extraSize;
                if (peekIntoNextBlock && destOffset + size + this.hdrSize > dest.length) {
                    throw new IOException("Attempted to read " + size + " bytes and " + this.hdrSize + " bytes of next header into a " + dest.length + "-byte array at offset " + destOffset);
                }
                if (!pread && this.streamLock.tryLock()) {
                    try {
                        istream.seek(fileOffset);
                        long realOffset = istream.getPos();
                        if (realOffset != fileOffset) {
                            throw new IOException("Tried to seek to " + fileOffset + " to read " + size + " bytes, but pos=" + realOffset + " after seek");
                        }
                        if (!peekIntoNextBlock) {
                            IOUtils.readFully((InputStream)istream, (byte[])dest, (int)destOffset, (int)size);
                            int n = -1;
                            return n;
                        }
                        if (!HFileBlock.readWithExtra((InputStream)istream, dest, destOffset, size, this.hdrSize)) {
                            int n = -1;
                            return n;
                        }
                        break block11;
                    }
                    finally {
                        this.streamLock.unlock();
                    }
                }
                int n = extraSize = peekIntoNextBlock ? this.hdrSize : 0;
                if (!HFileBlock.positionalReadWithExtra(istream, fileOffset, dest, destOffset, size, extraSize)) {
                    return -1;
                }
            }
            assert (peekIntoNextBlock);
            return Bytes.toInt(dest, destOffset + size + 8) + this.hdrSize;
        }
    }

    public static interface FSReader {
        public HFileBlock readBlockData(long var1, long var3, int var5, boolean var6) throws IOException;

        public BlockIterator blockRange(long var1, long var3);

        public void closeStreams() throws IOException;

        public HFileBlockDecodingContext getBlockDecodingContext();

        public HFileBlockDecodingContext getDefaultBlockDecodingContext();

        public void unbufferStream();
    }

    public static interface BlockIterator {
        public HFileBlock nextBlock() throws IOException;

        public HFileBlock nextBlockWithBlockType(BlockType var1) throws IOException;
    }

    public static interface BlockWritable {
        public BlockType getBlockType();

        public void writeToBlock(DataOutput var1) throws IOException;
    }

    public static class Writer {
        private State state = State.INIT;
        private final HFileDataBlockEncoder dataBlockEncoder;
        private HFileBlockEncodingContext dataBlockEncodingCtx;
        private HFileBlockDefaultEncodingContext defaultBlockEncodingCtx;
        private ByteArrayOutputStream baosInMemory;
        private BlockType blockType;
        private DataOutputStream userDataStream;
        private int unencodedDataSizeWritten;
        private byte[] onDiskBytesWithHeader;
        private byte[] onDiskChecksum;
        private byte[] uncompressedBytesWithHeader;
        private long startOffset;
        private long[] prevOffsetByType;
        private long prevOffset;
        private HFileContext fileContext;

        public Writer(HFileDataBlockEncoder dataBlockEncoder, HFileContext fileContext) {
            this.dataBlockEncoder = dataBlockEncoder != null ? dataBlockEncoder : NoOpDataBlockEncoder.INSTANCE;
            this.defaultBlockEncodingCtx = new HFileBlockDefaultEncodingContext(null, HConstants.HFILEBLOCK_DUMMY_HEADER, fileContext);
            this.dataBlockEncodingCtx = this.dataBlockEncoder.newDataBlockEncodingContext(HConstants.HFILEBLOCK_DUMMY_HEADER, fileContext);
            if (fileContext.getBytesPerChecksum() < 33) {
                throw new RuntimeException("Unsupported value of bytesPerChecksum.  Minimum is 33 but the configured value is " + fileContext.getBytesPerChecksum());
            }
            this.baosInMemory = new ByteArrayOutputStream();
            this.prevOffsetByType = new long[BlockType.values().length];
            for (int i = 0; i < this.prevOffsetByType.length; ++i) {
                this.prevOffsetByType[i] = -1L;
            }
            this.fileContext = fileContext;
        }

        public DataOutputStream startWriting(BlockType newBlockType) throws IOException {
            if (this.state == State.BLOCK_READY && this.startOffset != -1L) {
                this.prevOffsetByType[this.blockType.getId()] = this.startOffset;
            }
            this.startOffset = -1L;
            this.blockType = newBlockType;
            this.baosInMemory.reset();
            this.baosInMemory.write(HConstants.HFILEBLOCK_DUMMY_HEADER);
            this.state = State.WRITING;
            this.userDataStream = new DataOutputStream(this.baosInMemory);
            if (newBlockType == BlockType.DATA) {
                this.dataBlockEncoder.startBlockEncoding(this.dataBlockEncodingCtx, this.userDataStream);
            }
            this.unencodedDataSizeWritten = 0;
            return this.userDataStream;
        }

        public void write(Cell cell) throws IOException {
            this.expectState(State.WRITING);
            this.unencodedDataSizeWritten += this.dataBlockEncoder.encode(cell, this.dataBlockEncodingCtx, this.userDataStream);
        }

        DataOutputStream getUserDataStream() {
            this.expectState(State.WRITING);
            return this.userDataStream;
        }

        void ensureBlockReady() throws IOException {
            Preconditions.checkState(this.state != State.INIT, "Unexpected state: " + (Object)((Object)this.state));
            if (this.state == State.BLOCK_READY) {
                return;
            }
            this.finishBlock();
        }

        private void finishBlock() throws IOException {
            if (this.blockType == BlockType.DATA) {
                BufferGrabbingByteArrayOutputStream baosInMemoryCopy = new BufferGrabbingByteArrayOutputStream();
                this.baosInMemory.writeTo(baosInMemoryCopy);
                this.dataBlockEncoder.endBlockEncoding(this.dataBlockEncodingCtx, this.userDataStream, baosInMemoryCopy.buf, this.blockType);
                this.blockType = this.dataBlockEncodingCtx.getBlockType();
            }
            this.userDataStream.flush();
            this.uncompressedBytesWithHeader = this.baosInMemory.toByteArray();
            this.prevOffset = this.prevOffsetByType[this.blockType.getId()];
            this.state = State.BLOCK_READY;
            this.onDiskBytesWithHeader = this.blockType == BlockType.DATA || this.blockType == BlockType.ENCODED_DATA ? this.dataBlockEncodingCtx.compressAndEncrypt(this.uncompressedBytesWithHeader) : this.defaultBlockEncodingCtx.compressAndEncrypt(this.uncompressedBytesWithHeader);
            int numBytes = (int)ChecksumUtil.numBytes(this.onDiskBytesWithHeader.length, this.fileContext.getBytesPerChecksum());
            this.putHeader(this.onDiskBytesWithHeader, 0, this.onDiskBytesWithHeader.length + numBytes, this.uncompressedBytesWithHeader.length, this.onDiskBytesWithHeader.length);
            this.putHeader(this.uncompressedBytesWithHeader, 0, this.onDiskBytesWithHeader.length + numBytes, this.uncompressedBytesWithHeader.length, this.onDiskBytesWithHeader.length);
            this.onDiskChecksum = new byte[numBytes];
            ChecksumUtil.generateChecksums(this.onDiskBytesWithHeader, 0, this.onDiskBytesWithHeader.length, this.onDiskChecksum, 0, this.fileContext.getChecksumType(), this.fileContext.getBytesPerChecksum());
        }

        private void putHeader(byte[] dest, int offset, int onDiskSize, int uncompressedSize, int onDiskDataSize) {
            offset = this.blockType.put(dest, offset);
            offset = Bytes.putInt(dest, offset, onDiskSize - 33);
            offset = Bytes.putInt(dest, offset, uncompressedSize - 33);
            offset = Bytes.putLong(dest, offset, this.prevOffset);
            offset = Bytes.putByte(dest, offset, this.fileContext.getChecksumType().getCode());
            offset = Bytes.putInt(dest, offset, this.fileContext.getBytesPerChecksum());
            Bytes.putInt(dest, offset, onDiskDataSize);
        }

        public void writeHeaderAndData(FSDataOutputStream out) throws IOException {
            long offset = out.getPos();
            if (this.startOffset != -1L && offset != this.startOffset) {
                throw new IOException("A " + (Object)((Object)this.blockType) + " block written to a stream twice, first at offset " + this.startOffset + ", then at " + offset);
            }
            this.startOffset = offset;
            this.finishBlockAndWriteHeaderAndData((DataOutputStream)out);
        }

        protected void finishBlockAndWriteHeaderAndData(DataOutputStream out) throws IOException {
            this.ensureBlockReady();
            out.write(this.onDiskBytesWithHeader);
            out.write(this.onDiskChecksum);
        }

        byte[] getHeaderAndDataForTest() throws IOException {
            this.ensureBlockReady();
            byte[] output = new byte[this.onDiskBytesWithHeader.length + this.onDiskChecksum.length];
            System.arraycopy(this.onDiskBytesWithHeader, 0, output, 0, this.onDiskBytesWithHeader.length);
            System.arraycopy(this.onDiskChecksum, 0, output, this.onDiskBytesWithHeader.length, this.onDiskChecksum.length);
            return output;
        }

        public void release() {
            if (this.dataBlockEncodingCtx != null) {
                this.dataBlockEncodingCtx.close();
                this.dataBlockEncodingCtx = null;
            }
            if (this.defaultBlockEncodingCtx != null) {
                this.defaultBlockEncodingCtx.close();
                this.defaultBlockEncodingCtx = null;
            }
        }

        int getOnDiskSizeWithoutHeader() {
            this.expectState(State.BLOCK_READY);
            return this.onDiskBytesWithHeader.length + this.onDiskChecksum.length - 33;
        }

        int getOnDiskSizeWithHeader() {
            this.expectState(State.BLOCK_READY);
            return this.onDiskBytesWithHeader.length + this.onDiskChecksum.length;
        }

        int getUncompressedSizeWithoutHeader() {
            this.expectState(State.BLOCK_READY);
            return this.uncompressedBytesWithHeader.length - 33;
        }

        int getUncompressedSizeWithHeader() {
            this.expectState(State.BLOCK_READY);
            return this.uncompressedBytesWithHeader.length;
        }

        public boolean isWriting() {
            return this.state == State.WRITING;
        }

        public int blockSizeWritten() {
            if (this.state != State.WRITING) {
                return 0;
            }
            return this.unencodedDataSizeWritten;
        }

        ByteBuffer getUncompressedBufferWithHeader() {
            this.expectState(State.BLOCK_READY);
            return ByteBuffer.wrap(this.uncompressedBytesWithHeader);
        }

        ByteBuffer getOnDiskBufferWithHeader() {
            this.expectState(State.BLOCK_READY);
            return ByteBuffer.wrap(this.onDiskBytesWithHeader);
        }

        private void expectState(State expectedState) {
            if (this.state != expectedState) {
                throw new IllegalStateException("Expected state: " + (Object)((Object)expectedState) + ", actual state: " + (Object)((Object)this.state));
            }
        }

        public void writeBlock(BlockWritable bw, FSDataOutputStream out) throws IOException {
            bw.writeToBlock(this.startWriting(bw.getBlockType()));
            this.writeHeaderAndData(out);
        }

        public HFileBlock getBlockForCaching(CacheConfig cacheConf) {
            HFileContext newContext = new HFileContextBuilder().withBlockSize(this.fileContext.getBlocksize()).withBytesPerCheckSum(0).withChecksumType(ChecksumType.NULL).withCompression(this.fileContext.getCompression()).withDataBlockEncoding(this.fileContext.getDataBlockEncoding()).withHBaseCheckSum(this.fileContext.isUseHBaseChecksum()).withCompressTags(this.fileContext.isCompressTags()).withIncludesMvcc(this.fileContext.isIncludesMvcc()).withIncludesTags(this.fileContext.isIncludesTags()).build();
            return new HFileBlock(this.blockType, this.getOnDiskSizeWithoutHeader(), this.getUncompressedSizeWithoutHeader(), this.prevOffset, cacheConf.shouldCacheCompressed(this.blockType.getCategory()) ? this.getOnDiskBufferWithHeader() : this.getUncompressedBufferWithHeader(), true, this.startOffset, this.onDiskBytesWithHeader.length + this.onDiskChecksum.length, newContext);
        }

        public static class BufferGrabbingByteArrayOutputStream
        extends ByteArrayOutputStream {
            private byte[] buf;

            @Override
            public void write(byte[] b, int off, int len) {
                this.buf = b;
            }

            public byte[] getBuffer() {
                return this.buf;
            }
        }

        private static enum State {
            INIT,
            WRITING,
            BLOCK_READY;

        }
    }

    static class Header {
        static int BLOCK_MAGIC_INDEX = 0;
        static int ON_DISK_SIZE_WITHOUT_HEADER_INDEX = 8;
        static int UNCOMPRESSED_SIZE_WITHOUT_HEADER_INDEX = 12;
        static int PREV_BLOCK_OFFSET_INDEX = 16;
        static int CHECKSUM_TYPE_INDEX = 24;
        static int BYTES_PER_CHECKSUM_INDEX = 25;
        static int ON_DISK_DATA_SIZE_WITH_HEADER_INDEX = 29;

        Header() {
        }
    }
}

