/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.io.hfile;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.Map;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.HBaseConfiguration;
import org.apache.hadoop.hbase.HBaseTestingUtility;
import org.apache.hadoop.hbase.io.hfile.BlockCache;
import org.apache.hadoop.hbase.io.hfile.BlockCacheKey;
import org.apache.hadoop.hbase.io.hfile.BlockType;
import org.apache.hadoop.hbase.io.hfile.CacheConfig;
import org.apache.hadoop.hbase.io.hfile.Cacheable;
import org.apache.hadoop.hbase.io.hfile.CacheableDeserializer;
import org.apache.hadoop.hbase.io.hfile.CacheableDeserializerIdManager;
import org.apache.hadoop.hbase.io.hfile.CombinedBlockCache;
import org.apache.hadoop.hbase.io.hfile.LruBlockCache;
import org.apache.hadoop.hbase.io.hfile.bucket.BucketCache;
import org.apache.hadoop.hbase.io.util.HeapMemorySizeUtil;
import org.apache.hadoop.hbase.testclassification.LargeTests;
import org.apache.hadoop.hbase.util.Threads;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.junit.experimental.categories.Category;

@Category(value={LargeTests.class})
public class TestCacheConfig {
    private static final Log LOG = LogFactory.getLog(TestCacheConfig.class);
    private Configuration conf;

    @Before
    public void setUp() throws Exception {
        CacheConfig.GLOBAL_BLOCK_CACHE_INSTANCE = null;
        this.conf = HBaseConfiguration.create();
    }

    @After
    public void tearDown() throws Exception {
        CacheConfig.GLOBAL_BLOCK_CACHE_INSTANCE = null;
    }

    void basicBlockCacheOps(CacheConfig cc, boolean doubling, boolean sizing) {
        Assert.assertTrue((boolean)cc.isBlockCacheEnabled());
        Assert.assertTrue((false == cc.isInMemory() ? 1 : 0) != 0);
        BlockCache bc = cc.getBlockCache();
        BlockCacheKey bck = new BlockCacheKey("f", 0L);
        DataCacheEntry c = new DataCacheEntry();
        long initialBlockCount = bc.getBlockCount();
        bc.cacheBlock(bck, c, cc.isInMemory(), cc.isCacheDataInL1());
        Assert.assertEquals((long)(doubling ? 2L : 1L), (long)(bc.getBlockCount() - initialBlockCount));
        bc.evictBlock(bck);
        Assert.assertEquals((long)initialBlockCount, (long)bc.getBlockCount());
        if (sizing) {
            long originalSize = bc.getCurrentSize();
            bc.cacheBlock(bck, c, cc.isInMemory(), cc.isCacheDataInL1());
            Assert.assertTrue((bc.getCurrentSize() > originalSize ? 1 : 0) != 0);
            bc.evictBlock(bck);
            long size = bc.getCurrentSize();
            Assert.assertEquals((long)originalSize, (long)size);
        }
    }

    private long cacheDataBlock(CacheConfig cc, String filename) {
        BlockCacheKey bck = new BlockCacheKey(filename, 0L);
        DataCacheEntry c = new DataCacheEntry();
        cc.getBlockCache().cacheBlock(bck, c, cc.isInMemory(), cc.isCacheDataInL1());
        return cc.getBlockCache().getBlockCount();
    }

    @Test
    public void testCacheConfigDefaultLRUBlockCache() {
        CacheConfig cc = new CacheConfig(this.conf);
        Assert.assertTrue((boolean)cc.isBlockCacheEnabled());
        Assert.assertTrue((false == cc.isInMemory() ? 1 : 0) != 0);
        this.basicBlockCacheOps(cc, false, true);
        Assert.assertTrue((boolean)(cc.getBlockCache() instanceof LruBlockCache));
    }

    @Test
    public void testOffHeapBucketCacheConfig() {
        this.conf.set("hbase.bucketcache.ioengine", "offheap");
        this.doBucketCacheConfigTest();
    }

    @Test
    public void testOnHeapBucketCacheConfig() {
        this.conf.set("hbase.bucketcache.ioengine", "heap");
        this.doBucketCacheConfigTest();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testFileBucketCacheConfig() throws IOException {
        HBaseTestingUtility htu = new HBaseTestingUtility(this.conf);
        try {
            Path p = new Path(htu.getDataTestDir(), "bc.txt");
            FileSystem fs = FileSystem.get((Configuration)this.conf);
            fs.create(p).close();
            this.conf.set("hbase.bucketcache.ioengine", "file:" + p);
            this.doBucketCacheConfigTest();
        }
        finally {
            htu.cleanupTestDir();
        }
    }

    private void doBucketCacheConfigTest() {
        int bcSize = 100;
        this.conf.setInt("hbase.bucketcache.size", 100);
        CacheConfig cc = new CacheConfig(this.conf);
        this.basicBlockCacheOps(cc, false, false);
        Assert.assertTrue((boolean)(cc.getBlockCache() instanceof CombinedBlockCache));
        CombinedBlockCache cbc = (CombinedBlockCache)cc.getBlockCache();
        BlockCache[] bcs = cbc.getBlockCaches();
        Assert.assertTrue((boolean)(bcs[0] instanceof LruBlockCache));
        LruBlockCache lbc = (LruBlockCache)bcs[0];
        Assert.assertEquals((long)HeapMemorySizeUtil.getLruCacheSize(this.conf), (long)lbc.getMaxSize());
        Assert.assertTrue((boolean)(bcs[1] instanceof BucketCache));
        BucketCache bc = (BucketCache)bcs[1];
        Assert.assertEquals((long)100L, (long)(bc.getMaxSize() / 0x100000L));
    }

    @Test(timeout=10000L)
    public void testBucketCacheConfigL1L2Setup() {
        this.conf.set("hbase.bucketcache.ioengine", "offheap");
        this.conf.setFloat("hfile.block.cache.size", 0.001f);
        long lruExpectedSize = HeapMemorySizeUtil.getLruCacheSize(this.conf);
        int bcSize = 100;
        long bcExpectedSize = 0x6400000L;
        Assert.assertTrue((lruExpectedSize < bcExpectedSize ? 1 : 0) != 0);
        this.conf.setInt("hbase.bucketcache.size", 100);
        this.conf.setBoolean("hbase.bucketcache.combinedcache.enabled", false);
        CacheConfig cc = new CacheConfig(this.conf);
        this.basicBlockCacheOps(cc, false, false);
        Assert.assertTrue((boolean)(cc.getBlockCache() instanceof LruBlockCache));
        LruBlockCache lbc = (LruBlockCache)cc.getBlockCache();
        Assert.assertEquals((long)lruExpectedSize, (long)lbc.getMaxSize());
        BlockCache bc = lbc.getVictimHandler();
        Assert.assertEquals((long)bcExpectedSize, (long)((BucketCache)bc).getMaxSize());
        long initialL1BlockCount = lbc.getBlockCount();
        long initialL2BlockCount = bc.getBlockCount();
        DataCacheEntry c = new DataCacheEntry();
        BlockCacheKey bck = new BlockCacheKey("bck", 0L);
        lbc.cacheBlock(bck, c, false, false);
        Assert.assertEquals((long)(initialL1BlockCount + 1L), (long)lbc.getBlockCount());
        Assert.assertEquals((long)initialL2BlockCount, (long)bc.getBlockCount());
        final long justTooBigSize = lbc.acceptableSize() + 1L;
        lbc.cacheBlock(new BlockCacheKey("bck2", 0L), new DataCacheEntry(){

            @Override
            public long heapSize() {
                return justTooBigSize;
            }

            @Override
            public int getSerializedLength() {
                return (int)this.heapSize();
            }
        });
        while (initialL1BlockCount != lbc.getBlockCount()) {
            Threads.sleep(10L);
        }
        Assert.assertEquals((long)initialL1BlockCount, (long)lbc.getBlockCount());
        long count = bc.getBlockCount();
        Assert.assertTrue((initialL2BlockCount + 1L <= count ? 1 : 0) != 0);
    }

    @Test
    public void testCacheDataInL1() {
        this.conf.set("hbase.bucketcache.ioengine", "offheap");
        this.conf.setInt("hbase.bucketcache.size", 100);
        CacheConfig cc = new CacheConfig(this.conf);
        Assert.assertTrue((boolean)(cc.getBlockCache() instanceof CombinedBlockCache));
        CombinedBlockCache cbc = (CombinedBlockCache)cc.getBlockCache();
        this.cacheDataBlock(cc, "1");
        LruBlockCache lrubc = (LruBlockCache)cbc.getBlockCaches()[0];
        this.assertDataBlockCount(lrubc, 0);
        cc.setCacheDataInL1(true);
        this.cacheDataBlock(cc, "2");
        this.assertDataBlockCount(lrubc, 1);
        cc.setCacheDataInL1(false);
        this.cacheDataBlock(cc, "3");
        this.assertDataBlockCount(lrubc, 1);
    }

    private void assertDataBlockCount(LruBlockCache bc, int expected) {
        Map<BlockType, Integer> blocks = bc.getBlockTypeCountsForTest();
        Assert.assertEquals((long)expected, (long)(blocks == null ? 0L : (blocks.get((Object)BlockType.DATA) == null ? 0L : (long)blocks.get((Object)BlockType.DATA).intValue())));
    }

    static class MetaCacheEntry
    extends DataCacheEntry {
        MetaCacheEntry() {
        }

        @Override
        public BlockType getBlockType() {
            return BlockType.INTERMEDIATE_INDEX;
        }
    }

    static class DataCacheEntry
    implements Cacheable {
        private static final int SIZE = 1;
        private static DataCacheEntry SINGLETON = new DataCacheEntry();
        final CacheableDeserializer<Cacheable> deserializer;

        DataCacheEntry() {
            this(SINGLETON);
        }

        DataCacheEntry(Cacheable c) {
            this.deserializer = new Deserializer(c);
        }

        public String toString() {
            return "size=1, type=" + (Object)((Object)this.getBlockType());
        }

        @Override
        public long heapSize() {
            return 1L;
        }

        @Override
        public int getSerializedLength() {
            return 1;
        }

        @Override
        public void serialize(ByteBuffer destination) {
            LOG.info((Object)("Serialized " + this + " to " + destination));
        }

        @Override
        public CacheableDeserializer<Cacheable> getDeserializer() {
            return this.deserializer;
        }

        @Override
        public BlockType getBlockType() {
            return BlockType.DATA;
        }
    }

    static class IndexCacheEntry
    extends DataCacheEntry {
        private static IndexCacheEntry SINGLETON = new IndexCacheEntry();

        public IndexCacheEntry() {
            super(SINGLETON);
        }

        @Override
        public BlockType getBlockType() {
            return BlockType.ROOT_INDEX;
        }
    }

    static class Deserializer
    implements CacheableDeserializer<Cacheable> {
        private final Cacheable cacheable;
        private int deserializedIdentifier = CacheableDeserializerIdManager.registerDeserializer(this);

        Deserializer(Cacheable c) {
            this.cacheable = c;
        }

        @Override
        public int getDeserialiserIdentifier() {
            return this.deserializedIdentifier;
        }

        @Override
        public Cacheable deserialize(ByteBuffer b, boolean reuse) throws IOException {
            LOG.info((Object)("Deserialized " + b + ", reuse=" + reuse));
            return this.cacheable;
        }

        @Override
        public Cacheable deserialize(ByteBuffer b) throws IOException {
            LOG.info((Object)("Deserialized " + b));
            return this.cacheable;
        }
    }
}

