/**
 * Copyright 2019 Confluent Inc.
 *
 * <p>Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at</p>
 *
 * <p>http://www.apache.org/licenses/LICENSE-2.0</p>
 *
 * <p>Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.</p>
 */

package io.confluent.rest.auth;

import com.google.protobuf.InvalidProtocolBufferException;
import com.mapr.fs.proto.Security;
import com.mapr.security.MutableInt;
import io.confluent.rest.impersonation.Errors;

import javax.ws.rs.container.ContainerRequestContext;
import javax.ws.rs.core.Cookie;
import javax.ws.rs.core.HttpHeaders;
import java.io.IOException;
import java.net.HttpCookie;
import java.nio.charset.StandardCharsets;
import java.util.Base64;
import java.util.Collections;
import java.util.Optional;

import static org.apache.commons.lang3.StringUtils.removeStart;
import static org.apache.commons.lang3.StringUtils.substringAfter;
import static org.apache.commons.lang3.StringUtils.substringBefore;

public class MaprAuthenticationUtils {
  private static final String MAPR_NEGOTIATE = "MAPR-Negotiate";
  private static final String BASIC = "Basic";
  private static final String HADOOP_AUTH_COOKIE = "hadoop.auth";

  public static String getUserNameFromRequestContext(ContainerRequestContext request) {
    String authorization = request.getHeaderString(HttpHeaders.AUTHORIZATION);
    return findUserNameFromAuthentication(authorization)
        .orElseGet(() -> findUserNameFromRequestCookie(request)
            .orElseThrow(MaprAuthenticationUtils::onMissingUsername));
  }

  private static Optional<String> findUserNameFromRequestCookie(ContainerRequestContext request) {
    return Optional.ofNullable(request.getCookies())
        .map(cookies -> cookies.get(HADOOP_AUTH_COOKIE))
        .map(Cookie::getValue)
        .flatMap(MaprAuthenticationUtils::findUserNameFromHadoopAuthCookieValue);
  }

  public static String getUserNameFromAuthenticationOrCookie(String auth, String cookie) {
    return findUserNameFromAuthentication(auth)
        .orElseGet(() -> findUserNameFromCookie(cookie)
            .orElseThrow(MaprAuthenticationUtils::onMissingUsername));
  }

  private static RuntimeException onMissingUsername() {
    return Errors.serverLoginException(new IOException("Username is unavailable"));
  }

  public static Optional<String> findUserNameFromAuthentication(String auth) {
    String type = substringBefore(auth, " ");
    String value = substringAfter(auth, " ");
    if (BASIC.equals(type)) {
      return Optional.ofNullable(getUserNameFromBasicAuthentication(value.trim()));
    } else if (MAPR_NEGOTIATE.equals(type)) {
      return Optional.ofNullable(getUserNameFromMaprTicketAuthentication(value.trim()));
    }
    return Optional.empty();
  }

  private static String getUserNameFromBasicAuthentication(String base64Credentials) {
    Base64.Decoder decoder = Base64.getDecoder();
    String credentials = new String(decoder.decode(base64Credentials), StandardCharsets.UTF_8);
    return substringBefore(credentials, ":");
  }

  private static String getUserNameFromMaprTicketAuthentication(String authRequest) {
    try {
      byte[] base64decoded = org.apache.commons.codec.binary.Base64.decodeBase64(authRequest);
      Security.AuthenticationReqFull req = Security.AuthenticationReqFull.parseFrom(base64decoded);
      if (req != null && req.hasEncryptedTicket()) {
        byte[] encryptedTicket = req.getEncryptedTicket().toByteArray();
        MutableInt err = new MutableInt();
        Security.Ticket decryptedTicket =
                com.mapr.security.Security.DecryptTicket(encryptedTicket, err);
        if (err.GetValue() == 0 && decryptedTicket != null) {
          Security.CredentialsMsg userCreds = decryptedTicket.getUserCreds();
          return userCreds.getUserName();
        } else {
          String decryptError = "Error while decrypting ticket and key " + err.GetValue();
          throw Errors.maprTicketDecryptException(decryptError);
        }
      } else {
        String clientRequestError = "Malformed client request";
        throw Errors.maprTicketDecryptException(clientRequestError);
      }
    } catch (InvalidProtocolBufferException e) {
      throw Errors.maprTicketDecryptException("Bad server key", e);
    }
  }

  public static Optional<String> findUserNameFromCookie(String cookie) {
    return Optional.ofNullable(cookie)
        .map(HttpCookie::parse)
        .orElse(Collections.emptyList())
        .stream()
        .filter(httpCookie -> httpCookie.getName().equals(HADOOP_AUTH_COOKIE))
        .findFirst()
        .map(HttpCookie::getValue)
        .flatMap(MaprAuthenticationUtils::findUserNameFromHadoopAuthCookieValue);
  }

  private static Optional<String> findUserNameFromHadoopAuthCookieValue(String value) {
    String[] parameters = value.split("&");
    for (String parameter : parameters) {
      if (parameter.startsWith("u=")) {
        return Optional.of(removeStart(parameter, "u="));
      }
    }
    return Optional.empty();
  }
}
