/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.rest;

import com.google.common.base.Preconditions;
import io.confluent.rest.Application;
import io.confluent.rest.ApplicationServer;
import io.confluent.rest.RestConfig;
import io.confluent.rest.TestRestConfig;
import java.net.URI;
import java.time.Duration;
import java.util.List;
import java.util.Properties;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import javax.ws.rs.Consumes;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.client.Client;
import javax.ws.rs.client.ClientBuilder;
import javax.ws.rs.client.Entity;
import javax.ws.rs.core.Configurable;
import javax.ws.rs.core.Form;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import org.hamcrest.MatcherAssert;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.DisplayName;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.TestInfo;

public class RateLimitNetworkTrafficListenerTest {
    private static TestRestConfig testConfig;
    private static ApplicationServer<TestRestConfig> server;
    private static final String TEST_MESSAGE = "Test message";
    private ScheduledExecutorService executor;
    private TestApp app;
    private Client client;

    @BeforeEach
    public void setup(TestInfo info) throws Exception {
        Properties props = new Properties();
        props.setProperty("listeners", "http://0.0.0.0:0");
        if (info.getDisplayName().contains("NetworkTrafficRateLimitEnabled")) {
            props.put("network.traffic.rate.limit.enable", "true");
            props.put("network.traffic.rate.limit.bytes.per.sec", (Object)10000);
            if (info.getDisplayName().contains("Resilience4j")) {
                props.put("network.traffic.rate.limit.backend", "Resilience4j");
            }
        }
        testConfig = TestRestConfig.maprCompatible(props);
        server = new ApplicationServer((RestConfig)testConfig);
        this.app = new TestApp("/app");
        server.registerApplication((Application)this.app);
        server.start();
        this.executor = Executors.newScheduledThreadPool(4);
        this.client = ClientBuilder.newClient();
    }

    @AfterEach
    public void tearDown() throws Exception {
        server.stop();
        server.join();
        this.client.close();
        this.awaitTerminationAfterShutdown(this.executor);
    }

    @Test
    @DisplayName(value="NetworkTrafficRateLimitDisabled")
    public void testNetworkTrafficRateLimitDisabled_unlimited() throws Exception {
        long startTime = System.nanoTime();
        this.hammerAtConstantRate(this.app.getServer().getURI(), "/resource", Duration.ofMillis(1L), 10, 1000);
        double durationMillis = (double)(System.nanoTime() - startTime) / 1000000.0;
        MatcherAssert.assertThat((String)"Duration must be greater than 1 second", (durationMillis >= 1000.0 ? 1 : 0) != 0);
        MatcherAssert.assertThat((String)"Duration must be smaller than 5 seconds", (durationMillis < 5000.0 ? 1 : 0) != 0);
    }

    @Test
    @DisplayName(value="NetworkTrafficRateLimitEnabled")
    public void testNetworkTrafficRateLimitEnabled_Guava_slowdown() throws Exception {
        long startTime = System.nanoTime();
        this.hammerAtConstantRate(this.app.getServer().getURI(), "/resource", Duration.ofMillis(1L), 10, 1000);
        double durationMillis = (double)(System.nanoTime() - startTime) / 1000000.0;
        MatcherAssert.assertThat((String)"Duration must be greater than 10 seconds", (durationMillis >= (double)Duration.ofSeconds(10L).toMillis() ? 1 : 0) != 0);
    }

    @Test
    @DisplayName(value="NetworkTrafficRateLimitEnabled_Resilience4j")
    public void testNetworkTrafficRateLimitEnabled_Resilience4j_slowdown() throws Exception {
        long startTime = System.nanoTime();
        this.hammerAtConstantRate(this.app.getServer().getURI(), "/resource", Duration.ofMillis(1L), 10, 1000);
        double durationMillis = (double)(System.nanoTime() - startTime) / 1000000.0;
        MatcherAssert.assertThat((String)"Duration must be greater than 10 seconds", (durationMillis >= (double)Duration.ofSeconds(10L).toMillis() ? 1 : 0) != 0);
    }

    private int hammerAtConstantRate(URI server, String path, Duration rate, int warmupRequests, int totalRequests) {
        Preconditions.checkArgument((!rate.isNegative() ? 1 : 0) != 0, (Object)"rate must be non-negative");
        Preconditions.checkArgument((warmupRequests <= totalRequests ? 1 : 0) != 0, (Object)"warmupRequests must be at most totalRequests");
        List responses = IntStream.range(0, totalRequests).mapToObj(i -> this.executor.schedule(() -> this.client.target(server).path(path).request(new MediaType[]{MediaType.APPLICATION_FORM_URLENCODED_TYPE}).post(Entity.form((Form)new Form("message", TEST_MESSAGE))), (long)i * rate.toMillis(), TimeUnit.MILLISECONDS)).collect(Collectors.toList()).stream().map(future -> {
            try {
                return (Response)future.get();
            }
            catch (InterruptedException | ExecutionException e) {
                throw new RuntimeException(e);
            }
        }).collect(Collectors.toList());
        for (Response response2 : responses) {
            int status = response2.getStatus();
            if (status == 200 || status == 429) continue;
            Assertions.fail((String)String.format("Expected HTTP 200 or HTTP 429, but got HTTP %d instead: %s", status, response2.readEntity(String.class)));
        }
        return (int)responses.subList(warmupRequests, responses.size()).stream().filter(response -> response.getStatus() == Response.Status.OK.getStatusCode()).count();
    }

    private void awaitTerminationAfterShutdown(ExecutorService threadPool) {
        threadPool.shutdown();
        try {
            if (!threadPool.awaitTermination(60L, TimeUnit.SECONDS)) {
                threadPool.shutdownNow();
            }
        }
        catch (InterruptedException ex) {
            threadPool.shutdownNow();
            Thread.currentThread().interrupt();
        }
    }

    @Path(value="/")
    @Produces(value={"text/plain"})
    public static class RestResource {
        @POST
        @Path(value="/resource")
        @Consumes(value={"application/x-www-form-urlencoded"})
        @Produces(value={"*/*"})
        public String post() {
            return "Hello";
        }
    }

    private static class TestApp
    extends Application<TestRestConfig>
    implements AutoCloseable {
        TestApp(String path) {
            this(testConfig, path);
        }

        TestApp(TestRestConfig config, String path) {
            super((RestConfig)config, path);
        }

        public void setupResources(Configurable<?> config, TestRestConfig appConfig) {
            config.register(RestResource.class);
        }

        @Override
        public void close() throws Exception {
            this.stop();
        }
    }
}

