/*
 * Copyright 2022 Confluent Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.confluent.rest;

import java.nio.file.Path;
import java.nio.file.Paths;
import java.security.Security;
import java.util.List;
import java.util.Objects;

import org.apache.kafka.common.config.ConfigException;
import org.bouncycastle.jcajce.provider.BouncyCastleFipsProvider;
import org.bouncycastle.jsse.provider.BouncyCastleJsseProvider;
import org.conscrypt.OpenSSLProvider;
import org.eclipse.jetty.util.ssl.SslContextFactory;
import org.eclipse.jetty.util.ssl.SslContextFactory.Server;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class SslFactory {

  private static final Logger log = LoggerFactory.getLogger(SslFactory.class);

  private SslFactory() {}

  // CHECKSTYLE_RULES.OFF: CyclomaticComplexity|JavaNCSS|NPathComplexity
  public static SslContextFactory createSslContextFactory(SslConfig sslConfig) {
    SslContextFactory.Server sslContextFactory = new SslContextFactory.Server();

    if (!sslConfig.getKeyStoreType().isEmpty()) {
      sslContextFactory.setKeyStoreType(sslConfig.getKeyStoreType());
    } else {
      String clusterSslKeystoreType = KafkaRestSslPropertiesReader.getServerKeystoreType();
      if (clusterSslKeystoreType == null) {
        throw new ConfigException(RestConfig.SSL_KEYSTORE_TYPE_CONFIG);
      }
      sslContextFactory.setKeyStoreType(clusterSslKeystoreType);
      if (clusterSslKeystoreType.equalsIgnoreCase("BCFKS")) {
        Security.addProvider(new BouncyCastleFipsProvider());
        Security.addProvider(new BouncyCastleJsseProvider());
        sslContextFactory.setProvider(BouncyCastleJsseProvider.PROVIDER_NAME);
      }
    }
    if (!sslConfig.getKeyStorePath().isEmpty()) {
      sslContextFactory.setKeyStorePath(sslConfig.getKeyStorePath());
    } else {
      String clusterSslKeystoreLocation =
              KafkaRestSslPropertiesReader.getServerKeystoreLocation();
      if (clusterSslKeystoreLocation == null) {
        throw new ConfigException(RestConfig.SSL_KEYSTORE_LOCATION_CONFIG);
      }
      sslContextFactory.setKeyStorePath(clusterSslKeystoreLocation);
    }
    String sslKeystorePassword = sslConfig.getKeyStorePassword();
    if (!Objects.equals(sslKeystorePassword, "")) {
      sslContextFactory.setKeyStorePassword(sslKeystorePassword);
    } else {
      String clusterSslKeystorePassword =
              KafkaRestSslPropertiesReader.getServerKeystorePassword();
      if (Objects.equals(clusterSslKeystorePassword, "")) {
        throw new ConfigException(RestConfig.SSL_KEYSTORE_PASSWORD_CONFIG);
      }
      sslContextFactory.setKeyStorePassword(clusterSslKeystorePassword);
    }

    String sslKeyPassword = sslConfig.getKeyManagerPassword();
    if (!Objects.equals(sslKeyPassword, "")) {
      sslContextFactory.setKeyManagerPassword(sslKeyPassword);
    } else {
      String clusterSslKeyPassword = KafkaRestSslPropertiesReader.getServerKeyPassword();
      if (Objects.equals(clusterSslKeyPassword, "")) {
        throw new ConfigException(RestConfig.SSL_KEY_PASSWORD_CONFIG);
      }
      sslContextFactory.setKeyManagerPassword(clusterSslKeyPassword);
    }

    if (!sslConfig.getKeyManagerFactoryAlgorithm().isEmpty()) {
      sslContextFactory.setKeyManagerFactoryAlgorithm(sslConfig.getKeyManagerFactoryAlgorithm());
    }

    if (sslConfig.getReloadOnKeyStoreChange()) {
      Path watchLocation = Paths.get(sslConfig.getReloadOnKeyStoreChangePath());
      try {
        FileWatcher.onFileChange(watchLocation, () -> {
          // Need to reset the key store path for symbolic link case
          sslContextFactory.setKeyStorePath(sslConfig.getKeyStorePath());
          sslContextFactory.reload(scf -> {
            log.info("SSL cert auto reload begun: " + scf.getKeyStorePath());
          });
          log.info("SSL cert auto reload complete");
            }
        );
        log.info("Enabled SSL cert auto reload for: " + watchLocation);
      } catch (java.io.IOException e) {
        log.error("Cannot enable SSL cert auto reload", e);
      }
    }

    configureClientAuth(sslContextFactory, sslConfig);

    if (!sslConfig.getIncludeProtocols().isEmpty()) {
      sslContextFactory.setIncludeProtocols(
              sslConfig.getIncludeProtocols().toArray(new String[0]));
    }

    List<String> disabledProtocols = sslConfig.getExcludeProtocols();
    if (!disabledProtocols.isEmpty()) {
      sslContextFactory.setExcludeProtocols(disabledProtocols.toArray(new String[0]));
    }

    if (!sslConfig.getIncludeCipherSuites().isEmpty()) {
      sslContextFactory.setIncludeCipherSuites(
              sslConfig.getIncludeCipherSuites().toArray(new String[0]));
    }

    List<String> excludedCipherSuites = sslConfig.getExcludeCipherSuites();
    if (!excludedCipherSuites.isEmpty()) {
      sslContextFactory.setExcludeCipherSuites(excludedCipherSuites.toArray(new String[0]));
    }

    if (sslConfig.getEndpointIdentificationAlgorithm() != null) {
      sslContextFactory.setEndpointIdentificationAlgorithm(
              sslConfig.getEndpointIdentificationAlgorithm());
    }

    if (!sslConfig.getTrustStoreType().isEmpty()) {
      sslContextFactory.setTrustStoreType(sslConfig.getTrustStoreType());
    } else {
      String clusterSslTruststoreType = KafkaRestSslPropertiesReader.getServerTruststoreType();
      if (clusterSslTruststoreType == null) {
        throw new ConfigException(RestConfig.SSL_TRUSTSTORE_TYPE_CONFIG);
      }
      sslContextFactory.setTrustStoreType(clusterSslTruststoreType);
    }

    if (!sslConfig.getTrustStorePath().isEmpty()) {
      sslContextFactory.setTrustStorePath(sslConfig.getTrustStorePath());
      sslContextFactory.setTrustStorePassword(sslConfig.getTrustStorePassword());

      if (!sslConfig.getTrustManagerFactoryAlgorithm().isEmpty()) {
        sslContextFactory.setTrustManagerFactoryAlgorithm(
                sslConfig.getTrustManagerFactoryAlgorithm());
      }
    }

    sslContextFactory.setProtocol(sslConfig.getProtocol());
    if (!sslConfig.getProvider().isEmpty()) {
      configureSecurityProvider(sslContextFactory, sslConfig);
    }

    sslContextFactory.setRenegotiationAllowed(false);

    return sslContextFactory;
  }

  private static void configureClientAuth(
      SslContextFactory.Server sslContextFactory, SslConfig config) {
    switch (config.getClientAuth()) {
      case NEED:
        sslContextFactory.setNeedClientAuth(true);
        break;
      case WANT:
        sslContextFactory.setWantClientAuth(true);
        break;
      default:
    }
  }

  private static void configureSecurityProvider(Server sslContextFactory, SslConfig sslConfig) {
    sslContextFactory.setProvider(sslConfig.getProvider());
    if (SslConfig.TLS_CONSCRYPT.equalsIgnoreCase(sslConfig.getProvider())) {
      Security.addProvider(new OpenSSLProvider());
    }
  }
}
