/**
 * Copyright 2019 Confluent Inc.
 *
 * <p>Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at</p>
 *
 * <p>http://www.apache.org/licenses/LICENSE-2.0</p>
 *
 * <p>Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.</p>
 */

package io.confluent.rest.impersonation;

import io.confluent.rest.RestConfig;
import io.confluent.rest.auth.MaprAuthenticationUtils;

import java.security.PrivilegedAction;

public class ImpersonationUtils {
  private static final ImpersonatedExecutor EXECUTOR = new ImpersonatedExecutor();
  private static boolean isImpersonationEnabled;

  public static boolean isImpersonationEnabled() {
    return isImpersonationEnabled;
  }

  public static void initialize(RestConfig config) {
    isImpersonationEnabled = config.getBoolean(RestConfig.IMPERSONATION);
  }

  public static <T> T runAsUserIfImpersonationEnabled(PrivilegedAction<T> action,
                                                      String auth, String cookie) {
    if (isImpersonationEnabled) {
      return runAsUser(action, auth, cookie);
    } else {
      return action.run();
    }
  }

  public static ImpersonatedExecutor executor() {
    return EXECUTOR;
  }

  public static <T> T runAsUser(PrivilegedAction<T> action, String auth, String cookie) {
    String username = MaprAuthenticationUtils.getUserNameFromAuthenticationOrCookie(auth, cookie);
    return executor().runAs(username, action);
  }
}
