/*
 * Copyright 2018 Confluent Inc.
 *
 * Licensed under the Confluent Community License (the "License"); you may not use
 * this file except in compliance with the License.  You may obtain a copy of the
 * License at
 *
 * http://www.confluent.io/confluent-community-license
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OF ANY KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations under the License.
 */

package io.confluent.kafka.schemaregistry.rest.resources;

import io.confluent.kafka.schemaregistry.filter.RequirePermission;
import io.confluent.kafka.schemaregistry.filter.Permission;
import io.confluent.rest.impersonation.ImpersonationUtils;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.validation.constraints.NotNull;
import javax.ws.rs.Consumes;
import javax.ws.rs.HeaderParam;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.container.AsyncResponse;
import javax.ws.rs.container.Suspended;
import javax.ws.rs.core.HttpHeaders;

import io.confluent.kafka.schemaregistry.client.rest.Versions;
import io.confluent.kafka.schemaregistry.client.rest.entities.Schema;
import io.confluent.kafka.schemaregistry.client.rest.entities.requests.CompatibilityCheckResponse;
import io.confluent.kafka.schemaregistry.client.rest.entities.requests.RegisterSchemaRequest;
import io.confluent.kafka.schemaregistry.exceptions.InvalidSchemaException;
import io.confluent.kafka.schemaregistry.exceptions.InvalidVersionException;
import io.confluent.kafka.schemaregistry.exceptions.SchemaRegistryException;
import io.confluent.kafka.schemaregistry.exceptions.SchemaRegistryStoreException;
import io.confluent.kafka.schemaregistry.rest.VersionId;
import io.confluent.kafka.schemaregistry.rest.exceptions.Errors;
import io.confluent.kafka.schemaregistry.storage.KafkaSchemaRegistry;
import io.confluent.rest.annotations.PerformanceMetric;

@Path("/compatibility")
@Produces({Versions.SCHEMA_REGISTRY_V1_JSON_WEIGHTED,
           Versions.SCHEMA_REGISTRY_DEFAULT_JSON_WEIGHTED,
           Versions.JSON_WEIGHTED})
@Consumes({Versions.SCHEMA_REGISTRY_V1_JSON,
           Versions.SCHEMA_REGISTRY_DEFAULT_JSON,
           Versions.JSON, Versions.GENERIC_REQUEST})
public class CompatibilityResource {

  private static final Logger log = LoggerFactory.getLogger(CompatibilityResource.class);
  private final KafkaSchemaRegistry schemaRegistry;

  public CompatibilityResource(KafkaSchemaRegistry schemaRegistry) {
    this.schemaRegistry = schemaRegistry;
  }

  @POST
  @Path("/subjects/{subject: .+}/versions/{version}")
  @ApiOperation(value = "Test input schema against a particular version of a subject's schema for "
      + "compatibility.",
      notes = "the compatibility level applied for the check is the configured compatibility level "
          + "for the subject (http:get:: /config/(string: subject)). If this subject's "
          + "compatibility level was never changed, then the global compatibility level "
          + "applies (http:get:: /config).",
      response = CompatibilityCheckResponse.class
  )
  @ApiResponses(value = {
      @ApiResponse(code = 404, message = "Error code 40401 -- Subject not found\n"
          + "Error code 40402 -- Version not found"),
      @ApiResponse(code = 422, message = "Error code 42201 -- Invalid schema or schema type\n"
          + "Error code 42202 -- Invalid version"),
      @ApiResponse(code = 500, message = "Error code 50001 -- Error in the backend data store") })
  @PerformanceMetric("compatibility.subjects.versions.verify")
  @RequirePermission(Permission.READ)
  public void testCompatibilityBySubjectName(
      final @Suspended AsyncResponse asyncResponse,
      final @HeaderParam("Content-Type") String contentType,
      final @HeaderParam("Accept") String accept,
      @ApiParam(value = "Subject of the schema version against which compatibility is to be tested",
          required = true)@PathParam("subject") String subject,
      @ApiParam(value = "Version of the subject's schema against which compatibility is to be "
          + "tested. Valid values for versionId are between [1,2^31-1] or the string \"latest\"."
          + "\"latest\" checks compatibility of the input schema with the last registered schema "
          + "under the specified subject", required = true)@PathParam("version") String version,
      @ApiParam(value = "Schema", required = true)
      @NotNull RegisterSchemaRequest request,
      @HeaderParam(HttpHeaders.AUTHORIZATION) String auth,
      @HeaderParam(HttpHeaders.COOKIE) String cookie) {
    ImpersonationUtils.runAsUserIfImpersonationEnabled(() -> {
      testCompatibilityBySubjectName(asyncResponse, contentType, accept, subject, version,
              request);
      return null;
    }, auth, cookie);
  }

  private void testCompatibilityBySubjectName(AsyncResponse asyncResponse, String contentType,
                                        String accept, String subject,
                                        String version, RegisterSchemaRequest request) {
    log.info("Testing schema subject {} compatibility between existing version {} and "
             + "specified version {}, id {}, type {}",
             subject, version, request.getVersion(), request.getId(), request.getSchemaType());
    // returns true if posted schema is compatible with the specified version. "latest" is 
    // a special version
    boolean isCompatible = false;
    CompatibilityCheckResponse compatibilityCheckResponse = new CompatibilityCheckResponse();
    String errorMessage = "Error while retrieving list of all subjects";
    Schema schemaForSpecifiedVersion = null;
    VersionId versionId = parseVersionId(version);
    try {
      //Don't check compatibility against deleted schema
      schemaForSpecifiedVersion = schemaRegistry.get(subject, versionId.getVersionId(), false);
    } catch (InvalidVersionException e) {
      throw Errors.invalidVersionException(e.getMessage());
    } catch (SchemaRegistryException e) {
      throw Errors.storeException("Error while retrieving schema for subject "
                                  + subject + " and version "
                                  + versionId.getVersionId(), e);
    }
    registerWithError(subject, errorMessage);
    if (schemaForSpecifiedVersion == null) {
      if (versionId.isLatest()) {
        isCompatible = true;
        compatibilityCheckResponse.setIsCompatible(isCompatible);
        asyncResponse.resume(compatibilityCheckResponse);
      } else {
        throw Errors.versionNotFoundException(versionId.getVersionId());
      }
    } else {
      try {
        isCompatible = schemaRegistry.isCompatible(
            subject, new Schema(subject, request.getVersion(), request.getId(),
                request.getSchemaType(), request.getReferences(), request.getSchema()),
            schemaForSpecifiedVersion
        );
      } catch (InvalidSchemaException e) {
        throw Errors.invalidSchemaException("Invalid input schema " + request.getSchema(), e);
      } catch (SchemaRegistryStoreException e) {
        throw Errors.storeException(
            "Error while getting compatibility level for subject " + subject, e);
      } catch (SchemaRegistryException e) {
        throw Errors.schemaRegistryException(
            "Error while getting compatibility level for subject " + subject, e);
      }
      compatibilityCheckResponse.setIsCompatible(isCompatible);
      asyncResponse.resume(compatibilityCheckResponse);
    }
  }

  private static VersionId parseVersionId(String version) {
    final VersionId versionId;
    try {
      versionId = new VersionId(version);
    } catch (InvalidVersionException e) {
      throw Errors.invalidVersionException(e.getMessage());
    }
    return versionId;
  }

  private void registerWithError(final String subject, final String errorMessage) {
    try {
      if (!schemaRegistry.hasSubjects(subject, false)) {
        throw Errors.subjectNotFoundException(subject);
      }
    } catch (SchemaRegistryStoreException e) {
      throw Errors.storeException(errorMessage, e);
    } catch (SchemaRegistryException e) {
      throw Errors.schemaRegistryException(errorMessage, e);
    }
  }
}
