/*
 * Decompiled with CFR 0.152.
 */
package voldemort.store.readonly;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.util.ArrayList;
import java.util.List;
import org.apache.log4j.Logger;
import voldemort.VoldemortException;
import voldemort.store.readonly.ReadOnlyUtils;
import voldemort.utils.Utils;

public class ChunkedFileSet {
    private static Logger logger = Logger.getLogger(ChunkedFileSet.class);
    private final int numChunks;
    private final File baseDir;
    private final List<Integer> indexFileSizes;
    private final List<Integer> dataFileSizes;
    private final List<MappedByteBuffer> indexFiles;
    private final List<FileChannel> dataFiles;

    public ChunkedFileSet(File directory) {
        this.baseDir = directory;
        if (!Utils.isReadableDir(directory)) {
            throw new VoldemortException(directory.getAbsolutePath() + " is not a readable directory.");
        }
        this.indexFileSizes = new ArrayList<Integer>();
        this.dataFileSizes = new ArrayList<Integer>();
        this.indexFiles = new ArrayList<MappedByteBuffer>();
        this.dataFiles = new ArrayList<FileChannel>();
        if (this.baseDir.list() != null && this.baseDir.list().length == 0) {
            try {
                new File(this.baseDir, "0.index").createNewFile();
                new File(this.baseDir, "0.data").createNewFile();
                logger.info("No index or data files found, creating empty files 0.index and 0.data.");
            }
            catch (IOException e) {
                throw new VoldemortException("Error creating empty read-only files.", e);
            }
        }
        int chunkId = 0;
        while (true) {
            File index = new File(this.baseDir, Integer.toString(chunkId) + ".index");
            File data = new File(this.baseDir, Integer.toString(chunkId) + ".data");
            if (!index.exists() && !data.exists()) break;
            if (index.exists() ^ data.exists()) {
                throw new VoldemortException("One of the following does not exist: " + index.toString() + " and " + data.toString() + ".");
            }
            long indexLength = index.length();
            long dataLength = data.length();
            this.validateFileSizes(indexLength, dataLength);
            this.indexFileSizes.add((int)indexLength);
            this.dataFileSizes.add((int)dataLength);
            this.dataFiles.add(this.openChannel(data));
            this.indexFiles.add(this.mapFile(index));
            ++chunkId;
        }
        if (chunkId == 0) {
            throw new VoldemortException("No data chunks found in directory " + this.baseDir.toString());
        }
        this.numChunks = chunkId;
        logger.trace("Opened chunked file set for " + this.baseDir + " with " + this.indexFileSizes.size() + " chunks.");
    }

    public void validateFileSizes(long indexLength, long dataLength) {
        if (indexLength > Integer.MAX_VALUE || dataLength > Integer.MAX_VALUE) {
            throw new VoldemortException("Index or data file exceeds 2147483647 bytes.");
        }
        if (indexLength % 20L != 0L) {
            throw new VoldemortException("Invalid index file, file length must be a multiple of 20 but is only " + indexLength + " bytes.");
        }
        if (dataLength < 4L * indexLength / 20L) {
            throw new VoldemortException("Invalid data file, file length must not be less than num_index_entries * 4 bytes, but data file is only " + dataLength + " bytes.");
        }
    }

    public void close() {
        for (int chunk = 0; chunk < this.numChunks; ++chunk) {
            FileChannel channel = this.dataFileFor(chunk);
            try {
                channel.close();
                continue;
            }
            catch (IOException e) {
                logger.error("Error while closing file.", e);
            }
        }
    }

    private FileChannel openChannel(File file) {
        try {
            return new FileInputStream(file).getChannel();
        }
        catch (IOException e) {
            throw new VoldemortException(e);
        }
    }

    private MappedByteBuffer mapFile(File file) {
        try {
            FileChannel channel = new FileInputStream(file).getChannel();
            MappedByteBuffer buffer = channel.map(FileChannel.MapMode.READ_ONLY, 0L, file.length());
            channel.close();
            return buffer;
        }
        catch (IOException e) {
            throw new VoldemortException(e);
        }
    }

    public int getNumChunks() {
        return this.numChunks;
    }

    public int getChunkForKey(byte[] key) {
        return ReadOnlyUtils.chunk(key, this.numChunks);
    }

    public ByteBuffer indexFileFor(int chunk) {
        return this.indexFiles.get(chunk).duplicate();
    }

    public FileChannel dataFileFor(int chunk) {
        return this.dataFiles.get(chunk);
    }

    public int getIndexFileSize(int chunk) {
        return this.indexFileSizes.get(chunk);
    }

    public int getDataFileSize(int chunk) {
        return this.indexFileSizes.get(chunk);
    }
}

