/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.io.sstable;

import com.google.common.base.Function;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileDescriptor;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectOutputStream;
import java.nio.ByteBuffer;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.cassandra.cache.JMXInstrumentedCache;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.db.ColumnFamily;
import org.apache.cassandra.db.DecoratedKey;
import org.apache.cassandra.io.sstable.Descriptor;
import org.apache.cassandra.io.sstable.SSTableReader;
import org.apache.cassandra.utils.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SSTableTracker
implements Iterable<SSTableReader> {
    private static final Logger logger = LoggerFactory.getLogger(SSTableTracker.class);
    private volatile Set<SSTableReader> sstables;
    private final AtomicLong liveSize = new AtomicLong();
    private final AtomicLong totalSize = new AtomicLong();
    private final String ksname;
    private final String cfname;
    private final JMXInstrumentedCache<Pair<Descriptor, DecoratedKey>, Long> keyCache;
    private final JMXInstrumentedCache<DecoratedKey, ColumnFamily> rowCache;

    public SSTableTracker(String ksname, String cfname) {
        this.ksname = ksname;
        this.cfname = cfname;
        this.sstables = Collections.emptySet();
        this.keyCache = new JMXInstrumentedCache(ksname, cfname + "KeyCache", 0);
        this.rowCache = new JMXInstrumentedCache(ksname, cfname + "RowCache", 3);
    }

    public void saveKeyCache() throws IOException {
        Function<Pair<Descriptor, DecoratedKey>, ByteBuffer> function = new Function<Pair<Descriptor, DecoratedKey>, ByteBuffer>(){

            @Override
            public ByteBuffer apply(Pair<Descriptor, DecoratedKey> key) {
                return ((DecoratedKey)key.right).key;
            }
        };
        CacheWriter<Pair<Descriptor, DecoratedKey>, Long> writer = new CacheWriter<Pair<Descriptor, DecoratedKey>, Long>();
        writer.saveCache(this.keyCache, DatabaseDescriptor.getSerializedKeyCachePath(this.ksname, this.cfname), function);
    }

    public void saveRowCache() throws IOException {
        Function<DecoratedKey, ByteBuffer> function = new Function<DecoratedKey, ByteBuffer>(){

            @Override
            public ByteBuffer apply(DecoratedKey key) {
                return key.key;
            }
        };
        CacheWriter<DecoratedKey, ColumnFamily> writer = new CacheWriter<DecoratedKey, ColumnFamily>();
        writer.saveCache(this.rowCache, DatabaseDescriptor.getSerializedRowCachePath(this.ksname, this.cfname), function);
    }

    public synchronized void replace(Collection<SSTableReader> oldSSTables, Iterable<SSTableReader> replacements) {
        HashSet<SSTableReader> sstablesNew = new HashSet<SSTableReader>(this.sstables);
        for (SSTableReader sstable : replacements) {
            assert (sstable.getKeySamples() != null);
            sstablesNew.add(sstable);
            long size = sstable.bytesOnDisk();
            this.liveSize.addAndGet(size);
            this.totalSize.addAndGet(size);
            sstable.setTrackedBy(this);
        }
        long maxDataAge = -1L;
        for (SSTableReader sstable : oldSSTables) {
            boolean removed = sstablesNew.remove(sstable);
            assert (removed);
            sstable.markCompacted();
            maxDataAge = Math.max(maxDataAge, sstable.maxDataAge);
            this.liveSize.addAndGet(-sstable.bytesOnDisk());
        }
        this.sstables = Collections.unmodifiableSet(sstablesNew);
        this.updateCacheSizes();
    }

    public synchronized void add(Iterable<SSTableReader> sstables) {
        assert (sstables != null);
        this.replace(Collections.<SSTableReader>emptyList(), sstables);
    }

    public synchronized void markCompacted(Collection<SSTableReader> compacted) {
        this.replace(compacted, Collections.<SSTableReader>emptyList());
    }

    public synchronized void updateCacheSizes() {
        int rowCacheSize;
        int keyCacheSize;
        long keys = this.estimatedKeys();
        if (!this.keyCache.isCapacitySetManually() && (keyCacheSize = DatabaseDescriptor.getKeysCachedFor(this.ksname, this.cfname, keys)) != this.keyCache.getCapacity()) {
            if (logger.isDebugEnabled()) {
                logger.debug("key cache capacity for " + this.cfname + " is " + keyCacheSize);
            }
            this.keyCache.updateCapacity(keyCacheSize);
        }
        if (!this.rowCache.isCapacitySetManually() && (rowCacheSize = DatabaseDescriptor.getRowsCachedFor(this.ksname, this.cfname, keys)) != this.rowCache.getCapacity()) {
            if (logger.isDebugEnabled()) {
                logger.debug("row cache capacity for " + this.cfname + " is " + rowCacheSize);
            }
            this.rowCache.updateCapacity(rowCacheSize);
        }
    }

    public Set<SSTableReader> getSSTables() {
        return this.sstables;
    }

    public int size() {
        return this.sstables.size();
    }

    @Override
    public Iterator<SSTableReader> iterator() {
        return this.sstables.iterator();
    }

    public synchronized void clearUnsafe() {
        this.sstables = Collections.emptySet();
    }

    public JMXInstrumentedCache<DecoratedKey, ColumnFamily> getRowCache() {
        return this.rowCache;
    }

    public long estimatedKeys() {
        long n = 0L;
        for (SSTableReader sstable : this) {
            n += sstable.estimatedKeys();
        }
        return n;
    }

    public long getLiveSize() {
        return this.liveSize.get();
    }

    public long getTotalSize() {
        return this.totalSize.get();
    }

    public void spaceReclaimed(long size) {
        this.totalSize.addAndGet(-size);
    }

    public JMXInstrumentedCache<Pair<Descriptor, DecoratedKey>, Long> getKeyCache() {
        return this.keyCache;
    }

    protected class CacheWriter<K, V> {
        protected CacheWriter() {
        }

        public void saveCache(JMXInstrumentedCache<K, V> cache, File savedCachePath, Function<K, ByteBuffer> converter) throws IOException {
            long start = System.currentTimeMillis();
            String msgSuffix = savedCachePath.getName() + " for " + SSTableTracker.this.cfname + " of " + SSTableTracker.this.ksname;
            logger.info("saving " + msgSuffix);
            int count = 0;
            File tmpFile = File.createTempFile(savedCachePath.getName(), null, savedCachePath.getParentFile());
            FileOutputStream fout = new FileOutputStream(tmpFile);
            ObjectOutputStream out = new ObjectOutputStream(new BufferedOutputStream(fout));
            FileDescriptor fd = fout.getFD();
            for (Object key : cache.getKeySet()) {
                ByteBuffer bytes = converter.apply(key);
                out.writeInt(bytes.remaining());
                out.write(bytes.array(), bytes.position() + bytes.arrayOffset(), bytes.remaining());
                ++count;
            }
            out.flush();
            fd.sync();
            out.close();
            if (!tmpFile.renameTo(savedCachePath)) {
                throw new IOException("Unable to rename cache to " + savedCachePath);
            }
            if (logger.isDebugEnabled()) {
                logger.debug("saved " + count + " keys in " + (System.currentTimeMillis() - start) + " ms from " + msgSuffix);
            }
        }
    }
}

