package com.mapr.util.zookeeper;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Vector;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;

import org.apache.log4j.Logger;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.WatchedEvent;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.ZooKeeper;
import org.apache.zookeeper.Watcher.Event.KeeperState;
import org.apache.zookeeper.data.Stat;

import com.mapr.baseutils.zookeeper.ZKClosedException;
import com.mapr.baseutils.zookeeper.ZKUtils;

import com.mapr.fs.proto.Common.ServiceData;
import com.google.protobuf.InvalidProtocolBufferException;


public class ZKDataRetrieval implements Watcher {

	private static final Logger LOG = Logger.getLogger(ZKDataRetrieval.class);
	
	public static final String CLDB_SERVICE_MASTER_PATH = "/datacenter/controlnodes/cldb/active/CLDBMaster";
	public static final String HBASE_MASTER_SERVICE_MASTER_PATH = "/hbase/master";
	public static final String SERVER_PATH = "/servers";
	public static final String SERVICES_PATH = "/services";
	public static final String CONF_SERVICES_PATH = "/services_config";

	public static final int TIMEOUT_SERVER = 30 * 1000;
	private static final String CLDB_SERVICE_NAME = "cldb";
	private static final String HBASE_MASTER_SERVICE_NAME = "hbmaster";

  // Added for watching the RM HA ZNode.
  private boolean watchRMHAZNode = false;
  private String rmHAZNodeToWatch = null;
  private byte[] rmHAZNodeData = null;

  // List of nodes
	private List<String> serversInfo = new Vector<String>();
	// Map of service to list of nodes that were configured by warden
	private Map<String, List<String>> configServicesMap = new ConcurrentHashMap<String, List<String>>();
	// Map of service to list of nodes that are running currently by warden
	private Map<String, List<String>> servicesMap = new ConcurrentHashMap<String, List<String>>();
	// Map of service to master content 
	private Map<String, ServiceData> serviceToMasterMap = new ConcurrentHashMap<String, ServiceData>();
	// Map of configured service to its nodes,properties to figure out stuff for arbitrary services
	private Map<String, Map<String,Properties>> serviceToProperties =  new ConcurrentHashMap<String, Map<String,Properties>>();

	private Map<String, Map<String, ServiceData>> serviceToData =  new ConcurrentHashMap<String, Map<String,ServiceData>>();
	
	private CountDownLatch lock = new CountDownLatch(1);	
	private volatile boolean isSessionExpired = false;
	
	private String zkConnectString;
	private ZooKeeper s_zk;
	
	public ZKDataRetrieval(String zkConnectString) {
		this.zkConnectString = zkConnectString;
		initThread();
	}

  public ZKDataRetrieval(String zkConnectString, String rmHAZNode) {
    this(zkConnectString);
    this.watchRMHAZNode = true;
    this.rmHAZNodeToWatch = rmHAZNode;
  }

	public String getZKString() {
		return zkConnectString;
	}
	
	public synchronized ZooKeeper getZKObject() {
		return s_zk;
	}
	
	/**
	 * Inits hashes and connects to ZK
	 */
	private void init() {
		serversInfo = new Vector<String>();
		configServicesMap = new ConcurrentHashMap<String, List<String>>();
		servicesMap = new ConcurrentHashMap<String, List<String>>();
		serviceToMasterMap = new ConcurrentHashMap<String,  ServiceData>();
		
		if ( zkConnectString == null ) {
			LOG.fatal("Could not create ZooKeeper instance. Zookeeper connection string is missing");
			return;
		}
	    try {
    		s_zk = new ZooKeeper(zkConnectString, ZKUtils.TIMEOUT, this);
    		while ( !lock.await(ZKUtils.TIMEOUT, TimeUnit.MILLISECONDS) ) {
    			LOG.error("Could not connect to ZK within: " + ZKUtils.TIMEOUT + " ms. Check if ZK connection defined correctly: " +  zkConnectString + ". No data from ZK will be returned.");
    		}
    		LOG.info("Connected to ZK: " + zkConnectString);
		} catch (IOException e) {
 	    LOG.fatal("Could not create ZooKeeper instance. Due to IOException. No data from ZK with connect string: " + zkConnectString + " will be returned.", e);
		} catch (InterruptedException e) {
			LOG.error("InterruptedException during wait for ZK to be connected");
		}
	}
	
  /**
	 * Starts reinit thread to be able to reinit ZK in case of SessionExpiration
	 */
	private void initThread() {
		init();
		Thread reinitThread = new Thread(new Runnable(){

			@Override
			public void run() {
				while ( true ) {
					synchronized(s_zk) {
						while ( !isSessionExpired ) {
							try {
								s_zk.wait();
							} catch (InterruptedException e) {
								LOG.error("InterruptedException during wait for ZK for: " + zkConnectString);
							}
						}
					}
					// looks like session expired
					// we better close and start again
					zooKeeperReset();
					isSessionExpired = false;
				}
			}});

    // Make it a daemon thread as it should not cause JVM to hang around.
    reinitThread.setDaemon(true);
		reinitThread.start();
		
	}
	
	public Map<String, Map<String, ServiceData>> getServiceToNodeMap() {
		// trigger our map to fill in if not yet
		getServicesMap();
		return serviceToData;
	}
	
	public Map<String, ServiceData> getServiceToMasterMap() {
		// trigger our map to fill in if not yet
		getServicesMap();
		return serviceToMasterMap;
	}
	
	public Map<String, Map<String, Properties>> getServiceToPropertiesMap() {
		// trigger our map to fill in if not yet
		getConfigServicesMap();
		return serviceToProperties;
	}

  public byte[] getActiveRMZNodeData() {
    if(!watchRMHAZNode) {
      LOG.error("Not watching any RM HA ZNode, unable to get Active RM Data");
      return null;
    }

    if(rmHAZNodeData == null) {
      try {
        try {
          rmHAZNodeData = ZKUtils.getData(s_zk, rmHAZNodeToWatch, this, null, ZKUtils.ZK_RETRIALS);
        } catch (Exception e) {
          LOG.error("Unable to get Active RM Data at " + rmHAZNodeToWatch + " with error: " + e.getLocalizedMessage());
          ZKUtils.checkZKNodeForExistence(s_zk, rmHAZNodeToWatch, this, ZKUtils.ZK_RETRIALS);
          return null;
        }
      } catch (KeeperException e) {
        LOG.error("Unable to get Active RM Data at " + rmHAZNodeToWatch + " with error: " + e.getLocalizedMessage());
      } catch (InterruptedException e) {
        LOG.error("Unable to get Active RM Data at " + rmHAZNodeToWatch + " with error: " + e.getLocalizedMessage());
      } catch (ZKClosedException e) {
        // Most likely SessionExpirationException
        // need to reset ZK and call myself again
        LOG.error("Most likely SessionExpirationException. Need to reset ZK and call myself again", e);
      }
    } else {
      if ( LOG.isDebugEnabled() ) {
        LOG.debug("Getting CACHED RMHAZNodeData");
      }
    }

    return rmHAZNodeData;
  }

	public List<String> getServersInfo() {
		if ( !serversInfo.isEmpty()) {
			if ( LOG.isDebugEnabled() ) {
				LOG.debug("Getting CACHED list of servers");
			}
			return serversInfo;
		} 
		int retries = 0;
		while ( retries++ < ZKUtils.ZK_RETRIALS) {
		// list of servers
		LOG.info("Getting list of servers");
		List<String> nodes = new ArrayList<String>();
		try {
			List<String> znodes = ZKUtils.getZkNodeChildren(s_zk, SERVER_PATH, this, ZKUtils.ZK_RETRIALS);
			for ( String znode : znodes ) {
				byte[] znodeData = s_zk.getData(SERVER_PATH + "/" + znode, false, null);
				String znodeName = new String(znodeData);
				nodes.add(znodeName);
			}
			serversInfo = nodes;
			break;
		} catch (KeeperException e) {
			LOG.error("Can not get children of " + SERVER_PATH + "with error: "+ e.getLocalizedMessage());
		} catch (InterruptedException e) {
			LOG.error("Can not get children of " + SERVER_PATH + "with error: "+ e.getLocalizedMessage());
		} catch (ZKClosedException e) {
			// Most likely SessionExpirationException
			// need to reset ZK and call myself again
			LOG.error("Most likely SessionExpirationException. Need to reset ZK and call myself again", e);
		}
		}
		return serversInfo;
	}

	public Map<String, List<String>> getConfigServicesMap() {
		if ( !configServicesMap.isEmpty() ) {
			if ( LOG.isDebugEnabled() ) {
				LOG.debug("Getting CACHED configServicesMap");
			}
			return configServicesMap;
		} 
		int retries = 0;
		while ( retries++ < ZKUtils.ZK_RETRIALS) {
		try {
			LOG.info("Getting configServicesMap");
			List<String> services = ZKUtils.getZkNodeChildren(s_zk, CONF_SERVICES_PATH, this, ZKUtils.ZK_RETRIALS);
			for ( String service : services ) {
				List<String> nodes = ZKUtils.getZkNodeChildren(s_zk, CONF_SERVICES_PATH + "/" + service, this, ZKUtils.ZK_RETRIALS);
				configServicesMap.put(service, nodes);
				// need to get Properties for configured service
				Map<String, Properties> nodeProperties = getPropertiesHelper(CONF_SERVICES_PATH, service, nodes);
				serviceToProperties.put(service, nodeProperties);
			}
			break;
		} catch (KeeperException e) {
			LOG.error("Can not get children of " + CONF_SERVICES_PATH + " or it's children with error: "+ e.getLocalizedMessage());
		} catch (InterruptedException e) {
			LOG.error("Can not get children of " + CONF_SERVICES_PATH + " or it's children with error: "+ e.getLocalizedMessage());
		} catch (ZKClosedException e) {
			// Most likely SessionExpirationException
			// need to reset ZK and call myself again
			LOG.error("Most likely SessionExpirationException. Need to reset ZK and call myself again", e);
		}
		}
		return configServicesMap;
	}

	public ServiceData getServiceMasterData(String serviceName) {

		// Check if we have the global data first. Useful in clients that keep a watch on everything like Warden/webserver etc.
		// or even in a case where we are watching only a few services from the code below.
		if( !serviceToMasterMap.isEmpty() ) {
			if(LOG.isDebugEnabled()) {
				LOG.debug("Getting serviceData for master node of " + serviceName + " service from CACHED serviceToMasterMap");
			}
			ServiceData serviceMasterData = serviceToMasterMap.get(serviceName);
			if (serviceMasterData != null)
				return serviceToMasterMap.get(serviceName);
		}

		// Return from Zookeeper and leave a watch. Useful when the client only is interested in a specific service's master such as
		// ResourceManager from NodeManagers etc.
		int retries = 0;
		while ( retries++ < ZKUtils.ZK_RETRIALS) {
			try {
				LOG.info("Getting serviceData for master node of " + serviceName);
				if ( !serviceName.equalsIgnoreCase(CLDB_SERVICE_NAME) &&
				     !serviceName.equalsIgnoreCase(HBASE_MASTER_SERVICE_NAME)) {
					// for CLDB and hbmaster, let's not even put a watcher on "master", as it might be wrong
					try {
						byte [] byteData = ZKUtils.getData(s_zk, SERVICES_PATH + "/" + serviceName + "/" + ZKUtils.SERVICE_MASTER_NODE, this, null, ZKUtils.ZK_RETRIALS);
						addServiceDataToMasterMap(serviceName, byteData);
					} catch (KeeperException.NoNodeException e) {
						LOG.warn("Can not get children of " + SERVICES_PATH + "/" + serviceName + "/" + ZKUtils.SERVICE_MASTER_NODE + " with error: "+ e.getLocalizedMessage());
						ZKUtils.checkZKNodeForExistence(s_zk, SERVICES_PATH + "/" + serviceName + "/" + ZKUtils.SERVICE_MASTER_NODE, this, ZKUtils.ZK_RETRIALS);
					}
				}
				if ( serviceName.equalsIgnoreCase(CLDB_SERVICE_NAME)) {
					try {
						// if it is cldb look for different place to get master
						byte [] cldbByteData = ZKUtils.getData(s_zk, CLDB_SERVICE_MASTER_PATH, this, null, ZKUtils.ZK_RETRIALS);
						serviceToMasterMap.put(serviceName, parseCLDBMasterName(cldbByteData));
					} catch (KeeperException.NoNodeException e) {
						LOG.warn("Can not get data for " + CLDB_SERVICE_MASTER_PATH + e.getLocalizedMessage());
						// set watcher on CLDB Master
						ZKUtils.checkZKNodeForExistence(s_zk, CLDB_SERVICE_MASTER_PATH, this, ZKUtils.ZK_RETRIALS);
					}
				}

				if ( serviceName.equalsIgnoreCase(HBASE_MASTER_SERVICE_NAME)) {
					try {
						// if it is cldb look for different place to get master
						byte [] hbmasterByteData = ZKUtils.getData(s_zk, HBASE_MASTER_SERVICE_MASTER_PATH, this, null, ZKUtils.ZK_RETRIALS);
						serviceToMasterMap.put(serviceName, parseHbaseMasterName(hbmasterByteData));
					} catch (KeeperException.NoNodeException e) {
						LOG.warn("Can not get data for " + HBASE_MASTER_SERVICE_MASTER_PATH + e.getLocalizedMessage());
						// set watcher on Hbase Master
						ZKUtils.checkZKNodeForExistence(s_zk, HBASE_MASTER_SERVICE_MASTER_PATH, this, ZKUtils.ZK_RETRIALS);
					}
				}
			} catch (KeeperException e) {
				LOG.error("Can not get children of " + SERVICES_PATH + "/" + serviceName + "/" + ZKUtils.SERVICE_MASTER_NODE + " with error: "+ e.getLocalizedMessage());
			} catch (InterruptedException e) {
				LOG.error("Can not get children of " + SERVICES_PATH + "/" + serviceName + "/" + ZKUtils.SERVICE_MASTER_NODE + " with error: "+ e.getLocalizedMessage());
			} catch (ZKClosedException e) {
				// Most likely SessionExpirationException
				// need to reset ZK and call myself again
				LOG.error("Most likely SessionExpirationException. Need to reset ZK and call myself again", e);
			}
			break;
		}
		return serviceToMasterMap.get(serviceName);
	}

	public Map<String, List<String>> getServicesMap() {
		if ( !servicesMap.isEmpty()) {
			if ( LOG.isDebugEnabled() ) {
				LOG.debug("Getting CACHED servicesMap");
			}
			return servicesMap;
		}
		int retries = 0;
		while ( retries++ < ZKUtils.ZK_RETRIALS) {
		try {
			LOG.info("Getting servicesMap");
			List<String> services = ZKUtils.getZkNodeChildren(s_zk, SERVICES_PATH, this, ZKUtils.ZK_RETRIALS);
			for ( String service : services ) {
				List<String> nodes = ZKUtils.getZkNodeChildren(s_zk, SERVICES_PATH + "/" + service, this, ZKUtils.ZK_RETRIALS);
				serviceToData.put(service, getNodeDataHelper(SERVICES_PATH, service, nodes));
				if ( !service.equalsIgnoreCase(CLDB_SERVICE_NAME) &&
				     !service.equalsIgnoreCase(HBASE_MASTER_SERVICE_NAME)) {
				  // for CLDB/hbmaster let's not even put a watcher on "master", as it might be wrong
  				try {
  					byte [] byteData = ZKUtils.getData(s_zk, SERVICES_PATH + "/" + service + "/" + ZKUtils.SERVICE_MASTER_NODE, this, null, ZKUtils.ZK_RETRIALS);
  					addServiceDataToMasterMap(service, byteData);
  				} catch (KeeperException.NoNodeException e) {
  					LOG.warn("Can not get children of " + SERVICES_PATH + " or it's children with error: "+ e.getLocalizedMessage());
  					ZKUtils.checkZKNodeForExistence(s_zk, SERVICES_PATH + "/" + service + "/" + ZKUtils.SERVICE_MASTER_NODE, this, ZKUtils.ZK_RETRIALS);
  				}
				}
				// put them separately for kvstore and fileserver into per service map
				nodes.remove(ZKUtils.SERVICE_MASTER_NODE);
				servicesMap.put(service, nodes);
				if ( service.equalsIgnoreCase(CLDB_SERVICE_NAME)) {
					try {
					  // if it is cldb look for different place to get master
					  byte [] cldbByteData = ZKUtils.getData(s_zk, CLDB_SERVICE_MASTER_PATH, this, null, ZKUtils.ZK_RETRIALS);
					  serviceToMasterMap.put(service, parseCLDBMasterName(cldbByteData));
					} catch (KeeperException.NoNodeException e) {
						LOG.warn("Can not get data for " + CLDB_SERVICE_MASTER_PATH + e.getLocalizedMessage());
						// set watcher on CLDB Master
						ZKUtils.checkZKNodeForExistence(s_zk, CLDB_SERVICE_MASTER_PATH, this, ZKUtils.ZK_RETRIALS);
					}
				}

				if ( service.equalsIgnoreCase(HBASE_MASTER_SERVICE_NAME)) {
					try {
					  // if it is hbmaster look for different place to get master
					  byte [] hbmasterByteData = ZKUtils.getData(s_zk, HBASE_MASTER_SERVICE_MASTER_PATH, this, null, ZKUtils.ZK_RETRIALS);
					  serviceToMasterMap.put(service, parseHbaseMasterName(hbmasterByteData));
					} catch (KeeperException.NoNodeException e) {
						LOG.warn("Can not get data for " + HBASE_MASTER_SERVICE_MASTER_PATH + e.getLocalizedMessage());
						// set watcher on hbmaster Master
						ZKUtils.checkZKNodeForExistence(s_zk, HBASE_MASTER_SERVICE_MASTER_PATH, this, ZKUtils.ZK_RETRIALS);
					}
				}
			}
			break;
		} catch (KeeperException e) {
			LOG.error("Can not get children of " + SERVICES_PATH + " or it's children with error: "+ e.getLocalizedMessage());
		} catch (InterruptedException e) {
			LOG.error("Can not get children of " + SERVICES_PATH + " or it's children with error: "+ e.getLocalizedMessage());
		} catch (ZKClosedException e) {
			// Most likely SessionExpirationException
			// need to reset ZK and call myself again
			LOG.error("Most likely SessionExpirationException. Need to reset ZK and call myself again", e);
		}
		}
		return servicesMap;
	}

	@Override
	public void process(WatchedEvent event) {
		String path = event.getPath();
		LOG.info("Process path: " + path + ". Event state: " + event.getState() + ". Event type: " + event.getType());
		if ( event.getState() == KeeperState.SyncConnected ) {
			lock.countDown();
		}

		if ( event.getState() == KeeperState.Expired ) {
			// session expired - need to reset ZK
			LOG.warn("ZK Reset due to SessionExpiration for ZK: " + zkConnectString);
			synchronized (s_zk) {
				isSessionExpired = true;
				s_zk.notify();
			}
			return;
		}
		
		switch ( event.getType() ) {
		case NodeCreated:
			// need to handle master creation
			if ( path.endsWith(ZKUtils.SERVICE_MASTER_NODE)) {
				String [] servicesPath = path.split("/");
				if ( servicesPath.length > 1 ) {
					String service = servicesPath[servicesPath.length - 2];
					if ( !service.equalsIgnoreCase(CLDB_SERVICE_NAME) &&
					     !service.equalsIgnoreCase(HBASE_MASTER_SERVICE_NAME)) {
  					try {
  						Stat stat = ZKUtils.checkZKNodeForExistence(s_zk, path, this, ZKUtils.ZK_RETRIALS);
  						if ( stat != null ) {
  							byte [] byteData = ZKUtils.getData(s_zk, path, this, null, ZKUtils.ZK_RETRIALS);
  							addServiceDataToMasterMap(service, byteData);
  						}
  					} catch (KeeperException e) {
  						LOG.error("Can not get data from " + path + "with error: "+ e.getLocalizedMessage());
  					} catch (InterruptedException e) {
  						LOG.error("Can not get data from " + path + "with error: "+ e.getLocalizedMessage());
  					} catch (ZKClosedException e) {
  						// Most likely SessionExpirationException
  						// need to reset ZK and call myself again
  						LOG.error("Most likely SessionExpirationException. Need to reset ZK and call myself again", e);
  					}
  				}
				}
			}
			if ( path.equalsIgnoreCase(CLDB_SERVICE_MASTER_PATH)) {
				// deal with CLDB master creation
			  try {
  				try {
  				  // if it is cldb look for different place to get master
  				  byte [] cldbByteData = ZKUtils.getData(s_zk, CLDB_SERVICE_MASTER_PATH, this, null, ZKUtils.ZK_RETRIALS);
  				  serviceToMasterMap.put(CLDB_SERVICE_NAME, parseCLDBMasterName(cldbByteData));
  				} catch (KeeperException.NoNodeException e) {
  					LOG.warn("Can not get data for " + CLDB_SERVICE_MASTER_PATH + e.getLocalizedMessage());
  					// set watcher on CLDB Master
  					ZKUtils.checkZKNodeForExistence(s_zk, CLDB_SERVICE_MASTER_PATH, this, ZKUtils.ZK_RETRIALS);
  				} 
			  } catch (KeeperException e) {
				LOG.error("Can not get data from " + CLDB_SERVICE_MASTER_PATH + "with error: "+ e.getLocalizedMessage());
			  } catch (InterruptedException e) {
				LOG.error("Can not get data from " + CLDB_SERVICE_MASTER_PATH + "with error: "+ e.getLocalizedMessage());
			  } catch (ZKClosedException e) {
				// Most likely SessionExpirationException
				// need to reset ZK and call myself again
				LOG.error("Most likely SessionExpirationException. Need to reset ZK and call myself again", e);
			  }
			}

			if ( path.equalsIgnoreCase(HBASE_MASTER_SERVICE_MASTER_PATH)) {
				// deal with hbmaster master creation
			  try {
  				try {
  				  // if it is hbmaster look for different place to get master
  				  byte [] hbmasterByteData = ZKUtils.getData(s_zk, HBASE_MASTER_SERVICE_MASTER_PATH, this, null, ZKUtils.ZK_RETRIALS);
  				  serviceToMasterMap.put(HBASE_MASTER_SERVICE_NAME, parseHbaseMasterName(hbmasterByteData));
  				} catch (KeeperException.NoNodeException e) {
  					LOG.warn("Can not get data for " + HBASE_MASTER_SERVICE_MASTER_PATH + e.getLocalizedMessage());
  					// set watcher on hbmaster Master
  					ZKUtils.checkZKNodeForExistence(s_zk, HBASE_MASTER_SERVICE_MASTER_PATH, this, ZKUtils.ZK_RETRIALS);
  				}
                          } catch (KeeperException | InterruptedException e) {
                              LOG.error("Can not get data from " + HBASE_MASTER_SERVICE_MASTER_PATH + "with error: " +
                                  e.getClass().getName() + ": " + e.getLocalizedMessage());
			  } catch (ZKClosedException e) {
			      // Most likely SessionExpirationException
			      // need to reset ZK and call myself again
                              LOG.error("Most likely SessionExpirationException." +
                                  " Need to reset ZK and call myself again. " +
                                  e.getClass().getName() + ": " + e.getLocalizedMessage());
			  }
			}
      if(path.equalsIgnoreCase(rmHAZNodeToWatch)) {
        // Handle new RM Master Node creation.
        try {
          rmHAZNodeData = ZKUtils.getData(s_zk, path, this, null, ZKUtils.ZK_RETRIALS);
        } catch (KeeperException e) {
          LOG.error("Can not get data from " + path + "with error: "+ e.getLocalizedMessage());
        } catch (InterruptedException e) {
          LOG.error("Can not get data from " + path + "with error: "+ e.getLocalizedMessage());
        } catch (ZKClosedException e) {
          // Most likely SessionExpirationException
          // need to reset ZK and call myself again
          LOG.error("Most likely SessionExpirationException. Need to reset ZK and call myself again", e);
        }
      }
			break;
		case NodeDeleted:
			// need to handle master deletion
			if ( path.endsWith(ZKUtils.SERVICE_MASTER_NODE)) {
				String [] servicesPath = path.split("/");
				if ( servicesPath.length > 1 ) {
					String service = servicesPath[servicesPath.length - 2];
					if ( !service.equalsIgnoreCase(CLDB_SERVICE_NAME) &&
					     !service.equalsIgnoreCase(HBASE_MASTER_SERVICE_NAME)) {
  					serviceToMasterMap.remove(service);
  					try {
  						Stat stat = ZKUtils.checkZKNodeForExistence(s_zk, path, this, ZKUtils.ZK_RETRIALS);
  						if ( stat != null ) {
  							byte [] byteData = ZKUtils.getData(s_zk, path, this, null, ZKUtils.ZK_RETRIALS);
  							addServiceDataToMasterMap(service, byteData);
  						}
  					} catch (KeeperException e) {
  						LOG.error("Can not get data from " + path + "with error: "+ e.getLocalizedMessage());
  					} catch (InterruptedException e) {
  						LOG.error("Can not get data from " + path + "with error: "+ e.getLocalizedMessage());
  					} catch (ZKClosedException e) {
  						// Most likely SessionExpirationException
  						// need to reset ZK and call myself again
  						LOG.error("Most likely SessionExpirationException. Need to reset ZK and call myself again", e);
  					}
  				}
				}
			} 
			if ( path.equalsIgnoreCase(CLDB_SERVICE_MASTER_PATH)) {
				// just check for existence
				try {
					serviceToMasterMap.remove(CLDB_SERVICE_NAME);
					Stat stat = ZKUtils.checkZKNodeForExistence(s_zk, path, this, ZKUtils.ZK_RETRIALS);
					if ( stat != null ) {
						byte [] cldbByteData = ZKUtils.getData(s_zk, path, this, null, ZKUtils.ZK_RETRIALS);
		  				serviceToMasterMap.put(CLDB_SERVICE_NAME, parseCLDBMasterName(cldbByteData));
					}
				} catch (KeeperException e) {
					LOG.error("Can not get data from " + path + "with error: "+ e.getLocalizedMessage());
				} catch (InterruptedException e) {
					LOG.error("Can not get data from " + path + "with error: "+ e.getLocalizedMessage());
				} catch (ZKClosedException e) {
					// Most likely SessionExpirationException
					// need to reset ZK and call myself again
					LOG.error("Most likely SessionExpirationException. Need to reset ZK and call myself again", e);
				}

			}

			if ( path.equalsIgnoreCase(HBASE_MASTER_SERVICE_MASTER_PATH)) {
				// just check for existence
				try {
					serviceToMasterMap.remove(HBASE_MASTER_SERVICE_NAME);
					Stat stat = ZKUtils.checkZKNodeForExistence(s_zk, path, this, ZKUtils.ZK_RETRIALS);
					if ( stat != null ) {
						byte [] hbmasterByteData = ZKUtils.getData(s_zk, path, this, null, ZKUtils.ZK_RETRIALS);
		  				serviceToMasterMap.put(HBASE_MASTER_SERVICE_NAME, parseHbaseMasterName(hbmasterByteData));
					}
                                } catch (KeeperException | InterruptedException e) {
                                    LOG.error("Can not get data from " + path + "with error: " +
                                        e.getClass().getName() + ": " + e.getLocalizedMessage());
				} catch (ZKClosedException e) {
				    // Most likely SessionExpirationException
				    // need to reset ZK and call myself again
                                    LOG.error("Most likely SessionExpirationException." +
                                        " Need to reset ZK and call myself again. " +
                                        e.getClass().getName() + ": " + e.getLocalizedMessage());
				}

			}
      if(path.equalsIgnoreCase(rmHAZNodeToWatch)) {
        // Handle RM Master disappearance
        try {
          rmHAZNodeData = null;
          Stat stat = ZKUtils.checkZKNodeForExistence(s_zk, path, this, ZKUtils.ZK_RETRIALS);
          if ( stat != null ) {
            rmHAZNodeData = ZKUtils.getData(s_zk, path, this, null, ZKUtils.ZK_RETRIALS);
          }
        } catch (KeeperException e) {
          LOG.error("Can not get data from " + path + "with error: "+ e.getLocalizedMessage());
        } catch (InterruptedException e) {
          LOG.error("Can not get data from " + path + "with error: "+ e.getLocalizedMessage());
        } catch (ZKClosedException e) {
          // Most likely SessionExpirationException
          // need to reset ZK and call myself again
          LOG.error("Most likely SessionExpirationException. Need to reset ZK and call myself again", e);
        }
      }
			break;
		case NodeDataChanged:
			// master data changed
			if ( path.endsWith(ZKUtils.SERVICE_MASTER_NODE)) {
				String [] servicesPath = path.split("/");
				if ( servicesPath.length > 1 ) {
					String service = servicesPath[servicesPath.length - 2];
					// we actually did not set a watcher but just in case
					// we should not use warden's "master" data to set it, only CLDB true master
					if ( !service.equalsIgnoreCase(CLDB_SERVICE_NAME) &&
					     !service.equalsIgnoreCase(HBASE_MASTER_SERVICE_MASTER_PATH)) {
  					try {
  					  try {
    						byte [] byteData = ZKUtils.getData(s_zk, path, this, null, ZKUtils.ZK_RETRIALS);
    						addServiceDataToMasterMap(service, byteData);
  					  } catch (KeeperException.NoNodeException e) {
    						LOG.warn("Can not get data from " + path + " with error: "+ e.getLocalizedMessage());
    						ZKUtils.checkZKNodeForExistence(s_zk, path, this, ZKUtils.ZK_RETRIALS);
  					  } 
  					} catch (KeeperException e) {
  						LOG.error("Can not get data from " + path + "with error: "+ e.getLocalizedMessage());
  					} catch (InterruptedException e) {
  						LOG.error("Can not get data from " + path + "with error: "+ e.getLocalizedMessage());
  					} catch (ZKClosedException e) {
  						// Most likely SessionExpirationException
  						// need to reset ZK and call myself again
  						LOG.error("Most likely SessionExpirationException. Need to reset ZK and call myself again", e);
  					}
  				}
				}
			}
			if ( path.equalsIgnoreCase(CLDB_SERVICE_MASTER_PATH)) {
				try {
					  try {
						byte [] cldbByteData = ZKUtils.getData(s_zk, path, this, null, ZKUtils.ZK_RETRIALS);
						serviceToMasterMap.put(CLDB_SERVICE_NAME, parseCLDBMasterName(cldbByteData));
					  } catch (KeeperException.NoNodeException e) {
						LOG.warn("Can not get data from " + path + " with error: "+ e.getLocalizedMessage());
						ZKUtils.checkZKNodeForExistence(s_zk, path, this, ZKUtils.ZK_RETRIALS);
					  } 
					} catch (KeeperException e) {
						LOG.error("Can not get data from " + path + "with error: "+ e.getLocalizedMessage());
					} catch (InterruptedException e) {
						LOG.error("Can not get data from " + path + "with error: "+ e.getLocalizedMessage());
					} catch (ZKClosedException e) {
						// Most likely SessionExpirationException
						// need to reset ZK and call myself again
						LOG.error("Most likely SessionExpirationException. Need to reset ZK and call myself again", e);
					}
			}

			if ( path.equalsIgnoreCase(HBASE_MASTER_SERVICE_MASTER_PATH)) {
				try {
					  try {
						byte [] hbmasterByteData = ZKUtils.getData(s_zk, path, this, null, ZKUtils.ZK_RETRIALS);
						serviceToMasterMap.put(HBASE_MASTER_SERVICE_NAME, parseHbaseMasterName(hbmasterByteData));
					  } catch (KeeperException.NoNodeException e) {
                                                LOG.error("Can not get data from " + path + "with error: " +
                                                    e.getClass().getName() + ": " + e.getLocalizedMessage());
						ZKUtils.checkZKNodeForExistence(s_zk, path, this, ZKUtils.ZK_RETRIALS);
					  }
                                        } catch (KeeperException | InterruptedException e) {
                                            LOG.error("Can not get data from " + path + "with error: " +
                                                e.getClass().getName() + ": " + e.getLocalizedMessage());
					} catch (ZKClosedException e) {
				            // Most likely SessionExpirationException
				            // need to reset ZK and call myself again
                                            LOG.error("Most likely SessionExpirationException." +
                                                " Need to reset ZK and call myself again. " +
                                                e.getClass().getName() + ": " + e.getLocalizedMessage());
					}
			}
			if ( path.startsWith(CONF_SERVICES_PATH)) {
				// most likely properties of one of the services changed
				try {
					String [] servicesPath = path.split("/");
					// should be 3 levels of path
					if ( servicesPath.length >= 3 ) {
						String service = servicesPath[servicesPath.length - 2];
						String node = servicesPath[servicesPath.length - 1];
						Map<String, Properties> nodeProperties = new HashMap<String, Properties>(serviceToProperties.get(service));
						byte [] byteData = ZKUtils.getData(s_zk, path, this, null, ZKUtils.ZK_RETRIALS);
						ByteArrayInputStream bais = new ByteArrayInputStream(byteData);
						Properties properties = new Properties();
						try {
							properties.load(bais);
							nodeProperties.put(node, properties);
						} catch (IOException e) {
							LOG.error("Unable to load properties for path: " + path, e);
						}
						serviceToProperties.put(service, nodeProperties);
					}
				} catch (KeeperException e) {
					LOG.error("Can not get data from " + path + "with error: "+ e.getLocalizedMessage());
				} catch (InterruptedException e) {
					LOG.error("Can not get data from " + path + "with error: "+ e.getLocalizedMessage());
				} catch (ZKClosedException e) {
					// Most likely SessionExpirationException
					// need to reset ZK and call myself again
					LOG.error("Most likely SessionExpirationException. Need to reset ZK and call myself again", e);
				}
			} else if ( path.startsWith(SERVICES_PATH)) {
 				// most likely data of one of the services changed
				try {
					String [] servicesPath = path.split("/");
					// should be 3 levels of path
						if ( servicesPath.length >= 3 &&
						 	   !servicesPath[servicesPath.length - 1].equals(ZKUtils.SERVICE_MASTER_NODE)) {
							String service = servicesPath[servicesPath.length - 2];
							String node = servicesPath[servicesPath.length - 1];
              Map<String, ServiceData> nodeProperties;
              Map<String, ServiceData> curNodeProps = serviceToData.get(service);
              if (curNodeProps == null) {
                nodeProperties = new HashMap<String, ServiceData>();
              } else {
                nodeProperties = new HashMap<String, ServiceData>(curNodeProps);
              }
							byte [] byteData = ZKUtils.getData(s_zk, SERVICES_PATH + "/" + service + "/" + node, this, null, ZKUtils.ZK_RETRIALS);
							ServiceData hostInfo = ServiceData.parseFrom(byteData);
							nodeProperties.put(node, hostInfo);
							serviceToData.put(service, nodeProperties);
						}
				} catch (KeeperException e) {
					LOG.error("Can not get data from " + path + "with error: "+ e.getLocalizedMessage());
				} catch (InterruptedException e) {
					LOG.error("Can not get data from " + path + "with error: "+ e.getLocalizedMessage());
				} catch (ZKClosedException e) {
					// Most likely SessionExpirationException
					// need to reset ZK and call myself again
					LOG.error("Most likely SessionExpirationException. Need to reset ZK and call myself again", e);
				} catch (InvalidProtocolBufferException e) {
					LOG.error("Did not receive valid Protobuf");
				}
			}
      if(path.equalsIgnoreCase(rmHAZNodeToWatch)) {
        // RM Fail over occurred.
        try {
          try {
            rmHAZNodeData = ZKUtils.getData(s_zk, path, this, null, ZKUtils.ZK_RETRIALS);
          } catch (KeeperException.NoNodeException e) {
            LOG.warn("Can not get data from " + path + " with error: " + e.getLocalizedMessage());
            ZKUtils.checkZKNodeForExistence(s_zk, path, this, ZKUtils.ZK_RETRIALS);
          }
        } catch (KeeperException e) {
          LOG.error("Can not get data from " + path + "with error: "+ e.getLocalizedMessage());
        } catch (InterruptedException e) {
          LOG.error("Can not get data from " + path + "with error: "+ e.getLocalizedMessage());
        } catch (ZKClosedException e) {
          // Most likely SessionExpirationException
          // need to reset ZK and call myself again
          LOG.error("Most likely SessionExpirationException. Need to reset ZK and call myself again", e);
        }
      }
			break;
		case NodeChildrenChanged:
			// identify which ones are changing
		 if (path.startsWith(CONF_SERVICES_PATH)) {
			try {
				String servicePath = event.getPath();
				if ( servicePath.equalsIgnoreCase(CONF_SERVICES_PATH)) {
					// some service dir was removed or created
					// need to read everything here
					List<String> services = ZKUtils.getZkNodeChildren(s_zk, servicePath, this, ZKUtils.ZK_RETRIALS);
					for ( String service : services ) {
						List<String> nodes = ZKUtils.getZkNodeChildren(s_zk, servicePath + "/" + service, this, ZKUtils.ZK_RETRIALS);
						configServicesMap.put(service, nodes);
						// need to get Properties for configured service
						Map<String, Properties> nodeProperties = getPropertiesHelper(CONF_SERVICES_PATH, service, nodes);						serviceToProperties.put(service, nodeProperties);
						serviceToProperties.put(service, nodeProperties);
					}
				} else {
					List<String> nodes = ZKUtils.getZkNodeChildren(s_zk, servicePath, this, ZKUtils.ZK_RETRIALS);
					String [] servicesPath = servicePath.split("/");
					String service = servicesPath[servicesPath.length -1];
					configServicesMap.put(service, nodes);
					// need to get Properties for configured service
					Map<String, Properties> nodeProperties = getPropertiesHelper(CONF_SERVICES_PATH, service, nodes);
					serviceToProperties.put(service, nodeProperties);
				}
			} catch (KeeperException e) {
				LOG.error("Can not get children of " + event.getPath() + "with error: "+ e.getLocalizedMessage());
			} catch (InterruptedException e) {
				LOG.error("Can not get children of " + event.getPath() + "with error: "+ e.getLocalizedMessage());
			} catch (ZKClosedException e) {
				// Most likely SessionExpirationException
				// need to reset ZK and call myself again
				LOG.error("Most likely SessionExpirationException. Need to reset ZK and call myself again", e);
			}
			// list of config services children
			break;
		 } else if ( path.startsWith(SERVICES_PATH)) {
				try {
					String servicePath = event.getPath();
					if ( servicePath.equalsIgnoreCase(SERVICES_PATH) ) {
						List<String> services = ZKUtils.getZkNodeChildren(s_zk, servicePath, this, ZKUtils.ZK_RETRIALS);
						for ( String service : services ) {
							List<String> nodes = ZKUtils.getZkNodeChildren(s_zk, servicePath + "/" + service, this, ZKUtils.ZK_RETRIALS);
							if ( !service.equalsIgnoreCase(CLDB_SERVICE_NAME) &&
							     !service.equalsIgnoreCase(HBASE_MASTER_SERVICE_NAME)) {
  							try {
  								byte [] byteData = ZKUtils.getData(s_zk, servicePath + "/" + service + "/" + ZKUtils.SERVICE_MASTER_NODE, this, null, ZKUtils.ZK_RETRIALS);
  								addServiceDataToMasterMap(service, byteData);
  							} catch (KeeperException.NoNodeException e) {
  								LOG.warn("Can not get children of " + SERVICES_PATH + " or it's children with error: "+ e.getLocalizedMessage());
  								ZKUtils.checkZKNodeForExistence(s_zk, servicePath + "/" + service + "/" + ZKUtils.SERVICE_MASTER_NODE, this, ZKUtils.ZK_RETRIALS);
  							}
							} else {
							    if ( service.equalsIgnoreCase(CLDB_SERVICE_NAME)) {
								try {
								  // if it is cldb look for different place to get master
								  byte [] cldbByteData = ZKUtils.getData(s_zk, CLDB_SERVICE_MASTER_PATH, this, null, ZKUtils.ZK_RETRIALS);
								  serviceToMasterMap.put(service, parseCLDBMasterName(cldbByteData));
								} catch (KeeperException.NoNodeException e) {
									LOG.warn("Can not get data for " + CLDB_SERVICE_MASTER_PATH + e.getLocalizedMessage());
									// set watcher on CLDB Master
									ZKUtils.checkZKNodeForExistence(s_zk, CLDB_SERVICE_MASTER_PATH, this, ZKUtils.ZK_RETRIALS);
								}
                                                            }

							    if ( service.equalsIgnoreCase(HBASE_MASTER_SERVICE_NAME)) {
								try {
								  // if it is hbmaster look for different place to get master
								  byte [] hbmasterByteData = ZKUtils.getData(s_zk, HBASE_MASTER_SERVICE_MASTER_PATH, this, null, ZKUtils.ZK_RETRIALS);
								  serviceToMasterMap.put(service, parseHbaseMasterName(hbmasterByteData));
								} catch (KeeperException.NoNodeException e) {
									LOG.warn("Can not get data for " + HBASE_MASTER_SERVICE_MASTER_PATH + e.getLocalizedMessage());
									// set watcher on hbmaster Master
									ZKUtils.checkZKNodeForExistence(s_zk, HBASE_MASTER_SERVICE_MASTER_PATH, this, ZKUtils.ZK_RETRIALS);
								}
                                                            }
							}
							// put them separately for kvstore and fileserver into per service map
							nodes.remove(ZKUtils.SERVICE_MASTER_NODE);
							servicesMap.put(service, nodes);
							serviceToData.put(service, getNodeDataHelper(SERVICES_PATH, service, nodes));
						}
					}
					List<String> nodes = ZKUtils.getZkNodeChildren(s_zk, servicePath, this, ZKUtils.ZK_RETRIALS);
					String [] servicesPath = servicePath.split("/");
					String service = servicesPath[servicesPath.length -1];
					nodes.remove(ZKUtils.SERVICE_MASTER_NODE);
					servicesMap.put(service, nodes);
					serviceToData.put(service, getNodeDataHelper(SERVICES_PATH, service, nodes));
				} catch (KeeperException e) {
					LOG.error("Can not get children of " + event.getPath() + "with error: "+ e.getLocalizedMessage());
				} catch (InterruptedException e) {
					LOG.error("Can not get children of " + event.getPath() + "with error: "+ e.getLocalizedMessage());
				} catch (ZKClosedException e) {
					// Most likely SessionExpirationException
					// need to reset ZK and call myself again
					LOG.error("Most likely SessionExpirationException. Need to reset ZK and call myself again", e);
				}

				// list of nodes changed
				break;
			} else if (path.equalsIgnoreCase(SERVER_PATH)) {
				// list of servers
				List<String> nodes = new ArrayList<String>();
				try {
					List<String> znodes = ZKUtils.getZkNodeChildren(s_zk, SERVER_PATH, this, ZKUtils.ZK_RETRIALS);
					for ( String znode : znodes ) {
						byte[] znodeData = s_zk.getData(SERVER_PATH + "/" + znode, false, null);
						String znodeName = new String(znodeData);
						nodes.add(znodeName);
					}
					serversInfo = nodes;
					break;
				} catch (KeeperException e) {
					LOG.error("Can not get children of " + SERVER_PATH + "with error: "+ e.getLocalizedMessage());
				} catch (InterruptedException e) {
					LOG.error("Can not get children of " + SERVER_PATH + "with error: "+ e.getLocalizedMessage());
				} catch (ZKClosedException e) {
					// Most likely SessionExpirationException
					// need to reset ZK and call myself again
					LOG.error("Most likely SessionExpirationException. Need to reset ZK and call myself again", e);
				}
				break;
			}
			break;
		}
	}

	/**
	 * Helper to reset ZK in case client decides to - like SessionExpirationException
	 * @param zkConnectString
	 * @return
	 * @throws IOException
	 */
	private synchronized void zooKeeperReset() {
		LOG.info("Close zk for: " + zkConnectString);
		if ( s_zk != null ) {
			try {
				s_zk.close();
			} catch (InterruptedException e) {
				LOG.error("InterruptedException during ZK closure");
			}
		}
		lock = new CountDownLatch(1);
		init();
	}
	private void addServiceDataToMasterMap(String service, byte[] data) {
		try {
			ServiceData hostInfo = ServiceData.parseFrom(data);
			serviceToMasterMap.put(service, hostInfo);
		} catch(InvalidProtocolBufferException e) {
			LOG.error("Did not get valid ProtoBuf.", e);
		}
		
	}
	
	private static ServiceData parseCLDBMasterName(byte[] data) {
		if ( data == null ) {
			return null;
		}
		String masterData = new String(data);
		int i = masterData.indexOf("HostName: ");
		if ( i < 0 || i > masterData.length() ) {
			return null;
		}
		String CLDBMasterName = masterData.substring(i + "HostName: ".length(), masterData.length());
		ServiceData.Builder zkData = ServiceData.newBuilder();
		zkData.setHost(CLDBMasterName);
		return zkData.build();
	}

	private static ServiceData parseHbaseMasterName(byte[] data) {
		if ( data == null ) {
		    LOG.debug("parseHbaseMasterName:Did not get valid ProtoBuf.");
		    return null;
		}

                /*
                // pulling in hbase means that cli utilities now would depend on hbase jars - skipping
                ServerName hbaseServerName;
                try {
                    ProtobufUtil hbpb = new ProtobufUtil();
                    hbaseServerName =  hbpb.parseServerNameFrom(data);
                } catch (DeserializationException e) {
                    LOG.error("Did not get valid ProtoBuf.", e);
                    return null;
                }
                */

                /* the protobuf of the master node starts
                   after the magic PBUF marker
                   here is an example of what is returned from zk:
                   ff 00 00 00 14 6d 61 73 74 65 72 3a 31 36 30 30 30 63 10 58
                   b4 17 0c 03 3d 50 42 55 46 0a 1c 0a 10 71 61 2d 6e 6f 64 65
                   39 33 2e 71 61 2e 6c 61 62 10 80 7d 18 8c d2 dc b8 bf 2c 10
                   00 18 8a 7d

                   the protobuf section starts after 50 42 55 46:
                   0a 1c 0a 10 71 61 2d 6e 6f 64 65 39 33 2e 71 61 2e 6c 61 62
                   10 80 7d 18 8c d2 dc b8 bf 2c 10 00 18 8a 7d


                   running that through protoc gives:
                   echo 0a1c0a1071612d6e6f646539332e71612e6c616210807d188cd2dcb8bf2c1000188a7d | xxd -r -p | protoc --decode_raw
                   1 {
                     1: "qa-node93.qa.lab"
                     2: 16000
                     3: 1528858880268
                   }
                   2: 0
                   3: 16010

                   The first byte - 0x0a provides the member number 1 and type 2 (last 3 bits)
                   is a WireFormat.WIRETYPE_LENGTH_DELIMITED type, meaning the next 1-4( possibly 8)
                   bytes contains the length of the structure. The next 0x0a is the start of the first
                   element of the ServerName struct and is also of WireFormat.WIRETYPE_LENGTH_DELIMITED
                   type. Instead of writing a recursive routine to parse the whole protobuf, we will
                   just rely on the two tags and using the same logic as protobuf code to determine the
                   correct lengths to get the hostname.
                */

                int idx = 0;
                boolean foundPrefix = false;
                for (; idx < data.length; idx++) {
                    if (data[idx] == 'P') {
                        if ((data.length - idx > 4) && data[idx + 1] == 'B' &&
                            data[idx + 2] == 'U' && data[idx + 3] == 'F') {
                            foundPrefix = true;
                            break;
                        }
                    }
                    if (data.length - idx < 4) {
                        break;
                    }
                }

                if (foundPrefix) {
                    StringBuilder builder = new StringBuilder();
                    for (int x=0; x < data.length; x++) {
                        builder.append(String.format("%02x ", data[x]));
                    }
                    LOG.debug("parseHbaseMasterName: byte array is: " + builder);

                    idx += 4; // adjust for the PBUF marker
                    if (data[idx++] == 0x0a) {
                        byte outerLength = 0;
                        // eat the first length counter - we don't care about it
                        do {
                            outerLength = data[idx++];
                        } while  (outerLength < 0 && idx < data.length);

                        if (data[idx++] == 0x0a) {
                            // also number of bytes of length
                            int multiplier = 0;
                            int nameLength = 0;
                            byte tmp = 0;
                            // find the size of the string
                            do {
                                tmp = data[idx++];
                                if (tmp >= 0) {
                                    nameLength |= (tmp << multiplier*7);
                                    LOG.debug("len2 = " + nameLength);
                                } else {
                                    nameLength |= ((tmp & 0x7f)<< multiplier*7);
                                    LOG.debug("len2- = " + nameLength);
                                }
                                multiplier++;
                            } while (tmp < 0 && idx < data.length);
                            LOG.debug("len = " + nameLength + ", idx = " + idx + ", multiplier = " + multiplier);
                            String HbaseMasterName = new String(data, idx + multiplier - 1, nameLength);
		            LOG.debug("parseHbaseMasterName: returning Hbase Master = " + HbaseMasterName);
                            //String HbaseMasterName = new String("mfs74.qa.lab");
                            ServiceData.Builder zkData = ServiceData.newBuilder();
                            zkData.setHost(HbaseMasterName);
                            return zkData.build();
                        } else {
                            LOG.error("parseHbaseMasterName: Did not find second pb marker");
                            return null;
                        }
                    } else {
                        LOG.error("parseHbaseMasterName: Did not find first pb marker");
                        return null;
                    }
               } else {
                    LOG.error("parseHbaseMasterName: Did not find protoBuf marker.");
                    return null;
               }
	}

	private Map<String, ServiceData> getNodeDataHelper(String prefix, String service, List<String> nodes)
		throws KeeperException, InterruptedException, ZKClosedException {
		Map<String, ServiceData> nodeProperties = new HashMap<String, ServiceData>();
		for ( String node : nodes ) {
			try {
				byte [] byteData = ZKUtils.getData(s_zk, prefix + "/" + service + "/" + node, this, null, ZKUtils.ZK_RETRIALS);
				ServiceData dataInfo = ServiceData.parseFrom(byteData);
				nodeProperties.put(node, dataInfo);
			} catch (KeeperException.NoNodeException e) {
				LOG.warn("Can not get children of " + service + " or it's children with error: "+ e.getLocalizedMessage());
				ZKUtils.checkZKNodeForExistence(s_zk, prefix + "/" + service + "/" + node, this, ZKUtils.ZK_RETRIALS);
			} catch(InvalidProtocolBufferException e) {
			LOG.error("Did not get valid ProtoBuf.", e);
			}
		}
		return nodeProperties;
	}
	
	private Map<String, Properties> getPropertiesHelper(String prefix, String service, List<String> nodes) 
		throws KeeperException, InterruptedException, ZKClosedException {
		Map<String, Properties> nodeProperties = new HashMap<String, Properties>();
		for ( String node : nodes ) {
			byte [] byteData = ZKUtils.getData(s_zk, prefix + "/" + service + "/" + node, this, null, ZKUtils.ZK_RETRIALS);
			ByteArrayInputStream bais = new ByteArrayInputStream(byteData);
			Properties properties = new Properties();
			try {
				properties.load(bais);
				nodeProperties.put(node, properties);
			} catch (IOException e) {
				LOG.error("Unable to load properties for node: " + node + " and service: " + service, e);
			}
		}
		return nodeProperties;
	}
	
}
