package com.mapr.util;

import java.io.IOException;
import java.net.URI;

import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.Path;
import org.apache.log4j.Logger;

import com.mapr.fs.MapRFileSystem;

/**
 * Utility class that provides APIs to interact with MapRFS.
 * 
 * @author smarella
 *
 */
public class MapRFSUtil {
  private static final Logger LOG = Logger.getLogger(MapRFSUtil.class);

  /**
   * Preserve a static instance of {@link MapRFileSystem} for re-usability.
   */
  private static MapRFileSystem fs;
  
  public static MapRFileSystem getMapRFileSystem() {
    if (fs == null) {
      Configuration c = new Configuration();
      c.set("fs.default.name", "maprfs:///");    
      try {
        fs = new MapRFileSystem();
        URI uri = new URI(c.get("fs.default.name"));
        fs.initialize(uri, c);
      } catch (Exception e) {
        LOG.error("Encountered error while trying to initialize MapRFS", e);
      }
    }
    return fs;
  }
  
  public static boolean isDir(String path) {
    try {
      return !getMapRFileSystem().isFile(new Path(path));
    } catch (IOException e) {
      LOG.error("Exception while checking if the path specified is a dir. path: " + path, e);
      return false;
    }
  }

  public static boolean pathExists(String path) {
    try {
      return getMapRFileSystem().exists(new Path(path));
    } catch (IOException e) {
      LOG.error("Exception while checking for existence of path: " + path, e);
      return false;
    }
  }

  public static boolean create(String path, boolean isDir) {
    if (isDir) {
      try {
        return getMapRFileSystem().mkdirs(new Path(path));
      } catch (IOException e) {
        LOG.error("Exception while creating dir: " + path, e);
      }
    } else {
      try {
        return getMapRFileSystem().createNewFile(new Path(path));
      } catch (IOException e) {
        LOG.error("Exception while creating file: " + path, e);
      }
    }
    return false;
  }
  
  public static boolean delete(String path) {
    try {
      return getMapRFileSystem().delete(new Path(path));
    } catch (IOException e) {
      LOG.error("Exception while deleting path: " + path, e);    
    }
    return false;
  }
}
