/* Copyright (c) 2009 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.util;

import java.io.FileInputStream;
import java.io.IOException;
import java.util.*;

import javax.naming.*;
import javax.naming.NamingException;
import javax.naming.directory.*;

import org.apache.log4j.Logger;

public class LDAPUtil {
  
  //LDAP Configuration related parameters
  public static final String LDAP_PARAM_PREFIX = "mapr.ldap.";
  public static final String ParamLdapUseConf = LDAP_PARAM_PREFIX + "useconf";
  public static final String ParamLdapConfPath = LDAP_PARAM_PREFIX + "conf.path";
  public static final String ParamLdapServer = LDAP_PARAM_PREFIX + "server";
  public static final String ParamLdapPort = LDAP_PARAM_PREFIX + "port";
  public static final String ParamLdapSslRequired = LDAP_PARAM_PREFIX + "sslrequired";
  public static final String ParamLdapBindDn = LDAP_PARAM_PREFIX + "binddn";
  public static final String ParamLdapBindDnPasswd = LDAP_PARAM_PREFIX + "binddnpasswd";
  public static final String ParamLdapBaseDn = LDAP_PARAM_PREFIX + "basedn";
  public static final String ParamLdapUidAttrib = LDAP_PARAM_PREFIX + "attribute.uid";
  public static final String ParamLdapMailAttrib = LDAP_PARAM_PREFIX + "attribute.mail";
  public static final String ParamLdapAuthEnabled = LDAP_PARAM_PREFIX + "auth.enabled";
 
  private static final Logger LOG = Logger.getLogger(LDAPUtil.class);

  public static String getEmailAddress(Properties prop, String user,
                                       boolean type) throws IOException {

    String uri = null;
    String baseDn = null;
    String bindDn = null;
    String bindDnPasswd = null;

    /**
     * <MAPR_ERROR> Error during an LDAP search for user's email </MAPR_ERROR>
     */
    boolean ldapAuthEnable = Boolean.parseBoolean(prop.getProperty(ParamLdapAuthEnabled,"true").trim());
    if (LOG.isDebugEnabled()) {
      LOG.debug("LDAP authentication enabled "+ldapAuthEnable);
    }
    String useConf = prop.getProperty(ParamLdapUseConf);
    if (useConf != null && Integer.valueOf(useConf) > 0) {
      // Get LDAP settings from file?
      String ldapFile = prop.getProperty(ParamLdapConfPath);
      Properties ldapProp = new Properties();
      try {
        ldapProp.load(new FileInputStream(ldapFile));
      } catch (IOException e) {
        /**
        * <MAPR_ERROR>
        * Message:Could not open conf file: <file> <message>
        * Function:LDAPUtil.getEmailAddress()
        * Meaning:An error occurred during an LDAP search for the user's email address.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
        LOG.error("Could not open conf file: " + ldapFile + " "
            + e.getLocalizedMessage());
        return null;
      }

      uri = ldapProp.getProperty("URI");
      if (uri == null || uri.isEmpty()) {
        /**
        * <MAPR_ERROR>
        * Message:Could not fetch URI from conf file <file> 
        * Function:LDAPUtil.getEmailAddress()
        * Meaning:An error occurred during an LDAP search for the user's email address.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
        LOG.error("Could not fetch URI from conf file " + ldapFile);
        return null;
      }

      baseDn = ldapProp.getProperty("BASE");
      if (baseDn == null || baseDn.isEmpty()) {
        /**
        * <MAPR_ERROR>
        * Message:Could not fetch BaseDn from conf file <file> 
        * Function:LDAPUtil.getEmailAddress()
        * Meaning:An error occurred during an LDAP search for the user's email address.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
        LOG.error("Could not fetch BaseDn from conf file " + ldapFile);
        return null;
      }

    } else {
      // Get LDAP settings from Params configured from UI

      // Fetch Server info
      String server = prop.getProperty(ParamLdapServer);
      if (server == null || server.isEmpty()) {
        /**
        * <MAPR_ERROR>
        * Message:Server not defined
        * Function:LDAPUtil.getEmailAddress()
        * Meaning:An error occurred during an LDAP search for the user's email address.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
        LOG.error("Server not defined ");
        return null;
      }

      // Fetch BaseDn
      baseDn = prop.getProperty(ParamLdapBaseDn);
      if (baseDn == null || baseDn.isEmpty()) {
        LOG.error("BaseDn not defined ");
        return null;
      }

      // FetchBindDn 
      bindDn = prop.getProperty(ParamLdapBindDn);
      if (ldapAuthEnable && (bindDn == null || bindDn.isEmpty())) {
        LOG.error("BindDn not defined ");
        return null;
      }
      // Fetch BindDnPasswd
      bindDnPasswd = prop.getProperty(ParamLdapBindDnPasswd);

      // Fetch SSL info
      int ssl = Integer.valueOf(prop.getProperty(ParamLdapSslRequired, "0"));

      // Fetch port info
      String portStr = prop.getProperty(ParamLdapPort);
      if (portStr == null || portStr.isEmpty()) {
        // Use default ports
        portStr = ((ssl == 0) ? "389" : "636");
      } else {
        portStr = prop.getProperty(ParamLdapPort);
      }

      String protocol = ((ssl == 0) ? "ldap" : "ldaps");
      uri = protocol + "://" + server + ":" + portStr + "/";
    }

    // Fetch Uid Attrib
    String uidAttrib = prop.getProperty(ParamLdapUidAttrib);
    if (uidAttrib == null || uidAttrib.isEmpty()) {
      /**
      * <MAPR_ERROR>
      * Message:Uid Attribute not defined
      * Function:LDAPUtil.getEmailAddress()
      * Meaning:An error occurred during an LDAP search for the user's email address.
      * Resolution:Contact technical support.
      * </MAPR_ERROR>
      */
      LOG.error("Uid Attribute not defined ");
      return null;
    }

    // Fetch Mail Attrib
    String mailAttrib = prop.getProperty(ParamLdapMailAttrib);
    if (mailAttrib == null || mailAttrib.isEmpty()) {
      /**
      * <MAPR_ERROR>
      * Message:Mail Attribute not defined
      * Function:LDAPUtil.getEmailAddress()
      * Meaning:An error occurred during an LDAP search for the user's email address.
      * Resolution:Contact technical support.
      * </MAPR_ERROR>
      */
      LOG.error("Mail Attribute not defined ");
      return null;
    }

    // Connect to LDAP server.
    Hashtable<String, String> env = new Hashtable<String, String>(11);
    env.put(Context.INITIAL_CONTEXT_FACTORY, "com.sun.jndi.ldap.LdapCtxFactory");
    env.put(Context.PROVIDER_URL, uri);
    if(ldapAuthEnable) {
      env.put(Context.SECURITY_AUTHENTICATION, "simple");
      env.put(Context.SECURITY_PRINCIPAL, bindDn);
      if (bindDnPasswd != null && !bindDnPasswd.isEmpty()) {
        env.put(Context.SECURITY_CREDENTIALS, bindDnPasswd);
      }
    }

    InitialDirContext ctx = null;
    try {
      ctx = new InitialDirContext(env);
    } catch (Exception e) {
      /**
      * <MAPR_ERROR>
      * Message:Problem connecting to LDAP server: <message>
      * Function:LDAPUtil.getEmailAddress()
      * Meaning:An error occurred during an LDAP search for the user's email address.
      * Resolution:Contact technical support.
      * </MAPR_ERROR>
      */
      LOG.error("Problem connecting to LDAP server:" + e.getLocalizedMessage());
      return null;
    }

    String filter = "(" + uidAttrib + "=" + user + ")";
    String[] searchAttrs = { mailAttrib };

    SearchControls constraints = new SearchControls();
    constraints.setSearchScope(SearchControls.SUBTREE_SCOPE);
    constraints.setTimeLimit(30000);
    constraints.setReturningAttributes(searchAttrs);

    // Search for user's email given matching user ID.
    String email = null;
    try {

      NamingEnumeration<SearchResult> answer = 
                      ctx.search(baseDn, filter, constraints);
      while (answer.hasMore()) {
        SearchResult ldapResult = answer.next();
        Attribute emailAttr = ldapResult.getAttributes().get(mailAttrib);
        email = (String) emailAttr.get();
      }
      ctx.close();

    } catch (NamingException e) {
      /**
      * <MAPR_ERROR>
      * Message:Problem fetching Email for user <user> from LDAP server: <message>
      * Function:LDAPUtil.getEmailAddress()
      * Meaning:An error occurred during an LDAP search for the user's email address.
      * Resolution:Contact technical support.
      * </MAPR_ERROR>
      */
      LOG.error("Problem fetching Email for user " + user
          + " from LDAP server:" + e.getLocalizedMessage());
      return null;
    }

    return email;
  }
}
