package com.mapr.streams.tools;

import java.io.IOException;
import java.util.*;
import java.io.*;


import org.apache.kafka.clients.producer.Callback;
import org.apache.kafka.clients.producer.KafkaProducer;
import org.apache.kafka.clients.producer.RecordMetadata;
import org.apache.kafka.clients.producer.ProducerRecord;
import org.apache.kafka.clients.producer.ProducerConfig;
import com.mapr.fs.proto.Marlinserver.MarlinConfigDefaults;

import org.apache.kafka.clients.consumer.Consumer;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.apache.kafka.clients.consumer.ConsumerRecords;
import org.apache.kafka.clients.consumer.KafkaConsumer;
import org.apache.kafka.common.serialization.ByteArrayDeserializer;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.PartitionInfo;

public class MirrorMaker {
  public static int nPartitions = 4;
  public static String srcTopicName;
  public static String dstTopicName;
  public static String consumerConfig = null;
  public static String producerConfig;
  public static boolean stopOnErr=true;
  public static Properties consumerProps;
  public static Properties producerProps;

  public static void usage() {
    System.err.println("MirrorMaker -src <topic-full-name> -dst <topic-full-name>" + 
        " [-consumerConfig ccfile -producerConfig pcfile -stopOnError true|false]");
    System.exit(1);
  }

  public static void main(String args[]) throws IOException {
    System.out.println("Running mirror maker");

    for (int i = 0; i < args.length; ++i) {
      if (args[i].equals("-src")) {
        i++;
        if (i >= args.length) usage();
        srcTopicName = args[i];
      } else if (args[i].equals("-dst")) {
        i++;
        if (i >= args.length) usage();
        dstTopicName = args[i];
      } else if (args[i].equals("-consumerConfig")) {
        i++;
        if (i >= args.length) usage();
        consumerConfig = args[i];
      } else if (args[i].equals("-producerConfig")) {
        i++;
        if (i >= args.length) usage();
        producerConfig = args[i];
      } else if (args[i].equals("-stopOnError")) {
        i++;
        if (i >= args.length) usage();
        stopOnErr = Boolean.parseBoolean(args[i]);
      } else {
        usage();
      }
    }
   
    if (srcTopicName == null || dstTopicName == null) {
      usage();
    }

    initConsumer();
    initProducer();

    verifyAndPossiblyCreateDstTopic();

    Thread[] partitionWorkers = new Thread[nPartitions];
    for (int i = 0; i < nPartitions; i++) {
      KafkaProducer producer = new KafkaProducer<byte[], byte[]>(producerProps);
      KafkaConsumer consumer = new KafkaConsumer<byte[], byte[]>(consumerProps);
      PartitionWorker worker = new PartitionWorker(producer,
                                                   consumer, srcTopicName,
                                                   dstTopicName, i,
                                                   stopOnErr);
      partitionWorkers[i] = new Thread(worker);
      partitionWorkers[i].start();
    }

    try {
      for(int i = 0; i < partitionWorkers.length; i++) {
        partitionWorkers[i].join();
      }
    } catch ( InterruptedException e) {
      System.out.println("***** Joining partitionWorkers failed *****");
    }
  }


  public static void verifyAndPossiblyCreateDstTopic() {
    KafkaProducer producer = new KafkaProducer<byte[], byte[]>(producerProps);
    KafkaConsumer consumer = new KafkaConsumer<byte[], byte[]>(consumerProps);
    
    List<PartitionInfo> srcPartitions =
      consumer.partitionsFor(srcTopicName);
    
    List<PartitionInfo> dstPartitions; 
    try {
      dstPartitions = 
        producer.partitionsFor(dstTopicName);
    } catch (Exception e) {
      nPartitions = srcPartitions.size();
      producer.close();
      consumer.close();
      return;
    }

    if (dstPartitions.size() != srcPartitions.size()) {
      System.out.println("Source topic and destination topic do not have" +
          " equal number of partitions");
      System.exit(1);
    }

    nPartitions = dstPartitions.size();
    producer.close();
    consumer.close();
  }

  public static void initProducer() throws IOException {
    producerProps = new Properties();
    if (producerConfig != null) {
      producerProps.load(new FileInputStream(producerConfig));
    }

    if (producerProps.getProperty("bootstrap.servers") == null) {
      producerProps.put("bootstrap.servers", "localhost:9092");
    }
    
    if (producerProps.getProperty("key.serializer") == null) {
      producerProps.put("key.serializer", "org.apache.kafka.common.serialization.ByteArraySerializer");
    }

    if (producerProps.getProperty("value.serializer") == null) {
      producerProps.put("value.serializer", "org.apache.kafka.common.serialization.ByteArraySerializer");
    }
  
    if (producerProps.getProperty("streams.parallel.flushers.per.partition") != null) {
        producerProps.put("streams.parallel.flushers.per.partition", false);
    }  
  }

  public static void initConsumer() throws IOException {
    consumerProps = new Properties();
    if (consumerConfig != null) {
      consumerProps.load(new FileInputStream(consumerConfig));
    }
  
    if (consumerProps.getProperty("bootstrap.servers") == null) {
      consumerProps.put("bootstrap.servers", "localhost:9092");
    }

    if (consumerProps.getProperty("key.deserializer") == null) {
      consumerProps.put("key.deserializer",
          "org.apache.kafka.common.serialization.ByteArrayDeserializer");
    }

    if (consumerProps.getProperty("value.deserializer") == null) {
      consumerProps.put("value.deserializer",
          "org.apache.kafka.common.serialization.ByteArrayDeserializer");
    }

  }
  
  private static final class ProducerCallback implements Callback {
    boolean stopOnErr;

    public ProducerCallback(boolean stopOnErr) {
      this.stopOnErr = stopOnErr;
    }

    public void onCompletion(RecordMetadata metadata,
                             Exception exception) {
      if (exception != null) {
        exception.printStackTrace();
        System.out.println("Send failed");
        if (stopOnErr) {
          System.exit(1);
        }
      }
    }
  }

  private static final class PartitionWorker implements Runnable {

    KafkaProducer producer;
    KafkaConsumer consumer;
    String srcTopicName;
    String dstTopicName;
    int partitionId;
    int pollTimeout = 1000;
    boolean stopOnErr;

    public PartitionWorker(KafkaProducer producer, KafkaConsumer consumer, String srcTopicName, String dstTopicName, int
                           partitionId, boolean stopOnError)  {
      this.consumer = consumer;
      this.producer = producer;
      this.srcTopicName = srcTopicName;
      this.dstTopicName = dstTopicName;
      this.partitionId = partitionId;
      this.stopOnErr = stopOnErr;
    }

    public void run() {
      try {
        TopicPartition partition = new TopicPartition(srcTopicName, partitionId);
        List<TopicPartition> partsToSubscribe = new ArrayList<TopicPartition>();
        partsToSubscribe.add(partition);
        consumer.assign(partsToSubscribe);
        Callback cb = new ProducerCallback(stopOnErr);
       
        consumer.seekToBeginning(partition);
        while(true) {
          ConsumerRecords<byte[], byte[]> crecs = consumer.poll(pollTimeout);
          Iterator<ConsumerRecord<byte[], byte[]>> iter = crecs.iterator();
          while (iter.hasNext()) {
            ConsumerRecord<byte[], byte[]> crec = iter.next();
            byte[] key = crec.key();
            byte[] value = crec.value();
            ProducerRecord<byte[], byte[]> prec =
              new ProducerRecord<byte[], byte[]>(dstTopicName, partitionId, key, value);
            producer.send(prec, cb);
          }
        }
      } catch (Exception e) {
        e.printStackTrace();
        System.out.println("PartitionWorker failed");
      }
      producer.close();
      consumer.close();
    }
  }
}
