/* Copyright (c) 2015 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.streams.tools;

import java.util.Arrays;

import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.conf.Configured;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.util.GenericOptionsParser;
import org.apache.hadoop.util.Tool;
import org.apache.hadoop.util.ToolRunner;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.mapr.db.Admin;
import com.mapr.db.MapRDB;
import com.mapr.db.impl.TableDescriptorImpl;
import com.mapr.db.mapreduce.tools.Export;

import com.mapr.fs.MapRFileSystem;

public class ExportStream extends Configured implements Tool {
  private static final Logger LOG = LoggerFactory.getLogger(ExportStream.class);
  private static final String NAME = "ExportStream";

  private void Usage(String errMsg) {
    if (errMsg != null)
      System.err.println("ERROR: " + errMsg);

    System.err.println("Usage: " + NAME + "\n" +
                       "  -src <Input stream path>]\n" +
                       "  -dst <dest directory>\n" +
                       "  [-mapreduce <true|false> (default: false)]\n");
    System.exit(1);
  }

  @Override
  public int run(String[] args) throws Exception {
    String[] cmdArgs =
      new GenericOptionsParser(getConf(), args).getRemainingArgs();

    String srcPath = null;
    String dstPath = null;
    boolean mapreduceOptionPresent = false;
    for (int i = 0; i < cmdArgs.length; ++i) {
      if (cmdArgs[i].equalsIgnoreCase("-src")) {
        srcPath = cmdArgs[++i];
      } else if (cmdArgs[i].equalsIgnoreCase("-dst")) {
        dstPath = cmdArgs[++i];
      } else if (cmdArgs[i].equalsIgnoreCase("-mapreduce")) {
        boolean mapReduce = Boolean.valueOf(cmdArgs[++i]);
        mapreduceOptionPresent = true;
      } else {
        Usage("Unknown option " + cmdArgs[i]);
      }
    }
    if (srcPath == null)
      Usage("missing -src streamPath");
    if (dstPath == null)
      Usage("missing -dst directory");

    /* Verify that src represents a stream */
    MapRFileSystem mfs = (MapRFileSystem)(FileSystem.get(new Configuration()));
    if (!mfs.isStream(new Path(srcPath))) {
      Usage("src " + srcPath + " is not a stream");
    }

    int numExportArgs = args.length + (mapreduceOptionPresent ? 0 : 2);
    String [] exportTableArgs = Arrays.copyOf(args, numExportArgs);
    int argsIdx = args.length;

    if (!mapreduceOptionPresent) {
      // mapreduce is false by default for streams.
      exportTableArgs[argsIdx++] = "-mapreduce";
      exportTableArgs[argsIdx++] = "false";
    }

    return ToolRunner.run(getConf(), new Export(), exportTableArgs);
  }

  public static void main(String[] args) throws Exception {
    int ret = 0;
    try {
      ret = ToolRunner.run(new Configuration(), new ExportStream(), args);
    } catch (Exception e) {
      ret = 1;
      e.printStackTrace();
    }
    System.exit(ret);
  }
}
