/* Copyright (c) 2015 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.streams.tools;

import java.util.Arrays;

import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.conf.Configured;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.util.GenericOptionsParser;
import org.apache.hadoop.util.Tool;
import org.apache.hadoop.util.ToolRunner;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.mapr.db.Admin;
import com.mapr.db.MapRDB;
import com.mapr.db.impl.TableDescriptorImpl;
import com.mapr.db.mapreduce.tools.DiffTables;

import com.mapr.fs.MapRFileSystem;

public class DiffStreams extends Configured implements Tool {
  private static final Logger LOG = LoggerFactory.getLogger(DiffStreams.class);
  private static final String NAME = "DiffStreams";

  private void Usage(String errMsg) {
    if (errMsg != null)
      System.err.println("ERROR: " + errMsg);

    System.err.println("Usage: " + NAME + "\n" +
                       "  -src <source stream path>]\n" +
                       "  -dst <destination stream path>\n" +
                       "  -outdir <output directory>\n" +
                       "  [-first_exit] Exit when first difference is found\n" +
                       "  [-mapreduce <true|false> (default: false)]\n" +
                       "  [-numthreads <number of threads> (default: 16)\n");
    System.exit(1);
  }

  @Override
  public int run(String[] args) throws Exception {
    String[] cmdArgs =
      new GenericOptionsParser(getConf(), args).getRemainingArgs();

    String srcPath = null;
    String dstPath = null;
    String outDir = null;
    boolean mapreduceOptionPresent = false;
    for (int i = 0; i < cmdArgs.length; ++i) {
      if (cmdArgs[i].equalsIgnoreCase("-src")) {
        srcPath = cmdArgs[++i];
      } else if (cmdArgs[i].equalsIgnoreCase("-dst")) {
        dstPath = cmdArgs[++i];
      } else if (cmdArgs[i].equalsIgnoreCase("-outdir")) {
        outDir = cmdArgs[++i];
      } else if (cmdArgs[i].equalsIgnoreCase("-mapreduce")) {
        boolean mapReduce = Boolean.valueOf(cmdArgs[++i]);
        mapreduceOptionPresent = true;
      } else if (cmdArgs[i].equalsIgnoreCase("-first_exit")) {
        // no arg for this option
      } else if (cmdArgs[i].equalsIgnoreCase("-numthreads")) {
        int numThreads = Integer.parseInt(cmdArgs[++i]);
      } else {
        Usage("Unknown option " + cmdArgs[i]);
      }
    }

    if (srcPath == null || dstPath == null)
      Usage("missing -src or -dst stream");

    if (outDir == null)
      Usage("missing -outdir");

    /* Verify that both src & dst paths represent streams */
    MapRFileSystem mfs = (MapRFileSystem)(FileSystem.get(new Configuration()));
    if (!mfs.isStream(new Path(srcPath))) {
      Usage("src " + srcPath + " is not a stream");
    }

    if (!mfs.isStream(new Path(dstPath))) {
      Usage("dst " + dstPath + " is not a stream");
    }

    int numDiffTableArgs = args.length +
                           (mapreduceOptionPresent ? 0 : 2);
    String [] diffTablesArgs = Arrays.copyOf(args, numDiffTableArgs);
    int argsIdx = args.length;

    if (!mapreduceOptionPresent) {
      // mapreduce is false by default for streams.
      diffTablesArgs[argsIdx++] = "-mapreduce";
      diffTablesArgs[argsIdx++] = "false";
    }

    return ToolRunner.run(getConf(), new DiffTables(), diffTablesArgs);
  }

  public static void main(String[] args) throws Exception {
    int ret = 0;
    try {
      ret = ToolRunner.run(new Configuration(), new DiffStreams(), args);
    } catch (Exception e) {
      ret = 1;
      e.printStackTrace();
    }
    System.exit(ret);
  }
}
