package com.mapr.db.mapreduce.tools.impl;

import static com.mapr.db.impl.Constants.COLUMNSPECCONF;
import static com.mapr.db.mapreduce.TableInputFormat.INPUT_TABLE;

import java.io.IOException;
import java.net.URISyntaxException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.zip.Checksum;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.conf.Configured;
import org.apache.hadoop.fs.CommonConfigurationKeysPublic;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.io.LongWritable;
import org.apache.hadoop.mapreduce.Job;
import org.apache.hadoop.mapreduce.Mapper;
import org.apache.hadoop.mapreduce.Reducer;
import org.apache.hadoop.mapreduce.lib.output.FileOutputFormat;
import org.apache.hadoop.mapreduce.lib.output.LazyOutputFormat;
import org.apache.hadoop.mapreduce.lib.output.SequenceFileOutputFormat;
//import org.apache.hadoop.mapreduce.v2.jobhistory.JHAdminConfig;
import org.apache.hadoop.util.PureJavaCrc32;
import org.apache.hadoop.util.Tool;
import org.apache.hadoop.util.ToolRunner;
import org.apache.hadoop.yarn.conf.YarnConfiguration;
import org.ojai.FieldPath;
import org.ojai.Value;

import com.mapr.db.Admin;
import com.mapr.db.MapRDB;
import com.mapr.db.impl.IdCodec;
import com.mapr.db.impl.MapRDBTableImplHelper;
import com.mapr.db.impl.TableDescriptorImpl;
import com.mapr.db.mapreduce.TableInputFormat;
import com.mapr.db.mapreduce.impl.ByteBufWritableComparable;
import com.mapr.db.mapreduce.impl.ClusterTablePath;
import com.mapr.db.mapreduce.impl.DiffTableUtils;
import com.mapr.db.mapreduce.impl.DocEmptySerialization;
import com.mapr.db.mapreduce.impl.MapReduceUtilMethods;
import com.mapr.db.mapreduce.impl.RangeChecksumInputFormat;
import com.mapr.db.mapreduce.impl.TableSplit;
import com.mapr.db.rowcol.DBDocumentImpl;
import com.mapr.db.rowcol.KeyValueWithTS;
import com.mapr.db.rowcol.RowcolCodec;
import com.mapr.db.rowcol.SerializationAction;
import com.mapr.db.rowcol.SerializedFamilyInfo;
import com.mapr.org.apache.hadoop.hbase.util.Bytes;

public class RangeChecksum extends Configured implements Tool{

  private static final Log LOG = LogFactory.getLog(RangeChecksum.class);
  public final static String NAME = "ComputeChecksum";
  public final static int ISROOT = 1;
  public final static int ISNOTROOT = 0;
  public final static int ISARRAYELEMENT = 1;
  public final static int ISNOTARRAYELEMENT = 0;

  static String jobDisplayName = null;

  static String tableName = null;

  static String inputSplitKeyRangeFileName = null;   // input key range file, this is roughly how user want to divide region
  static boolean needSplit = false;
  static String subKeyRangeFileName = null;     // input sub key range file, each region is a chunk for checksum calculation
  static String includeKeyRangeFileName = null; // the key ranges that should be included in this mapreduce job.
  static String columnSpec = null;
  static boolean excludedEmbeddedFamily;

  static String outputDir = null;
  static boolean exit = false;
  static boolean diffChecksum = false;

  public static class RangeChecksumMapper extends
                                Mapper<Value, DBDocumentImpl, ByteBufWritableComparable, LongWritable> {

    private boolean hasRows_ = false;
    private Checksum checksum_ = new PureJavaCrc32();
    private String keyRangeFileName_ = null;
    private String keyRangeDirName_ = null;
    private List<ByteBufWritableComparable> keysInThisMapper_ = null;
    private int cIdx_ = -1;   //start key index of current segment
    private int nIdx_ = -1;   //start key index of next segment

    private Map<Integer, FieldPath> tableIdToPathMap_ = new HashMap <Integer, FieldPath>();
    private Map<FieldPath, Integer> tablePathToIdMap_ = new HashMap <FieldPath, Integer>();
    private Map<Integer, List<String>> projectionMap_ = null;

    @Override
    public void setup(Context context) throws IOException,
        InterruptedException {
      super.setup(context);

      // get startRow and endRow
      TableSplit tableSplit = (TableSplit)context.getInputSplit();
      final byte[] startRow = tableSplit.getStartRow();
      final byte[] endRow = tableSplit.getStopRow();
      if (LOG.isInfoEnabled()) {
        LOG.info("Mapper Setup: startRow(" + Bytes.toStringBinary(startRow) + ") endRow(" + Bytes.toStringBinary(endRow) + ")");
      }
      // read segment keys file and directory name from configuration
      Configuration conf = context.getConfiguration();
      keyRangeDirName_ = conf.get("KeyRangeDirName", null);

      //First segment, file name is "neginf"
      if (startRow.length == 0) {
        keyRangeFileName_ = DiffTableUtils.FIRSTKEYNAME;
      } else {
        keyRangeFileName_ = DiffTableUtils.KEYPREFIX + Bytes.toStringBinary(startRow);
      }
      LOG.info("keyRangeDirName("+keyRangeDirName_+")\n DiffTableUtils.KEYPREFIX("+DiffTableUtils.KEYPREFIX+")\n "
          +"startRow("+Bytes.toStringBinary(startRow)+")\n keyRangeFileName_("+keyRangeFileName_+")");
      keysInThisMapper_ = DiffTableUtils.readKeyRange(conf, keyRangeDirName_ + "/" + keyRangeFileName_);
      if (keysInThisMapper_.size() == 0) {
        throw new IOException(
                "No segment keys found for tablet with start at row (" + Bytes.toStringBinary(startRow) + ")");
      }
      //First segment start key must match with the tablet start key.
      if (!Bytes.equals(keysInThisMapper_.get(0).getBytes(), startRow)) {
        throw new IOException(
                "First segment key(" + keysInThisMapper_.get(0)
                + ") is NOT the tablet start key(" + Bytes.toStringBinary(startRow) + ")");
      }
      cIdx_ = 0;
      if (keysInThisMapper_.size() > 1) {
        nIdx_ = 1;
      }

      String tName = conf.get(INPUT_TABLE, null);
      if (tName == null) {
        throw new IOException("Input table cannot be null!");
      }

      tableIdToPathMap_.clear();
      tablePathToIdMap_.clear();
      DiffTableComparator.getIdToFieldPathMap(tName, tableIdToPathMap_, tablePathToIdMap_);

      String cols = conf.get(COLUMNSPECCONF);
      if (cols != null) {
        projectionMap_ = MapRDBTableImplHelper.getMultipleCFQualifiers(tablePathToIdMap_, excludedEmbeddedFamily, cols.split(","));
      }

      LOG.info("Checksum table " + tName +"has id-to-path map " + tableIdToPathMap_);
      checksum_.reset();
    }

    public void updateChecksum(ByteBufWritableComparable row, DBDocumentImpl docValue) {

      ByteBuffer rowKey = IdCodec.encode(docValue.getId());
      LOG.info("row from mapper("+Bytes.toStringBinary(row.getBytes())+"), rowkey(id) from doc("+Bytes.toStringBinary(rowKey.array())+")");
      checksum_.update(rowKey.array(), 0, rowKey.limit());  //use limit since the ByteBufWritableComparable use limit for getLength.

      SerializedFamilyInfo[] docData =
          RowcolCodec.encode(docValue, tablePathToIdMap_, false /*isBulkLoad*/, true /*useEncoded*/);

      // Sort based on the column family names so that all matching ones come together
      Arrays.sort(docData, new DBDocComparator(tableIdToPathMap_));

      int index = 0;
      while (index < docData.length) {

        SerializedFamilyInfo i = null;
        if (index < docData.length) {
          i = docData[index];
          if (projectionMap_ != null &&
            !projectionMap_.containsKey(i.getFamilyId())) {
          index++;
          continue;
          }
        }

        // There is additional family in one table and its with valid data
        if (i == null || i.getAction() == SerializationAction.NO_ACTION) {
          // Do not update checksum if i does not exist or "do NOT do anything flag is on"
          index++;
          continue;
        }

        DBDocumentImpl familyDoc = (DBDocumentImpl) RowcolCodec.decode(i.getByteBuffer(),
                null, true /*excludeId*/, true /*decodeTimestamp*/, true /*preserveDeleteFlags*/);
        //Do not update checksum if no family doc
        if (familyDoc == null ) {
          index++;
          continue;
        }

        // after the familyName, continue to process the smaller family
        FieldPath family = tableIdToPathMap_.get(i.getFamilyId());
        String familyStr = family.asPathString();
        checksum_.update(familyStr.getBytes(), 0, familyStr.length());

        //No projection or entire family is part of projection
        if (projectionMap_ == null || projectionMap_.get(i.getFamilyId()).isEmpty()) {
          KeyValueWithTS.updateChecksumKeyValue(checksum_, familyDoc, true /*matchTimestamp*/, null /*parentDelTime*/);
        } else {
          List<String> projFPList = projectionMap_.get(i.getFamilyId());
          for (String fp : projFPList) {
            KeyValueWithTS.updateChecksumKeyValue(checksum_, familyDoc.getKeyValue(fp), true /*matchTimestamp*/, null /*parentDelTime*/);
          }
        }
        index++;
      }
      LOG.debug("row: " + row + ", update checksum to: " + checksum_.getValue());
    }

    @Override
    public void map(Value rowValue, DBDocumentImpl docValue, Context context)
        throws IOException, InterruptedException {

      if (!hasRows_) {
        hasRows_ = true;
      }
      ByteBufWritableComparable rowkey = new ByteBufWritableComparable(IdCodec.encode(rowValue));
      //Make sure rowkey is larger that the start key of current segment.
      if (keysInThisMapper_.get(cIdx_).compareTo(rowkey) > 0) {
        throw new IOException(
                "rowkey(" + rowkey + ") is larger that its segment start key("
                + keysInThisMapper_.get(cIdx_) + ")");

      } else if ((nIdx_ != -1) && (keysInThisMapper_.get(nIdx_).compareTo(rowkey) <= 0)) {

        //This rowkey is bigger than this segment.
        //Output this segment's checksum
        context.write(keysInThisMapper_.get(cIdx_), new LongWritable(checksum_.getValue()));
        if (LOG.isInfoEnabled()) {
          LOG.info("Segment with start key: " + keysInThisMapper_.get(cIdx_)
                  + ", checksum: " + checksum_.getValue());
        }
        checksum_.reset();

        //Move to next segment
        ++cIdx_;
        if (cIdx_ >= keysInThisMapper_.size()) {
          throw new IOException(
                    "No more segments to cover rowkey(" + rowkey + ")");
        }
        nIdx_ = cIdx_+1;
        if (nIdx_ == keysInThisMapper_.size()) {
          nIdx_ = -1;  // last segment
        }

        //Add this rowkey to the checksum
        updateChecksum(rowkey, docValue);
      } else {
        //Add this rowkey to the checksum
        updateChecksum(rowkey, docValue);
      }
    }

    @Override
    protected void cleanup(Context context) throws IOException,
        InterruptedException {

      if (!hasRows_) {
        LOG.info("This Split does not contain any row, will log default checksum for empty segment");
      }

      //Write the checksum of last segment
      context.write(keysInThisMapper_.get(cIdx_), new LongWritable(checksum_.getValue()));
      if (LOG.isInfoEnabled()) {
        LOG.info("Segment with start key: " + keysInThisMapper_.get(cIdx_)
                + ", checksum: " + checksum_.getValue());
      }
      cIdx_ = -1;
      nIdx_ = -1;
      checksum_.reset();
      tableIdToPathMap_.clear();
      tablePathToIdMap_.clear();
    }
  }

  public static class RangeCheckSumReducer
        extends Reducer<ByteBufWritableComparable, LongWritable, ByteBufWritableComparable, LongWritable>  {
    @Override
    public void reduce(ByteBufWritableComparable key, Iterable<LongWritable> values, Context context)
        throws IOException, InterruptedException {

      int i = 0;
      for (LongWritable val : values) {
        if (i == 0) {
          context.write(key, val);
          if (LOG.isInfoEnabled()) {
            LOG.info("checksum for start key(" + key + "):(" + Long.toString(val.get()) + ")");
          }
        } else {
          //Mapper should only produce one checksum for one key-range. If we find an extra checksum
          //for a key-range, something is wrong.
          LOG.error("extra " + Integer.toString(i) + " checksum for start key(" + key  + "):(" + Long.toString(val.get()) + ")");
        }
        ++i;
      }
    }
  }

  public static String getOutputPathName(String dir, String table) {
    return new String(dir + "/" + table.replace("/", "").replace(":", "") + ".checksum");
  }


  public static Job createSubmittableJob(Configuration conf, String[] args)
      throws IOException, URISyntaxException {
    if (!doCommandLine(args)) {
      return null;
    }
    conf.set(INPUT_TABLE, tableName);

    ClusterTablePath ctPath = ClusterTablePath.parse(tableName);

    //submit to the correct cluster
    if (ctPath.clusterName != null) {

      //set "fs.defaultFS" to clusterName
      conf.set(CommonConfigurationKeysPublic.FS_DEFAULT_NAME_KEY, ctPath.getClusterUri().toString());

      String hsAddress = DiffTableUtils.getServiceAddress(conf, ctPath.getClusterUri().toString(), DiffTableUtils.ServiceName.HISTORYSERVERNAME);
      String rmAddress = DiffTableUtils.getServiceAddress(conf, ctPath.getClusterUri().toString(), DiffTableUtils.ServiceName.RESOURCEMANAGERNAME);

      if (rmAddress != null) {
        //set "yarn.resourcemanager.address" to rmAddress
        conf.set(YarnConfiguration.RM_ADDRESS, rmAddress);
      }
      if (hsAddress != null) {
        // set "mapreduce.jobhistory.address" to hsAddress
        //conf.set(JHAdminConfig.MR_HISTORY_ADDRESS, hsAddress);
        conf.set("mapreduce.jobhistory.address", hsAddress);
      }
      LOG.info("Job for table(" + tableName + ") will be submitted to cluster(" + ctPath.clusterName +")");
    } else {
      LOG.info("Job for table(" + tableName + ") will be submitted to local cluster.");
    }

    if (inputSplitKeyRangeFileName == null) {
      inputSplitKeyRangeFileName = outputDir + "/" + DiffTableUtils.getTabletKeyFileName(tableName);
      DiffTableUtils.writeTabletKeyRange(inputSplitKeyRangeFileName, tableName);
    }

    String segKeyDirName =  outputDir + "/" + DiffTableUtils.getSegKeyDirName(tableName);

    if (subKeyRangeFileName == null) {
      subKeyRangeFileName = outputDir + "/" + DiffTableUtils.getSegKeyFileName(tableName);
      DiffTableUtils.writeSegKeyRange(subKeyRangeFileName, tableName);
    }

    String outputSplitKeyRangeFileName = null;  // output key range file, each region is adjusted to line up with input sub key ranges
    if (needSplit) {
      outputSplitKeyRangeFileName = outputDir + "/" + DiffTableUtils.getReSplitedTabletKeyFileName(tableName);
      DiffTableUtils.splitSubRegionKeysbyRegionKeys(inputSplitKeyRangeFileName,
          subKeyRangeFileName, outputSplitKeyRangeFileName, segKeyDirName);
    } else {
      outputSplitKeyRangeFileName = inputSplitKeyRangeFileName;
    }

    // Set the conf for splits
    conf.set("KeyRangeDirName", segKeyDirName);
    conf.set(RangeChecksumInputFormat.SPLITFILENAME, outputSplitKeyRangeFileName);

    if (includeKeyRangeFileName != null) {
      conf.set(RangeChecksumInputFormat.INCLUDEDREGIONFILENAME, includeKeyRangeFileName);
    }

    //Set the conf for json table
    conf.setBoolean(TableInputFormat.EXCLUDE_EMBEDDEDFAMILY, excludedEmbeddedFamily);
    if (columnSpec != null) {
      conf.set(COLUMNSPECCONF, columnSpec);
      /* projection need to be applied in record reader */
      conf.set(TableInputFormat.FIELD_PATH, columnSpec);
    }
    conf.setStrings("io.serializations", conf.get("io.serializations"),
        DocEmptySerialization.class.getName());


    Job job = Job.getInstance(conf, NAME);
    if (jobDisplayName != null) {
      job.setJobName(jobDisplayName);
    } else {
      job.setJobName(NAME);
    }
    job.setJarByClass(RangeChecksum.class);

    job.setInputFormatClass(RangeChecksumInputFormat.class);
    job.setMapperClass(RangeChecksumMapper.class);
    job.setOutputKeyClass(ByteBufWritableComparable.class);
    job.setOutputValueClass(LongWritable.class);
    FileOutputFormat.setOutputPath(job, new Path(getOutputPathName(outputDir, tableName)));
    LazyOutputFormat.setOutputFormatClass(job, SequenceFileOutputFormat.class);
    job.setSpeculativeExecution(false);


    job.setReducerClass(RangeCheckSumReducer.class);    // reducer class
    job.setNumReduceTasks(1);

    return job;
  }

  public static String[] convertFromDiffTablesArg(boolean useSrcTable,
          final String splitKRFile,     // region split file
          final String subKRFile,       // sub region to calculate checksum
          final String includeKRFile,
          final String outChecksumDir,
          final String inputJobDisplayName,
          final String[] diffArgs) {

    if (diffArgs.length == 0 && splitKRFile == null && subKRFile == null && includeKRFile == null) {
      return new String[0];
    }
    if (subKRFile != null && splitKRFile == null) {
      throw new IllegalArgumentException("Input sub key range file of checksum calculation is missing.");
    }
    if (includeKRFile != null && splitKRFile == null) {
      throw new IllegalArgumentException("Input key range file of must-be-included regions is missing.");
    }

    boolean hasOutdir = false;
    ArrayList<String> rangeArgs = new ArrayList<String>();
    for (int i = 0; i < diffArgs.length; ++i) {
      String cmd = diffArgs[i];
      if (cmd.equalsIgnoreCase("-columns") ||
          cmd.equalsIgnoreCase("-exclude_embedded_families") ) {
        if (!checkNextArg(diffArgs, i)) {
            break;
        }
        rangeArgs.add(cmd);
        rangeArgs.add(diffArgs[i + 1]);
        continue;
      } else if (cmd.equalsIgnoreCase("-src") && useSrcTable) {
        if (!checkNextArg(diffArgs, i)) {
              break;
        }
        rangeArgs.add("-table");
        rangeArgs.add(diffArgs[i + 1]);
        continue;
      } else if (cmd.equalsIgnoreCase("-dst") && !useSrcTable) {
        if (!checkNextArg(diffArgs, i)) {
              break;
        }
        rangeArgs.add("-table");
        rangeArgs.add(diffArgs[i + 1]);
        continue;
      } else if (cmd.equalsIgnoreCase("-outdir")) {
        if (!checkNextArg(diffArgs, i)) {
              break;
        }
        if (outChecksumDir != null) {
          rangeArgs.add(cmd);
          rangeArgs.add(outChecksumDir);
          hasOutdir = true;
          continue;
        } else {
          throw new IllegalArgumentException("Output directory " + diffArgs[i + 1] +
                                             " but not set in DiffTableWithCrc but not set in RangeChecksum");
        }
      }
    }
    if (splitKRFile != null) {
      rangeArgs.add("-split_keyrange");
      rangeArgs.add(splitKRFile);
    }
    if (includeKRFile != null) {
      rangeArgs.add("-keyrange_included");
      rangeArgs.add(includeKRFile);
    }
    if (subKRFile != null) {
      rangeArgs.add("-sub_keyrange");
      rangeArgs.add(subKRFile);
    }
    if (inputJobDisplayName != null) {
      rangeArgs.add("-job_name");
      rangeArgs.add(inputJobDisplayName);
    }
    //Outdir has not been set.
    if (outChecksumDir != null && !hasOutdir) {
        rangeArgs.add("-outdir");
        rangeArgs.add(outChecksumDir);
    }
    String[] retArr = new String[rangeArgs.size()];
    retArr = rangeArgs.toArray(retArr);
    if (LOG.isInfoEnabled()) {
      LOG.info(rangeArgs.toString());
    }
    return retArr;
  }

  private static boolean doCommandLine(final String[] args) {
    if (args.length == 0) {
      return true;
    } else if (args.length < 2) {
      printUsage(null);
      return false;
    }
    try {
      for (int i = 0; i < args.length; i++) {
        String cmd = args[i];
        if (cmd.equalsIgnoreCase("-h") || cmd.startsWith("--h")) {
          printUsage(null);
          return false;
        } else if (cmd.equalsIgnoreCase("-table")) {
          if (!checkNextArg(args, i))
            return false;
          tableName = args[i + 1];
          continue;
        } else if (cmd.equalsIgnoreCase("-split_keyrange")) {
          if (!checkNextArg(args, i))
            return false;
          inputSplitKeyRangeFileName = args[i + 1];
          continue;
        } else if (cmd.equalsIgnoreCase("-keyrange_included")) {
          if (!checkNextArg(args, i))
            return false;
          includeKeyRangeFileName = args[i + 1];
          continue;
        } else if (cmd.equalsIgnoreCase("-sub_keyrange")) {
          if (!checkNextArg(args, i))
            return false;
          needSplit = true;
          subKeyRangeFileName = args[i + 1];
          continue;
        } else if (args[i].equalsIgnoreCase("-columns")) {
          if (!checkNextArg(args, i))
            return false;
          columnSpec = args[i+1];
          continue;
        } else if (cmd.equalsIgnoreCase("-exclude_embedded_families")){
          if (!checkNextArg(args, i))
            return false;
          excludedEmbeddedFamily = Boolean.valueOf(args[i + 1]);
          continue;
        }  else if (cmd.equalsIgnoreCase("-outdir")) {
          if (!checkNextArg(args, i))
            return false;
          outputDir = args[i + 1];
          continue;
        } else if (cmd.equalsIgnoreCase("-job_name")) {
          if (!checkNextArg(args, i))
            return false;
          jobDisplayName = args[i + 1];
          continue;
        }
      }

      if(tableName == null){
        printUsage(null);
        return false;
      }
      if(includeKeyRangeFileName != null && inputSplitKeyRangeFileName == null){
          printUsage("inputSplitKeyRangeFileName is provided, but includeKeyRangeFileName is missing.");
          return false;
      }

      columnSpec = MapReduceUtilMethods.processColumnSpec(columnSpec, tableName);
      LOG.info("Compute checksum for " + (columnSpec != null ? columnSpec : "all")
          + "column families from " + tableName);

      Admin admin = MapRDB.newAdmin();
      TableDescriptorImpl desc = (TableDescriptorImpl) admin.getTableDescriptor(tableName);
      if (desc.isStream()) {
        excludedEmbeddedFamily = true;
      }

    } catch (Exception e) {
      e.printStackTrace();
      printUsage("Can't start because " + e.getMessage());
      return false;
    }

    return true;
  }

  private static boolean checkNextArg(String[] args, int i) {
    if (i + 1 >= args.length) {
      printUsage("Missing arguements after " + args[i]);
      return false;
    }
    return true;
  }

  private static void printUsage(final String errorMsg) {

    if (errorMsg != null && errorMsg.length() > 0) {
      System.err.println("ERROR: " + errorMsg);
    }
    System.err.println("Usage: hadoop com.mapr.db.mapreduce.tools.impl.RangeChecksum -table tableName [options].\n"
                     + "Notes: the output directory will contain the calculated checksum file.\n"
                     + "Options:\n"
                     + "  [-outdir <output_checksum_file_directory>]  directory for checksum output file.\n"
                     + "  [-split_keyrange <split_keyrange_file>] The key ranges that mapper task should be divided.\n"
                     + "     If this is not provided, the program will use mapr tablets as the key ranges.\n"
                     + "  [-keyrange_included <keyrange_included_file>] The key ranges that mapper task must run.\n"
                     + "     The key ranges must be a subset of the key ranges in split_keyrange file.\n"
                     + "  [-columns <comma separated list of field paths> ]\n"
                     + "  [-exclude_embedded_families <true|false>] (default: false)\n"
                     +    "  Don't include the  other column families with path embedded in specified columns.");
  }

  @Override
  public int run(String[] args) throws Exception {
    Configuration conf = this.getConf();
    Job job = createSubmittableJob(conf, args);
    int ret = 0;
    if (job != null) {
      ret = job.waitForCompletion(true) ? 0 : 1;
    }
    return ret;
  }

  public static void main(String[] args) throws Exception {
    Configuration conf = new Configuration();
    int ret = 0;
    try {
      ret = ToolRunner.run(conf, new RangeChecksum(), args);
    } catch (Exception e) {
      ret = 1;
      e.printStackTrace();
    }
    System.exit(ret);
  }

}
